/*******************************************************************************
* Copyright (C) 2021 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
!  Content:
!    Construction of natural cubic spline and interpolation for
!    the vector-valued function saved in the non-consecutive memory
!******************************************************************************/

#include <stdio.h>

#include "mkl.h"
#include "errcheck.inc"
#include "generatedata.inc"
#include "rescheck.inc"


#define N            10 // number of break points
#define NY           3 // number of functions

#define NSCOEFF      (N - 1) * DF_PP_CUBIC    // total number of spline
                                              // coefficients
#define NSITE        15   // total number of interpolation sites
#define LLIM_X      -10.0 // left limit of interpolation interval
#define RLIM_X       10.0 // right limit of interpolation interval


int main(void)
{
    DFTaskPtr task;                     // Data Fitting task descriptor
    MKL_INT sorder;                     // spline order
    MKL_INT stype;                      // spline type
    MKL_INT nx;                         // number of break points
    MKL_INT xhint;                      // additional info about break points
    MKL_INT ny;                         // number of functions
    MKL_INT yhint;                      // additional info about function
    MKL_INT scoeffhint;                 // additional info about spline
                                        // coefficients
    MKL_INT bc_type;                    // boundary conditions type
    MKL_INT ic_type;                    // internal conditions type
    MKL_INT nsite;                      // total number of interpolation sites
    MKL_INT sitehint;                   // additional info about interpolation
                                        // sites
    MKL_INT rhint;                      // interpolation results storage format
    double left = LLIM_X;               // left limit of the interpolation
                                        // interval
    double right = RLIM_X;              // right limit of the interpolation
                                        // interval
    double x[N];                        // array of break points
    double* y[NY];                      // function values
    double *ic;                         // internal conditions
    double *bc;                         // boundary conditions
    double* scoeff[NY];                 // array of spline coefficients
    double site[NSITE];                 // array of interpolation sites
    double r[NY * NSITE];               // spline evaluation results
    double ref_r[NY][NSITE];            // reference interpolation results
    double result[NY][NSITE];           // array of interpolation sites for checking
                                        // interpolation results
    double freq;

    int i, j, idx, errcode = 0;

    /***** Initializing parameters for Data Fitting task *****/
    sorder     = DF_PP_CUBIC;
    stype      = DF_PP_NATURAL;

    /***** Parameters describing interpolation interval *****/
    nx         = N;
    xhint      = DF_NO_HINT;

    /***** Parameters describing function *****/
    ny         = NY;
    yhint      = DF_1ST_COORDINATE;

    /***** Function memory allocation *****/
    for ( i = 0; i < ny; ++i )
        y[i] = malloc(sizeof(double) * nx);

    /***** Parameters describing spline coefficients storage *****/
    scoeffhint = DF_1ST_COORDINATE;

    /***** Spline coefficients memory allocation *****/
    for ( i = 0; i < ny; ++i )
        scoeff[i] = malloc(sizeof(double) * NSCOEFF);

    /***** Parameters describing boundary conditions type *****/
    bc_type    = DF_BC_FREE_END;
    bc         = 0;

    /***** Parameters describing internal conditions type *****/
    ic_type    = DF_NO_IC;
    ic         = 0;

    /***** Parameters describing interpolation sites *****/
    nsite      = NSITE;
    sitehint   = DF_SORTED_DATA;

    /**** Parameter describing interpolation results storage *****/
    rhint      = DF_MATRIX_STORAGE_COLS;

    /***** Generate array of uniformly distributed break points *****/
    errcode = dUniformRandSortedData(x, left, right, (int)nx);
    CheckDfError(errcode);

    /***** Generate function y = sin(2 * Pi * freq * x) *****/
    for( i = 0; i < ny; ++i ) {
        freq = (i + 1) * 1.3;
        errcode = dSinDataNotUniformGrid(y[i], x, freq, (int)nx);
        CheckDfError(errcode);
    }

    /***** Generate interpolation sites *****/
    errcode = dUniformRandSortedData( site, left, right, (int)nsite );
    CheckDfError(errcode);

    /***** Create new task and configure *****/
    errcode = dfdNewTask1D(&task, nx, x, xhint, ny, y[0], yhint);
    CheckDfError(errcode);

    /***** Setting functions *****/
    for( idx = 1; idx < ny; idx++ ) {
        errcode = dfdEditIdxPtr(task, DF_Y, idx, y[idx]);
        CheckDfError(errcode);
    }

    /***** Edit task parameters for natural cubic spline construction *****/
    errcode = dfdEditPPSpline1D( task, sorder, stype, bc_type, bc,
        ic_type, ic, scoeff[0], scoeffhint );
    CheckDfError(errcode);

    /***** Setting coefficients *****/
    for( idx = 1; idx < ny; idx++ ) {
        errcode = dfdEditIdxPtr(task, DF_PP_SCOEFF, idx, scoeff[idx]);
        CheckDfError(errcode);
    }

    /***** Construct natural cubic spline using STD method *****/
    errcode = dfdConstruct1D( task, DF_PP_SPLINE, DF_METHOD_STD );
    CheckDfError(errcode);

    /***** Interpolate using PP method *****/
    errcode = dfdInterpolate1D( task, DF_INTERP, DF_METHOD_PP,
                                    nsite, site, sitehint, 0,
                                    0, 0, r, rhint, 0 );
    CheckDfError(errcode);

    /***** Delete Data Fitting task *****/
    errcode = dfDeleteTask( &task );
    CheckDfError(errcode);

    /***** Check results of interpolation *****/
    for ( i = 0; i < ny; ++i )
        for ( j = 0; j < nsite; ++j )
            result[i][j] = r[i * nsite + j];

    for( i = 0; i < ny; ++i ) {
        errcode = dCheckCubInterpRes( nx, x, 1, scoeff[i], nsite, site, 0,
                                    0, result[i], ref_r[i] );
        CheckDfError(errcode);
    }

    /***** Print results *****/
    printf("Number of break points : %d\n", (int)nx);

    /***** Print given function *****/
    printf("\n i   x(i)");
    for( i = 0; i < ny; i++ )
    {
        printf("        y%d(i)",i);
    }
    printf("\n");

    for( i = 0; i < nx; i++ )
    {
        printf(" %1d %+10lf", i, x[i]);
        for( j = 0; j < ny; j++ )
        {
            printf("   %+10lf", y[j][i]);
        }
        printf("\n");
    }

    /***** Print computed spline coefficients *****/
    printf("\nCoefficients are calculated for a polynomial of the form:\n\n");
    printf("Pi(x) = Ai + Bi*(x - x(i)) + Ci*(x - x(i))^2 + Di*(x - x(i))^3\n");
    printf("    where x(i) <= x < x(i+1)\n");
    printf("\nSpline coefficients for Y:\n");
    printf(" i      Ai              Bi              Ci              Di      \n");
    for( i = 0; i < ny; i++ )
    {
        printf("   for function y%d:\n",i);
        for( j = 0; j < nx - 1; j++ )
        {
            printf(" %1d %+13.6f   %+13.6f   %+13.6f   %+13.6f\n",
                    j,
                    scoeff[i][j * sorder + 0],
                    scoeff[i][j * sorder + 1],
                    scoeff[i][j * sorder + 2],
                    scoeff[i][j * sorder + 3]);
        }
    }

    /***** Print interpolation results ******/
    printf("\n  i    Sites       Spline value\n");
    printf("                   Computed    Expected\n");
    for( i = 0; i < ny; i++ )
    {
        printf("   for function y%d:\n",i);
        for ( j = 0; j < nsite; j++ )
        {
            printf(" %2d %+11.6lf %+11.6lf %+11.6lf\n", j,
                site[j], r[i * nsite + j], ref_r[i][j]);
        }
    }

    return 0;
}
