/*
 * Copyright (C) 2005 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

#include <sys/malloc.h>

#include "lwpmudrv_defines.h"
#include "lwpmudrv_types.h"

#include "rise_errors.h"
#include "lwpmudrv_ecb.h"
#include "lwpmudrv.h"
#include "inc/control.h"
#if defined(DRV_IA32) || defined(DRV_EM64T)
#include "core2.h"
#include "silvermont.h"
#include "perfver4.h"
#include "unc_gt.h"
#include "haswellunc_sa.h"
#include "valleyview_sochap.h"
#endif
#include "utility.h"

volatile int config_done;
extern  DISPATCH_NODE   unc_msr_dispatch;
extern  DISPATCH_NODE   unc_pci_dispatch;
extern  DISPATCH_NODE   unc_mmio_single_bar_dispatch;
extern  DISPATCH_NODE   unc_mmio_multiple_bar_dispatch;
extern  DISPATCH_NODE   unc_mmio_pmm_dispatch;
extern  DISPATCH_NODE   unc_power_dispatch;
extern  DISPATCH_NODE   unc_rdt_dispatch;
extern  U32             drv_type;

extern VOID
UTILITY_Read_TSC (
    U64* pTsc
)
{
    *(pTsc) = rdtsc();

    return;
}

#if defined(DRV_IA32) || defined(DRV_EM64T)
/* ------------------------------------------------------------------------- */
/*!
 * @fn       VOID UTILITY_Read_Cpuid
 *
 * @brief    executes the cpuid_function of cpuid and returns values
 *
 * @param  IN   cpuid_function
 *         OUT  rax  - results of the cpuid instruction in the
 *         OUT  rbx  - corresponding registers
 *         OUT  rcx
 *         OUT  rdx
 *
 * @return   none
 *
 * <I>Special Notes:</I>
 *              <NONE>
 *
 */
extern VOID
UTILITY_Read_Cpuid (
    U64   cpuid_function,
    U64  *rax_value,
    U64  *rbx_value,
    U64  *rcx_value,
    U64  *rdx_value
)
{
    U32 regs[4];

    cpuid_count((U32)cpuid_function, (U32)*rcx_value, regs);

    *rax_value = regs[0];
    *rbx_value = regs[1];
    *rcx_value = regs[2];
    *rdx_value = regs[3];
}
#endif

/* ------------------------------------------------------------------------- */
/*!
 * @fn       VOID UTILITY_Configure_CPU
 *
 * @brief    Reads the CPU information from the hardware
 *
 * @param    param   dispatch_id -  The id of the dispatch table.
 *
 * @return   Pointer to the correct dispatch table for the CPU architecture
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
extern  DISPATCH
UTILITY_Configure_CPU (
    U32 dispatch_id
)
{
    DISPATCH     dispatch = NULL;

    SEP_DRV_LOG_TRACE_IN("Dispatch_id: %u.", dispatch_id);

    switch (dispatch_id) {

#if defined(DRV_IA32) || defined(DRV_EM64T)
        case 2:
            dispatch = &corei7_dispatch;
            SEP_DRV_LOG_INIT("Set up the Core i7(TM) processor dispatch table.");
            break;
        case 3:
            SEP_DRV_LOG_INIT("Set up the Core i7(TM) dispatch table.");
            dispatch = &corei7_dispatch_htoff_mode;
            break;
        case 6:
            SEP_DRV_LOG_INIT("Set up the Silvermont dispatch table.");
            dispatch = &silvermont_dispatch;
            break;
        case 7:
            SEP_DRV_LOG_INIT("Set up the perfver4 HTON dispatch table such as Skylake.");
            dispatch = &perfver4_dispatch;
            break;
        case 8:
            SEP_DRV_LOG_INIT("Set up the perfver4 HTOFF dispatch table such as Skylake.");
            dispatch = &perfver4_dispatch_htoff_mode;
            break;
        case 11:
            dispatch = &perfver4_dispatch_nonht_mode;
            SEP_DRV_LOG_INIT("Set up the perfver4 NONHT dispatch table such as Icelake.");
            break;
        case 100:
            SEP_DRV_LOG_INIT("Set up the MSR based uncore dispatch table.");
            dispatch = &unc_msr_dispatch;
            break;
        case 110:
            SEP_DRV_LOG_INIT("Set up the PCI Based Uncore dispatch table.");
            dispatch = &unc_pci_dispatch;
            break;
        case 120:
            SEP_DRV_LOG_INIT("Set up the MMIO based single bar uncore dispatch table.");
            dispatch = &unc_mmio_single_bar_dispatch;
            break;
        case 122:
            SEP_DRV_LOG_INIT("Set up the MMIO based multiple bar uncore dispatch table.");
            dispatch = &unc_mmio_multiple_bar_dispatch;
            break;
        case 123:
            SEP_DRV_LOG_INIT("Set up the MMIO based uncore dispatch table for PMM.");
            dispatch = &unc_mmio_pmm_dispatch;
            break;
        case 130:
            SEP_DRV_LOG_INIT("Set up the Uncore Power dispatch table.");
            dispatch = &unc_power_dispatch;
            break;
        case 131:
            SEP_DRV_LOG_INIT("Set up the RDT dispatch table.");
            dispatch = &unc_rdt_dispatch;
            break;
        case 230:
            SEP_DRV_LOG_INIT("Set up the Haswell SA dispatch table.");
            dispatch = &hswunc_sa_dispatch;
            break;
        case 700:
            SEP_DRV_LOG_INIT("Set up the Valleyview SA dispatch table.");
            dispatch = &valleyview_visa_dispatch;
            break;
#endif
        default:
            dispatch = NULL;
            SEP_DRV_LOG_ERROR("Architecture not supported (dispatch_id=%d).", dispatch_id);
            break;
    }

    SEP_DRV_LOG_TRACE_OUT("Res: %p.", dispatch);
    return dispatch;
}

#if defined(DRV_IA32) || defined(DRV_EM64T)
/* ------------------------------------------------------------------------- */
/*!
 * @fn          U64 SYS_Read_PMC_opt (IN int ctr_addr, IN U32 is_fixed_reg)
 * @brief       Wrapper function of read perfmon counters
 *
 * @param       ctr_addr - counter address
 *              is_fixed_reg - flag to indicate if counter is fixed or GP
 *
 * @return      Counter value
 *
 * <I>Special Notes:</I>
 *      Counter relative index from base is specified in bits [29:0]
 *      If fixed register is requested, bit 30 of input operand must be additionally set
 *
 */
extern U64
SYS_Read_PMC_opt (
    U32 ctr_addr,
    U32 is_fixed_ctr
)
{
    U64 val       = 0;
    int counter   = 0;

    if (is_fixed_ctr) {
        counter  = (1 << RDPMC_COUNTER_TYPE_BIT_SHIFT);
        counter |= ctr_addr - IA32_FIXED_CTR0;
    }
    else {
        counter |= ctr_addr - IA32_PMC0;
    }
    SEP_DRV_LOG_REGISTER_IN("Will read counter 0x%x, ctr_index %d.", ctr_addr, counter);
    val = rdpmc(counter);
    SEP_DRV_LOG_REGISTER_OUT("Has read counter 0x%x: %llu.", ctr_addr, val);

    return val;
}


extern U64
SYS_Read_MSR (
    U32   msr
)
{
    U64 val;

    SEP_DRV_LOG_REGISTER_IN("Will read MSR 0x%x.", msr);

    if (!msr) {
        SEP_DRV_LOG_WARNING("Ignoring MSR address is 0.");
        return 0ULL;
    }

    val = rdmsr(msr);
    SEP_DRV_LOG_REGISTER_OUT("Has read MSR 0x%x: 0x%llx.", msr, val);

    return val;
}

extern void
SYS_Write_MSR (
    U32   msr,
    U64   val
)
{
    SEP_DRV_LOG_REGISTER_IN("Will write MSR 0x%x: 0x%llx.", msr, val);

    if (!msr) {
        SEP_DRV_LOG_WARNING("Ignoring MSR address is 0.");
        return;
    }

    wrmsr(msr, val);
    SEP_DRV_LOG_REGISTER_OUT("Wrote MSR 0x%x: 0x%llx.", msr, val);

    return;
}


#endif


/*
 ************************************
 *  DRIVER LOG BUFFER DECLARATIONS  *
 ************************************
 */

volatile U8         active_ioctl;

DRV_LOG_BUFFER      driver_log_buffer;

static const char* drv_log_categories[DRV_NB_LOG_CATEGORIES] = {
    "load",
    "init",
    "detection",
    "error",
    "state change",
    "mark",
    "debug",
    "flow",
    "alloc",
    "interrupt",
    "trace",
    "register",
    "notification",
    "warning"
};

#define DRV_LOG_NB_DRIVER_STATES 9
static const char* drv_log_states[DRV_LOG_NB_DRIVER_STATES] = {
    "Uninitialized",
    "Reserved",
    "Idle",
    "Paused",
    "Stopped",
    "Running",
    "Pausing",
    "Prepare_Stop",
    "Terminating"
};


/* ------------------------------------------------------------------------- */
/*!
 * @fn       static VOID utility_Driver_Log_Kprint_Helper (U8 category, const char**  category_string,
 *                                                  U8 secondary, const char** secondary_string_1,
 *                                                  const char**  secondary_string_2, const char**  secondary_string_3,
 *                                                  const char**  secondary_string_4)
 *
 * @brief    Helper function for printing log messages to the system log.
 *
 * @param    IN     category            -  message category
 *           IN/OUT category_string     -  location where to place a pointer to the category's name
 *           IN     secondary           -  secondary field value for the message
 *           IN/OUT secondary_string_1  -  location where to place a pointer to the 1st part of the secondary info's decoded information
 *           IN/OUT secondary_string_2  -  location where to place a pointer to the 2nd part of the secondary info's decoded information
 *           IN/OUT secondary_string_3  -  location where to place a pointer to the 3rd part of the secondary info's decoded information
 *           IN/OUT secondary_string_4  -  location where to place a pointer to the 4th part of the secondary info's decoded information
 *
 * @return   none
 *
 * <I>Special Notes:</I>
 *           Allows a single format string to be used for all categories (instead of category-specific format
 *           strings) when calling printk, simplifying the print routine and reducing potential errors.
 *           There is a performance cost to this approach (forcing printk to process empty strings), but it
 *           should be dwarved by the cost of calling printk in the first place.
 *           NB: none of the input string pointers may be NULL!
 */
static VOID
utility_Driver_Log_Kprint_Helper (
    U8            category,
    const char**  category_string,
    U8            secondary,
    const char**  secondary_string_1,
    const char**  secondary_string_2,
    const char**  secondary_string_3,
    const char**  secondary_string_4
)
{
    if (category >= DRV_NB_LOG_CATEGORIES) {
        *category_string = "Unknown category";
    }
    else {
        *category_string = drv_log_categories[category];
    }

    *secondary_string_1  = "";
    *secondary_string_2  = "";
    *secondary_string_3  = "";
    *secondary_string_4  = "";

    switch (category) {
        case DRV_LOG_CATEGORY_FLOW:
        case DRV_LOG_CATEGORY_TRACE:
        case DRV_LOG_CATEGORY_INTERRUPT:    // we should *never* be kprinting from an interrupt context...
            if (secondary != DRV_LOG_NOTHING) {
                *secondary_string_1 = ", ";
                if (secondary == DRV_LOG_FLOW_IN) {
                    *secondary_string_2 = "Entering";
                }
                else if (secondary == DRV_LOG_FLOW_OUT) {
                    *secondary_string_2 = "Leaving";
                }
            }
            break;
        case DRV_LOG_CATEGORY_STATE_CHANGE:
            {
                U8 orig_state, dest_state;

                orig_state = (secondary & 0xF0) >> 4;
                dest_state =  secondary & 0x0F;

                *secondary_string_1 = ", ";

                if (orig_state < DRV_LOG_NB_DRIVER_STATES) {
                    *secondary_string_2 = drv_log_states[orig_state];
                }
                else {
                    *secondary_string_2 = "Unknown_state";
                }

                *secondary_string_3 = " -> ";

                if (dest_state < DRV_LOG_NB_DRIVER_STATES) {
                    *secondary_string_4 = drv_log_states[dest_state];
                }
                else {
                    *secondary_string_4 = "Unknown_state";
                }
            }
            break;

        default:
            break;
    }

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn       static inline VOID utility_Log_Write (
 *                                    U8 destination, U8 category, U8 secondary,
 *                                    const char* function_name, U32 func_name_len,
 *                                    U32 line_number, U64 tsc, U8 ioctl, U16 processor_id,
 *                                    U8 driver_state, U16 nb_active_interrupts,
 *                                    U16 nb_active_notifications,
 *                                    const char* format_string, ...)
 *
 * @brief    Checks whether and where the message should be logged, and logs it as appropriate.
 *
 * @param    U8          destination             - whether to write to the primary (0) or the auxiliary log buffer (1)
 *           U8          category                - message category
 *           U8          secondary               - secondary information field for the message
 *           const char* function_name           - name of the calling function
 *           U32         func_name_len           - length of the name of the calling function (more efficient
 *                                                 to pass it as parameter than finding it back at runtime)
 *           U32         line_number             - line number of the call site
 *           U64         tsc                     - time stamp value to use
 *           U8          ioctl                   - current active ioctl
 *           U16         processor_id            - id of the active core/thread
 *           U8          driver_state            - current driver state
 *           U16         nb_active_interrupts    - number of interrupts currently being processed
 *           U16         nb_active_notifications - number of notifications currently being processed
 *           const char* format_string           - classical format string for printf-like functions
 *           ...                                 - elements to print
 *
 * @return   none
 *
 * <I>Special Notes:</I>
 *           Writes the specified message to the specified log buffer.
 *           The order of writes (integrity tag at the beginning, overflow tag at the very end) matters
 *           to ensure the logged information can be detected to be only partially written if applicable).
 *           Much of the needed information (active core, driver state, tsc..) is passed through the
 *           stack (instead of obtained inside utility_Log_Write) to guarantee entries representing the
 *           same message (or log call) in different channels use consistent information, letting the
 *           decoder reliably identify duplicates.
 */
static inline VOID
utility_Log_Write (
    U8          destination,
    U8          category,
    U8          secondary,
    const char* function_name,
    U32         function_name_length,
    U32         line_number,
    U64         tsc,
    U8          ioctl,
    U16         processor_id,
    U8          driver_state,
    U16         nb_active_interrupts,
    U16         nb_active_notifications,
    const char* format_string,
    va_list     args
)
{
    U32             entry_id;
    U16             overflow_tag;
    DRV_LOG_ENTRY   entry;
    char*           target_func_buffer;
    U32             local_func_name_length;
    U32             i;

    if (destination == 0) { // primary buffer
        entry_id     = __sync_add_and_fetch(&DRV_LOG_BUFFER_pri_entry_index(DRV_LOG()), 1);
        overflow_tag = (U16)(entry_id / DRV_LOG_MAX_NB_PRI_ENTRIES);
        entry        = DRV_LOG_BUFFER_entries(DRV_LOG())        +
                        entry_id % DRV_LOG_MAX_NB_PRI_ENTRIES;
    }
    else {
        entry_id     = __sync_add_and_fetch(&DRV_LOG_BUFFER_aux_entry_index(DRV_LOG()), 1);
        overflow_tag = (U16)(entry_id / DRV_LOG_MAX_NB_AUX_ENTRIES);
        entry        = DRV_LOG_BUFFER_entries(DRV_LOG())        +
                        DRV_LOG_MAX_NB_PRI_ENTRIES              +
                        entry_id % DRV_LOG_MAX_NB_AUX_ENTRIES;
    }

    DRV_LOG_COMPILER_MEM_BARRIER();
    DRV_LOG_ENTRY_integrity_tag(entry) = overflow_tag;
    DRV_LOG_COMPILER_MEM_BARRIER();

    if (format_string && *format_string) {          // setting this one first to try to increase MLP
        vsnprintf(DRV_LOG_ENTRY_message(entry),
                          DRV_LOG_MESSAGE_LENGTH,
                          format_string,
                          args);
    }
    else {
        DRV_LOG_ENTRY_message(entry)[0] = 0;
    }

    target_func_buffer     = DRV_LOG_ENTRY_function_name(entry);
    local_func_name_length = function_name_length < DRV_LOG_FUNCTION_NAME_LENGTH ?
                             function_name_length : DRV_LOG_FUNCTION_NAME_LENGTH;
    for (i = 0; i < local_func_name_length - 1; i++) {
        target_func_buffer[i] = function_name[i];
    }
    target_func_buffer[i] = 0;

    DRV_LOG_ENTRY_category(entry)                = category;
    DRV_LOG_ENTRY_secondary_info(entry)          = secondary;
    DRV_LOG_ENTRY_line_number(entry)             = line_number;
    DRV_LOG_ENTRY_active_drv_operation(entry)    = ioctl;
    DRV_LOG_ENTRY_processor_id(entry)            = processor_id;
    DRV_LOG_ENTRY_driver_state(entry)            = driver_state;
    DRV_LOG_ENTRY_nb_active_interrupts(entry)    = nb_active_interrupts;
    DRV_LOG_ENTRY_nb_active_notifications(entry) = nb_active_notifications;
    DRV_LOG_ENTRY_tsc(entry)                     = tsc;

    DRV_LOG_COMPILER_MEM_BARRIER();
    DRV_LOG_ENTRY_temporal_tag(entry)   = overflow_tag;
    DRV_LOG_COMPILER_MEM_BARRIER();

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn       extern void UTILITY_Log (U8 category, U8 in_notification, U8 secondary,
 *                                    const char* function_name, U32 func_name_len,
 *                                    U32 line_number, const char* format_string, ...)
 *
 * @brief    Checks whether and where the message should be logged, and logs it as appropriate.
 *
 * @param    U8          category        - message category
 *           U8          in_notification - whether or not we are in a notification/OS callback context
 *                                         (this information cannot be reliably obtained without passing
 *                                         it through the stack)
 *           U8          secondary       - secondary information field for the message
 *           const char* function_name   - name of the calling function
 *           U32         func_name_len   - length of the name of the calling function (more efficient
 *                                         to pass it as parameter than finding it back at runtime)
 *           U32         line_number     - line number of the call site
 *           const char* format_string   - classical format string for printf-like functions
 *           ...                         - elements to print
 *
 * @return   none
 *
 * <I>Special Notes:</I>
 *           Takes a snapshot of various elements (TSC, driver state, etc.) to ensure a single log call
 *           writes consistent information to all applicable channels (i.e. favoring consistency over
 *           instantaneous accuracy). See utility_Log_Write for details.
 */
extern VOID
UTILITY_Log (
    U8          category,
    U8          in_notification,
    U8          secondary,
    const char* function_name,
    U32         func_name_len,
    U32         line_number,
    const char* format_string,
    ...
)
{
    U64     tsc_snapshot;
    U8      ioctl_snapshot;
    U8      driver_state_snapshot;
    U16     processor_id_snapshot;
    U16     nb_active_interrupts_snapshot;
    U16     nb_active_notifications_snapshot;
    U8      category_verbosity;
    U8      in_interrupt;
    U8      is_enabled;

    category_verbosity    = DRV_LOG_VERBOSITY(category);
    processor_id_snapshot = curcpu;
    in_interrupt          = ((pcb && CPU_STATE_in_interrupt(&pcb[processor_id_snapshot])) +
                               (category == DRV_LOG_CATEGORY_INTERRUPT));
    is_enabled            = in_interrupt                          * !!(category_verbosity & LOG_CONTEXT_INTERRUPT) +
                            in_notification                       * !!(category_verbosity & LOG_CONTEXT_NOTIFICATION) +
                            (!in_interrupt * !in_notification)    * !!(category_verbosity & LOG_CONTEXT_REGULAR);


    if (is_enabled) {
        va_list args;
        U32     i;

        ioctl_snapshot                   = active_ioctl;
        driver_state_snapshot            = GET_DRIVER_STATE();
        nb_active_interrupts_snapshot    = DRV_LOG_BUFFER_nb_active_interrupts(DRV_LOG());
        nb_active_notifications_snapshot = DRV_LOG_BUFFER_nb_active_notifications(DRV_LOG());
        UTILITY_Read_TSC                 (&tsc_snapshot);

        va_start     (args, format_string);

        for (i = 0; i < 2; i++) {
            if (category_verbosity & (1 << i)) {
                va_list args_copy;
                va_copy(args_copy, args);
                utility_Log_Write(
                    i,          // 0 for primary log, 1 for auxiliary log
                    category,
                    secondary,
                    function_name,
                    func_name_len,
                    line_number,
                    tsc_snapshot,
                    ioctl_snapshot,
                    processor_id_snapshot,
                    driver_state_snapshot,
                    nb_active_interrupts_snapshot,
                    nb_active_notifications_snapshot,
                    format_string,
                    args_copy
                );
                va_end(args_copy);
            }
        }
        if (category_verbosity & LOG_CHANNEL_PRINTK ||
            category_verbosity & LOG_CHANNEL_TRACEK) {
#define     DRV_LOG_DEBUG_ARRAY_SIZE  512
            char tmp_array[DRV_LOG_DEBUG_ARRAY_SIZE];
            U32  nb_written_characters;
            const char *category_s, *sec1_s, *sec2_s, *sec3_s, *sec4_s;
            va_list args_copy;
            utility_Driver_Log_Kprint_Helper(
                category,
                &category_s,
                secondary,
                &sec1_s,
                &sec2_s,
                &sec3_s,
                &sec4_s
            );

            nb_written_characters = snprintf(tmp_array,
                DRV_LOG_DEBUG_ARRAY_SIZE - 1,
                SEP_MSG_PREFIX " [%s%s%s%s%s] [%s@%d]: ",
                category_s,
                sec1_s,
                sec2_s,
                sec3_s,
                sec4_s,
                function_name,
                line_number);

            if (nb_written_characters > 0) {
                va_copy(args_copy, args);
                nb_written_characters += vsnprintf(tmp_array + nb_written_characters,
                    DRV_LOG_DEBUG_ARRAY_SIZE - nb_written_characters - 1,
                    format_string,
                    args_copy);
                va_end(args_copy);
#undef      DRV_LOG_DEBUG_ARRAY_SIZE

                tmp_array[nb_written_characters++] = '\n';
                tmp_array[nb_written_characters++] = 0;

                if ((category_verbosity & LOG_CHANNEL_PRINTK) * !in_interrupt * !in_notification) {
                    if (!curthread->td_critnest) {
                        switch (category) {
                            case DRV_LOG_CATEGORY_ERROR:
                                printf("%s", tmp_array);
                                break;
                            case DRV_LOG_CATEGORY_WARNING:
                                log(LOG_WARNING, "%s", tmp_array);
                                break;
                            default:
                                log(LOG_INFO, "%s", tmp_array);
                                break;
                        }
                    }
                }

                if (category_verbosity & LOG_CHANNEL_TRACEK) {
                    // not implemented in FreeBSD
                }
            }
        }

        va_end       (args);
    }

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn       extern DRV_STATUS UTILITY_Driver_Log_Init (void)
 *
 * @brief    Allocates and initializes the driver log buffer.
 *
 * @param    none
 *
 * @return   OS_SUCCESS on success, OS_NO_MEM on error.
 *
 * <I>Special Notes:</I>
 *           Should be (successfully) run before any non-LOAD log calls.
 *           Allocates memory without going through CONTROL_Allocate (to avoid
 *           complicating the instrumentation of CONTROL_* functions): calling
 *           UTILITY_Driver_Log_Free is necessary to free the log structure.
 *           Falls back to vmalloc when contiguous physical memory cannot be
 *           allocated. This does not impact runtime behavior, but may impact
 *           the easiness of retrieving the log from a core dump if the system
 *           crashes.
 */
extern DRV_STATUS
UTILITY_Driver_Log_Init (
    void
)
{
    struct timespec cur_time;
    U32             size = sizeof(*driver_log_buffer);
    U8              using_contiguous_physical_memory;
    U32             bitness;

                                                                                        // allocating outside the regular function to restrict the area of the driver
    driver_log_buffer = (PVOID)contigmalloc(size, M_SEP, M_NOWAIT,
                                            0ul, ~0ul, PAGE_SIZE, 1024 * 1024 * 1024);  // where the log might not be initialized

    if (driver_log_buffer) {
        using_contiguous_physical_memory = 1;
    }
    else {
        driver_log_buffer = malloc(size, M_SEP, M_WAITOK);

        if (!driver_log_buffer) {
            return OS_NO_MEM;
        }

        using_contiguous_physical_memory = 0;
    }

    memset(driver_log_buffer, DRV_LOG_FILLER_BYTE, sizeof(*driver_log_buffer)); // we don't want zero-filled pages (so that the buffer's pages don't get ommitted in some crash dumps)

    DRV_LOG_COMPILER_MEM_BARRIER();
    DRV_LOG_BUFFER_header_signature(driver_log_buffer)[0] = DRV_LOG_SIGNATURE_0;
    DRV_LOG_BUFFER_footer_signature(driver_log_buffer)[0] = DRV_LOG_SIGNATURE_6;
    DRV_LOG_BUFFER_header_signature(driver_log_buffer)[3] = DRV_LOG_SIGNATURE_3;
    DRV_LOG_BUFFER_footer_signature(driver_log_buffer)[3] = DRV_LOG_SIGNATURE_3;

    DRV_LOG_COMPILER_MEM_BARRIER();
    DRV_LOG_BUFFER_header_signature(driver_log_buffer)[2] = DRV_LOG_SIGNATURE_2;
    DRV_LOG_BUFFER_footer_signature(driver_log_buffer)[2] = DRV_LOG_SIGNATURE_4;
    DRV_LOG_BUFFER_header_signature(driver_log_buffer)[1] = DRV_LOG_SIGNATURE_1;
    DRV_LOG_BUFFER_footer_signature(driver_log_buffer)[1] = DRV_LOG_SIGNATURE_5;


    DRV_LOG_COMPILER_MEM_BARRIER();
    DRV_LOG_BUFFER_header_signature(driver_log_buffer)[7] = DRV_LOG_SIGNATURE_7;
    DRV_LOG_BUFFER_footer_signature(driver_log_buffer)[7] = DRV_LOG_SIGNATURE_7;
    DRV_LOG_BUFFER_header_signature(driver_log_buffer)[5] = DRV_LOG_SIGNATURE_5;
    DRV_LOG_BUFFER_footer_signature(driver_log_buffer)[5] = DRV_LOG_SIGNATURE_1;

    DRV_LOG_COMPILER_MEM_BARRIER();
    DRV_LOG_BUFFER_header_signature(driver_log_buffer)[6] = DRV_LOG_SIGNATURE_6;
    DRV_LOG_BUFFER_footer_signature(driver_log_buffer)[6] = DRV_LOG_SIGNATURE_0;
    DRV_LOG_BUFFER_header_signature(driver_log_buffer)[4] = DRV_LOG_SIGNATURE_4;
    DRV_LOG_BUFFER_footer_signature(driver_log_buffer)[4] = DRV_LOG_SIGNATURE_2;


    DRV_LOG_BUFFER_log_size(driver_log_buffer)            = sizeof(*driver_log_buffer);
    DRV_LOG_BUFFER_max_nb_pri_entries(driver_log_buffer)  = DRV_LOG_MAX_NB_PRI_ENTRIES;
    DRV_LOG_BUFFER_max_nb_aux_entries(driver_log_buffer)  = DRV_LOG_MAX_NB_AUX_ENTRIES;
    getnanotime(&cur_time);
    DRV_LOG_BUFFER_init_time(driver_log_buffer)           = cur_time.tv_sec;
    DRV_LOG_BUFFER_disambiguator(driver_log_buffer)       = 0;
    DRV_LOG_BUFFER_log_version(driver_log_buffer)         = DRV_LOG_VERSION;
    DRV_LOG_BUFFER_pri_entry_index(driver_log_buffer)     = (U32)((S32)-1);
    DRV_LOG_BUFFER_aux_entry_index(driver_log_buffer)     = (U32)((S32)-1);

#if defined(DRV_EM64T)
    bitness = 64;
#else
    bitness = 32;
#endif

    snprintf(
        DRV_LOG_BUFFER_driver_version(driver_log_buffer),
        DRV_LOG_DRIVER_VERSION_SIZE,
        "[%u-bit FreeBSD] SEP v%d.%d . API %d. type %u.",
            bitness,
            SEP_MAJOR_VERSION,
            SEP_MINOR_VERSION,
            SEP_API_VERSION,
            drv_type);

    DRV_LOG_BUFFER_driver_state(driver_log_buffer)                = GET_DRIVER_STATE();
    DRV_LOG_BUFFER_active_drv_operation(driver_log_buffer)        = active_ioctl;
    DRV_LOG_BUFFER_nb_drv_operations(driver_log_buffer)           = 0;
    DRV_LOG_BUFFER_nb_interrupts(driver_log_buffer)               = 0;
    DRV_LOG_BUFFER_nb_active_interrupts(driver_log_buffer)        = 0;
    DRV_LOG_BUFFER_nb_notifications(driver_log_buffer)            = 0;
    DRV_LOG_BUFFER_nb_active_notifications(driver_log_buffer)     = 0;
    DRV_LOG_BUFFER_nb_driver_state_transitions(driver_log_buffer) = 0;

    DRV_LOG_VERBOSITY(DRV_LOG_CATEGORY_LOAD)            = DRV_LOG_DEFAULT_LOAD_VERBOSITY;
    DRV_LOG_VERBOSITY(DRV_LOG_CATEGORY_INIT)            = DRV_LOG_DEFAULT_INIT_VERBOSITY;
    DRV_LOG_VERBOSITY(DRV_LOG_CATEGORY_DETECTION)       = DRV_LOG_DEFAULT_DETECTION_VERBOSITY;
    DRV_LOG_VERBOSITY(DRV_LOG_CATEGORY_ERROR)           = DRV_LOG_DEFAULT_ERROR_VERBOSITY;
    DRV_LOG_VERBOSITY(DRV_LOG_CATEGORY_STATE_CHANGE)    = DRV_LOG_DEFAULT_STATE_CHANGE_VERBOSITY;
    DRV_LOG_VERBOSITY(DRV_LOG_CATEGORY_MARK)            = DRV_LOG_DEFAULT_MARK_VERBOSITY;
    DRV_LOG_VERBOSITY(DRV_LOG_CATEGORY_DEBUG)           = DRV_LOG_DEFAULT_DEBUG_VERBOSITY;
    DRV_LOG_VERBOSITY(DRV_LOG_CATEGORY_FLOW)            = DRV_LOG_DEFAULT_FLOW_VERBOSITY;
    DRV_LOG_VERBOSITY(DRV_LOG_CATEGORY_ALLOC)           = DRV_LOG_DEFAULT_ALLOC_VERBOSITY;
    DRV_LOG_VERBOSITY(DRV_LOG_CATEGORY_INTERRUPT)       = DRV_LOG_DEFAULT_INTERRUPT_VERBOSITY;
    DRV_LOG_VERBOSITY(DRV_LOG_CATEGORY_TRACE)           = DRV_LOG_DEFAULT_TRACE_VERBOSITY;
    DRV_LOG_VERBOSITY(DRV_LOG_CATEGORY_REGISTER)        = DRV_LOG_DEFAULT_REGISTER_VERBOSITY;
    DRV_LOG_VERBOSITY(DRV_LOG_CATEGORY_NOTIFICATION)    = DRV_LOG_DEFAULT_NOTIFICATION_VERBOSITY;
    DRV_LOG_VERBOSITY(DRV_LOG_CATEGORY_WARNING)         = DRV_LOG_DEFAULT_WARNING_VERBOSITY;

    DRV_LOG_BUFFER_contiguous_physical_memory(driver_log_buffer) = using_contiguous_physical_memory;

    SEP_DRV_LOG_LOAD("Initialized driver log using %scontiguous physical memory.",
        DRV_LOG_BUFFER_contiguous_physical_memory(driver_log_buffer) ? "" : "non-");

    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn       extern DRV_STATUS UTILITY_Driver_Log_Free (void)
 *
 * @brief    Frees the driver log buffer.
 *
 * @param    none
 *
 * @return   OS_SUCCESS on success, OS_NO_MEM on error.
 *
 * <I>Special Notes:</I>
 *           Should be done before unloading the driver.
 *           See UTILITY_Driver_Log_Init for details.
 */
extern void
UTILITY_Driver_Log_Free (
    VOID
)
{
    U32 size = sizeof(*driver_log_buffer);

    if (driver_log_buffer) {

        if (DRV_LOG_BUFFER_contiguous_physical_memory(driver_log_buffer)) {
            contigfree(driver_log_buffer, size, M_SEP);
        }
        else {
            free(driver_log_buffer, M_SEP);
        }

        driver_log_buffer = NULL;
    }
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn       extern void UTILITY_Driver_Set_Active_Ioctl (U32 ioctl)
 *
 * @brief    Sets the 'active_ioctl' global to the specified value.
 *
 * @param    U32 ioctl - ioctl/drvop code to use
 *
 * @return   none
 *
 * <I>Special Notes:</I>
 *           Used to keep track of the IOCTL operation currently being processed.
 *           This information is saved in the log buffer (globally), as well as
 *           in every log entry.
 *           NB: only IOCTLs for which grabbing the ioctl mutex is necessary
 *           should be kept track of this way.
 */
extern void
UTILITY_Driver_Set_Active_Ioctl (
    U32 ioctl
)
{
    active_ioctl = ioctl;
    if (ioctl) {
        DRV_LOG_BUFFER_nb_drv_operations(driver_log_buffer)++;
    }
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn       extern const char** UTILITY_Log_Category_Strings (void)
 *
 * @brief    Accessor function for the log category string array
 *
 * @param    none
 *
 * @return   none
 *
 * <I>Special Notes:</I>
 *           Only needed for cosmetic purposes when adjusting category verbosities.
 */
extern const char**
UTILITY_Log_Category_Strings (
    void
)
{
    return drv_log_categories;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn       extern U32 UTILITY_Change_Driver_State (U32 allowed_prior_states, U32 state, const char* func, U32 line_number)
 *
 * @brief    Updates the driver state (if the transition is legal).
 *
 * @param    U32 allowed_prior_states   - the bitmask representing the states from which the transition is allowed to occur
 *           U32 state                  - the destination state
 *           const char* func           - the callsite's function's name
 *           U32 line_number            - the callsite's line number
 *
 * @return   1 in case of success, 0 otherwise
 *
 * <I>Special Notes:</I>
 *
 */
extern U32
UTILITY_Change_Driver_State (
    U32         allowed_prior_states,
    U32         state,
    const char* func,
    U32         line_number
)
{
    U32 res;
    U32 previous_state;
    U32 nb_attempts     = 0;

    SEP_DRV_LOG_TRACE_IN("Prior states: 0x%x, state: %u, func: %p, line: %u.", allowed_prior_states, state, func, line_number);

    if (state >= DRV_LOG_NB_DRIVER_STATES) {
        SEP_DRV_LOG_ERROR("Illegal destination state %d (%s@%u)!", state, func, line_number);
        res = 0;
        goto clean_return;
    }

    do {
        previous_state = GET_DRIVER_STATE();
        nb_attempts++;
        SEP_DRV_LOG_TRACE("Attempt #%d to transition to state %s.", nb_attempts, drv_log_states[state]);

        if (DRIVER_STATE_IN(previous_state, allowed_prior_states)) {
            res = atomic_cmpset_int(&GET_DRIVER_STATE(),
                                     previous_state,
                                     state);
        }
        else {
            SEP_DRV_LOG_ERROR("Invalid transition [%s -> %s] (%s@%u)!", drv_log_states[previous_state], drv_log_states[state], func, line_number);
            res = 0;
            goto clean_return;
        }

    } while (!res);

    SEP_DRV_LOG_STATE_TRANSITION(previous_state, state, "From %s@%u.", func, line_number);

    clean_return:
    SEP_DRV_LOG_TRACE_OUT("Res: %u.", res);
    return res;
}
