/*
 * Copyright (C) 2005 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

#ifndef _OUTPUT_H_
#define _OUTPUT_H_

#include <sys/param.h>
#include <sys/lock.h>
#include <sys/mutex.h>
#include <sys/conf.h>

typedef off_t loff_t;

/*
 * Initial allocation
 * Size of buffer     = 512KB (2^19)
 * number of buffers  = 2
 * The max size of the buffer cannot exceed 1<<22 i.e. 4MB
 */
#define OUTPUT_SMALL_BUFFER        (1<<15)
#define OUTPUT_LARGE_BUFFER        (1<<19)
#define OUTPUT_EMON_BUFFER         (1<<25)
#define OUTPUT_MEMORY_THRESHOLD    0x8000000

extern  U32                   output_buffer_size;
#define OUTPUT_BUFFER_SIZE    output_buffer_size
#define OUTPUT_NUM_BUFFERS    2

/*
 *  Data type declarations and accessors macros
 */
typedef struct {
    struct mtx    buffer_lock;
    U32           remaining_buffer_size;
    U32           current_buffer;
    U32           next_buffer[OUTPUT_NUM_BUFFERS];
    U32           buffer_full[OUTPUT_NUM_BUFFERS];
    U8           *buffer[OUTPUT_NUM_BUFFERS];
} OUTPUT_NODE, *OUTPUT;

#define OUTPUT_buffer_lock(x)            (x)->buffer_lock
#define OUTPUT_remaining_buffer_size(x)  (x)->remaining_buffer_size
#define OUTPUT_total_buffer_size(x)      OUTPUT_BUFFER_SIZE
#define OUTPUT_buffer(x,y)               (x)->buffer[(y)]
#define OUTPUT_buffer_full(x,y)          (x)->buffer_full[(y)]
#define OUTPUT_current_buffer(x)         (x)->current_buffer

/*
 *  Add an array of control buffer for per-cpu
 */
typedef struct {
    struct cv        cv;
    struct mtx       mtx;
    OUTPUT_NODE      outbuf;
    U32              sample_count;
} BUFFER_DESC_NODE, *BUFFER_DESC;

#define BUFFER_DESC_cv(a)             (a)->cv
#define BUFFER_DESC_mtx(a)            (a)->mtx
#define BUFFER_DESC_outbuf(a)         (a)->outbuf
#define BUFFER_DESC_sample_count(a)   (a)->sample_count

extern BUFFER_DESC   cpu_buf;  // actually an array of BUFFER_DESC_NODE
extern BUFFER_DESC   unc_buf;
extern BUFFER_DESC   module_buf;
extern BUFFER_DESC   emon_buf;

/*
 *  Interface Functions
 */

extern int     OUTPUT_Module_Fill (PVOID data, U16 size, U32 in_notification);
extern int     OUTPUT_Initialize (void);
extern int     OUTPUT_Initialize_UNC (void);
extern int     OUTPUT_Initialize_EMON (void);
extern int     OUTPUT_Destroy (VOID);
extern int     OUTPUT_Flush (VOID);
extern int     OUTPUT_Flush_EMON (VOID);
extern d_read_t OUTPUT_Module_Read;
extern d_read_t OUTPUT_Sample_Read;
extern d_read_t OUTPUT_Uncore_Read;
extern d_read_t OUTPUT_Emon_Read;
extern void*   OUTPUT_Reserve_Buffer_Space(BUFFER_DESC bd, U32 size, boolean_t *signal_full, U32 in_notification);

#endif
