##
## Copyright 2016 Intel Corporation.
##
##
## This software and the related documents are Intel copyrighted materials, and your use of them is governed by
## the express license under which they were provided to you ('License'). Unless the License provides otherwise,
## you may not use, modify, copy, publish, distribute, disclose or transmit this software or the related
## documents without Intel's prior written permission.
## This software and the related documents are provided as is, with no express or implied warranties, other than
## those that are expressly stated in the License.
##

#
# Base commands
#
ARFLAGS ?= rvcs

#
# System configuration detection
#
SYSTEM_SFX := $(shell uname)
ifeq ($(SYSTEM_SFX),Darwin)
	G_SYSTEM := macOS
else
	G_SYSTEM := Linux
endif

G_GCC_VER := $(shell $(CC) --version | grep ^gcc | sed 's/^.* //g')

ifndef $(ARCH)
	ARCH_SFX := $(shell uname -m)
	ifeq ($(ARCH_SFX),x86_64)
		ARCH := intel64
	endif
endif

ifeq ($(ARCH),intel64)
	G_ARCH := intel64
endif

ifeq ($(G_ARCH),)
$(error ARCH is invalid)
endif

#
# Base build configuration
#
ifndef $(CONF)
	ifneq ($(TBBROOT),)
		CONF := release_tbb
	else
		CONF := release
	endif
endif

ifeq ($(CONF),debug)
	G_CONF      := debug
	G_USETBB    := 0
endif
ifeq ($(CONF),debug_tbb)
	G_CONF      := debug
	G_USETBB    := 1
endif
ifeq ($(CONF),release)
	G_CONF      := release
	G_USETBB    := 0
endif
ifeq ($(CONF),release_tbb)
	G_CONF      := release
	G_USETBB    := 1
endif
ifeq ($(G_CONF),)
$(error CONF is invalid)
endif

ifeq ($(G_ARCH),quark)
	G_USETBB    := 0
endif

ifeq ($(G_CONF),debug)
	G_CFLAGS   += -g -O0 -std=c99 -pedantic -D_DEBUG
	G_CXXFLAGS += -g -O0 -D_DEBUG -std=c++11
endif
ifeq ($(G_CONF),release)
	G_CFLAGS   += -D_FORTIFY_SOURCE=2 -O2 -std=c99 -pedantic
	G_CXXFLAGS += -D_FORTIFY_SOURCE=2 -O2 -std=c++11
endif

# Alias for OpenMP flag:
# * "-qopenmp" is used in icx/icpx
# * "-fopenmp" is used in gcc/g++ and clang/clang++
ifeq ($(CC),icx)
	G_C_OPENMP := -qopenmp
else
	G_C_OPENMP := -fopenmp
endif
ifeq ($(CXX),icpx)
	G_CXX_OPENMP := -qopenmp
else
	G_CXX_OPENMP := -fopenmp
endif

# Old/new (v2) directory layout machinery.
# If IPP_PRESERVE_OLD_LAYOUT is defined, the old directory layout is used.
# By default the new (v2) directory layout is used.
ifeq ($(IPP_PRESERVE_OLD_LAYOUT),)
	G_INCLUDE_RELATIVE_PATH := include/ipp
	G_LIB_RELATIVE_PATH64   := lib
else
	G_INCLUDE_RELATIVE_PATH := include
	G_LIB_RELATIVE_PATH64   := lib/intel64
	G_CFLAGS   += -DIPP_PRESERVE_OLD_LAYOUT
	G_CXXFLAGS += -DIPP_PRESERVE_OLD_LAYOUT
endif

ifeq ($(ARCH),intel64)
	G_IPP_PATH_SUFFIX ?= $(G_LIB_RELATIVE_PATH64)
	G_TBB_PATH_SUFFIX ?= $(G_LIB_RELATIVE_PATH64)/gcc4.7
	G_TBB_PATH_SUFFIX_2 ?= $(G_LIB_RELATIVE_PATH64)/gcc4.8
endif

#
# Paths configuration
#
G_MAKE_ROOT      := $(shell pwd)/
G_BUILD_ROOT     := $(G_MAKE_ROOT)/_build/
$(shell mkdir -p $(G_BUILD_ROOT))


$(info Checking components )
#
# Test compilers
#
G_CCOMPILER   := ""
G_CXXCOMPILER := ""
TEST_CCOUT_CLANG  := $(shell $(CC)  -v 2>&1 | grep clang )
TEST_CCOUT_GCC    := $(shell $(CC)  -v 2>&1 | grep "gcc version" )
TEST_CCOUT_ICPC   := $(shell $(CC)  --version 2>&1 | grep ICC )
TEST_CXXOUT_CLANG := $(shell $(CXX) -v 2>&1 | grep clang )
TEST_CXXOUT_GCC   := $(shell $(CXX) -v 2>&1 | grep "gcc version" )
TEST_CXXOUT_ICPC  := $(shell $(CXX) --version 2>&1 | grep ICC )
ifneq ($(TEST_CCOUT_CLANG),)
G_CCOMPILER := clang
$(info C compiler................ $(TEST_CCOUT_CLANG) )
else
ifneq ($(TEST_CCOUT_GCC),)
G_CCOMPILER := gcc
$(info C compiler................ $(TEST_CCOUT_GCC) )
else
ifneq ($(TEST_CCOUT_ICPC),)
G_CCOMPILER := icpc
$(info C compiler................ $(TEST_CCOUT_ICPC) )
endif
endif
endif

ifneq ($(TEST_CXXOUT_CLANG),)
G_CXXCOMPILER := clang
$(info C++ compiler.............. $(TEST_CXXOUT_CLANG) )
else
ifneq ($(TEST_CXXOUT_GCC),)
G_CXXCOMPILER := gcc
$(info C++ compiler.............. $(TEST_CXXOUT_GCC) )
else
ifneq ($(TEST_CXXOUT_ICPC),)
G_CXXCOMPILER := icpc
$(info C++ compiler.............. $(TEST_CXXOUT_ICPC) )
endif
endif
endif

#
# Test OpenGL
#
G_USERENDERER := 0
ifneq ($(RENDERER),0)
TEST_RENDERER_PREFIX := ${G_BUILD_ROOT}/test_ren
$(shell printf "\
#include <X11/Xlib.h>\n\
#include <GL/glx.h>\n\
\n\
int main(int argc, char *argv[])\n\
{\n\
	argc = argc;\n\
	argv = argv;\n\
	return 0;\n\
}\n\
" > ${G_BUILD_ROOT}/test_ren.c)

ifeq ($(G_ARCH),intel64)
TEST_CCOUT64  := $(shell $(CC)  -m64 -o ${TEST_RENDERER_PREFIX}_64.o -c ${TEST_RENDERER_PREFIX}.c 2>&1)
TEST_CCOUT64  := $(shell $(CC)  -m64 -o /dev/null ${TEST_RENDERER_PREFIX}_64.o -lGL -lX11 2>&1)
else
TEST_CCOUT32  := $(shell $(CC)  -m32 -o ${TEST_RENDERER_PREFIX}_32.o -c ${TEST_RENDERER_PREFIX}.c 2>&1)
TEST_CCOUT32  := $(shell $(CC)  -m32 -o /dev/null ${TEST_RENDERER_PREFIX}_32.o -lGL -lX11 2>&1)
endif
$(shell $(RM) $(TEST_RENDERER_PREFIX).c)
$(shell $(RM) $(TEST_RENDERER_PREFIX)_64.o)
$(shell $(RM) $(TEST_RENDERER_PREFIX)_32.o)
ifeq ($(strip $(TEST_CCOUT32)),)
ifeq ($(strip $(TEST_CCOUT64)),)
$(info OpenGL.................... yes )
	G_USERENDERER := 1
else
$(info OpenGL.................... no )
endif
else
$(info OpenGL.................... no )
endif
endif
#
# Test OpenMP
#
G_OMP_SUPPORTED := 0
TEST_CCOUT  := $(shell printf "\#include <omp.h>" | $(CC)  $(G_C_OPENMP) -o /dev/null -xc -c - 2>&1)
TEST_CXXOUT := $(shell printf "\#include <omp.h>" | $(CXX) $(G_CXX_OPENMP) -o /dev/null -xc++ -c - 2>&1)
ifeq ($(strip $(TEST_CCOUT)),)
ifeq ($(strip $(TEST_CXXOUT)),)
$(info OpenMP.................... yes )
	G_OMP_SUPPORTED := 1
else
$(info OpenMP.................... no )
endif
else
$(info OpenMP.................... no )
endif

#
# Test Intel(R) TBB
#
ifneq ($(G_USETBB),0)
G_USETBB := 0
TEST_CXXOUT := $(shell printf "\#include <tbb/version.h>" | $(CXX) -I${TBBROOT}/include -o /dev/null -xc++ -c -std=c++11 - 2>&1)
ifeq ($(strip $(TEST_CXXOUT)),)
$(info Intel(R) TBB.............. yes )
	G_USETBB := 1
else
$(info Intel(R) TBB.............. no )
endif
endif

ifeq ($(G_USETBB),1)
	G_CONF_TBB := $(G_CONF)_tbb
	G_LIB_SFX    := _tbb
else
	G_CONF_TBB := $(G_CONF)
	G_LIB_SFX    := _omp
endif

#
# Test Intel(R) IPP for OpenCV
#
G_ICV := 0
TEST_CCOUT  := $(shell printf "\#include <ippicv.h>" | $(CC) -I$(IPPROOT)/include -o /dev/null -xc -c - 2>&1)
ifeq ($(strip $(TEST_CCOUT)),)
	G_ICV := 1
endif

#
# Test -fstack-protector-strong
#
G_FSPS_SUPPORTED := 0
TEST_CCOUT  := $(shell printf "" | $(CC)  -fstack-protector-strong -o /dev/null -xc -c - 2>&1)
TEST_CXXOUT := $(shell printf "" | $(CXX) -fstack-protector-strong -o /dev/null -xc++ -c - 2>&1)
ifeq ($(strip $(TEST_CCOUT)),)
ifeq ($(strip $(TEST_CXXOUT)),)
$(info -fstack-protector-strong.. yes )
	G_FSPS_SUPPORTED := 1
else
$(info -fstack-protector-strong.. no )
endif
else
$(info -fstack-protector-strong.. no )
endif

#
# Test Intel(R) CET
#
ifneq ($(G_USECET),0)
G_USECET := 0
TEST_CCOUT := $(shell printf "" | $(CC) -fcf-protection -o /dev/null -xc++ -xc -c - 2>&1)
ifeq ($(strip $(TEST_CCOUT)),)
$(info Intel(R) CET.............. yes )
	G_USECET := 1
else
$(info Intel(R) CET.............. no )
endif
endif

$(info )

# Requred compilation flags:
# G_LIB_CFLAGS   - Library C flags
# G_LIB_LDFLAGS  - Library linker flags
# G_EXE_CFLAGS   - Executable C flags
# G_EXE_LDFLAGS  - Executable linker flags
G_LIB_CFLAGS   := -fPIC -Wformat -Wformat-security
G_LIB_LDFLAGS  := -z noexecstack -z relro -z now
G_EXE_CFLAGS   := -fPIC -Wformat -Wformat-security
G_EXE_LDFLAGS  := -pie -z noexecstack -z relro -z now

ifeq ($(G_FSPS_SUPPORTED),1)
	G_LIB_CFLAGS += -fstack-protector-strong
	G_EXE_CFLAGS += -fstack-protector-strong
else
	G_LIB_CFLAGS += -fstack-protector
	G_EXE_CFLAGS += -fstack-protector
endif

G_LIB_CFLAGS += -D_XOPEN_SOURCE=500 -Dlinux
G_EXE_CFLAGS += -D_XOPEN_SOURCE=500 -Dlinux

ifeq ($(G_USECET),1)
	G_LIB_CFLAGS += -fcf-protection
	G_EXE_CFLAGS += -fcf-protection
endif

#
# Export global vars
#
export G_SYSTEM
export G_MAKE_ROOT
export G_BUILD_ROOT
export G_ARCH
export G_CONF
export G_CONF_TBB
export G_USETBB
export G_USERENDERER
export G_CFLAGS
export G_CXXFLAGS
export G_C_OPENMP
export G_CXX_OPENMP
export G_INCLUDE_RELATIVE_PATH
export G_IPP_PATH_SUFFIX
export G_TBB_PATH_SUFFIX
export G_TBB_PATH_SUFFIX_2
export G_GEN_PATH_SUFFIX
export G_OMP_SUPPORTED
export G_ICV

export G_LIB_CFLAGS
export G_LIB_LDFLAGS
export G_EXE_CFLAGS
export G_EXE_LDFLAGS

export ARFLAGS
export CC
export CXX

export G_LIB_SFX
