/* This file is part of the 'atomes' software

'atomes' is free software: you can redistribute it and/or modify it under the terms
of the GNU Affero General Public License as published by the Free Software Foundation,
either version 3 of the License, or (at your option) any later version.

'atomes' is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU Affero General Public License along with 'atomes'.
If not, see <https://www.gnu.org/licenses/>

Copyright (C) 2022-2024 by CNRS and University of Strasbourg */

/*!
* @file charmm36_na.c
* @short CHARMM-charmm36_na force field, from 'charmm36_na'
* @author Sébastien Le Roux <sebastien.leroux@ipcms.unistra.fr>
*/

/*
* This file: 'charmm36_na.c'
*
* Contains:
*

  CHARMM-charmm36_na force field, data from taken from:

       - the file: 'charmm36_na'
       - distributed in: 'toppar_c36_jul16_mass_orig.tgz'

  This file contains several tables:

    Atoms      : charmm36_na_atoms[42][5]
    Bonds      : charmm36_na_bonds[89][5]
    Angles     : charmm36_na_angles[226][8]
    Dihedrals  : charmm36_na_dihedrals[502][8]
    Impropers  : charmm36_na_impropers[15][8]
    Non bonded : charmm36_na_vdw[42][6]

*/

#include "global.h"

int charmm36_na_objects[14] = {42, 0, 0, 89, 0, 0, 226, 0, 502, 0 , 0, 15, 42, 0};
int charmm36_na_dim[14] = {5, 0, 0, 5, 0, 0, 8, 0, 8, 0, 0, 8, 6, 0};

/*
 Atoms
  0= Element, 1= Mass, 2= Key, 3= Description, 4= charmm36_na info
*/
char * charmm36_na_atoms[42][5]= {{"H", "1.008000", "HN1", "Nucleic acid amine proton", "91"},
                            {"H", "1.008000", "HN2", "Nucleic acid ring nitrogen proton", "92"},
                            {"H", "1.008000", "HN3", "Nucleic acid aromatic carbon proton", "93"},
                            {"H", "1.008000", "HN4", "Nucleic acid phosphate hydroxyl proton", "94"},
                            {"H", "1.008000", "HN5", "Nucleic acid ribose hydroxyl proton", "95"},
                            {"H", "1.008000", "HN6", "Nucleic acid ribose aliphatic proton", "96"},
                            {"H", "1.008000", "HN7", "Nucleic acid proton (equivalent to protein HA)", "97"},
                            {"H", "1.008000", "HN8", "Bound to CN8 in nucleic acids/model compounds", "98"},
                            {"H", "1.008000", "HN9", "Bound to CN9 in nucleic acids/model compounds", "99"},
                            {"C", "12.011000", "CN1", "Nucleic acid carbonyl carbon", "100"},
                            {"C", "12.011000", "CN1T", "Nucleic acid carbonyl carbon (T/U C2)", "101"},
                            {"C", "12.011000", "CN2", "Nucleic acid aromatic carbon to amide", "102"},
                            {"C", "12.011000", "CN3", "Nucleic acid aromatic carbon", "103"},
                            {"C", "12.011000", "CN3T", "Nucleic acid aromatic carbon, Thy C5", "104"},
                            {"C", "12.011000", "CN4", "Nucleic acid purine C8 and ADE C2", "105"},
                            {"C", "12.011000", "CN5", "Nucleic acid purine C4 and C5", "106"},
                            {"C", "12.011000", "CN5G", "Nucleic acid guanine C5", "107"},
                            {"C", "12.011000", "CN7", "Nucleic acid carbon (equivalent to protein CT1)", "108"},
                            {"C", "12.011000", "CN7B", "Nucleic acid aliphatic carbon for C1'", "109"},
                            {"C", "12.011000", "CN8", "Nucleic acid carbon (equivalent to protein CT2)", "110"},
                            {"C", "12.011000", "CN8B", "Nucleic acid carbon (equivalent to protein CT2)", "111"},
                            {"C", "12.011000", "CN9", "Nucleic acid carbon (equivalent to protein CT3)", "112"},
                            {"N", "14.007000", "NN1", "Nucleic acid amide nitrogen", "113"},
                            {"N", "14.007000", "NN2", "Nucleic acid protonated ring nitrogen", "114"},
                            {"N", "14.007000", "NN2B", "From NN2, for N9 in GUA different from ADE", "115"},
                            {"N", "14.007000", "NN2U", "Nucleic acid protonated ring nitrogen, ura N3", "116"},
                            {"N", "14.007000", "NN2G", "Nucleic acid protonated ring nitrogen, gua N1", "117"},
                            {"N", "14.007000", "NN3", "Nucleic acid unprotonated ring nitrogen", "118"},
                            {"N", "14.007000", "NN3A", "Nucleic acid unprotonated ring nitrogen, ade N1 and N3", "119"},
                            {"N", "14.007000", "NN3G", "Nucleic acid unprotonated ring nitrogen, gua N3", "120"},
                            {"N", "14.007000", "NN4", "Nucleic acid purine N7", "121"},
                            {"N", "14.007000", "NN6", "Nucleic acid sp3 amine nitrogen (equiv to protein nh3)", "122"},
                            {"O", "15.999400", "ON1", "Nucleic acid carbonyl oxygen", "123"},
                            {"O", "15.999400", "ON1C", "Nucleic acid carbonyl oxygen, cyt O2", "124"},
                            {"O", "15.999400", "ON2", "Nucleic acid phosphate ester oxygen", "125"},
                            {"O", "15.999400", "ON3", "Nucleic acid =O in phosphate", "126"},
                            {"O", "15.999400", "ON4", "Nucleic acid phosphate hydroxyl oxygen", "127"},
                            {"O", "15.999400", "ON5", "Nucleic acid ribose hydroxyl oxygen", "128"},
                            {"O", "15.999400", "ON6", "Nucleic acid deoxyribose ring oxygen", "129"},
                            {"O", "15.999400", "ON6B", "Nucleic acid ribose ring oxygen", "130"},
                            {"P", "30.974000", "P", "phosphorus", "131"},
                            {"P", "30.974000", "P2", "phosphorus, adm, 2011 DNA update", "132"}};

/*
 Quadratic bonds
  0= Key_a, 1= Key_b, 2= R0 (A), 3= Kb (kcal mol^-1 A^2), 4= charmm36_na info

  V(R) = Kb x (R - R0)^2
*/
char * charmm36_na_bonds[89][5]= {{"CN8", "NN6", "200.000", "1.480", "methylammonium"},
                             {"NN6", "HN1", "403.000", "1.040", "methylammonium"},
                             {"ON6", "CN8B", "260.0", "1.420", "susil"},
                             {"CN8", "CN8B", "222.50", "1.528", "Alkanes, sacred"},
                             {"CN1", "CN3", "302.0", "1.409", "adm jr. 11/97"},
                             {"CN1", "CN3T", "302.0", "1.403", "adm jr. 11/97"},
                             {"CN1", "CN5G", "302.0", "1.360", "adm jr. 11/97"},
                             {"CN1", "NN2", "380.0", "1.367", "adm jr. 11/97"},
                             {"CN1T", "NN2B", "302.0", "1.348", "adm jr. 11/97"},
                             {"CN1", "NN2G", "340.0", "1.396", "adm jr. 11/97"},
                             {"CN1", "NN2U", "340.0", "1.389", "adm jr. 11/97"},
                             {"CN1T", "NN2U", "340.0", "1.383", "adm jr. 11/97"},
                             {"CN1", "NN3", "350.0", "1.335", "adm jr. 11/97"},
                             {"CN1T", "ON1", "860.0", "1.230", "jjp1/adm jr. 7/95"},
                             {"CN1", "ON1", "660.0", "1.234", "par_a4 adm jr. 10/2/91"},
                             {"CN1", "ON1C", "620.0", "1.245", "adm jr. 10/2/91"},
                             {"CN2", "CN3", "320.0", "1.406", "adm jr. 11/97"},
                             {"CN2", "CN5", "360.0", "1.358", "adm jr. 11/97"},
                             {"CN2", "NN1", "360.0", "1.366", "JWK, adm jr. 10/2/91"},
                             {"CN2", "NN2G", "400.0", "1.392", " "},
                             {"CN2", "NN3", "450.0", "1.343", " "},
                             {"CN2", "NN3A", "400.0", "1.342", "adm jr. 11/97"},
                             {"CN2", "NN3G", "320.0", "1.326", "adm jr. 11/97"},
                             {"CN3", "CN3", "500.0", "1.326", "adm jr. 11/97"},
                             {"CN3", "CN3T", "560.0", "1.320", "adm jr. 11/97"},
                             {"CN3T", "CN9", "230.0", "1.478", "adm jr. 11/97"},
                             {"CN3", "HN3", "350.0", "1.09", "JWK"},
                             {"CN3T", "HN3", "350.0", "1.09", "JWK"},
                             {"CN3", "NN2", "302.0", "1.343", "adm jr. 11/97"},
                             {"CN3", "NN2B", "320.0", "1.343", "adm jr. 11/97"},
                             {"CN4", "HN3", "380.0", "1.09", "JWK par_a7 9/30/91"},
                             {"CN4", "NN2", "320.0", "1.374", "adm jr. 11/97"},
                             {"CN4", "NN2B", "300.0", "1.378", "adm jr. 11/97"},
                             {"CN4", "NN3A", "420.0", "1.322", "adm jr. 11/97"},
                             {"CN4", "NN4", "400.0", "1.305", "adm jr. 11/97"},
                             {"CN5", "CN5", "310.0", "1.361", "adm jr. 11/97"},
                             {"CN5", "CN5G", "320.0", "1.350", "adm jr. 11/97"},
                             {"CN5", "NN2", "300.0", "1.375", "adm jr. 11/97"},
                             {"CN5", "NN2B", "302.0", "1.375", "adm jr. 11/97"},
                             {"CN5", "NN3A", "350.0", "1.312", "JWK par_a8 9/30/91"},
                             {"CN5", "NN3G", "350.0", "1.315", "adm jr. 11/97"},
                             {"CN5", "NN4", "310.0", "1.355", "adm jr. 11/97"},
                             {"CN5G", "NN4", "310.0", "1.365", "adm jr. 11/97"},
                             {"CN8", "CN8", "222.50", "1.528", "sacred"},
                             {"CN8", "CN9", "222.50", "1.528", "sacred"},
                             {"CN8", "NN2", "400.0", "1.460", "ADM JR."},
                             {"CN8", "ON5", "428.0", "1.42", "MeOH"},
                             {"CN9", "HN9", "322.0", "1.111", " "},
                             {"CN9", "ON2", "340.0", "1.43", "ADM Jr."},
                             {"HN1", "NN1", "488.0", "1.00", "JWK, adm jr. 7/24/91"},
                             {"HN2", "NN2", "474.0", "1.01", "JWK"},
                             {"HN2", "NN2B", "474.0", "1.01", "adm jr. 11/97"},
                             {"HN2", "NN2G", "471.0", "1.01", "JWK, par_a12 9/30/91"},
                             {"HN2", "NN2U", "474.0", "1.01", "JWK, adm jr. 7/24/91"},
                             {"HN4", "ON4", "545.0", "0.960", "ADM Jr."},
                             {"ON2", "P", "270.0", "1.60", "ADM Jr."},
                             {"ON2", "P2", "270.0", "1.60", "ADM Jr., adm, 2011 DNA update"},
                             {"ON3", "P", "580.0", "1.48", "ADM Jr."},
                             {"ON3", "P2", "580.0", "1.48", "ADM Jr., adm, 2011 DNA update"},
                             {"ON4", "P", "237.0", "1.58", "ADM Jr."},
                             {"ON4", "P2", "237.0", "1.58", "ADM Jr., adm, 2011 DNA update"},
                             {"CN7B", "ON6", "260.0", "1.420", "From exp"},
                             {"CN7B", "CN8", "200.0", "1.518", "From exp"},
                             {"CN7", "ON6", "240.0", "1.446", "Fom exp."},
                             {"CN7", "CN7", "222.5", "1.529", "From exp"},
                             {"CN7", "CN8", "222.5", "1.516", "From exp."},
                             {"CN7", "CN9", "222.5", "1.516", "for 5MET, From alkanes"},
                             {"CN7", "HN7", "309.0", "1.111", "sacred"},
                             {"CN8", "HN8", "309.0", "1.111", "sacred"},
                             {"CN7B", "HN7", "309.0", "1.111", "From CN8 HN7 (NF)"},
                             {"CN7B", "ON6B", "260.0", "1.420", "From CN7B ON6"},
                             {"CN7", "ON6B", "240.0", "1.480", "From CN7 ON6"},
                             {"CN7B", "CN7B", "200.0", "1.450", " "},
                             {"CN7", "CN7B", "222.5", "1.460", "Specific to RNA"},
                             {"CN7", "CN8B", "222.5", "1.512", "From exp."},
                             {"CN8B", "ON2", "320.0", "1.44", "From exp"},
                             {"CN8B", "ON5", "428.0", "1.42", "CN8 ON2, adm jr., 8/30/98"},
                             {"CN7", "ON2", "310.0", "1.433", "From exp"},
                             {"CN7B", "ON2", "310.0", "1.433", "From exp, for NADPH and bkbmod"},
                             {"CN7", "ON5", "428.0", "1.42", "CN8 ON2, adm jr., 8/30/98"},
                             {"CN9", "NN2", "400.0", "1.456", "adm jr."},
                             {"CN8", "NN2B", "400.0", "1.458", "adm jr."},
                             {"CN9", "NN2B", "400.0", "1.458", "adm jr."},
                             {"CN7B", "NN2", "220.0", "1.456", " "},
                             {"CN7B", "NN2B", "220.0", "1.458", " "},
                             {"CN8B", "HN8", "309.0", "1.111", "sacred"},
                             {"ON5", "HN5", "545.0", "0.960", "MeOH"},
                             {"CN7B", "ON5", "428.0", "1.400", "check adm jr.,"},
                             {"CN8", "ON2", "340.0", "1.44", " "}};

/*
 Quadratic angles:
  0= Key_a, 1= Key_b, 2= Key_c, 3= Ktheta (kcal mol^-1 rad^2), 4= Theta0 (deg)

  V(Theta) = Ktheta * (Theta - Theta0)^2

 Urey-Bradley angles:
  5= Kub (kcal mol^-1 A^2), 6= S0 (A), 7= charmm36_na info

  V(S) = Kub x (S - S0)^2
*/
char * charmm36_na_angles[226][8]= {{"CN7", "CN8", "CN8", "58.35", "113.60", "11.16", "2.561", " "},
                              {"CN8", "CN7", "CN8", "58.35", "113.60", "11.16", "2.561", " "},
                              {"CN8", "CN8", "CN8", "58.35", "113.60", "11.16", "2.561", " "},
                              {"HN1", "NN6", "CN8", "30.00", "109.50", "20.00", "2.074", " "},
                              {"NN6", "CN8", "HN8", "45.00", "107.50", "35.00", "2.101", " "},
                              {"CN7", "CN8", "ON2", "115.00", "109.70", " ", " ", "exocyclic angles"},
                              {"NN6", "CN8", "CN8", "67.70", "110.00", " ", " ", " "},
                              {"HN1", "NN6", "HN1", "44.00", "109.50", " ", " ", " "},
                              {"ON2", "CN8", "CN8", "115.0", "109.7", " ", " ", "exocyclic angles"},
                              {"CN7", "ON6", "CN8B", "110.0", "109.0", " ", " ", " "},
                              {"ON6", "CN8B", "CN8", "90.0", "106.0", " ", " ", " "},
                              {"CN8B", "CN8", "CN7", "80.0", "106.0", " ", " ", " "},
                              {"ON6", "CN8B", "HN8", "45.2", "107.24", " ", " ", " "},
                              {"HN8", "CN8B", "CN8", "34.53", "110.10", "22.53", "2.179", "alkane"},
                              {"HN8", "CN8", "CN8B", "34.53", "110.10", "22.53", "2.179", "alkane"},
                              {"CN2", "NN3A", "CN4", "90.0", "117.8", " ", " ", "adm jr. 11/97"},
                              {"NN3A", "CN4", "NN3A", "60.0", "133.0", " ", " ", " "},
                              {"CN4", "NN3A", "CN5", "90.0", "110.1", " ", " ", " "},
                              {"CN5", "CN5", "NN3A", "60.0", "127.4", " ", " ", "bridgeC4"},
                              {"CN2", "CN5", "CN5", "60.0", "121.0", " ", " ", "bridgeC5"},
                              {"CN5", "CN2", "NN3A", "60.0", "110.7", " ", " ", " "},
                              {"CN5", "CN5", "NN2", "100.0", "105.7", " ", " ", "bridgeC4"},
                              {"CN5", "CN5", "NN4", "100.0", "110.0", " ", " ", "bridgeC5"},
                              {"CN4", "NN4", "CN5", "120.0", "104.6", " ", " ", " "},
                              {"NN2", "CN4", "NN4", "100.0", "113.4", " ", " ", " "},
                              {"CN4", "NN2", "CN5", "100.0", "106.3", " ", " ", " "},
                              {"NN2", "CN5", "NN3A", "100.0", "126.9", " ", " ", " "},
                              {"CN2", "CN5", "NN4", "100.0", "129.0", " ", " ", " "},
                              {"HN3", "CN4", "NN3A", "38.0", "113.5", " ", " ", " "},
                              {"NN3A", "CN2", "NN1", "50.0", "130.7", " ", " ", " "},
                              {"CN5", "CN2", "NN1", "50.0", "118.6", " ", " ", " "},
                              {"CN2", "NN1", "HN1", "40.0", "121.5", " ", " ", "C,A,G"},
                              {"HN1", "NN1", "HN1", "31.0", "117.0", " ", " ", " "},
                              {"NN4", "CN4", "HN3", "39.0", "124.8", " ", " ", "G,A"},
                              {"NN2", "CN4", "HN3", "39.0", "121.8", " ", " ", " "},
                              {"CN5", "NN2", "HN2", "30.0", "129.4", " ", " ", " "},
                              {"CN4", "NN2", "HN2", "30.0", "125.0", " ", " ", " "},
                              {"CN1", "NN2G", "CN2", "70.0", "131.1", " ", " ", "adm jr. 11/97"},
                              {"NN2G", "CN2", "NN3G", "70.0", "122.2", " ", " ", " "},
                              {"CN2", "NN3G", "CN5", "90.0", "109.4", " ", " ", " "},
                              {"CN5G", "CN5", "NN3G", "70.0", "129.9", " ", " ", "bridgeC4"},
                              {"CN1", "CN5G", "CN5", "70.0", "119.6", " ", " ", "bridgeC5"},
                              {"CN5G", "CN1", "NN2G", "70.0", "107.8", " ", " ", " "},
                              {"CN5G", "CN5", "NN2B", "100.0", "104.6", " ", " ", "bridgeC4"},
                              {"CN5", "CN5G", "NN4", "100.0", "111.4", " ", " ", "bridgeC5"},
                              {"CN4", "NN4", "CN5G", "120.0", "103.8", " ", " ", " "},
                              {"NN2B", "CN4", "NN4", "100.0", "113.0", " ", " ", " "},
                              {"CN4", "NN2B", "CN5", "100.0", "107.2", " ", " ", " "},
                              {"NN2B", "CN5", "NN3G", "140.0", "125.5", " ", " ", "bridgeC4"},
                              {"CN1", "CN5G", "NN4", "125.0", "129.0", " ", " ", "bridgeC5"},
                              {"CN1", "NN2G", "HN2", "45.0", "113.3", " ", " ", "h1"},
                              {"CN2", "NN2G", "HN2", "45.0", "115.6", " ", " ", " "},
                              {"NN1", "CN2", "NN2G", "95.0", "115.4", " ", " ", "n2"},
                              {"NN1", "CN2", "NN3G", "95.0", "122.4", " ", " ", " "},
                              {"NN2G", "CN1", "ON1", "50.0", "127.5", " ", " ", "o6"},
                              {"CN5G", "CN1", "ON1", "50.0", "124.7", " ", " ", " "},
                              {"HN3", "CN4", "NN2B", "40.0", "122.2", " ", " ", "h8 (NN4 CN4 HN3 124.8)"},
                              {"CN4", "NN2B", "HN2", "30.0", "124.6", " ", " ", "h9"},
                              {"CN5", "NN2B", "HN2", "30.0", "129.3", " ", " ", " "},
                              {"CN1", "NN2", "CN3", "50.0", "124.1", " ", " ", "adm jr. 11/97"},
                              {"NN2", "CN1", "NN3", "50.0", "116.8", " ", " ", " "},
                              {"CN1", "NN3", "CN2", "85.0", "119.1", " ", " ", " "},
                              {"CN3", "CN2", "NN3", "85.0", "119.3", " ", " ", " "},
                              {"CN2", "CN3", "CN3", "85.0", "117.8", " ", " ", " "},
                              {"CN3", "CN3", "NN2", "85.0", "122.9", " ", " ", " "},
                              {"CN1", "NN2", "HN2", "37.0", "121.2", " ", " ", "h1"},
                              {"CN3", "NN2", "HN2", "37.0", "114.7", " ", " ", " "},
                              {"NN2", "CN1", "ON1C", "130.0", "119.4", " ", " ", "o2"},
                              {"NN3", "CN1", "ON1C", "130.0", "123.8", " ", " ", " "},
                              {"NN3", "CN2", "NN1", "81.0", "122.3", " ", " ", "n4"},
                              {"CN3", "CN2", "NN1", "81.0", "118.4", " ", " ", " "},
                              {"CN2", "CN3", "HN3", "38.0", "120.1", " ", " ", "h5"},
                              {"CN3", "CN3", "HN3", "38.0", "122.1", " ", " ", " "},
                              {"HN3", "CN3", "NN2", "44.0", "115.0", " ", " ", "h6"},
                              {"CN1T", "NN2B", "CN3", "70.0", "122.0", " ", " ", "adm jr. 11/97"},
                              {"NN2B", "CN1T", "NN2U", "50.0", "114.0", " ", " ", " "},
                              {"CN1T", "NN2U", "CN1", "50.0", "130.2", " ", " ", " "},
                              {"NN2U", "CN1", "CN3", "70.0", "112.6", " ", " ", " "},
                              {"CN1", "CN3", "CN3", "100.0", "117.6", " ", " ", " "},
                              {"CN3", "CN3", "NN2B", "100.0", "123.6", " ", " ", " "},
                              {"CN1T", "NN2B", "HN2", "40.5", "122.0", " ", " ", "h1"},
                              {"CN3", "NN2B", "HN2", "32.0", "116.0", " ", " ", " "},
                              {"NN2B", "CN1T", "ON1", "100.0", "121.6", " ", " ", "o2"},
                              {"NN2U", "CN1T", "ON1", "100.0", "124.4", " ", " ", " "},
                              {"CN1T", "NN2U", "HN2", "40.5", "114.4", " ", " ", "h3"},
                              {"CN1", "NN2U", "HN2", "40.5", "115.4", " ", " ", " "},
                              {"NN2U", "CN1", "ON1", "100.0", "121.9", " ", " ", "o4"},
                              {"CN3", "CN1", "ON1", "100.0", "125.5", " ", " ", " "},
                              {"CN1", "CN3", "HN3", "30.0", "120.3", " ", " ", "h5"},
                              {"HN3", "CN3", "NN2B", "30.0", "114.3", " ", " ", "h6"},
                              {"CN3T", "CN1", "NN2U", "70.0", "113.5", " ", " ", "adm jr. 11/97"},
                              {"CN1", "CN3T", "CN3", "120.0", "116.7", " ", " ", " "},
                              {"CN3T", "CN3", "NN2B", "120.0", "123.6", " ", " ", "!T"},
                              {"CN3T", "CN1", "ON1", "100.0", "124.6", " ", " ", "o4"},
                              {"CN1", "CN3T", "CN9", "38.0", "118.7", " ", " ", "c5 methyl"},
                              {"CN3", "CN3T", "CN9", "38.0", "124.6", " ", " ", " "},
                              {"CN3T", "CN3", "HN3", "30.0", "122.1", " ", " ", "h6"},
                              {"CN1T", "NN2B", "CN9", "70.0", "116.0", " ", " ", "adm jr."},
                              {"CN3", "NN2B", "CN9", "70.0", "122.0", " ", " ", "adm jr. 7/24/91"},
                              {"CN1", "NN2", "CN9", "70.0", "115.4", " ", " ", "adm jr."},
                              {"CN3", "NN2", "CN9", "70.0", "120.5", " ", " ", "adm jr. 7/24/91"},
                              {"CN5", "NN2", "CN9", "70.0", "125.9", " ", " ", "adm jr."},
                              {"CN4", "NN2", "CN9", "70.0", "127.8", " ", " ", "adm jr."},
                              {"CN5", "NN2B", "CN9", "70.0", "125.9", " ", " ", "adm jr."},
                              {"CN4", "NN2B", "CN9", "70.0", "126.9", " ", " ", "adm jr."},
                              {"CN5", "NN2B", "CN8", "70.0", "125.9", " ", " ", "adm jr."},
                              {"CN4", "NN2B", "CN8", "70.0", "126.9", " ", " ", "adm jr."},
                              {"NN2B", "CN8", "CN9", "70.0", "113.7", " ", " ", "adm jr."},
                              {"CN1T", "NN2B", "CN7B", "45.0", "118.4", " ", " ", "FC from A"},
                              {"CN3", "NN2B", "CN7B", "45.0", "119.6", " ", " ", " "},
                              {"CN1", "NN2", "CN7B", "45.0", "120.0", " ", " ", "FC from A"},
                              {"CN3", "NN2", "CN7B", "45.0", "115.9", " ", " ", " "},
                              {"CN5", "NN2", "CN7B", "45.0", "126.1", " ", " ", " "},
                              {"CN4", "NN2", "CN7B", "45.0", "127.6", " ", " ", " "},
                              {"CN5", "NN2B", "CN7B", "45.0", "126.5", " ", " ", " "},
                              {"CN4", "NN2B", "CN7B", "45.0", "126.3", " ", " ", " "},
                              {"ON6", "CN7B", "NN2", "110.0", "108.0", " ", " ", "DNA"},
                              {"ON6B", "CN7B", "NN2", "110.0", "112.0", " ", " ", "RNA"},
                              {"CN8", "CN7B", "NN2", "110.0", "113.7", " ", " ", " "},
                              {"CN7B", "CN7B", "NN2", "110.0", "111.0", " ", " ", "RNA"},
                              {"ON6", "CN7B", "NN2B", "110.0", "108.0", " ", " ", "(DNA) FC from A"},
                              {"ON6B", "CN7B", "NN2B", "110.0", "112.0", " ", " ", "(RNA) FC from A"},
                              {"CN8", "CN7B", "NN2B", "110.0", "113.7", " ", " ", " "},
                              {"CN7B", "CN7B", "NN2B", "110.0", "111.0", " ", " ", "RNA"},
                              {"HN7", "CN7B", "NN2", "43.0", "111.0", " ", " ", " "},
                              {"HN7", "CN7B", "NN2B", "43.0", "111.0", " ", " ", "HN7 CN7B NN2"},
                              {"CN9", "CN8", "HN8", "34.6", "110.10", "22.53", "2.179", "Alkanes, sacred"},
                              {"CN9", "CN7", "HN7", "34.6", "110.10", "22.53", "2.179", "Alkanes, sacred"},
                              {"HN8", "CN8", "NN2", "33.43", "110.1", " ", " ", "9-M-ADE(THY), ADM"},
                              {"HN8", "CN8", "ON5", "45.9", "108.89", " ", " ", "Adm Jr. MeOH"},
                              {"CN3", "CN9", "HN9", "33.43", "110.10", "22.53", "2.179", "Alkanes, sacred"},
                              {"CN3T", "CN9", "HN9", "33.43", "110.10", "22.53", "2.179", "Alkanes, sacred"},
                              {"CN8", "CN9", "HN9", "34.60", "110.10", "22.53", "2.179", "Alkanes, sacred"},
                              {"HN9", "CN9", "CN7", "33.43", "110.1", "22.53", "2.179", "Alkanes, sacred"},
                              {"HN9", "CN9", "NN2", "33.43", "110.1", " ", " ", "9-M-A(T), adm jr."},
                              {"HN9", "CN9", "NN2B", "33.43", "110.1", " ", " ", "9-M-G(C), adm jr."},
                              {"HN8", "CN8", "NN2B", "33.43", "110.1", " ", " ", "9-E-G, adm jr."},
                              {"HN9", "CN9", "ON2", "60.0", "109.5", " ", " ", "ADM Jr."},
                              {"CN9", "ON2", "P", "20.0", "120.0", "35.", "2.33", "ADM Jr."},
                              {"HN4", "ON4", "P", "30.0", "115.0", "40.0", "2.35", "ADM Jr."},
                              {"HN4", "ON4", "P2", "30.0", "115.0", "40.0", "2.35", "ADM Jr. , adm, 2011 DNA update"},
                              {"HN5", "ON5", "CN8", "57.5", "106.0", " ", " ", "Adm Jr. MeOH"},
                              {"HN5", "ON5", "CN9", "57.5", "106.0", " ", " ", "Adm Jr. MeOH"},
                              {"ON2", "P", "ON2", "80.0", "104.3", " ", " ", "ADM Jr."},
                              {"ON2", "P2", "ON2", "80.0", "104.3", " ", " ", "ADM Jr., adm, 2011 DNA update"},
                              {"ON2", "P", "ON4", "48.1", "108.0", " ", " ", "ADM Jr."},
                              {"ON2", "P2", "ON4", "48.1", "108.0", " ", " ", "ADM Jr., adm, 2011 DNA update"},
                              {"ON3", "P", "ON4", "98.9", "108.23", " ", " ", "ADM Jr."},
                              {"ON3", "P2", "ON4", "98.9", "108.23", " ", " ", "ADM Jr., adm, 2011 DNA update"},
                              {"ON4", "P", "ON4", "98.9", "104.0", " ", " ", "ADM Jr."},
                              {"ON4", "P2", "ON4", "98.9", "104.0", " ", " ", "ADM Jr., adm, 2011 DNA update"},
                              {"CN7", "CN8", "ON5", "75.7", "110.10", " ", " ", "adm jr. MeOH"},
                              {"HN9", "CN9", "HN9", "35.500", "108.40", "5.40", "1.802", "update, adm jr. 3/2/92"},
                              {"CN7", "ON6", "CN7B", "110.0", "108.0", " ", " ", "NF, 11/97, C4'O4'C1'"},
                              {"ON6", "CN7B", "CN8", "90.0", "102.0", " ", " ", "NF, 11/97, C4'O4'C1'"},
                              {"CN7B", "CN8", "CN7", "80.00", "100.0", " ", " ", "NF, 11/97, C1'C2'C3'"},
                              {"CN8", "CN7", "CN7", "60.00", "102.0", "8.0", "2.561", "11/97, C2'C3'C4'"},
                              {"CN9", "CN7", "CN7", "60.00", "102.0", "8.0", "2.561", "5MET, adm jr."},
                              {"CN7", "CN7", "ON6", "100.0", "104.0", " ", " ", "NF, 11/97, C3'C4'O4'"},
                              {"HN7", "CN7", "ON6", "45.2", "107.24", " ", " ", " "},
                              {"HN7", "CN7B", "ON6", "45.2", "107.24", " ", " ", " "},
                              {"HN7", "CN7", "CN7", "40.0", "108.00", " ", " ", " "},
                              {"CN7B", "CN8", "HN8", "33.4", "110.10", "22.53", "2.179", "following terms directly"},
                              {"CN8", "CN7B", "HN7", "33.4", "110.10", "22.53", "2.179", "from alkanes"},
                              {"HN7", "CN7", "CN8", "34.5", "110.1", "22.53", "2.179", " "},
                              {"HN8", "CN8", "CN7", "34.53", "110.10", "22.53", "2.179", " "},
                              {"HN8", "CN8", "CN8", "34.53", "110.10", "22.53", "2.179", " "},
                              {"HN8", "CN8", "HN8", "35.5", "109.00", "5.40", "1.802", " "},
                              {"HN7", "CN7", "HN7", "35.5", "109.00", "5.40", "1.802", " "},
                              {"CN7", "ON6B", "CN7B", "110.0", "115.0", " ", " ", "From CN7 ON6 CN7B"},
                              {"CN7", "CN7", "ON6B", "100.0", "110.0", " ", " ", "From CN7 CN7 ON6"},
                              {"ON6B", "CN7B", "CN7B", "90.0", "106.0", " ", " ", "030998"},
                              {"CN7B", "CN7B", "CN7", "110.0", "96.0", " ", " ", " "},
                              {"CN7B", "CN7", "CN7", "60.0", "100.0", "8.00", "2.561", "11/97, C2'C3'C4'"},
                              {"HN7", "CN7", "ON6B", "45.2", "107.24", " ", " ", " "},
                              {"HN7", "CN7B", "ON6B", "45.2", "107.24", " ", " ", " "},
                              {"CN7B", "CN7B", "HN7", "33.4", "110.10", "22.53", "2.179", "following terms directly"},
                              {"HN7", "CN7B", "HN7", "35.5", "109.00", "5.40", "1.802", " "},
                              {"ON6", "CN7", "CN8B", "90.0", "108.2", " ", " ", "11/97, O4'C4'C5'"},
                              {"ON6", "CN7", "CN9", "90.0", "108.2", " ", " ", "5MET, adm jr."},
                              {"CN7", "CN7", "CN8B", "45.0", "110.0", " ", " ", "11/97, C3'C4'C5'"},
                              {"CN8", "CN7", "CN8B", "58.35", "113.60", "11.16", "2.561", "from alkane, 25P1"},
                              {"CN7", "CN8B", "ON2", "70.0", "108.4", " ", " ", "11/97, C4'C5'O5'"},
                              {"CN7", "CN7", "ON2", "115.0", "109.7", " ", " ", "11/97, C4'C3'O3'"},
                              {"CN7B", "CN7B", "ON2", "115.0", "109.7", " ", " ", "11/97, C4'C3'O3' for NADPH and bkbmod"},
                              {"CN8", "CN7", "ON2", "115.0", "109.7", " ", " ", "11/97, C2'C3'O3'"},
                              {"CN8B", "ON2", "P", "20.0", "120.0", "35.00", "2.33", "11/97, C5'O5'P"},
                              {"CN8B", "ON2", "P2", "20.0", "120.0", "35.00", "2.33", "11/97, C5'O5'P, adm, 2011 DNA update"},
                              {"CN7", "ON2", "P", "20.0", "120.0", "35.00", "2.33", "11/97, C3'O3'P"},
                              {"CN7", "ON2", "P2", "20.0", "120.0", "35.00", "2.33", "11/97, C3'O3'P, adm, 2011 DNA update"},
                              {"CN7B", "ON2", "P", "20.0", "120.0", "35.00", "2.33", "11/97, C3'O3'P, for NADPH and bkbmod"},
                              {"CN7B", "ON2", "P2", "20.0", "120.0", "35.00", "2.33", "11/97, C3'O3'P, for NADPH and bkbmod, adm, 2011 DNA update"},
                              {"HN7", "CN7", "CN8B", "34.5", "110.1", "22.53", "2.179", "From HN7 CN7 CN8"},
                              {"HN8", "CN8B", "ON2", "60.0", "109.5", " ", " ", "From HN7 CN8 ON2"},
                              {"HN5", "ON5", "CN8B", "57.5", "106.0", " ", " ", "From HN5 ON5 CN8"},
                              {"HN8", "CN8B", "HN8", "35.5", "109.0", "5.40", "1.802", "Alkanes, sacred"},
                              {"HN8", "CN8B", "CN7", "34.53", "110.1", "22.53", "2.179", "Alkanes, sacred"},
                              {"HN7", "CN7", "ON2", "60.0", "109.5", " ", " ", "adm jr. from HN7 CN8 ON2"},
                              {"HN7", "CN7B", "ON2", "60.0", "109.5", " ", " ", "adm jr. from HN7 CN8 ON2, for NADPH and bkbmod"},
                              {"CN7", "CN8B", "ON5", "75.7", "110.10", " ", " ", "From CN7 CN8B ON5"},
                              {"CN8B", "CN7", "ON5", "90.0", "108.2", " ", " ", "phosphoramidate, carbocyclic"},
                              {"HN8", "CN8B", "ON5", "45.9", "108.89", " ", " ", "From HN7 CN8 ON5"},
                              {"ON5", "CN7", "CN8", "75.7", "110.0", " ", " ", "from CHARMM22"},
                              {"ON5", "CN7", "CN7", "75.7", "110.1", " ", " ", " "},
                              {"HN7", "CN7", "ON5", "60.0", "109.5", " ", " ", " "},
                              {"HN5", "ON5", "CN7", "57.5", "109.0", " ", " ", " "},
                              {"ON6B", "CN7", "CN8B", "90.0", "108.2", " ", " ", " "},
                              {"ON6B", "CN7", "CN9", "90.0", "108.2", " ", " ", "for 5MET patch, adm jr."},
                              {"ON2", "CN7", "CN7B", "90.0", "110.0", " ", " ", " "},
                              {"ON5", "CN7", "CN7B", "90.0", "110.0", " ", " ", "From ON5 CN7 CN8"},
                              {"ON5", "CN7B", "CN7B", "80.0", "108.4", " ", " ", " "},
                              {"ON5", "CN7B", "CN7", "90.0", "108.0", " ", " ", " "},
                              {"HN7", "CN7B", "ON5", "60.0", "109.5", " ", " ", " "},
                              {"HN5", "ON5", "CN7B", "57.5", "109.0", " ", " ", " "},
                              {"HN7", "CN7B", "CN7", "34.53", "110.10", "22.53", "2.179", " "},
                              {"HN7", "CN7", "CN7B", "34.53", "110.10", "22.53", "2.179", " "},
                              {"CN8", "ON2", "P", "20.0", "120.0", "35.", "2.33", "adm jr."},
                              {"CN8", "ON2", "P2", "20.0", "120.0", "35.", "2.33", "adm jr., adm 2011 DNA update"},
                              {"ON2", "P", "ON3", "98.9", "111.6", " ", " ", "adm jr."},
                              {"ON2", "P2", "ON3", "98.9", "111.6", " ", " ", "adm jr., adm, 2011 DNA update"},
                              {"ON3", "P", "ON3", "120.0", "120.0", " ", " ", "adm jr."},
                              {"ON3", "P2", "ON3", "120.0", "120.0", " ", " ", "adm jr., adm, 2011 DNA update"},
                              {"HN8", "CN8", "ON2", "60.0", "109.5", " ", " ", "adm jr."},
                              {"ON5", "P", "ON3", "98.9", "111.6", " ", " ", "From ON2 P ON3"},
                              {"ON6", "CN7B", "CN7", "120.0", "106.25", " ", " ", " "},
                              {"CN7B", "CN7", "CN8", "58.35", "113.6", "11.16", "2.561", " "}};

/*
 Dihedrals
  0-3= Keys, 4= Kchi (kcal mol^-1), 5= n (multi), 6= delta (deg), 7= charmm36_na info

  V(chi) = Kchi x (1 + cos (n x (chi) - delta))
*/
char * charmm36_na_dihedrals[502][8]= {{"X", "CN8", "ON2", "X", "-0.10", "3", "0.0", "phosphate ester"},
                                 {"X", "CN7", "CN8", "X", "0.20", "3", "0.0", "alkane"},
                                 {"X", "CN8", "NN6", "X", "0.10", "3", "0.0", "methylammonium"},
                                 {"CN7", "ON6", "CN8B", "HN8", "0.195", "1", "0.0", " "},
                                 {"ON6", "CN8B", "CN8", "HN8", "0.195", "1", "0.0", " "},
                                 {"HN7", "CN7", "ON6", "CN8B", "0.195", "3", "0.0", " "},
                                 {"CN8B", "CN8", "CN7", "HN7", "0.195", "3", "0.0", " "},
                                 {"HN8", "CN8B", "CN8", "HN8", "0.195", "3", "0.0", " "},
                                 {"HN8", "CN8B", "CN8", "CN7", "0.195", "3", "0.0", " "},
                                 {"CN8B", "CN7", "ON6", "CN8B", "0.5", "5", "0.0", "min at 150 310 max at 25 200"},
                                 {"CN8B", "CN7", "ON6", "CN8B", "0.1", "3", "180.0", " "},
                                 {"CN8B", "CN7", "ON6", "CN8B", "0.5", "1", "0.0", " "},
                                 {"CN8B", "CN8", "CN7", "ON5", "0.4", "5", "0.0", " "},
                                 {"CN8B", "CN8", "CN7", "ON5", "0.4", "3", "0.0", " "},
                                 {"CN8B", "CN8", "CN7", "ON5", "0.7", "2", "0.0", "energy at 0 (c3'endo), adm"},
                                 {"CN8B", "CN8", "CN7", "ON5", "0.5", "1", "180.0", "energy at 0 (c3'endo), adm"},
                                 {"CN8B", "CN8", "CN7", "ON2", "0.4", "5", "0.0", "for oligonuclotide"},
                                 {"CN8B", "CN8", "CN7", "ON2", "0.4", "3", "0.0", " "},
                                 {"CN8B", "CN8", "CN7", "ON2", "0.7", "2", "0.0", " "},
                                 {"CN8B", "CN8", "CN7", "ON2", "0.5", "1", "180.0", " "},
                                 {"CN7", "ON6", "CN8B", "CN8", "0.6", "6", "180.0", " "},
                                 {"CN7", "ON6", "CN8B", "CN8", "0.6", "3", "0.0", " "},
                                 {"ON6", "CN8B", "CN8", "CN7", "0.7", "5", "180.0", "90, shifts c2endo minimum towards 200"},
                                 {"ON6", "CN8B", "CN8", "CN7", "0.4", "4", "0.0", "90, shifts c2endo minimum towards 200"},
                                 {"ON6", "CN8B", "CN8", "CN7", "0.4", "3", "180.0", "90, shifts c2endo minimum towards 200"},
                                 {"CN7", "CN7", "CN8", "CN8B", "0.5", "4", "0.0", " "},
                                 {"CN7", "CN7", "CN8", "CN8B", "0.1", "3", "0.0", "energy in 150 to 250 range"},
                                 {"CN8B", "ON6", "CN7", "CN7", "0.5", "3", "0.0", "surface in 200-360 region"},
                                 {"ON2", "P2", "ON2", "CN7", "0.90", "1", "180.0", "adm, 2011 DNA update new param, zeta, kat2 set"},
                                 {"ON2", "P2", "ON2", "CN7", "0.40", "2", "180.0", "adm, 2011 DNA update new param, zeta, kat2 set"},
                                 {"ON2", "P2", "ON2", "CN7", "0.20", "3", "180.0", "adm, 2011 DNA update new param, zeta, kat2 set"},
                                 {"ON2", "P", "ON2", "CN7", "1.20", "1", "180.0", "DMP, adm jr., RNA"},
                                 {"ON2", "P", "ON2", "CN7", "0.10", "2", "180.0", "DMP, adm jr., RNA"},
                                 {"ON2", "P", "ON2", "CN7", "0.10", "3", "180.0", "DMP, adm jr., RNA"},
                                 {"ON2", "P", "ON2", "CN7", "0.00", "6", "0.0", "DMP, adm jr., RNA"},
                                 {"ON2", "P", "ON2", "CN8", "1.20", "1", "180.0", "DMP, adm jr."},
                                 {"ON2", "P", "ON2", "CN8", "0.10", "2", "180.0", "DMP, adm jr."},
                                 {"ON2", "P", "ON2", "CN8", "0.10", "3", "180.0", "DMP, adm jr."},
                                 {"ON2", "P", "ON2", "CN8", "0.00", "6", "0.0", "DMP, adm jr."},
                                 {"ON2", "P2", "ON2", "CN8", "1.20", "1", "180.0", "DMP, adm jr., adm 2011 DNA update"},
                                 {"ON2", "P2", "ON2", "CN8", "0.10", "2", "180.0", "DMP, adm jr., adm 2011 DNA update"},
                                 {"ON2", "P2", "ON2", "CN8", "0.10", "3", "180.0", "DMP, adm jr., adm 2011 DNA update"},
                                 {"ON2", "P2", "ON2", "CN8", "0.00", "6", "0.0", "DMP, adm jr., adm 2011 DNA update"},
                                 {"ON2", "P2", "ON2", "CN8B", "1.20", "1", "180.0", "DMP, adm jr., adm, 2011 DNA update"},
                                 {"ON2", "P2", "ON2", "CN8B", "0.10", "2", "180.0", "DMP, adm jr., adm, 2011 DNA update"},
                                 {"ON2", "P2", "ON2", "CN8B", "0.10", "3", "180.0", "DMP, adm jr., adm, 2011 DNA update"},
                                 {"ON2", "P2", "ON2", "CN8B", "0.00", "6", "0.0", "DMP, adm jr., adm, 2011 DNA update"},
                                 {"ON2", "P", "ON2", "CN8B", "1.20", "1", "180.0", "DMP, adm jr., RNA"},
                                 {"ON2", "P", "ON2", "CN8B", "0.10", "2", "180.0", "DMP, adm jr., RNA"},
                                 {"ON2", "P", "ON2", "CN8B", "0.10", "3", "180.0", "DMP, adm jr., RNA"},
                                 {"ON2", "P", "ON2", "CN8B", "0.00", "6", "0.0", "DMP, adm jr., RNA"},
                                 {"ON2", "P", "ON2", "CN9", "1.20", "1", "180.0", " "},
                                 {"ON2", "P", "ON2", "CN9", "0.10", "2", "180.0", " "},
                                 {"ON2", "P", "ON2", "CN9", "0.10", "3", "180.0", " "},
                                 {"ON2", "P", "ON2", "CN9", "0.00", "6", "0.0", " "},
                                 {"ON2", "P2", "ON2", "CN9", "1.20", "1", "180.0", "adm, 2011 DNA update"},
                                 {"ON2", "P2", "ON2", "CN9", "0.10", "2", "180.0", "adm, 2011 DNA update"},
                                 {"ON2", "P2", "ON2", "CN9", "0.10", "3", "180.0", "adm, 2011 DNA update"},
                                 {"ON2", "P2", "ON2", "CN9", "0.00", "6", "0.0", "adm, 2011 DNA update"},
                                 {"ON3", "P", "ON2", "CN7", "0.10", "3", "0.0", "O1P-P-O3'-C3'"},
                                 {"ON3", "P2", "ON2", "CN7", "0.10", "3", "0.0", "O1P-P-O3'-C3', adm, 2011 DNA update"},
                                 {"ON3", "P", "ON2", "CN7B", "0.10", "3", "0.0", "NADPH and bkbmod"},
                                 {"ON3", "P", "ON2", "CN8", "0.10", "3", "0.0", " "},
                                 {"ON3", "P2", "ON2", "CN8", "0.10", "3", "0.0", "adm 2011 DNA update"},
                                 {"ON3", "P", "ON2", "CN8B", "0.10", "3", "0.0", "O1P-P-O5'-C5'"},
                                 {"ON3", "P2", "ON2", "CN8B", "0.10", "3", "0.0", "O1P-P-O5'-C5', adm, 2011 DNA update"},
                                 {"ON3", "P", "ON2", "CN9", "0.10", "3", "0.0", " "},
                                 {"ON3", "P2", "ON2", "CN9", "0.10", "3", "0.0", "adm, 2011 DNA update"},
                                 {"ON4", "P", "ON2", "CN7", "0.95", "2", "0.0", "adm jr."},
                                 {"ON4", "P", "ON2", "CN7", "0.50", "3", "0.0", "adm jr."},
                                 {"ON4", "P2", "ON2", "CN7", "0.95", "2", "0.0", "adm jr., adm, 2011 DNA update"},
                                 {"ON4", "P2", "ON2", "CN7", "0.50", "3", "0.0", "adm jr., adm, 2011 DNA update"},
                                 {"ON4", "P", "ON2", "CN8", "0.95", "2", "0.0", "adm jr."},
                                 {"ON4", "P", "ON2", "CN8", "0.50", "3", "0.0", "adm jr."},
                                 {"ON4", "P", "ON2", "CN8B", "0.95", "2", "0.0", "adm jr."},
                                 {"ON4", "P", "ON2", "CN8B", "0.50", "3", "0.0", "adm jr."},
                                 {"ON4", "P2", "ON2", "CN8B", "0.95", "2", "0.0", "adm jr., adm, 2011 DNA update"},
                                 {"ON4", "P2", "ON2", "CN8B", "0.50", "3", "0.0", "adm jr., adm, 2011 DNA update"},
                                 {"ON4", "P", "ON2", "CN9", "0.95", "2", "0.0", "adm jr."},
                                 {"ON4", "P", "ON2", "CN9", "0.50", "3", "0.0", "adm jr."},
                                 {"ON4", "P2", "ON2", "CN9", "0.95", "2", "0.0", "adm jr., adm, 2011 DNA update"},
                                 {"ON4", "P2", "ON2", "CN9", "0.50", "3", "0.0", "adm jr., adm, 2011 DNA update"},
                                 {"X", "ON4", "P", "X", "0.30", "3", "0.0", "adm jr."},
                                 {"X", "ON4", "P2", "X", "0.30", "3", "0.0", "adm jr., adm, 2011 DNA update"},
                                 {"P", "ON2", "CN7", "HN7", "0.000", "3", "0.0", "H-C3'-O3'-P"},
                                 {"P2", "ON2", "CN7", "HN7", "0.000", "3", "0.0", "H-C3'-O3'-P, adm, 2011 DNA update"},
                                 {"P", "ON2", "CN7B", "HN7", "0.000", "3", "0.0", "NADPH and bkbmod"},
                                 {"P", "ON2", "CN8B", "HN8", "0.000", "3", "0.0", "H-C5'-O5'-P"},
                                 {"P2", "ON2", "CN8B", "HN8", "0.000", "3", "0.0", "H-C5'-O5'-P, adm, 2011 DNA update"},
                                 {"P", "ON2", "CN8", "HN8", "0.000", "3", "0.0", " "},
                                 {"P", "ON2", "CN9", "HN9", "0.000", "3", "0.0", " "},
                                 {"P2", "ON2", "CN9", "HN9", "0.000", "3", "0.0", "adm, 2011 DNA update"},
                                 {"CN9", "CN8", "CN8", "CN9", "0.15", "1", "0.0", " "},
                                 {"CN9", "CN8", "CN8", "CN8", "0.15", "1", "0.0", " "},
                                 {"NN2B", "CN1T", "NN2U", "CN1", "1.5", "2", "180.0", "adm jr. 11/97"},
                                 {"CN1T", "NN2U", "CN1", "CN3", "1.5", "2", "180.0", "adm jr. 11/97"},
                                 {"NN2U", "CN1", "CN3", "CN3", "1.5", "2", "180.0", "adm jr. 11/97"},
                                 {"CN1", "CN3", "CN3", "NN2B", "6.0", "2", "180.0", "adm jr. 11/97"},
                                 {"CN3", "CN3", "NN2B", "CN1T", "1.5", "2", "180.0", "adm jr. 11/97"},
                                 {"CN3", "NN2B", "CN1T", "NN2U", "1.5", "2", "180.0", "adm jr. 11/97"},
                                 {"HN3", "CN3", "CN3", "HN3", "3.0", "2", "180.0", "adm jr. 11/97"},
                                 {"HN3", "CN3", "CN1", "ON1", "6.0", "2", "180.0", "adm jr. 11/97"},
                                 {"ON1", "CN1T", "NN2B", "HN2", "0.0", "2", "180.0", "adm jr. 11/97"},
                                 {"ON1", "CN1", "NN2U", "HN2", "0.0", "2", "180.0", "adm jr. 11/97"},
                                 {"ON1", "CN1T", "NN2U", "HN2", "0.0", "2", "180.0", "adm jr. 11/97"},
                                 {"HN2", "NN2B", "CN3", "HN3", "1.5", "2", "180.0", "adm jr. 11/97"},
                                 {"NN2B", "CN1T", "NN2U", "HN2", "3.8", "2", "180.0", "adm jr. 11/97"},
                                 {"CN3", "CN1", "NN2U", "HN2", "3.8", "2", "180.0", "adm jr. 11/97"},
                                 {"CN3", "CN3", "NN2B", "HN2", "1.6", "2", "180.0", "adm jr. 11/97"},
                                 {"NN2U", "CN1T", "NN2B", "HN2", "1.6", "2", "180.0", "adm jr. 11/97"},
                                 {"CN1T", "NN2B", "CN3", "CN3T", "1.8", "2", "180.0", "adm jr. 11/97"},
                                 {"NN2U", "CN1", "CN3T", "CN3", "1.8", "2", "180.0", "adm jr. 11/97"},
                                 {"CN1", "CN3T", "CN3", "NN2B", "3.0", "2", "180.0", "adm jr. 11/97"},
                                 {"NN2B", "CN1", "CN3T", "CN9", "5.6", "2", "180.0", "adm jr. 11/97"},
                                 {"NN2B", "CN3", "CN3T", "CN9", "5.6", "2", "180.0", "adm jr. 11/97"},
                                 {"CN1", "CN3T", "CN9", "HN9", "0.46", "3", "0.0", "adm jr. 11/97"},
                                 {"CN3", "CN3T", "CN9", "HN9", "0.46", "3", "0.0", "adm jr. 11/97"},
                                 {"CN3T", "CN1", "NN2U", "HN2", "4.8", "2", "180.0", "adm jr. 11/97"},
                                 {"CN3", "NN2", "CN1", "NN3", "0.6", "2", "180.0", "adm jr. 11/97"},
                                 {"NN2", "CN1", "NN3", "CN2", "0.6", "2", "180.0", "adm jr. 11/97"},
                                 {"CN1", "NN3", "CN2", "CN3", "6.0", "2", "180.0", "adm jr. 11/97"},
                                 {"NN3", "CN2", "CN3", "CN3", "0.6", "2", "180.0", "adm jr. 11/97"},
                                 {"CN2", "CN3", "CN3", "NN2", "6.0", "2", "180.0", "adm jr. 11/97"},
                                 {"CN3", "CN3", "NN2", "CN1", "0.6", "2", "180.0", "adm jr. 11/97"},
                                 {"NN3", "CN2", "NN1", "HN1", "1.0", "2", "180.0", "adm jr. 11/97"},
                                 {"CN3", "CN2", "NN1", "HN1", "1.0", "2", "180.0", "adm jr. 11/97"},
                                 {"NN1", "CN2", "NN3", "CN1", "2.0", "2", "180.0", "adm jr. 11/97"},
                                 {"NN1", "CN2", "CN3", "CN3", "2.0", "2", "180.0", "adm jr. 11/97"},
                                 {"NN1", "CN2", "CN3", "HN3", "2.0", "2", "180.0", "adm jr. 11/97"},
                                 {"ON1C", "CN1", "NN2", "HN2", "3.0", "2", "180.0", "adm jr. 11/97"},
                                 {"ON1C", "CN1", "NN3", "CN2", "1.6", "2", "180.0", "adm jr. 11/97"},
                                 {"ON1C", "CN1", "NN2", "CN3", "1.6", "2", "180.0", "adm jr. 11/97"},
                                 {"NN3", "CN2", "CN3", "HN3", "3.4", "2", "180.0", "adm jr. 11/97"},
                                 {"NN2", "CN3", "CN3", "HN3", "3.4", "2", "180.0", "adm jr. 11/97"},
                                 {"CN2", "CN3", "CN3", "HN3", "4.6", "2", "180.0", "adm jr. 11/97"},
                                 {"CN1", "NN2", "CN3", "HN3", "4.6", "2", "180.0", "adm jr. 11/97"},
                                 {"X", "CN2", "NN3", "X", "2.0", "2", "180.0", "adm jr. 11/97"},
                                 {"CN2", "NN3A", "CN4", "NN3A", "1.8", "2", "180.0", "adm jr. 11/97, 6-mem"},
                                 {"NN3A", "CN4", "NN3A", "CN5", "2.0", "2", "180.0", " "},
                                 {"CN4", "NN3A", "CN5", "CN5", "1.8", "2", "180.0", " "},
                                 {"NN3A", "CN5", "CN5", "CN2", "2.0", "2", "180.0", "treated 2x"},
                                 {"CN5", "CN5", "CN2", "NN3A", "1.8", "2", "180.0", " "},
                                 {"CN5", "CN2", "NN3A", "CN4", "10.0", "2", "180.0", " "},
                                 {"CN5", "CN5", "NN4", "CN4", "6.0", "2", "180.0", "5-mem"},
                                 {"CN5", "NN4", "CN4", "NN2", "14.0", "2", "180.0", " "},
                                 {"NN4", "CN4", "NN2", "CN5", "6.0", "2", "180.0", " "},
                                 {"CN4", "NN2", "CN5", "CN5", "6.0", "2", "180.0", " "},
                                 {"NN2", "CN5", "CN5", "NN4", "14.0", "2", "180.0", "treated 2x"},
                                 {"CN2", "NN3A", "CN4", "HN3", "8.5", "2", "180.0", "H2"},
                                 {"CN5", "NN3A", "CN4", "HN3", "8.5", "2", "180.0", "H2"},
                                 {"CN5", "NN4", "CN4", "HN3", "5.2", "2", "180.0", "H8"},
                                 {"CN5", "NN2", "CN4", "HN3", "5.2", "2", "180.0", "H8"},
                                 {"CN5", "CN5", "NN2", "HN2", "1.2", "2", "180.0", "H9"},
                                 {"NN4", "CN4", "NN2", "HN2", "1.2", "2", "180.0", "H9"},
                                 {"HN2", "NN2", "CN4", "HN3", "0.0", "2", "180.0", "H8-C-N-H9"},
                                 {"CN4", "NN3A", "CN2", "NN1", "4.0", "2", "180.0", "N6"},
                                 {"CN5", "CN5", "CN2", "NN1", "4.0", "2", "180.0", "N6"},
                                 {"NN4", "CN5", "CN2", "NN1", "0.0", "2", "180.0", "N6"},
                                 {"CN5", "CN2", "NN1", "HN1", "0.5", "2", "180.0", "6-NH2"},
                                 {"NN3A", "CN2", "NN1", "HN1", "0.5", "2", "180.0", " "},
                                 {"NN3A", "CN5", "CN5", "NN4", "7.0", "2", "180.0", "adm jr. 11/97"},
                                 {"CN2", "CN5", "CN5", "NN2", "7.0", "2", "180.0", " "},
                                 {"NN3A", "CN2", "CN5", "NN4", "2.0", "2", "180.0", " "},
                                 {"CN2", "CN5", "NN4", "CN4", "2.0", "2", "180.0", " "},
                                 {"CN4", "NN3A", "CN5", "NN2", "2.0", "2", "180.0", " "},
                                 {"NN3A", "CN5", "NN2", "CN4", "2.0", "2", "180.0", " "},
                                 {"CN1", "NN2G", "CN2", "NN3G", "0.2", "2", "180.0", "jr. 11/97, 6-mem"},
                                 {"NN2G", "CN2", "NN3G", "CN5", "2.0", "2", "180.0", " "},
                                 {"CN2", "NN3G", "CN5", "CN5G", "0.2", "2", "180.0", " "},
                                 {"NN3G", "CN5", "CN5G", "CN1", "2.0", "2", "180.0", " "},
                                 {"CN5", "CN5G", "CN1", "NN2G", "0.2", "2", "180.0", " "},
                                 {"CN5G", "CN1", "NN2G", "CN2", "0.2", "2", "180.0", " "},
                                 {"CN5", "CN5G", "NN4", "CN4", "6.0", "2", "180.0", " "},
                                 {"CN5G", "NN4", "CN4", "NN2B", "16.0", "2", "180.0", " "},
                                 {"NN4", "CN4", "NN2B", "CN5", "6.0", "2", "180.0", " "},
                                 {"CN4", "NN2B", "CN5", "CN5G", "6.0", "2", "180.0", " "},
                                 {"NN2B", "CN5", "CN5G", "NN4", "10.0", "2", "180.0", " "},
                                 {"ON1", "CN1", "CN5G", "CN5", "14.0", "2", "180.0", "O6"},
                                 {"ON1", "CN1", "CN5G", "NN4", "0.0", "2", "180.0", " "},
                                 {"ON1", "CN1", "NN2G", "CN2", "14.0", "2", "180.0", " "},
                                 {"ON1", "CN1", "NN2G", "HN2", "0.0", "2", "180.0", " "},
                                 {"NN1", "CN2", "NN2G", "CN1", "4.0", "2", "180.0", "N2"},
                                 {"NN1", "CN2", "NN3G", "CN5", "4.0", "2", "180.0", " "},
                                 {"NN1", "CN2", "NN2G", "HN2", "0.0", "2", "180.0", " "},
                                 {"NN2G", "CN2", "NN1", "HN1", "1.2", "2", "180.0", " "},
                                 {"NN3G", "CN2", "NN1", "HN1", "1.2", "2", "180.0", " "},
                                 {"HN2", "NN2G", "CN1", "CN5G", "3.6", "2", "180.0", "H1"},
                                 {"HN2", "NN2G", "CN2", "NN3G", "3.6", "2", "180.0", " "},
                                 {"HN3", "CN4", "NN4", "CN5G", "5.6", "2", "180.0", "H8"},
                                 {"HN3", "CN4", "NN2B", "CN5", "5.6", "2", "180.0", " "},
                                 {"HN3", "CN4", "NN2B", "HN2", "0.0", "2", "180.0", " "},
                                 {"HN2", "NN2B", "CN5", "CN5G", "1.2", "2", "180.0", "H9"},
                                 {"HN2", "NN2B", "CN5", "NN3G", "1.2", "2", "180.0", " "},
                                 {"HN2", "NN2B", "CN4", "NN4", "1.2", "2", "180.0", " "},
                                 {"NN3G", "CN5", "CN5G", "NN4", "10.0", "2", "180.0", "jr. 11/97"},
                                 {"CN1", "CN5G", "CN5", "NN2", "10.0", "2", "180.0", " "},
                                 {"NN2G", "CN1", "CN5G", "NN4", "2.0", "2", "180.0", " "},
                                 {"CN1", "CN5G", "NN4", "CN4", "2.0", "2", "180.0", " "},
                                 {"CN2", "NN3G", "CN5", "NN2B", "2.0", "2", "180.0", " "},
                                 {"NN3G", "CN5", "NN2B", "CN4", "2.0", "2", "180.0", " "},
                                 {"X", "CN1", "NN3", "X", "1.0", "2", "180.0", "c22"},
                                 {"X", "CN1", "NN2", "X", "0.9", "2", "180.0", "c22"},
                                 {"X", "CN1T", "NN2B", "X", "0.9", "2", "180.0", "From X CN1 NN2 X, for thymines"},
                                 {"X", "CN1", "NN2G", "X", "0.9", "2", "180.0", "c22"},
                                 {"X", "CN1", "NN2U", "X", "0.9", "2", "180.0", "c22"},
                                 {"X", "CN1T", "NN2U", "X", "0.9", "2", "180.0", "c22"},
                                 {"X", "CN3", "NN2", "X", "1.0", "2", "180.0", "c22"},
                                 {"X", "CN3", "NN2B", "X", "1.0", "2", "180.0", "From X CN3 NN2 X, for thymines"},
                                 {"X", "CN3", "CN3", "X", "1.0", "2", "180.0", "c22"},
                                 {"X", "CN3", "CN3T", "X", "1.0", "2", "180.0", "adm jr. 11/97"},
                                 {"X", "CN1", "CN3", "X", "1.0", "2", "180.0", "c22"},
                                 {"X", "CN1", "CN3T", "X", "1.0", "2", "180.0", "adm jr. 11/97"},
                                 {"X", "CN2", "CN3", "X", "0.8", "2", "180.0", "c22"},
                                 {"X", "CN1", "CN5G", "X", "1.0", "2", "180.0", "adm jr. 11/97"},
                                 {"X", "CN2", "NN2G", "X", "1.0", "2", "180.0", " "},
                                 {"X", "CN2", "CN5", "X", "1.0", "2", "180.0", " "},
                                 {"X", "CN4", "NN2", "X", "1.5", "2", "180.0", " "},
                                 {"X", "CN4", "NN2B", "X", "1.5", "2", "180.0", "From X CN4 NN2 X"},
                                 {"X", "CN4", "NN3A", "X", "3.5", "2", "180.0", " "},
                                 {"X", "CN4", "NN4", "X", "2.0", "2", "180.0", "A,G"},
                                 {"X", "CN5", "CN5", "X", "0.0", "2", "180.0", " "},
                                 {"X", "CN5G", "CN5", "X", "0.0", "2", "180.0", "adm jr. 11/97"},
                                 {"X", "CN5", "NN2", "X", "1.5", "2", "180.0", " "},
                                 {"X", "CN5", "NN2B", "X", "1.5", "2", "180.0", "From X CN5 NN2 X"},
                                 {"X", "CN5", "NN3A", "X", "1.0", "2", "180.0", " "},
                                 {"X", "CN5", "NN3G", "X", "1.0", "2", "180.0", "adm jr. 11/97"},
                                 {"X", "CN5", "NN4", "X", "1.0", "2", "180.0", " "},
                                 {"X", "CN5G", "NN4", "X", "1.0", "2", "180.0", "adm jr. 11/97"},
                                 {"X", "CN2", "NN3A", "X", "1.0", "2", "180.0", " "},
                                 {"X", "CN2", "NN3G", "X", "1.0", "2", "180.0", "adm jr. 11/97"},
                                 {"CN1", "NN2", "CN9", "HN9", "0.19", "3", "0.0", "1-M-C"},
                                 {"CN3", "NN2", "CN9", "HN9", "0.00", "3", "0.0", "1-M-C"},
                                 {"CN4", "NN2", "CN9", "HN9", "0.00", "3", "0.0", "9-M-A"},
                                 {"CN5", "NN2", "CN9", "HN9", "0.19", "3", "0.0", "9-M-A"},
                                 {"CN1", "NN2B", "CN9", "HN9", "0.19", "3", "0.0", "1-M-U"},
                                 {"CN1T", "NN2B", "CN9", "HN9", "0.19", "3", "0.0", "1-M-T"},
                                 {"CN3", "NN2B", "CN9", "HN9", "0.00", "3", "0.0", "1-M-T/U"},
                                 {"CN4", "NN2B", "CN9", "HN9", "0.00", "3", "0.0", "9-M-G"},
                                 {"CN5", "NN2B", "CN9", "HN9", "0.19", "3", "0.0", "9-M-G"},
                                 {"CN4", "NN2B", "CN8", "HN8", "0.00", "3", "0.0", "9-E-G"},
                                 {"CN5", "NN2B", "CN8", "HN8", "0.19", "3", "0.0", "9-E-G"},
                                 {"CN4", "NN2B", "CN8", "CN9", "0.00", "3", "0.0", "9-E-G"},
                                 {"CN5", "NN2B", "CN8", "CN9", "0.19", "3", "0.0", "9-E-G"},
                                 {"X", "CN8", "CN8", "X", "0.15", "3", "0.0", "Alkanes (0.2 to 0.15)"},
                                 {"X", "CN8", "CN9", "X", "0.15", "3", "0.0", "Alkanes (0.2 to 0.15)"},
                                 {"HN7", "CN7B", "CN7B", "ON2", "0.195", "3", "0.0", "NADPH and bkbmod"},
                                 {"ON2", "CN7B", "CN7B", "NN2", "0.0", "3", "0.0", "NADPH and bkbmod"},
                                 {"CN7", "CN7B", "ON6", "CN7", "0.6", "6", "180.0", " "},
                                 {"CN7B", "CN7", "CN7", "CN7", "0.4", "6", "0.0", "good for amplitudes"},
                                 {"CN7B", "CN7", "CN7", "CN9", "0.4", "6", "0.0", "good for amplitudes, 5MET"},
                                 {"CN7", "CN7", "CN7", "ON6", "0.6", "6", "0.0", " "},
                                 {"CN7", "CN7", "CN7B", "ON6", "0.6", "6", "0.0", " "},
                                 {"ON2", "CN7", "CN7", "CN7", "0.8", "6", "0.0", " "},
                                 {"ON2", "CN7", "CN7", "CN7", "0.4", "5", "0.0", "Moves the barrier right"},
                                 {"ON2", "CN7", "CN7", "CN7", "2.0", "3", "180.0", " "},
                                 {"ON2", "CN7B", "CN7", "CN7", "0.8", "6", "0.0", " "},
                                 {"ON2", "CN7B", "CN7", "CN7", "0.4", "5", "0.0", "Moves the barrier right"},
                                 {"ON2", "CN7B", "CN7", "CN7", "2.0", "3", "180.0", " "},
                                 {"ON5", "CN7", "CN7", "CN7", "0.8", "6", "0.0", " "},
                                 {"ON5", "CN7", "CN7", "CN7", "0.4", "5", "0.0", "Moves the barrier right"},
                                 {"ON5", "CN7", "CN7", "CN7", "2.0", "3", "180.0", " "},
                                 {"ON5", "CN7", "CN7", "ON5", "0.0", "3", "0.0", " "},
                                 {"ON5", "CN7", "CN7", "ON2", "0.0", "3", "0.0", " "},
                                 {"ON2", "CN7", "CN7B", "ON6", "0.5", "6", "0.0", "for amplitudes"},
                                 {"ON2", "CN7", "CN7B", "ON6", "0.3", "5", "0.0", "on amplitudes"},
                                 {"ON2", "CN7", "CN7B", "ON6", "0.6", "4", "180.0", "c2'endo"},
                                 {"ON2", "CN7", "CN7B", "ON6", "0.2", "3", "0.0", " "},
                                 {"CN7", "CN7", "CN7", "CN8B", "0.5", "4", "180.0", "lowers 180 deg."},
                                 {"CN7B", "NN2", "CN4", "HN3", "0.3", "2", "180.0", "NF"},
                                 {"CN7B", "NN2", "CN5", "CN5", "11.0", "2", "180.0", "adm jr."},
                                 {"CN7B", "NN2", "CN4", "NN4", "11.0", "2", "180.0", "adm jr."},
                                 {"CN7B", "NN2", "CN4", "NN3A", "11.0", "2", "180.0", "adm jr."},
                                 {"ON6", "CN7B", "NN2", "CN5", "1.1", "1", "180.0", " "},
                                 {"ON6", "CN7B", "NN2", "CN4", "1.1", "1", "0.0", "NF"},
                                 {"ON6B", "CN7B", "NN2", "CN5", "1.1", "1", "180.0", " "},
                                 {"ON6B", "CN7B", "NN2", "CN4", "1.1", "1", "0.0", " "},
                                 {"CN8", "CN7B", "NN2", "CN5", "0.3", "3", "0.0", "NF"},
                                 {"CN8", "CN7B", "NN2", "CN4", "0.0", "3", "180.0", "NF"},
                                 {"CN7B", "CN7B", "NN2", "CN5", "0.3", "3", "0.0", "NF"},
                                 {"CN7B", "CN7B", "NN2", "CN4", "0.0", "3", "180.0", "NF"},
                                 {"HN7", "CN7B", "NN2", "CN5", "0.0", "3", "0.0", "NF"},
                                 {"HN7", "CN7B", "NN2", "CN4", "0.195", "3", "0.0", "NF"},
                                 {"CN7B", "NN2", "CN3", "HN3", "0.3", "2", "180.0", "NF"},
                                 {"CN7B", "NN2", "CN1", "ON1C", "11.0", "2", "180.0", "adm jr. from A"},
                                 {"CN7B", "NN2", "CN1", "NN3", "11.0", "2", "180.0", "adm jr."},
                                 {"CN7B", "NN2", "CN3", "CN3", "11.0", "2", "180.0", "adm jr."},
                                 {"ON6", "CN7B", "NN2", "CN1", "0.0", "3", "0.0", " "},
                                 {"ON6", "CN7B", "NN2", "CN3", "1.0", "1", "0.0", "NF"},
                                 {"ON6B", "CN7B", "NN2", "CN1", "0.0", "3", "0.0", " "},
                                 {"ON6B", "CN7B", "NN2", "CN3", "1.0", "1", "0.0", " "},
                                 {"CN8", "CN7B", "NN2", "CN1", "1.0", "3", "0.0", " "},
                                 {"CN8", "CN7B", "NN2", "CN3", "0.0", "3", "180.0", "NF 030697"},
                                 {"CN7B", "CN7B", "NN2", "CN1", "1.0", "3", "0.0", " "},
                                 {"CN7B", "CN7B", "NN2", "CN3", "0.0", "3", "180.0", " "},
                                 {"HN7", "CN7B", "NN2", "CN1", "0.0", "3", "0.0", "NF"},
                                 {"HN7", "CN7B", "NN2", "CN3", "0.195", "3", "0.0", "NF"},
                                 {"CN7B", "NN2B", "CN3", "HN3", "0.3", "2", "180.0", "NF"},
                                 {"CN7B", "NN2B", "CN1T", "ON1", "11.0", "2", "180.0", "adm jr. from A"},
                                 {"CN7B", "NN2B", "CN1T", "NN2U", "11.0", "2", "180.0", "adm jr."},
                                 {"CN7B", "NN2B", "CN3", "CN3T", "11.0", "2", "180.0", "adm jr."},
                                 {"ON6", "CN7B", "NN2B", "CN1", "0.0", "3", "0.0", " "},
                                 {"ON6", "CN7B", "NN2B", "CN1T", "0.7", "3", "0.0", " "},
                                 {"ON6", "CN7B", "NN2B", "CN1T", "0.8", "1", "180.0", " "},
                                 {"ON6", "CN7B", "NN2B", "CN3", "0.9", "1", "0.0", "NF"},
                                 {"ON6B", "CN7B", "NN2B", "CN1", "0.0", "3", "0.0", " "},
                                 {"ON6B", "CN7B", "NN2B", "CN1T", "0.7", "3", "0.0", " "},
                                 {"ON6B", "CN7B", "NN2B", "CN1T", "0.8", "1", "180.0", " "},
                                 {"ON6B", "CN7B", "NN2B", "CN3", "0.9", "1", "0.0", " "},
                                 {"CN8", "CN7B", "NN2B", "CN1T", "0.2", "3", "180.0", " "},
                                 {"CN8", "CN7B", "NN2B", "CN3", "0.0", "3", "180.0", "NF"},
                                 {"CN7B", "CN7B", "NN2B", "CN1T", "0.2", "3", "180.0", " "},
                                 {"CN7B", "CN7B", "NN2B", "CN3", "0.0", "3", "180.0", " "},
                                 {"HN7", "CN7B", "NN2B", "CN1T", "0.0", "3", "0.0", "NF"},
                                 {"HN7", "CN7B", "NN2B", "CN3", "0.195", "3", "0.0", "NF"},
                                 {"CN7B", "NN2B", "CN4", "HN3", "0.3", "2", "180.0", "NF"},
                                 {"CN7B", "NN2B", "CN4", "NN4", "11.0", "2", "180.0", "adm jr."},
                                 {"CN7B", "NN2B", "CN5", "CN5G", "11.0", "2", "180.0", "adm jr. from U"},
                                 {"CN7B", "NN2B", "CN5", "NN3G", "11.0", "2", "180.0", "adm jr."},
                                 {"ON6", "CN7B", "NN2B", "CN5", "0.2", "3", "0.0", " "},
                                 {"ON6", "CN7B", "NN2B", "CN5", "1.1", "1", "180.0", " "},
                                 {"ON6", "CN7B", "NN2B", "CN4", "1.4", "1", "0.0", "NF"},
                                 {"ON6B", "CN7B", "NN2B", "CN5", "0.2", "3", "0.0", " "},
                                 {"ON6B", "CN7B", "NN2B", "CN5", "1.1", "1", "180.0", " "},
                                 {"ON6B", "CN7B", "NN2B", "CN4", "1.4", "1", "0.0", " "},
                                 {"CN8", "CN7B", "NN2B", "CN5", "0.0", "3", "0.0", "NF"},
                                 {"CN8", "CN7B", "NN2B", "CN4", "0.0", "3", "180.0", "NF 030697"},
                                 {"CN7B", "CN7B", "NN2B", "CN5", "0.0", "3", "0.0", "NF"},
                                 {"CN7B", "CN7B", "NN2B", "CN4", "0.0", "3", "180.0", " "},
                                 {"HN7", "CN7B", "NN2B", "CN5", "0.0", "3", "0.0", "NF"},
                                 {"HN7", "CN7B", "NN2B", "CN4", "0.195", "3", "0.0", "NF"},
                                 {"CN7", "ON6", "CN7B", "NN2", "0.0", "3", "0.0", " "},
                                 {"CN7", "ON6", "CN7B", "NN2B", "0.0", "3", "0.0", " "},
                                 {"CN7", "ON6B", "CN7B", "NN2", "0.0", "3", "0.0", " "},
                                 {"CN7", "ON6B", "CN7B", "NN2B", "0.0", "3", "0.0", " "},
                                 {"CN7", "CN8", "CN7B", "NN2", "0.0", "3", "0.0", " "},
                                 {"CN7", "CN8", "CN7B", "NN2B", "0.0", "3", "0.0", " "},
                                 {"HN8", "CN8", "CN7B", "NN2", "0.0", "3", "0.0", " "},
                                 {"HN8", "CN8", "CN7B", "NN2B", "0.0", "3", "0.0", " "},
                                 {"CN7", "CN7B", "CN7B", "NN2", "0.0", "3", "0.0", " "},
                                 {"CN7", "CN7B", "CN7B", "NN2B", "0.0", "3", "0.0", " "},
                                 {"HN7", "CN7B", "CN7B", "NN2", "0.0", "3", "0.0", " "},
                                 {"HN7", "CN7B", "CN7B", "NN2B", "0.0", "3", "0.0", " "},
                                 {"CN7", "CN8B", "ON2", "P", "0.2", "1", "120.0", "C4'-C5'-O5'-P, adm jr."},
                                 {"CN7", "CN8B", "ON2", "P2", "0.2", "1", "120.0", "C4'-C5'-O5'-P, adm jr., adm, 2011 DNA update"},
                                 {"CN7", "CN8B", "ON5", "HN5", "1.3300", "1", "0.00", " "},
                                 {"CN7", "CN8B", "ON5", "HN5", "0.1800", "2", "0.00", " "},
                                 {"CN7", "CN8B", "ON5", "HN5", "0.3200", "3", "0.00", " "},
                                 {"HN8", "CN8B", "ON5", "HN5", "0.0", "3", "0.0", " "},
                                 {"CN7", "CN7", "CN8B", "ON2", "0.20", "4", "180.0", "adm jr."},
                                 {"CN7", "CN7", "CN8B", "ON2", "0.80", "3", "180.0", "C3'-C4'-C5'-O5'"},
                                 {"CN7", "CN7", "CN8B", "ON2", "0.40", "2", "0.0", " "},
                                 {"CN7", "CN7", "CN8B", "ON2", "2.50", "1", "180.0", " "},
                                 {"CN8", "CN7", "CN8B", "ON2", "0.2", "3", "180.0", "from gam, carbocyclic and 25P1"},
                                 {"CN7", "CN7", "CN8B", "ON5", "0.20", "4", "180.0", "adm jr."},
                                 {"CN7", "CN7", "CN8B", "ON5", "0.80", "3", "180.0", "C3'-C4'-C5'-O5'"},
                                 {"CN7", "CN7", "CN8B", "ON5", "0.40", "2", "0.0", " "},
                                 {"CN7", "CN7", "CN8B", "ON5", "2.50", "1", "180.0", " "},
                                 {"ON6", "CN7", "CN8B", "ON2", "3.4", "1", "180.0", "O4'-C4'-C5'-O5',influences +60"},
                                 {"ON6B", "CN7", "CN8B", "ON2", "3.4", "1", "180.0", "RNA"},
                                 {"ON6", "CN7", "CN8B", "ON5", "3.4", "1", "180.0", " "},
                                 {"ON6B", "CN7", "CN8B", "ON5", "3.4", "1", "180.0", "RNA"},
                                 {"HN8", "CN8B", "CN7", "CN7", "0.195", "3", "0.0", " "},
                                 {"HN8", "CN8B", "CN7", "CN8", "0.195", "1", "0.0", "carbocylic, 25P1"},
                                 {"HN8", "CN8B", "CN7", "ON6", "0.195", "1", "0.0", " "},
                                 {"HN8", "CN8B", "CN7", "ON6B", "0.195", "1", "0.0", "RNA"},
                                 {"HN7", "CN7", "CN8B", "ON2", "0.195", "3", "0.0", "H-C4'-C5'-O5'"},
                                 {"HN7", "CN7", "CN8B", "ON5", "0.195", "3", "0.0", " "},
                                 {"HN8", "CN8", "CN8", "ON6", "0.195", "1", "0.0", " "},
                                 {"CN9", "CN7", "CN7", "CN8B", "0.5", "4", "180.0", "-> CN9"},
                                 {"HN7", "CN7", "CN9", "HN9", "0.195", "3", "0.0", "-> CN9"},
                                 {"CN7", "CN7", "CN9", "HN9", "0.195", "3", "0.0", "-> CN9"},
                                 {"ON6", "CN7", "CN9", "HN9", "0.195", "3", "0.0", "-> CN9"},
                                 {"HN7", "CN7", "CN7", "CN9", "0.195", "3", "0.0", "-> CN9"},
                                 {"ON2", "CN7", "CN7", "CN9", "0.2", "4", "0.0", "-> CN9"},
                                 {"ON2", "CN7", "CN7", "CN9", "0.8", "3", "180.0", "-> CN9"},
                                 {"CN8", "CN7", "CN7", "CN9", "0.5", "4", "180.0", "-> CN9"},
                                 {"CN8", "CN7", "CN7", "CN8B", "0.5", "4", "180.0", "lowers 180 deg."},
                                 {"CN7B", "CN7", "CN7", "CN8B", "0.2", "4", "180.0", "RNA"},
                                 {"ON2", "CN7", "CN7", "CN8B", "0.2", "4", "0.0", " "},
                                 {"ON2", "CN7", "CN7", "CN8B", "0.8", "3", "180.0", "P [100,250]"},
                                 {"ON5", "CN7", "CN7", "CN8B", "0.2", "4", "0.0", " "},
                                 {"ON5", "CN7", "CN7", "CN8B", "0.8", "3", "180.0", " "},
                                 {"ON2", "CN7", "CN7", "ON6", "0.5", "6", "0.0", "good for amplitudes"},
                                 {"ON2", "CN7", "CN7", "ON6", "0.3", "5", "0.0", "impact on amplitudes"},
                                 {"ON2", "CN7", "CN7", "ON6", "0.6", "4", "180.0", "increases c2'endo"},
                                 {"ON2", "CN7", "CN7", "ON6", "0.2", "3", "0.0", " "},
                                 {"ON2", "CN7", "CN7", "ON6B", "0.4", "6", "0.0", "RNA, good for amplitudes"},
                                 {"ON2", "CN7", "CN7", "ON6B", "0.0", "5", "0.0", "RNA, impact on amplitudes"},
                                 {"ON2", "CN7", "CN7", "ON6B", "0.0", "4", "180.0", "RNA, increases c2'endo"},
                                 {"ON2", "CN7", "CN7", "ON6B", "1.6", "3", "0.0", "RNA, increases C2'endo"},
                                 {"ON2", "CN7B", "CN7B", "ON6B", "0.4", "6", "0.0", "RNA, good for amplitudes"},
                                 {"ON2", "CN7B", "CN7B", "ON6B", "0.0", "5", "0.0", "RNA, impact on amplitudes"},
                                 {"ON2", "CN7B", "CN7B", "ON6B", "0.0", "4", "180.0", "RNA, increases c2'endo"},
                                 {"ON2", "CN7B", "CN7B", "ON6B", "1.6", "3", "0.0", "RNA, increases C2'endo"},
                                 {"ON5", "CN7", "CN7", "ON6", "0.5", "6", "0.0", " "},
                                 {"ON5", "CN7", "CN7", "ON6", "0.3", "5", "0.0", " "},
                                 {"ON5", "CN7", "CN7", "ON6", "0.6", "4", "180.0", " "},
                                 {"ON5", "CN7", "CN7", "ON6", "0.2", "3", "0.0", " "},
                                 {"ON5", "CN7", "CN7", "ON6B", "0.4", "6", "0.0", " "},
                                 {"ON5", "CN7", "CN7", "ON6B", "0.0", "5", "0.0", " "},
                                 {"ON5", "CN7", "CN7", "ON6B", "0.0", "4", "180.0", " "},
                                 {"ON5", "CN7", "CN7", "ON6B", "1.6", "3", "0.0", "increases c2'endo"},
                                 {"CN7B", "ON6", "CN7", "CN8B", "0.8", "3", "0.0", "P [30,80]"},
                                 {"CN7B", "ON6B", "CN7", "CN8B", "2.0", "3", "0.0", "To lower barrier in RNA"},
                                 {"CN7B", "ON6B", "CN7", "CN9", "2.0", "3", "0.0", "To lower barrier in RNA, 5MET"},
                                 {"ON2", "CN7", "CN8", "CN7B", "0.8", "6", "0.0", " "},
                                 {"ON2", "CN7", "CN8", "CN7B", "0.4", "5", "0.0", "Moves the barrier right"},
                                 {"ON2", "CN7", "CN8", "CN7B", "2.0", "3", "180.0", " "},
                                 {"ON2", "CN7", "CN7B", "CN7B", "0.6", "6", "0.0", "RNA"},
                                 {"ON2", "CN7", "CN7B", "CN7B", "0.0", "5", "0.0", "RNA c2/c3 endo in RNA"},
                                 {"ON2", "CN7", "CN7B", "CN7B", "1.6", "3", "180.0", " "},
                                 {"ON5", "CN7", "CN8", "CN7B", "0.8", "6", "0.0", " "},
                                 {"ON5", "CN7", "CN8", "CN7B", "0.4", "5", "0.0", " "},
                                 {"ON5", "CN7", "CN8", "CN7B", "2.0", "3", "180.0", " "},
                                 {"ON5", "CN7", "CN7B", "CN7B", "0.6", "6", "0.0", "RNA, c2/c3 endo"},
                                 {"ON5", "CN7", "CN7B", "CN7B", "0.0", "5", "0.0", "RNA"},
                                 {"ON5", "CN7", "CN7B", "CN7B", "1.6", "3", "180.0", "RNA"},
                                 {"ON2", "CN7", "CN8", "HN8", "0.195", "3", "0.0", " "},
                                 {"ON5", "CN7", "CN8", "HN8", "0.195", "3", "180.0", " "},
                                 {"ON2", "CN7", "CN7B", "HN7", "0.195", "3", "0.0", "RNA"},
                                 {"ON5", "CN7", "CN7B", "HN7", "0.195", "3", "180.0", "RNA"},
                                 {"HN7", "CN7", "CN7", "ON2", "0.195", "3", "0.0", " "},
                                 {"HN7", "CN7", "CN7", "ON5", "0.195", "3", "0.0", " "},
                                 {"CN7", "CN7", "ON2", "P2", "0.6", "5", "0.0", "adm, 2011 DNA update"},
                                 {"CN7", "CN7", "ON2", "P2", "0.2", "4", "0.0", "locat of 200 mimima, adm, 2011 DNA update"},
                                 {"CN7", "CN7", "ON2", "P2", "0.0", "3", "180.0", "barE beteen minima, adm, 2011 DNA update"},
                                 {"CN7", "CN7", "ON2", "P2", "0.4", "2", "0.0", "relE of 200 vs 275 min, adm, 2011 DNA update"},
                                 {"CN7", "CN7", "ON2", "P2", "1.9", "1", "180.0", "adm, 2011 DNA update"},
                                 {"CN7", "CN7", "ON2", "P", "0.6", "5", "0.0", "RNA"},
                                 {"CN7", "CN7", "ON2", "P", "0.2", "4", "0.0", "locat of 200 mimima, RNA"},
                                 {"CN7", "CN7", "ON2", "P", "0.0", "3", "180.0", "barE beteen minima, RNA"},
                                 {"CN7", "CN7", "ON2", "P", "0.4", "2", "0.0", "relE of 200 vs 275 min, RNA"},
                                 {"CN7", "CN7", "ON2", "P", "1.9", "1", "180.0", "RNA"},
                                 {"CN8", "CN7", "ON2", "P", "2.5", "1", "180.0", "3-terminal phosphate"},
                                 {"CN8", "CN7", "ON2", "P2", "1.9", "1", "180.0", "adm, 2011 DNA update new param, eps"},
                                 {"CN7B", "CN7", "ON2", "P", "2.5", "1", "180.0", "RNA"},
                                 {"CN7B", "CN7B", "ON2", "P", "2.5", "1", "180.0", "NADPH and bkbmod"},
                                 {"CN7", "CN7B", "ON2", "P", "2.5", "1", "180.0", "NADPH and bkbmod"},
                                 {"CN7", "CN7", "ON5", "HN5", "0.5", "3", "0.0", " "},
                                 {"CN7", "CN7", "ON5", "HN5", "0.3", "2", "180.0", " "},
                                 {"CN7", "CN7", "ON5", "HN5", "1.5", "1", "0.0", " "},
                                 {"CN8", "CN7", "ON5", "HN5", "0.5", "3", "0.0", " "},
                                 {"CN8", "CN7", "ON5", "HN5", "1.0", "2", "180.0", " "},
                                 {"CN8", "CN7", "ON5", "HN5", "0.3", "1", "0.0", " "},
                                 {"CN7B", "CN7", "ON5", "HN5", "0.8", "3", "0.0", "RNA"},
                                 {"CN7B", "CN7", "ON5", "HN5", "0.5", "1", "0.0", "RNA"},
                                 {"HN7", "CN7", "ON5", "HN5", "0.0", "3", "0.0", " "},
                                 {"HN7", "CN7", "CN8B", "HN8", "0.195", "3", "0.0", "H-C4'-C5'-H"},
                                 {"HN7", "CN7", "CN7", "CN8B", "0.195", "3", "0.0", "H-C3'-C4'-C5'"},
                                 {"CN8", "CN7B", "ON6", "CN7", "0.6", "6", "180.0", " "},
                                 {"CN8", "CN7", "CN7", "ON6", "1.0", "4", "0.0", "adm, 2011 DNA update new param, C2'-C3'-C4'-O4'; lowers c3'endo"},
                                 {"CN8", "CN7", "CN7", "ON6", "0.3", "5", "180.0", "adm, 2011 DNA update new param, C2'-C3'-C4'-O4'; position of minima"},
                                 {"CN8", "CN7", "CN7", "ON6", "0.3", "6", "180.0", "adm, 2011 DNA update new param, C2'-C3'-C4'-O4'; position of minima"},
                                 {"CN7B", "CN7B", "ON6B", "CN7", "0.0", "6", "0.0", "RNA, Lowers barrier"},
                                 {"CN7B", "CN7", "CN7", "ON6B", "0.0", "3", "0.0", "RNA"},
                                 {"CN7", "CN8", "CN7B", "ON6", "0.6", "6", "0.0", "C3'-C2'-C1'-O4', adjust barrier"},
                                 {"CN7", "CN7B", "CN7B", "ON6B", "0.4", "6", "0.0", "RNA"},
                                 {"CN7B", "CN8", "CN7", "CN7", "0.4", "6", "0.0", "good for amplitudes"},
                                 {"CN7B", "CN7B", "CN7", "CN7", "0.0", "6", "0.0", "RNA"},
                                 {"CN7", "CN7", "ON6", "CN7B", "0.6", "6", "180.0", "C3'-C4'-O4'-C1'"},
                                 {"CN7", "CN7", "ON6B", "CN7B", "0.0", "6", "180.0", "RNA"},
                                 {"HN7", "CN7", "CN7", "CN8", "0.0", "3", "0.0", " "},
                                 {"HN7", "CN7", "CN8", "CN7B", "0.195", "3", "0.0", " "},
                                 {"HN7", "CN7B", "CN8", "CN7", "0.195", "3", "0.0", " "},
                                 {"HN7", "CN7", "CN7", "ON6", "0.195", "3", "180.0", "useful"},
                                 {"HN8", "CN8", "CN7B", "ON6", "0.195", "3", "0.0", " "},
                                 {"HN7", "CN7", "CN7", "HN7", "0.195", "3", "0.0", " "},
                                 {"HN7", "CN7B", "CN8", "HN8", "0.195", "3", "0.0", " "},
                                 {"HN7", "CN7", "CN8", "HN8", "0.195", "3", "0.0", " "},
                                 {"HN8", "CN8", "CN7", "CN7", "0.195", "3", "0.0", "useful *cccc*"},
                                 {"HN7", "CN7", "ON6", "CN7B", "0.195", "3", "0.0", " "},
                                 {"HN7", "CN7B", "ON6", "CN7", "0.000", "3", "0.0", " "},
                                 {"HN7", "CN7", "CN7", "ON6B", "0.195", "3", "180.0", "RNA"},
                                 {"HN9", "CN9", "CN7", "ON6B", "0.195", "3", "180.0", "RNA, 5MET"},
                                 {"HN8", "CN8", "CN7B", "ON6B", "0.195", "3", "0.0", "RNA"},
                                 {"HN7", "CN7B", "ON6B", "CN7", "0.000", "3", "0.0", "RNA"},
                                 {"HN7", "CN7", "ON6B", "CN7B", "0.195", "3", "0.0", "RNA"},
                                 {"HN7", "CN7", "CN7B", "CN7B", "0.195", "3", "0.0", "RNA, H-C3'-C2'-C1'"},
                                 {"HN7", "CN7B", "CN7B", "CN7", "0.195", "3", "0.0", "RNA, H-C1'-C2'-C3'"},
                                 {"HN7", "CN7B", "CN7B", "ON6B", "0.195", "3", "0.0", "RNA, H-C2'-C1'-O4'"},
                                 {"NN2", "CN7B", "CN7B", "ON5", "0.000", "3", "0.0", "Adenine and cytosine"},
                                 {"NN2B", "CN7B", "CN7B", "ON5", "0.000", "3", "0.0", "Guanine and uracil"},
                                 {"ON5", "CN7B", "CN7B", "HN7", "0.000", "3", "0.0", " "},
                                 {"HN7", "CN7B", "CN7B", "HN7", "0.000", "3", "0.0", " "},
                                 {"CN7", "CN7", "CN7B", "ON5", "0.000", "3", "0.0", " "},
                                 {"ON6B", "CN7B", "CN7B", "ON5", "0.000", "3", "0.0", " "},
                                 {"ON5", "CN7B", "CN7", "ON2", "0.000", "3", "0.0", " "},
                                 {"ON5", "CN7", "CN7B", "ON2", "0.000", "3", "0.0", " "},
                                 {"ON5", "CN7B", "CN7", "ON5", "0.000", "3", "0.0", " "},
                                 {"HN7", "CN7B", "ON5", "HN5", "0.000", "3", "0.0", " "},
                                 {"HN5", "ON5", "CN7B", "CN7B", "0.000", "6", "180.0", "ejd, 2010 RNA update"},
                                 {"HN5", "ON5", "CN7B", "CN7B", "0.400", "3", "0.0", "shifts min, ejd, 2010 RNA update"},
                                 {"HN5", "ON5", "CN7B", "CN7B", "0.400", "2", "0.0", "ejd, 2010 RNA update"},
                                 {"HN5", "ON5", "CN7B", "CN7B", "0.800", "1", "0.0", "height of right barrier and can shift min, ejd, 2010 RNA update"},
                                 {"HN5", "ON5", "CN7B", "CN7", "0.200", "3", "0.0", "ejd, 2010 RNA update"},
                                 {"HN5", "ON5", "CN7B", "CN7", "0.000", "2", "180.0", "ejd, 2010 RNA update"},
                                 {"HN5", "ON5", "CN7B", "CN7", "2.000", "1", "0.0", "height of left side barrier, ejd, 2010 RNA update"},
                                 {"CN7B", "ON6", "CN7", "CN9", "0.0", "3", "0.0", "added for sugar model compounds"},
                                 {"HN7", "CN7", "CN7B", "ON5", "0.195", "3", "0.0", "for nadp/nadph (NOT!), adm jr."},
                                 {"HN7", "CN7B", "CN7", "CN7", "0.195", "3", "0.0", "for nadp/nadph (NOT!), adm jr."},
                                 {"HN7", "CN7", "CN7", "CN7B", "0.195", "3", "0.0", "for nadp/nadph (NOT!), adm jr."},
                                 {"HN7", "CN7", "CN7B", "HN7", "0.195", "3", "0.0", "for nadp/nadph (NOT!), adm jr."}};

/*
 Impropers
  0-3= Keys, 4= Kpsi (kcal mol^-1 rad^-2), 5= psi0 (deg), 6= charmm36_na info

  V(psi) = Kpsi x (psi - psi0)^2
*/
char * charmm36_na_impropers[15][8]= {{"HN2", "X", "X", "NN2", "1.0", "0", "0.0", "adm jr. 11/97"},
                                {"NN2B", "CN4", "CN5", "HN2", "7.0", "0", "0.0", "adm jr. 11/97"},
                                {"HN1", "X", "X", "NN1", "4.0", "0", "0.0", "adm jr. 11/97"},
                                {"NN1", "CN2", "HN1", "HN1", "6.0", "0", "0.0", "adm jr. 11/97"},
                                {"CN1", "X", "X", "ON1", "90.0", "0", "0.0", " "},
                                {"CN1T", "X", "X", "ON1", "90.0", "0", "0.0", " "},
                                {"CN1", "NN2G", "CN5G", "ON1", "90.0", "0", "0.0", " "},
                                {"CN1T", "NN2B", "NN2U", "ON1", "110.0", "0", "0.0", "adm jr. 11/97"},
                                {"CN1", "NN2U", "CN3T", "ON1", "90.0", "0", "0.0", "adm jr. 11/97"},
                                {"CN1", "X", "X", "ON1C", "80.0", "0", "0.0", "par_32, adm jr. 10/2/91"},
                                {"CN2", "X", "X", "NN1", "90.0", "0", "0.0", " "},
                                {"CN2", "NN3G", "NN2G", "NN1", "40.0", "0", "0.0", " "},
                                {"CN2", "NN3A", "CN5", "NN1", "40.0", "0", "0.0", " "},
                                {"CN2", "NN3", "CN3", "NN1", "60.0", "0", "0.0", " "},
                                {"CN9", "X", "X", "CN3T", "14.0", "0", "0.0", "adm jr. 11/97"}};

/*
 Non-bonded
  0= Key, 1= epsilon (kcal mol^-1), 2= Rmin/2 (A), 3= epsilon[1-4], 4= Rmin[1-4]/2, 5= charmm36_na info

  V(rij) =  Eps(ij) x [(Rmin(ij)/rij)^12 - 2 x (Rmin(ij)/rij)^6]
  With:
    Esp(ij) = sqrt(epsilon([i) x epsilon[j])
    Rmin(ij)= (Rmin[i] + Rmin[j])/2
*/
char * charmm36_na_vdw[42][6]= {{"HN1", "-0.0460", "0.2245", " ", " ", " "},
                          {"HN2", "-0.0460", "0.2245", " ", " ", " "},
                          {"HN3", "-0.046", "1.1000", " ", " ", "aromatic Hvdw"},
                          {"HN4", "-0.0460", "0.2245", " ", " ", " "},
                          {"HN5", "-0.0460", "0.2245", " ", " ", " "},
                          {"HN6", "-0.0220", "1.3200", " ", " ", " "},
                          {"HN7", "-0.0220", "1.3200", " ", " ", " "},
                          {"HN8", "-0.0280", "1.3400", " ", " ", "bound to CN8"},
                          {"HN9", "-0.0240", "1.3400", " ", " ", "bound to CN9"},
                          {"NN1", "-0.20", "1.85", " ", " ", " "},
                          {"NN2", "-0.20", "1.85", " ", " ", " "},
                          {"NN2B", "-0.20", "1.85", " ", " ", "NN2, for N9 in guanines"},
                          {"NN2G", "-0.20", "1.85", " ", " ", " "},
                          {"NN2U", "-0.20", "1.85", " ", " ", " "},
                          {"NN3", "-0.20", "1.85", " ", " ", " "},
                          {"NN3A", "-0.20", "1.85", " ", " ", " "},
                          {"NN3G", "-0.20", "1.85", " ", " ", " "},
                          {"NN4", "-0.20", "1.85", " ", " ", " "},
                          {"NN6", "-0.20", "1.85", " ", " ", " "},
                          {"ON1", "-0.1200", "1.70", " ", " ", " "},
                          {"ON1C", "-0.1200", "1.70", " ", " ", " "},
                          {"ON2", "-0.1521", "1.77", " ", " ", " "},
                          {"ON3", "-0.1200", "1.70", " ", " ", " "},
                          {"ON4", "-0.1521", "1.77", " ", " ", " "},
                          {"ON5", "-0.1521", "1.77", " ", " ", " "},
                          {"ON6", "-0.1521", "1.77", " ", " ", " "},
                          {"ON6B", "-0.1521", "1.77", " ", " ", " "},
                          {"CN1", "-0.10", "1.9000", " ", " ", " "},
                          {"CN1T", "-0.10", "1.9000", " ", " ", " "},
                          {"CN2", "-0.10", "1.9000", " ", " ", " "},
                          {"CN3", "-0.09", "1.9000", " ", " ", " "},
                          {"CN3T", "-0.09", "1.9000", " ", " ", "adm jr."},
                          {"CN4", "-0.075", "1.9000", " ", " ", " "},
                          {"CN5", "-0.075", "1.9000", " ", " ", " "},
                          {"CN5G", "-0.075", "1.9000", " ", " ", " "},
                          {"CN7", "-0.02", "2.275", "-0.01", "1.90", "to protein CT1"},
                          {"CN7B", "-0.02", "2.275", "-0.01", "1.90", "to protein CT1"},
                          {"CN8", "-0.0560", "2.010", "-0.01", "1.90", " "},
                          {"CN8B", "-0.0560", "2.010", "-0.01", "1.90", " "},
                          {"CN9", "-0.0780", "2.040", "-0.01", "1.90", " "},
                          {"P", "-0.585", "2.15", " ", " ", " "},
                          {"P2", "-0.585", "2.15", " ", " ", " "}};
