#include "FFReadStream.h"
#include "FFReadHandler.h"
#include "avm_fourcc.h"
#include "avm_output.h"
#include "utils.h"

#include "avm_avformat.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

AVM_BEGIN_NAMESPACE;

static int get_fcc(enum CodecID id)
{
    const struct id2fcc {
	enum CodecID id;
	uint32_t fcc;
    } id2fcct[] = {
	{ CODEC_ID_MPEG1VIDEO, RIFFINFO_MPG1 },
	{ CODEC_ID_MPEG1VIDEO, mmioFOURCC('P', 'I', 'M', '1') },
	{ CODEC_ID_MPEG2VIDEO, RIFFINFO_MPG2 },
	{ CODEC_ID_H263, mmioFOURCC('H', '2', '6', '3') },
	{ CODEC_ID_H263P, mmioFOURCC('H', '2', '6', '3') },
	{ CODEC_ID_H263I, mmioFOURCC('I', '2', '6', '3') },
	{ CODEC_ID_H264, mmioFOURCC('H', '2', '6', '4') },
	{ CODEC_ID_MSMPEG4V3, mmioFOURCC('D', 'I', 'V', '3') },
	{ CODEC_ID_MPEG4, mmioFOURCC('D', 'I', 'V', 'X') },
	{ CODEC_ID_MSMPEG4V2, mmioFOURCC('M', 'P', '4', '2') },
	{ CODEC_ID_FLV1, mmioFOURCC('F', 'L', 'V', '1') },
	{ CODEC_ID_MJPEG, mmioFOURCC('M', 'J', 'P', 'G') },
	{ CODEC_ID_DTS, 0x10 },
	{ CODEC_ID_MP2, 0x50 },
	{ CODEC_ID_MP3, 0x55 },
	{ CODEC_ID_AC3, WAVE_FORMAT_DVM },
	{ CODEC_ID_AAC,  'M' | ('P' << 8) }, // MP4A
	{ CODEC_ID_DVVIDEO, mmioFOURCC('D', 'V', 'S', 'D') },
	{ CODEC_ID_DVAUDIO, ('D' << 8) | 'A' },
	{ CODEC_ID_NONE }
    };
    //printf("TRANSLATE  %d    0x%x\n", id, id);
    for (const struct id2fcc* p = id2fcct; p->id; p++)
	if (p->id == id)
	    return p->fcc;
    //printf("FAILED\n");
    return  0;
}

FFReadStream::FFReadStream(FFReadHandler* handler, uint_t sid, AVStream* avs)
    : m_pHandler(handler), m_pAvStream(avs), m_pAvContext(0), m_uiSId(sid),
    m_Packets(5000), m_uiPosition(0), m_dTimestamp(0.),
    m_uiKeyChunks(0), m_uiKeySize(0), m_uiKeyMaxSize(0), m_uiKeyMinSize(0),
    m_uiDeltaSize(0), m_uiDeltaMaxSize(0), m_uiDeltaMinSize(0)
{
    m_StartTime = (m_pHandler->m_pContext->start_time != (int64_t)AV_NOPTS_VALUE)
	? m_pHandler->m_pContext->start_time : 0;
    m_dLength = (double)m_pHandler->m_pContext->duration / (double)AV_TIME_BASE;

    AVM_WRITE("FF stream", "Starttime:%" PRId64 "  Duration:%fs\n", m_StartTime, m_dLength);
    //printf("codec %d	 %.4s\n", avs->codec->codec_id, &avs->codec->codec_tag);
    //printf("CODECRA %d  %d   %d\n", avs->codec->frame_rate, avs->codec->frame_rate_base, avs->r_frame_rate_base);
    if (0 && avs->codec->codec_id == CODEC_ID_MPEG1VIDEO)
    {
	m_pAvContext = avcodec_alloc_context();
	//AVCodec* codec = avcodec_find_encoder(avs->codec->codec_id);
	if (m_pAvContext)
	{
	    AVCodec* codec = avcodec_find_decoder(avs->codec->codec_id);
	    if (codec && avcodec_open(m_pAvContext, codec) == 0)
	    {
		m_pAvContext->flags |= CODEC_FLAG_TRUNCATED;
		m_pAvContext->hurry_up = 5;
		//printf("Opened hurryup decoder %p  %p\n", codec, m_pAvContext->codec->decode);
	    }
	    else
	    {
		avcodec_close(m_pAvContext);
		m_pAvContext = 0;
	    }
	}
    }
}

FFReadStream::~FFReadStream()
{
    if (m_pAvContext)
    {
	avcodec_close(m_pAvContext);
	free(m_pAvContext);
    }
}

double FFReadStream::CacheSize() const
{
    return 1.; //(double)m_Packets.size() / (double)m_Packets.capacity() + 0.2;
}

void FFReadStream::ClearCache()
{
}

size_t FFReadStream::GetHeader(void* header, size_t size) const
{
    return 0;
}

framepos_t FFReadStream::GetPrevKeyFrame(framepos_t frame) const
{
    return ERR;
}

framepos_t FFReadStream::GetNextKeyFrame(framepos_t frame) const
{
    return ERR;
}

framepos_t FFReadStream::GetNearestKeyFrame(framepos_t frame) const
{
    return ERR;
}

framepos_t FFReadStream::GetLength() const
{
    return 0x7fffffff;
}

double FFReadStream::GetFrameTime() const
{
    return 1/24.;
}

double FFReadStream::GetLengthTime() const
{
    return m_dLength;
}

StreamInfo* FFReadStream::GetStreamInfo() const
{
    AVStream* avs = m_pHandler->m_pContext->streams[m_uiSId];
    uint_t s;
    if (m_StreamInfo.m_p->m_dLengthTime == 0.0)
    {
	m_StreamInfo.m_p->setKfFrames(m_uiKeyMaxSize, m_uiKeyMinSize,
				      m_uiKeyChunks, m_uiKeySize);

	m_StreamInfo.m_p->setFrames(m_uiDeltaMaxSize,
				    (m_uiDeltaMinSize > m_uiDeltaMaxSize)
				    // usually no delta frames
				    ? m_uiDeltaMaxSize : m_uiDeltaMinSize,
				    (uint_t)(m_Offsets.size() - m_uiKeyChunks),
				    m_uiStreamSize - m_uiKeySize);

	m_StreamInfo.m_p->m_dLengthTime = GetLengthTime();
	m_StreamInfo.m_p->m_iQuality = 0;
	m_StreamInfo.m_p->m_iSampleSize = 1;//m_Header.dwSampleSize;

	switch (avs->codec->codec_type)
	{
	case CODEC_TYPE_AUDIO:
	    m_StreamInfo.m_p->setAudio(avs->codec->channels,
				       avs->codec->sample_rate,
				       avs->codec->frame_bits);
	    m_StreamInfo.m_p->m_Type = StreamInfo::Audio;
	    m_StreamInfo.m_p->m_uiFormat = avs->codec->codec_tag;
	    AVM_WRITE("FF stream", "Audio Format:  %.4s (0x%x)\n",
		      (const char*)&avs->codec->codec_tag, avs->codec->codec_tag);
	    break;
	case CODEC_TYPE_VIDEO:
	    m_StreamInfo.m_p->setVideo(avs->codec->width, avs->codec->height,
				       0, (float)avs->codec->sample_aspect_ratio.num /
				       (float)avs->codec->sample_aspect_ratio.den);
	    m_StreamInfo.m_p->m_Type = StreamInfo::Video;
	    m_StreamInfo.m_p->m_uiFormat = avs->codec->codec_tag;
	    AVM_WRITE("FF stream", "Codec tag format %.4s\n", (char*) &m_StreamInfo.m_p->m_uiFormat);
	    break;
	default:
	    return 0;
	}
	if (m_StreamInfo.m_p->m_uiFormat == 0) {
	    m_StreamInfo.m_p->m_uiFormat = get_fcc(avs->codec->codec_id);

	    if (m_StreamInfo.m_p->m_uiFormat == 0)
		AVM_WRITE("FF stream", "StreamInfo extension neeeded\n");
	}
    }

    return new StreamInfo(m_StreamInfo);
}

double FFReadStream::GetTime(framepos_t lSample) const
{
    if (lSample == ERR)
	return m_dTimestamp;

    return 0.;
}

size_t FFReadStream::GetSampleSize() const
{
    return 0;
}

IStream::StreamType FFReadStream::GetType() const
{
    switch (m_pHandler->m_pContext->streams[m_uiSId]->codec->codec_type)
    {
    case CODEC_TYPE_AUDIO: return IStream::Audio;
    case CODEC_TYPE_VIDEO: return IStream::Video;
    default: return IStream::Other;
    }
}

size_t FFReadStream::GetFormat(void *format, size_t size) const
{
    AVStream* avs = m_pHandler->m_pContext->streams[m_uiSId];

    int tag = get_fcc(avs->codec->codec_id);
    switch (avs->codec->codec_type)
    {
    case CODEC_TYPE_AUDIO:
	if (format && size >= sizeof(WAVEFORMATEX))
	{
	    WAVEFORMATEX& wfx = *(WAVEFORMATEX*)format;
	    FFMPEGWAVEFORMATEX& ffwfx = *(FFMPEGWAVEFORMATEX*)format;
	    if (tag == 0) {
		if (size < sizeof(FFMPEGWAVEFORMATEX))
		    return 0;
		memset(&ffwfx, 0, sizeof(ffwfx));
                // using extension - there is no free place in original WAVEFORMATEX
		wfx.wFormatTag = WAVE_FORMAT_EXTENSIBLE;
		avm_set_le32(&ffwfx.wfex.SubFormat.f1, CodecInfo::FFMPEG);
		avm_set_le32(&ffwfx.dwCodecID, avs->codec->codec_id);
	    } else {
		memset(&wfx, 0, sizeof(wfx));
		wfx.wFormatTag = (uint16_t)tag;
	    }

	    //if (avs->codec->codec_tag == 0) wf->wFormatTag = av_codec_get_fourcc(avs->codec->codec_id);
	    //printf("CODEC  %x   %x   %x\n", wf.wFormatTag, avs->codec->codec_id, avs->codec->codec_tag);
	    wfx.nChannels = (uint16_t)avs->codec->channels;
	    wfx.nSamplesPerSec = avs->codec->sample_rate;
	    wfx.nAvgBytesPerSec = avs->codec->bit_rate / 8;
	    //printf("SAMP:%d AVGBYTES:%d  ALIGN:%d    BITS:%d\n", avs->codec->sample_rate, avs->codec->bit_rate, avs->codec->block_align, avs->codec->bits_per_coded_sample);
	    wfx.nBlockAlign = (uint16_t)avs->codec->block_align;
	    wfx.wBitsPerSample = (uint16_t)avs->codec->bits_per_coded_sample;
	    wfx.cbSize = tag ? 0 : sizeof(FFMPEGWAVEFORMATEX) - sizeof(WAVEFORMATEX);
	    if (avs->codec->extradata
		&& size >= (sizeof(WAVEFORMATEX) + wfx.cbSize + avs->codec->extradata_size))
	    {
		memcpy((uint8_t*)(&wfx + 1) + wfx.cbSize, avs->codec->extradata, avs->codec->extradata_size);
		wfx.cbSize = (uint16_t)(wfx.cbSize + avs->codec->extradata_size);
	    }

	    if (avs->codec->codec_id == CODEC_ID_AAC) {
                // hmm currenly hack - ffmpeg seems to fail properly detect this stream
		wfx.nChannels = 2;
		wfx.nSamplesPerSec = 44100;
		wfx.nAvgBytesPerSec = 16000;
		wfx.wBitsPerSample = 16;
	    }

	    ///printf("AUDIO %d   %x   %x   %d\n", avs->codec->extradata_size, wf->wFormatTag, avs->codec->codec_tag, avs->codec->codec_id);
	}
	return (tag ? sizeof(WAVEFORMATEX) : sizeof(FFMPEGWAVEFORMATEX))
		+ ((avs->codec->extradata) ? avs->codec->extradata_size : 0);
    case CODEC_TYPE_VIDEO:
	if (format && size >= sizeof(BITMAPINFOHEADER))
	{
	    memset(format, 0, sizeof(BITMAPINFOHEADER));
	    BITMAPINFOHEADER& bih = *(BITMAPINFOHEADER*)format;
	    //printf("FORMAT %p  %d   %d\n", format, m_uiSId, avc->width);
	    bih.biSize = sizeof(BITMAPINFOHEADER);
	    bih.biWidth = avs->codec->width;
	    bih.biHeight = avs->codec->height;
	    bih.biPlanes = 1;
	    bih.biCompression = get_fcc(avs->codec->codec_id);
	    bih.biBitCount = (uint16_t)avs->codec->bits_per_coded_sample;
	    // hack which might be eventually usefull
	    //memcpy(&bih->biXPelsPerMeter, &m_pHandler->m_pContext, sizeof(void*));
	    if (!bih.biCompression) {
		bih.biCompression = CodecInfo::FFMPEG;
		bih.biSizeImage = avs->codec->codec_id;
	    }
	    if (avs->codec->extradata && size >= (sizeof(BITMAPINFOHEADER) + avs->codec->extradata_size))
	    {
		bih.biSize += avs->codec->extradata_size;
		memcpy(&bih + 1, avs->codec->extradata, avs->codec->extradata_size);
		//printf("COPY EXTRA %d\n", avs->extradata_size);
		//for (unsigned i = 0; i < size; i++) printf("%d  0x%x\n", i, ((uint8_t*)format)[i]);
	    }
	    //BitmapInfo(*bih).Print();
	}
	return sizeof(BITMAPINFOHEADER)
	    + ((avs->codec->extradata) ? avs->codec->extradata_size : 0);
    default:
	return 0;
    }
}

bool FFReadStream::IsKeyFrame(framepos_t frame) const
{
    if (frame == ERR && m_Packets.size())
        return (m_Packets.front()->GetFlags() & KEYFRAME);

    AVM_WRITE("FF stream", "IsKeyFrame unimplemented!\n");
    return true;
}

StreamPacket* FFReadStream::ReadPacket()
{
    for (size_t i = 0; i < m_Packets.capacity() - 2; ++i)
    {
	if (m_Packets.size() > 0)
	    break;

	if (m_pHandler->readPacket() < 0)
	{
	    if (m_dLength < m_dTimestamp)
		m_dLength = m_dTimestamp;
	    return 0;
	}
    }
    //printf("PACKET SIZE   %d	 %d\n", m_Packets.size(), m_Packets.capacity());
    if (!m_Packets.size())
	return 0;

    Locker locker(m_pHandler->m_Mutex);
    StreamPacket* p = m_Packets.front();
    m_Packets.pop();
    m_dTimestamp = (double)p->GetTimestamp() / (double)AV_TIME_BASE;

    if (m_dLength < m_dTimestamp)
	m_dLength = m_dTimestamp;

    //printf("FFREADSTREAMPACKET  %f   sid:%d  qsz:%" PRIsz " s:%d   frm: %d\n", m_dTimestamp, m_uiSId, m_Packets.size(), p->size, m_pAvStream->codec->frame_number);
    return p;
}

int FFReadStream::Seek(framepos_t pos)
{
    AVM_WRITE("FF stream", "SeekPos: %d\n", pos);
    if (pos == 0)
    {
	m_pHandler->seek(0);
	return 0;
    }
    return -1;
}

int FFReadStream::SeekTime(double time)
{
    AVM_WRITE("FF stream", "SeekTime: %f\n", time);
    return m_pHandler->seek(time);
    if (time < 1.)
    {
	if (m_pAvStream->codec->codec_type == CODEC_TYPE_AUDIO)
	    // check if more streams are available
	    // and seek only with the video
	    return 0;
	m_pHandler->seek(0);
	return 0;
    }
    return -1;
}

int FFReadStream::SkipFrame()
{
    AVM_WRITE("FF stream", "SkipFrame\n");
    StreamPacket* p = ReadPacket();
    if (p)
	p->Release();
    return p ? 0 : -1;
}

int FFReadStream::SkipTo(double pos)
{
    AVM_WRITE("FF stream", "SkipTo: %f\n", pos);
    while (pos < GetTime())
	SkipFrame();
    return 0;
}

AVM_END_NAMESPACE;
