% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stat-alluvium.r
\name{stat_alluvium}
\alias{stat_alluvium}
\title{Alluvial positions}
\usage{
stat_alluvium(
  mapping = NULL,
  data = NULL,
  geom = "alluvium",
  position = "identity",
  decreasing = NULL,
  reverse = NULL,
  absolute = NULL,
  discern = FALSE,
  negate.strata = NULL,
  aggregate.y = NULL,
  cement.alluvia = NULL,
  lode.guidance = NULL,
  lode.ordering = NULL,
  aes.bind = NULL,
  infer.label = FALSE,
  min.y = NULL,
  max.y = NULL,
  na.rm = FALSE,
  show.legend = NA,
  inherit.aes = TRUE,
  ...
)
}
\arguments{
\item{mapping}{Set of aesthetic mappings created by \code{\link[ggplot2:aes]{aes()}}. If specified and
\code{inherit.aes = TRUE} (the default), it is combined with the default mapping
at the top level of the plot. You must supply \code{mapping} if there is no plot
mapping.}

\item{data}{The data to be displayed in this layer. There are three
options:

If \code{NULL}, the default, the data is inherited from the plot
data as specified in the call to \code{\link[ggplot2:ggplot]{ggplot()}}.

A \code{data.frame}, or other object, will override the plot
data. All objects will be fortified to produce a data frame. See
\code{\link[ggplot2:fortify]{fortify()}} for which variables will be created.

A \code{function} will be called with a single argument,
the plot data. The return value must be a \code{data.frame}, and
will be used as the layer data. A \code{function} can be created
from a \code{formula} (e.g. \code{~ head(.x, 10)}).}

\item{geom}{The geometric object to use display the data; override the
default.}

\item{position}{Position adjustment, either as a string naming the adjustment
(e.g. \code{"jitter"} to use \code{position_jitter}), or the result of a call to a
position adjustment function. Use the latter if you need to change the
settings of the adjustment.}

\item{decreasing}{Logical; whether to arrange the strata at each axis in the
order of the variable values (\code{NA}, the default), in ascending order of
totals (largest on top, \code{FALSE}), or in descending order of totals (largest
on bottom, \code{TRUE}).}

\item{reverse}{Logical; if \code{decreasing} is \code{NA}, whether to arrange the
strata at each axis in the reverse order of the variable values, so that
they match the order of the values in the legend. Ignored if \code{decreasing}
is not \code{NA}. Defaults to \code{TRUE}.}

\item{absolute}{Logical; if some cases or strata are negative, whether to
arrange them (respecting \code{decreasing} and \code{reverse}) using negative or
absolute values of \code{y}.}

\item{discern}{Passed to \code{\link[=to_lodes_form]{to_lodes_form()}} if \code{data} is in alluvia format.}

\item{negate.strata}{A vector of values of the \code{stratum} aesthetic to be
treated as negative (will ignore missing values with a warning).}

\item{aggregate.y}{Deprecated alias for \code{cement.alluvia}.}

\item{cement.alluvia}{Logical value indicating whether to aggregate \code{y}
values over equivalent alluvia before computing lode and flow positions.}

\item{lode.guidance}{The function to prioritize the axis variables for
ordering the lodes within each stratum, or else a character string
identifying the function. Character options are "zigzag", "frontback",
"backfront", "forward", and "backward" (see \code{\link{lode-guidance-functions}}).}

\item{lode.ordering}{\strong{Deprecated in favor of the \code{order} aesthetic.} A list
(of length the number of axes) of integer vectors (each of length the
number of rows of \code{data}) or NULL entries (indicating no imposed ordering),
or else a numeric matrix of corresponding dimensions, giving the preferred
ordering of alluvia at each axis. This will be used to order the lodes
within each stratum by sorting the lodes first by stratum, then by the
provided vectors, and lastly by remaining factors (if the vectors contain
duplicate entries and therefore do not completely determine the lode
orderings).}

\item{aes.bind}{At what grouping level, if any, to prioritize differentiation
aesthetics when ordering the lodes within each stratum. Defaults to
\code{"none"} (no aesthetic binding) with intermediate option \code{"flows"} to bind
aesthetics after stratifying by axes linked to the index axis (the one
adjacent axis in \code{stat_flow()}; all remaining axes in \code{stat_alluvium()})
and strongest option \code{"alluvia"} to bind aesthetics after stratifying by
the index axis but before stratifying by linked axes (only available for
\code{stat_alluvium()}). Stratification by any axis is done with respect to the
strata at that axis, after separating positive and negative strata,
consistent with the values of \code{decreasing}, \code{reverse}, and \code{absolute}.
Thus, if \code{"none"}, then lode orderings will not depend on aesthetic
variables. All aesthetic variables are used, in the order in which they are
specified in \code{aes()}.}

\item{infer.label}{Logical; whether to assign the \code{stratum} or \code{alluvium}
variable to the \code{label} aesthetic. Defaults to \code{FALSE}, and requires that
no \code{label} aesthetic is assigned. This parameter is intended for use only
with data in alluva form, which are converted to lode form before the
statistical transformation. Deprecated; use
\code{\link[ggplot2:aes_eval]{ggplot2::after_stat()}} instead.}

\item{min.y, max.y}{Numeric; bounds on the heights of the strata to be
rendered. Use these bounds to exclude strata outside a certain range, for
example when labeling strata using \code{\link[ggplot2:geom_text]{ggplot2::geom_text()}}.}

\item{na.rm}{Logical:
if \code{FALSE}, the default, \code{NA} lodes are not included;
if \code{TRUE}, \code{NA} lodes constitute a separate category,
plotted in grey (regardless of the color scheme).}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped.
\code{FALSE} never includes, and \code{TRUE} always includes.
It can also be a named logical vector to finely select the aesthetics to
display.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics,
rather than combining with them. This is most useful for helper functions
that define both data and aesthetics and shouldn't inherit behaviour from
the default plot specification, e.g. \code{\link[ggplot2:borders]{borders()}}.}

\item{...}{Additional arguments passed to \code{\link[ggplot2:layer]{ggplot2::layer()}}.}
}
\description{
Given a dataset with alluvial structure, \code{stat_alluvium} calculates the
centroids (\code{x} and \code{y}) and heights (\code{ymin} and \code{ymax}) of the lodes, the
intersections of the alluvia with the strata. It leverages the \code{group}
aesthetic for plotting purposes (for now).
}
\section{Aesthetics}{

\code{stat_alluvium}, \code{stat_flow}, and \code{stat_stratum} require one
of two sets of aesthetics:
\itemize{
\item \strong{\code{x}} and at least one of \strong{\code{alluvium}} and \strong{\code{stratum}}
\item any number of \strong{\verb{axis[0-9]*}} (\code{axis1}, \code{axis2}, etc.)
}

Use \code{x}, \code{alluvium}, and/or \code{stratum} for data in lodes format
and \verb{axis[0-9]*} for data in alluvia format (see \code{\link{alluvial-data}}).
Arguments to parameters inconsistent with the format will be ignored.
Additionally, each \verb{stat_*()} accepts the following optional
aesthetics:
\itemize{
\item \code{y}
\item \code{weight}
\item \code{order}
\item \code{group}
\item \code{label}
}

\code{y} controls the heights of the alluvia,
and may be aggregated across equivalent observations.
\code{weight} applies to the computed variables (see that section below)
but does not affect the positional aesthetics.
\code{order}, recognized by \code{stat_alluvium()} and \code{stat_flow()}, is used to
arrange the lodes within each stratum. It tolerates duplicates and takes
precedence over the differentiation aesthetics (when \code{aes.bind} is not
\code{"none"}) and lode guidance with respect to the remaining axes. (It replaces
the deprecated parameter \code{lode.ordering}.)
\code{group} is used internally; arguments are ignored.
\code{label} is used to label the strata or lodes and must take a unique value
across the observations within each stratum or lode.

These and any other aesthetics are aggregated as follows:
Numeric aesthetics, including \code{y}, are summed.
Character and factor aesthetics, including \code{label},
are assigned to strata or lodes provided they take unique values across the
observations within each (and are otherwise assigned \code{NA}).
}

\section{Computed variables}{
 These can be used with
\code{\link[ggplot2:aes_eval]{ggplot2::after_stat()}} to \href{https://ggplot2.tidyverse.org/reference/aes_eval.html}{control aesthetic evaluation}.
\describe{
\item{\code{n}}{number of cases in lode}
\item{\code{count}}{cumulative weight of lode}
\item{\code{prop}}{weighted proportion of lode}
\item{\code{stratum}}{value of variable used to define strata}
\item{\code{deposit}}{order in which (signed) strata are deposited}
\item{\code{lode}}{lode label distilled from alluvia
(\code{stat_alluvium()} and \code{stat_flow()} only)}
\item{\code{flow}}{direction of flow \code{"to"} or \code{"from"} from its axis
(\code{stat_flow()} only)}
}
The numerical variables \code{n}, \code{count}, and \code{prop} are calculated after the
data are grouped by \code{x} and weighted by \code{weight} (in addition to \code{y}).
The integer variable \code{deposit} is used internally to sort the data before
calculating heights. The character variable \code{lode} is obtained from
\code{alluvium} according to \code{distill}.
}

\section{Package options}{

\code{stat_stratum}, \code{stat_alluvium}, and \code{stat_flow} order strata and lodes
according to the values of several parameters, which must be held fixed
across every layer in an alluvial plot. These package-specific options set
global values for these parameters that will be defaulted to when not
manually set:
\itemize{
\item \code{ggalluvial.decreasing} (each \verb{stat_*}): defaults to \code{NA}.
\item \code{ggalluvial.reverse} (each \verb{stat_*}): defaults to \code{TRUE}.
\item \code{ggalluvial.absolute} (each \verb{stat_*}): defaults to \code{TRUE}.
\item \code{ggalluvial.cement.alluvia} (\code{stat_alluvium}): defaults to \code{FALSE}.
\item \code{ggalluvial.lode.guidance} (\code{stat_alluvium}): defaults to \code{"zigzag"}.
\item \code{ggalluvial.aes.bind} (\code{stat_alluvium} and \code{stat_flow}): defaults to
\code{"none"}.
}

See \code{\link[base:options]{base::options()}} for how to use options.
}

\section{Defunct parameters}{

The previously defunct parameters \code{weight} and \code{aggregate.wts} have been
discontinued. Use \code{y} and \code{cement.alluvia} instead.
}

\examples{
# illustrate positioning
ggplot(as.data.frame(Titanic),
       aes(y = Freq,
           axis1 = Class, axis2 = Sex, axis3 = Age,
           color = Survived)) +
  stat_stratum(geom = "errorbar") +
  geom_line(stat = "alluvium") +
  stat_alluvium(geom = "pointrange") +
  geom_text(stat = "stratum", aes(label = after_stat(stratum))) +
  scale_x_discrete(limits = c("Class", "Sex", "Age"))

# lode ordering examples
gg <- ggplot(as.data.frame(Titanic),
             aes(y = Freq,
                 axis1 = Class, axis2 = Sex, axis3 = Age)) +
  geom_stratum() +
  geom_text(stat = "stratum", aes(label = after_stat(stratum))) +
  scale_x_discrete(limits = c("Class", "Sex", "Age"))
# use of lode controls
gg + geom_flow(aes(fill = Survived, alpha = Sex), stat = "alluvium",
               lode.guidance = "forward")
# prioritize aesthetic binding
gg + geom_flow(aes(fill = Survived, alpha = Sex), stat = "alluvium",
               aes.bind = "alluvia", lode.guidance = "forward")
# use of custom lode order
gg + geom_flow(aes(fill = Survived, alpha = Sex, order = sample(x = 32)),
               stat = "alluvium")
# use of custom luide guidance function
lode_custom <- function(n, i) {
  stopifnot(n == 3)
  switch(
    i,
    `1` = 1:3,
    `2` = c(2, 3, 1),
    `3` = 3:1
  )
}
gg + geom_flow(aes(fill = Survived, alpha = Sex), stat = "alluvium",
               aes.bind = "flow", lode.guidance = lode_custom)

# omit missing elements & reverse the `y` axis
ggplot(ggalluvial::majors,
       aes(x = semester, stratum = curriculum, alluvium = student, y = 1)) +
  geom_alluvium(fill = "darkgrey", na.rm = TRUE) +
  geom_stratum(aes(fill = curriculum), color = NA, na.rm = TRUE) +
  theme_bw() +
  scale_y_reverse()

\donttest{
# alluvium cementation examples
gg <- ggplot(ggalluvial::majors,
             aes(x = semester, stratum = curriculum, alluvium = student,
                 fill = curriculum)) +
  geom_stratum()
# diagram with outlined alluvia and labels
gg + geom_flow(stat = "alluvium", color = "black") +
  geom_text(aes(label = after_stat(lode)), stat = "alluvium")
# cemented diagram with default distillation (first most common alluvium)
gg +
  geom_flow(stat = "alluvium", color = "black", cement.alluvia = TRUE) +
  geom_text(aes(label = after_stat(lode)), stat = "alluvium",
            cement.alluvia = TRUE)
# cemented diagram with custom label distillation
gg +
  geom_flow(stat = "alluvium", color = "black", cement.alluvia = TRUE) +
  geom_text(aes(label = after_stat(lode)), stat = "alluvium",
            cement.alluvia = TRUE,
            distill = function(x) paste(x, collapse = "; "))
}

\donttest{
data(babynames, package = "babynames")
# a discontiguous alluvium
bn <- subset(babynames, prop >= .01 & sex == "F" & year > 1962 & year < 1968)
ggplot(data = bn,
       aes(x = year, alluvium = name, y = prop)) +
  geom_alluvium(aes(fill = name, color = name == "Tammy"),
                decreasing = TRUE, show.legend = FALSE) +
  scale_color_manual(values = c("#00000000", "#000000"))
# expanded to include missing values
bn2 <- merge(bn,
             expand.grid(year = unique(bn$year), name = unique(bn$name)),
             all = TRUE)
ggplot(data = bn2,
       aes(x = year, alluvium = name, y = prop)) +
  geom_alluvium(aes(fill = name, color = name == "Tammy"),
                decreasing = TRUE, show.legend = FALSE) +
  scale_color_manual(values = c("#00000000", "#000000"))
# with missing values filled in with zeros
bn2$prop[is.na(bn2$prop)] <- 0
ggplot(data = bn2,
       aes(x = year, alluvium = name, y = prop)) +
  geom_alluvium(aes(fill = name, color = name == "Tammy"),
                decreasing = TRUE, show.legend = FALSE) +
  scale_color_manual(values = c("#00000000", "#000000"))
}

# use negative y values to encode deaths versus survivals
titanic <- as.data.frame(Titanic)
titanic <- transform(titanic, Lives = Freq * (-1) ^ (Survived == "No"))
ggplot(subset(titanic, Class != "Crew"),
       aes(axis1 = Class, axis2 = Sex, axis3 = Age, y = Lives)) +
  geom_alluvium(aes(alpha = Survived, fill = Class), absolute = FALSE) +
  geom_stratum(absolute = FALSE) +
  geom_text(stat = "stratum", aes(label = after_stat(stratum)),
            absolute = FALSE) +
  scale_x_discrete(limits = c("Class", "Sex", "Age"), expand = c(.1, .05)) +
  scale_alpha_discrete(range = c(.25, .75), guide = "none")

# faceting with common alluvia
ggplot(titanic, aes(y = Freq, axis1 = Class, axis2 = Sex, axis3 = Age)) +
  facet_wrap(~ Survived) +
  geom_alluvium() +
  geom_stratum() +
  geom_text(stat = "stratum", aes(label = after_stat(stratum)))
ggplot(transform(alluvial::Refugees, id = 1),
       aes(y = refugees, x = year, alluvium = id)) +
  facet_wrap(~ country) +
  geom_alluvium(alpha = .75, color = "darkgrey") +
  scale_x_continuous(breaks = seq(2004, 2012, 4))
}
\seealso{
\code{\link[ggplot2:layer]{ggplot2::layer()}} for additional arguments and \code{\link[=geom_alluvium]{geom_alluvium()}},
\code{\link[=geom_lode]{geom_lode()}}, and \code{\link[=geom_flow]{geom_flow()}} for the corresponding geoms.

Other alluvial stat layers: 
\code{\link{stat_flow}()},
\code{\link{stat_stratum}()}
}
\concept{alluvial stat layers}
