// Copyright 2015-2020 Benjamin Fry <benjaminfry@me.com>
//
// Licensed under the Apache License, Version 2.0, <LICENSE-APACHE or
// http://apache.org/licenses/LICENSE-2.0> or the MIT license <LICENSE-MIT or
// http://opensource.org/licenses/MIT>, at your option. This file may not be
// copied, modified, or distributed except according to those terms.

use std::future::Future;
use std::time::Duration;

use async_trait::async_trait;
use trust_dns_resolver::proto::Time;

/// AsyncStd backed timer implementation
#[derive(Clone, Copy)]
pub struct AsyncStdTime;

#[async_trait]
impl Time for AsyncStdTime {
    async fn delay_for(duration: Duration) {
        async_std::task::sleep(duration).await
    }

    async fn timeout<F: 'static + Future + Send>(
        duration: Duration,
        future: F,
    ) -> Result<F::Output, std::io::Error> {
        async_std::future::timeout(duration, future)
            .await
            .map_err(move |_| std::io::Error::new(std::io::ErrorKind::TimedOut, "future timed out"))
    }
}
