"""
    sphinx.application
    ~~~~~~~~~~~~~~~~~~

    Sphinx application class and extensibility interface.

    Gracefully adapted from the TextPress system by Armin.

    :copyright: Copyright 2007-2020 by the Sphinx team, see AUTHORS.
    :license: BSD, see LICENSE for details.
"""

import os
import pickle
import platform
import sys
import warnings
from collections import deque
from io import StringIO
from os import path
from typing import Any, Callable, Dict, IO, List, Optional, Tuple, Union

from docutils import nodes
from docutils.nodes import Element, TextElement
from docutils.parsers import Parser
from docutils.parsers.rst import Directive, roles
from docutils.transforms import Transform
from pygments.lexer import Lexer

import sphinx
from sphinx import package_dir, locale
from sphinx.config import Config
from sphinx.deprecation import RemovedInSphinx40Warning
from sphinx.domains import Domain, Index
from sphinx.environment import BuildEnvironment
from sphinx.environment.collectors import EnvironmentCollector
from sphinx.errors import ApplicationError, ConfigError, VersionRequirementError
from sphinx.events import EventManager
from sphinx.extension import Extension
from sphinx.highlighting import lexer_classes, lexers
from sphinx.locale import __
from sphinx.project import Project
from sphinx.registry import SphinxComponentRegistry
from sphinx.roles import XRefRole
from sphinx.theming import Theme
from sphinx.util import docutils
from sphinx.util import logging
from sphinx.util import progress_message
from sphinx.util.build_phase import BuildPhase
from sphinx.util.console import bold  # type: ignore
from sphinx.util.i18n import CatalogRepository
from sphinx.util.logging import prefixed_warnings
from sphinx.util.osutil import abspath, ensuredir, relpath
from sphinx.util.tags import Tags
from sphinx.util.typing import RoleFunction, TitleGetter

if False:
    # For type annotation
    from docutils.nodes import Node  # NOQA
    from typing import Type  # for python3.5.1
    from sphinx.builders import Builder


builtin_extensions = (
    'sphinx.addnodes',
    'sphinx.builders.changes',
    'sphinx.builders.epub3',
    'sphinx.builders.dirhtml',
    'sphinx.builders.dummy',
    'sphinx.builders.gettext',
    'sphinx.builders.html',
    'sphinx.builders.latex',
    'sphinx.builders.linkcheck',
    'sphinx.builders.manpage',
    'sphinx.builders.singlehtml',
    'sphinx.builders.texinfo',
    'sphinx.builders.text',
    'sphinx.builders.xml',
    'sphinx.config',
    'sphinx.domains.c',
    'sphinx.domains.changeset',
    'sphinx.domains.citation',
    'sphinx.domains.cpp',
    'sphinx.domains.index',
    'sphinx.domains.javascript',
    'sphinx.domains.math',
    'sphinx.domains.python',
    'sphinx.domains.rst',
    'sphinx.domains.std',
    'sphinx.directives',
    'sphinx.directives.code',
    'sphinx.directives.other',
    'sphinx.directives.patches',
    'sphinx.extension',
    'sphinx.parsers',
    'sphinx.registry',
    'sphinx.roles',
    'sphinx.transforms',
    'sphinx.transforms.compact_bullet_list',
    'sphinx.transforms.i18n',
    'sphinx.transforms.references',
    'sphinx.transforms.post_transforms',
    'sphinx.transforms.post_transforms.code',
    'sphinx.transforms.post_transforms.images',
    'sphinx.util.compat',
    'sphinx.versioning',
    # collectors should be loaded by specific order
    'sphinx.environment.collectors.dependencies',
    'sphinx.environment.collectors.asset',
    'sphinx.environment.collectors.metadata',
    'sphinx.environment.collectors.title',
    'sphinx.environment.collectors.toctree',
    # 1st party extensions
    'sphinxcontrib.applehelp',
    'sphinxcontrib.devhelp',
    'sphinxcontrib.htmlhelp',
    'sphinxcontrib.serializinghtml',
    'sphinxcontrib.qthelp',
    # Strictly, alabaster theme is not a builtin extension,
    # but it is loaded automatically to use it as default theme.
    'alabaster',
)

ENV_PICKLE_FILENAME = 'environment.pickle'

logger = logging.getLogger(__name__)


class Sphinx:
    """The main application class and extensibility interface.

    :ivar srcdir: Directory containing source.
    :ivar confdir: Directory containing ``conf.py``.
    :ivar doctreedir: Directory for storing pickled doctrees.
    :ivar outdir: Directory for storing build documents.
    """

    def __init__(self, srcdir: str, confdir: str, outdir: str, doctreedir: str,
                 buildername: str, confoverrides: Dict = None,
                 status: IO = sys.stdout, warning: IO = sys.stderr,
                 freshenv: bool = False, warningiserror: bool = False, tags: List[str] = None,
                 verbosity: int = 0, parallel: int = 0, keep_going: bool = False) -> None:
        self.phase = BuildPhase.INITIALIZATION
        self.verbosity = verbosity
        self.extensions = {}                    # type: Dict[str, Extension]
        self.builder = None                     # type: Builder
        self.env = None                         # type: BuildEnvironment
        self.project = None                     # type: Project
        self.registry = SphinxComponentRegistry()
        self.html_themes = {}                   # type: Dict[str, str]

        # validate provided directories
        self.srcdir = abspath(srcdir)
        self.outdir = abspath(outdir)
        self.doctreedir = abspath(doctreedir)
        self.confdir = confdir
        if self.confdir:  # confdir is optional
            self.confdir = abspath(self.confdir)
            if not path.isfile(path.join(self.confdir, 'conf.py')):
                raise ApplicationError(__("config directory doesn't contain a "
                                          "conf.py file (%s)") % confdir)

        if not path.isdir(self.srcdir):
            raise ApplicationError(__('Cannot find source directory (%s)') %
                                   self.srcdir)

        if path.exists(self.outdir) and not path.isdir(self.outdir):
            raise ApplicationError(__('Output directory (%s) is not a directory') %
                                   self.outdir)

        if self.srcdir == self.outdir:
            raise ApplicationError(__('Source directory and destination '
                                      'directory cannot be identical'))

        self.parallel = parallel

        if status is None:
            self._status = StringIO()      # type: IO
            self.quiet = True
        else:
            self._status = status
            self.quiet = False

        if warning is None:
            self._warning = StringIO()     # type: IO
        else:
            self._warning = warning
        self._warncount = 0
        self.keep_going = warningiserror and keep_going
        if self.keep_going:
            self.warningiserror = False
        else:
            self.warningiserror = warningiserror
        logging.setup(self, self._status, self._warning)

        self.events = EventManager(self)

        # keep last few messages for traceback
        # This will be filled by sphinx.util.logging.LastMessagesWriter
        self.messagelog = deque(maxlen=10)  # type: deque

        # say hello to the world
        logger.info(bold(__('Running Sphinx v%s') % sphinx.__display_version__))

        # notice for parallel build on macOS and py38+
        if sys.version_info > (3, 8) and platform.system() == 'Darwin' and parallel > 1:
            logger.info(bold(__("For security reason, parallel mode is disabled on macOS and "
                                "python3.8 and above. For more details, please read "
                                "https://github.com/sphinx-doc/sphinx/issues/6803")))

        # status code for command-line application
        self.statuscode = 0

        # read config
        self.tags = Tags(tags)
        if self.confdir is None:
            self.config = Config({}, confoverrides or {})
        else:
            self.config = Config.read(self.confdir, confoverrides or {}, self.tags)

        # initialize some limited config variables before initialize i18n and loading
        # extensions
        self.config.pre_init_values()

        # set up translation infrastructure
        self._init_i18n()

        # check the Sphinx version if requested
        if self.config.needs_sphinx and self.config.needs_sphinx > sphinx.__display_version__:
            raise VersionRequirementError(
                __('This project needs at least Sphinx v%s and therefore cannot '
                   'be built with this version.') % self.config.needs_sphinx)

        # set confdir to srcdir if -C given (!= no confdir); a few pieces
        # of code expect a confdir to be set
        if self.confdir is None:
            self.confdir = self.srcdir

        # load all built-in extension modules
        for extension in builtin_extensions:
            self.setup_extension(extension)

        # load all user-given extension modules
        for extension in self.config.extensions:
            self.setup_extension(extension)

        # preload builder module (before init config values)
        self.preload_builder(buildername)

        if not path.isdir(outdir):
            with progress_message(__('making output directory')):
                ensuredir(outdir)

        # the config file itself can be an extension
        if self.config.setup:
            prefix = __('while setting up extension %s:') % "conf.py"
            with prefixed_warnings(prefix):
                if callable(self.config.setup):
                    self.config.setup(self)
                else:
                    raise ConfigError(
                        __("'setup' as currently defined in conf.py isn't a Python callable. "
                           "Please modify its definition to make it a callable function. "
                           "This is needed for conf.py to behave as a Sphinx extension.")
                    )

        # now that we know all config values, collect them from conf.py
        self.config.init_values()
        self.events.emit('config-inited', self.config)

        # create the project
        self.project = Project(self.srcdir, self.config.source_suffix)
        # create the builder
        self.builder = self.create_builder(buildername)
        # set up the build environment
        self._init_env(freshenv)
        # set up the builder
        self._init_builder()

    def _init_i18n(self) -> None:
        """Load translated strings from the configured localedirs if enabled in
        the configuration.
        """
        if self.config.language is None:
            self.translator, has_translation = locale.init([], None)
        else:
            logger.info(bold(__('loading translations [%s]... ') % self.config.language),
                        nonl=True)

            # compile mo files if sphinx.po file in user locale directories are updated
            repo = CatalogRepository(self.srcdir, self.config.locale_dirs,
                                     self.config.language, self.config.source_encoding)
            for catalog in repo.catalogs:
                if catalog.domain == 'sphinx' and catalog.is_outdated():
                    catalog.write_mo(self.config.language)

            locale_dirs = list(repo.locale_dirs)  # type: List[Optional[str]]
            locale_dirs += [None]
            locale_dirs += [path.join(package_dir, 'locale')]

            self.translator, has_translation = locale.init(locale_dirs, self.config.language)
            if has_translation or self.config.language == 'en':
                # "en" never needs to be translated
                logger.info(__('done'))
            else:
                logger.info(__('not available for built-in messages'))

    def _init_env(self, freshenv: bool) -> None:
        filename = path.join(self.doctreedir, ENV_PICKLE_FILENAME)
        if freshenv or not os.path.exists(filename):
            self.env = BuildEnvironment()
            self.env.setup(self)
            self.env.find_files(self.config, self.builder)
        else:
            try:
                with progress_message(__('loading pickled environment')):
                    with open(filename, 'rb') as f:
                        self.env = pickle.load(f)
                        self.env.setup(self)
            except Exception as err:
                logger.info(__('failed: %s'), err)
                self._init_env(freshenv=True)

    def preload_builder(self, name: str) -> None:
        self.registry.preload_builder(self, name)

    def create_builder(self, name: str) -> "Builder":
        if name is None:
            logger.info(__('No builder selected, using default: html'))
            name = 'html'

        return self.registry.create_builder(self, name)

    def _init_builder(self) -> None:
        self.builder.set_environment(self.env)
        self.builder.init()
        self.events.emit('builder-inited')

    # ---- main "build" method -------------------------------------------------

    def build(self, force_all: bool = False, filenames: List[str] = None) -> None:
        self.phase = BuildPhase.READING
        try:
            if force_all:
                self.builder.compile_all_catalogs()
                self.builder.build_all()
            elif filenames:
                self.builder.compile_specific_catalogs(filenames)
                self.builder.build_specific(filenames)
            else:
                self.builder.compile_update_catalogs()
                self.builder.build_update()

            if self._warncount and self.keep_going:
                self.statuscode = 1

            status = (__('succeeded') if self.statuscode == 0
                      else __('finished with problems'))
            if self._warncount:
                if self.warningiserror:
                    if self._warncount == 1:
                        msg = __('build %s, %s warning (with warnings treated as errors).')
                    else:
                        msg = __('build %s, %s warnings (with warnings treated as errors).')
                else:
                    if self._warncount == 1:
                        msg = __('build %s, %s warning.')
                    else:
                        msg = __('build %s, %s warnings.')

                logger.info(bold(msg % (status, self._warncount)))
            else:
                logger.info(bold(__('build %s.') % status))

            if self.statuscode == 0 and self.builder.epilog:
                logger.info('')
                logger.info(self.builder.epilog % {
                    'outdir': relpath(self.outdir),
                    'project': self.config.project
                })
        except Exception as err:
            # delete the saved env to force a fresh build next time
            envfile = path.join(self.doctreedir, ENV_PICKLE_FILENAME)
            if path.isfile(envfile):
                os.unlink(envfile)
            self.events.emit('build-finished', err)
            raise
        else:
            self.events.emit('build-finished', None)
        self.builder.cleanup()

    # ---- general extensibility interface -------------------------------------

    def setup_extension(self, extname: str) -> None:
        """Import and setup a Sphinx extension module.

        Load the extension given by the module *name*.  Use this if your
        extension needs the features provided by another extension.  No-op if
        called twice.
        """
        logger.debug('[app] setting up extension: %r', extname)
        self.registry.load_extension(self, extname)

    def require_sphinx(self, version: str) -> None:
        """Check the Sphinx version if requested.

        Compare *version* (which must be a ``major.minor`` version string, e.g.
        ``'1.1'``) with the version of the running Sphinx, and abort the build
        when it is too old.

        .. versionadded:: 1.0
        """
        if version > sphinx.__display_version__[:3]:
            raise VersionRequirementError(version)

    # event interface
    def connect(self, event: str, callback: Callable, priority: int = 500) -> int:
        """Register *callback* to be called when *event* is emitted.

        For details on available core events and the arguments of callback
        functions, please see :ref:`events`.

        Registered callbacks will be invoked on event in the order of *priority* and
        registration.  The priority is ascending order.

        The method returns a "listener ID" that can be used as an argument to
        :meth:`disconnect`.

        .. versionchanged:: 3.0

           Support *priority*
        """
        listener_id = self.events.connect(event, callback, priority)
        logger.debug('[app] connecting event %r (%d): %r [id=%s]',
                     event, priority, callback, listener_id)
        return listener_id

    def disconnect(self, listener_id: int) -> None:
        """Unregister callback by *listener_id*."""
        logger.debug('[app] disconnecting event: [id=%s]', listener_id)
        self.events.disconnect(listener_id)

    def emit(self, event: str, *args: Any,
             allowed_exceptions: Tuple["Type[Exception]", ...] = ()) -> List:
        """Emit *event* and pass *arguments* to the callback functions.

        Return the return values of all callbacks as a list.  Do not emit core
        Sphinx events in extensions!

        .. versionchanged:: 3.1

           Added *allowed_exceptions* to specify path-through exceptions
        """
        return self.events.emit(event, *args, allowed_exceptions=allowed_exceptions)

    def emit_firstresult(self, event: str, *args: Any,
                         allowed_exceptions: Tuple["Type[Exception]", ...] = ()) -> Any:
        """Emit *event* and pass *arguments* to the callback functions.

        Return the result of the first callback that doesn't return ``None``.

        .. versionadded:: 0.5
        .. versionchanged:: 3.1

           Added *allowed_exceptions* to specify path-through exceptions
        """
        return self.events.emit_firstresult(event, *args,
                                            allowed_exceptions=allowed_exceptions)

    # registering addon parts

    def add_builder(self, builder: "Type[Builder]", override: bool = False) -> None:
        """Register a new builder.

        *builder* must be a class that inherits from :class:`~sphinx.builders.Builder`.

        If *override* is True, the given *builder* is forcedly installed even if
        a builder having the same name is already installed.

        .. versionchanged:: 1.8
           Add *override* keyword.
        """
        self.registry.add_builder(builder, override=override)

    # TODO(stephenfin): Describe 'types' parameter
    def add_config_value(self, name: str, default: Any, rebuild: Union[bool, str],
                         types: Any = ()) -> None:
        """Register a configuration value.

        This is necessary for Sphinx to recognize new values and set default
        values accordingly.  The *name* should be prefixed with the extension
        name, to avoid clashes.  The *default* value can be any Python object.
        The string value *rebuild* must be one of those values:

        * ``'env'`` if a change in the setting only takes effect when a
          document is parsed -- this means that the whole environment must be
          rebuilt.
        * ``'html'`` if a change in the setting needs a full rebuild of HTML
          documents.
        * ``''`` if a change in the setting will not need any special rebuild.

        .. versionchanged:: 0.6
           Changed *rebuild* from a simple boolean (equivalent to ``''`` or
           ``'env'``) to a string.  However, booleans are still accepted and
           converted internally.

        .. versionchanged:: 0.4
           If the *default* value is a callable, it will be called with the
           config object as its argument in order to get the default value.
           This can be used to implement config values whose default depends on
           other values.
        """
        logger.debug('[app] adding config value: %r',
                     (name, default, rebuild) + ((types,) if types else ()))
        if rebuild in (False, True):
            rebuild = 'env' if rebuild else ''
        self.config.add(name, default, rebuild, types)

    def add_event(self, name: str) -> None:
        """Register an event called *name*.

        This is needed to be able to emit it.
        """
        logger.debug('[app] adding event: %r', name)
        self.events.add(name)

    def set_translator(self, name: str, translator_class: "Type[nodes.NodeVisitor]",
                       override: bool = False) -> None:
        """Register or override a Docutils translator class.

        This is used to register a custom output translator or to replace a
        builtin translator.  This allows extensions to use custom translator
        and define custom nodes for the translator (see :meth:`add_node`).

        If *override* is True, the given *translator_class* is forcedly installed even if
        a translator for *name* is already installed.

        .. versionadded:: 1.3
        .. versionchanged:: 1.8
           Add *override* keyword.
        """
        self.registry.add_translator(name, translator_class, override=override)

    def add_node(self, node: "Type[Element]", override: bool = False,
                 **kwargs: Tuple[Callable, Callable]) -> None:
        """Register a Docutils node class.

        This is necessary for Docutils internals.  It may also be used in the
        future to validate nodes in the parsed documents.

        Node visitor functions for the Sphinx HTML, LaTeX, text and manpage
        writers can be given as keyword arguments: the keyword should be one or
        more of ``'html'``, ``'latex'``, ``'text'``, ``'man'``, ``'texinfo'``
        or any other supported translators, the value a 2-tuple of ``(visit,
        depart)`` methods.  ``depart`` can be ``None`` if the ``visit``
        function raises :exc:`docutils.nodes.SkipNode`.  Example:

        .. code-block:: python

           class math(docutils.nodes.Element): pass

           def visit_math_html(self, node):
               self.body.append(self.starttag(node, 'math'))
           def depart_math_html(self, node):
               self.body.append('</math>')

           app.add_node(math, html=(visit_math_html, depart_math_html))

        Obviously, translators for which you don't specify visitor methods will
        choke on the node when encountered in a document to translate.

        If *override* is True, the given *node* is forcedly installed even if
        a node having the same name is already installed.

        .. versionchanged:: 0.5
           Added the support for keyword arguments giving visit functions.
        """
        logger.debug('[app] adding node: %r', (node, kwargs))
        if not override and docutils.is_node_registered(node):
            logger.warning(__('node class %r is already registered, '
                              'its visitors will be overridden'),
                           node.__name__, type='app', subtype='add_node')
        docutils.register_node(node)
        self.registry.add_translation_handlers(node, **kwargs)

    def add_enumerable_node(self, node: "Type[Element]", figtype: str,
                            title_getter: TitleGetter = None, override: bool = False,
                            **kwargs: Tuple[Callable, Callable]) -> None:
        """Register a Docutils node class as a numfig target.

        Sphinx numbers the node automatically. And then the users can refer it
        using :rst:role:`numref`.

        *figtype* is a type of enumerable nodes.  Each figtypes have individual
        numbering sequences.  As a system figtypes, ``figure``, ``table`` and
        ``code-block`` are defined.  It is able to add custom nodes to these
        default figtypes.  It is also able to define new custom figtype if new
        figtype is given.

        *title_getter* is a getter function to obtain the title of node.  It
        takes an instance of the enumerable node, and it must return its title
        as string.  The title is used to the default title of references for
        :rst:role:`ref`.  By default, Sphinx searches
        ``docutils.nodes.caption`` or ``docutils.nodes.title`` from the node as
        a title.

        Other keyword arguments are used for node visitor functions. See the
        :meth:`.Sphinx.add_node` for details.

        If *override* is True, the given *node* is forcedly installed even if
        a node having the same name is already installed.

        .. versionadded:: 1.4
        """
        self.registry.add_enumerable_node(node, figtype, title_getter, override=override)
        self.add_node(node, override=override, **kwargs)

    def add_directive(self, name: str, cls: "Type[Directive]", override: bool = False) -> None:
        """Register a Docutils directive.

        *name* must be the prospective directive name.  *cls* is a directive
        class which inherits ``docutils.parsers.rst.Directive``.  For more
        details, see `the Docutils docs
        <http://docutils.sourceforge.net/docs/howto/rst-directives.html>`_ .

        For example, a custom directive named ``my-directive`` would be added
        like this:

        .. code-block:: python

           from docutils.parsers.rst import Directive, directives

           class MyDirective(Directive):
               has_content = True
               required_arguments = 1
               optional_arguments = 0
               final_argument_whitespace = True
               option_spec = {
                   'class': directives.class_option,
                   'name': directives.unchanged,
               }

               def run(self):
                   ...

           def setup(app):
               add_directive('my-directive', MyDirective)

        If *override* is True, the given *cls* is forcedly installed even if
        a directive named as *name* is already installed.

        .. versionchanged:: 0.6
           Docutils 0.5-style directive classes are now supported.
        .. deprecated:: 1.8
           Docutils 0.4-style (function based) directives support is deprecated.
        .. versionchanged:: 1.8
           Add *override* keyword.
        """
        logger.debug('[app] adding directive: %r', (name, cls))
        if not override and docutils.is_directive_registered(name):
            logger.warning(__('directive %r is already registered, it will be overridden'),
                           name, type='app', subtype='add_directive')

        docutils.register_directive(name, cls)

    def add_role(self, name: str, role: Any, override: bool = False) -> None:
        """Register a Docutils role.

        *name* must be the role name that occurs in the source, *role* the role
        function. Refer to the `Docutils documentation
        <http://docutils.sourceforge.net/docs/howto/rst-roles.html>`_ for
        more information.

        If *override* is True, the given *role* is forcedly installed even if
        a role named as *name* is already installed.

        .. versionchanged:: 1.8
           Add *override* keyword.
        """
        logger.debug('[app] adding role: %r', (name, role))
        if not override and docutils.is_role_registered(name):
            logger.warning(__('role %r is already registered, it will be overridden'),
                           name, type='app', subtype='add_role')
        docutils.register_role(name, role)

    def add_generic_role(self, name: str, nodeclass: Any, override: bool = False) -> None:
        """Register a generic Docutils role.

        Register a Docutils role that does nothing but wrap its contents in the
        node given by *nodeclass*.

        If *override* is True, the given *nodeclass* is forcedly installed even if
        a role named as *name* is already installed.

        .. versionadded:: 0.6
        .. versionchanged:: 1.8
           Add *override* keyword.
        """
        # Don't use ``roles.register_generic_role`` because it uses
        # ``register_canonical_role``.
        logger.debug('[app] adding generic role: %r', (name, nodeclass))
        if not override and docutils.is_role_registered(name):
            logger.warning(__('role %r is already registered, it will be overridden'),
                           name, type='app', subtype='add_generic_role')
        role = roles.GenericRole(name, nodeclass)
        docutils.register_role(name, role)

    def add_domain(self, domain: "Type[Domain]", override: bool = False) -> None:
        """Register a domain.

        Make the given *domain* (which must be a class; more precisely, a
        subclass of :class:`~sphinx.domains.Domain`) known to Sphinx.

        If *override* is True, the given *domain* is forcedly installed even if
        a domain having the same name is already installed.

        .. versionadded:: 1.0
        .. versionchanged:: 1.8
           Add *override* keyword.
        """
        self.registry.add_domain(domain, override=override)

    def add_directive_to_domain(self, domain: str, name: str,
                                cls: "Type[Directive]", override: bool = False) -> None:
        """Register a Docutils directive in a domain.

        Like :meth:`add_directive`, but the directive is added to the domain
        named *domain*.

        If *override* is True, the given *directive* is forcedly installed even if
        a directive named as *name* is already installed.

        .. versionadded:: 1.0
        .. versionchanged:: 1.8
           Add *override* keyword.
        """
        self.registry.add_directive_to_domain(domain, name, cls, override=override)

    def add_role_to_domain(self, domain: str, name: str, role: Union[RoleFunction, XRefRole],
                           override: bool = False) -> None:
        """Register a Docutils role in a domain.

        Like :meth:`add_role`, but the role is added to the domain named
        *domain*.

        If *override* is True, the given *role* is forcedly installed even if
        a role named as *name* is already installed.

        .. versionadded:: 1.0
        .. versionchanged:: 1.8
           Add *override* keyword.
        """
        self.registry.add_role_to_domain(domain, name, role, override=override)

    def add_index_to_domain(self, domain: str, index: "Type[Index]", override: bool = False
                            ) -> None:
        """Register a custom index for a domain.

        Add a custom *index* class to the domain named *domain*.  *index* must
        be a subclass of :class:`~sphinx.domains.Index`.

        If *override* is True, the given *index* is forcedly installed even if
        an index having the same name is already installed.

        .. versionadded:: 1.0
        .. versionchanged:: 1.8
           Add *override* keyword.
        """
        self.registry.add_index_to_domain(domain, index)

    def add_object_type(self, directivename: str, rolename: str, indextemplate: str = '',
                        parse_node: Callable = None, ref_nodeclass: "Type[TextElement]" = None,
                        objname: str = '', doc_field_types: List = [], override: bool = False
                        ) -> None:
        """Register a new object type.

        This method is a very convenient way to add a new :term:`object` type
        that can be cross-referenced.  It will do this:

        - Create a new directive (called *directivename*) for documenting an
          object.  It will automatically add index entries if *indextemplate*
          is nonempty; if given, it must contain exactly one instance of
          ``%s``.  See the example below for how the template will be
          interpreted.
        - Create a new role (called *rolename*) to cross-reference to these
          object descriptions.
        - If you provide *parse_node*, it must be a function that takes a
          string and a docutils node, and it must populate the node with
          children parsed from the string.  It must then return the name of the
          item to be used in cross-referencing and index entries.  See the
          :file:`conf.py` file in the source for this documentation for an
          example.
        - The *objname* (if not given, will default to *directivename*) names
          the type of object.  It is used when listing objects, e.g. in search
          results.

        For example, if you have this call in a custom Sphinx extension::

           app.add_object_type('directive', 'dir', 'pair: %s; directive')

        you can use this markup in your documents::

           .. rst:directive:: function

              Document a function.

           <...>

           See also the :rst:dir:`function` directive.

        For the directive, an index entry will be generated as if you had prepended ::

           .. index:: pair: function; directive

        The reference node will be of class ``literal`` (so it will be rendered
        in a proportional font, as appropriate for code) unless you give the
        *ref_nodeclass* argument, which must be a docutils node class.  Most
        useful are ``docutils.nodes.emphasis`` or ``docutils.nodes.strong`` --
        you can also use ``docutils.nodes.generated`` if you want no further
        text decoration.  If the text should be treated as literal (e.g. no
        smart quote replacement), but not have typewriter styling, use
        ``sphinx.addnodes.literal_emphasis`` or
        ``sphinx.addnodes.literal_strong``.

        For the role content, you have the same syntactical possibilities as
        for standard Sphinx roles (see :ref:`xref-syntax`).

        If *override* is True, the given object_type is forcedly installed even if
        an object_type having the same name is already installed.

        .. versionchanged:: 1.8
           Add *override* keyword.
        """
        self.registry.add_object_type(directivename, rolename, indextemplate, parse_node,
                                      ref_nodeclass, objname, doc_field_types,
                                      override=override)

    def add_crossref_type(self, directivename: str, rolename: str, indextemplate: str = '',
                          ref_nodeclass: "Type[TextElement]" = None, objname: str = '',
                          override: bool = False) -> None:
        """Register a new crossref object type.

        This method is very similar to :meth:`add_object_type` except that the
        directive it generates must be empty, and will produce no output.

        That means that you can add semantic targets to your sources, and refer
        to them using custom roles instead of generic ones (like
        :rst:role:`ref`).  Example call::

           app.add_crossref_type('topic', 'topic', 'single: %s',
                                 docutils.nodes.emphasis)

        Example usage::

           .. topic:: application API

           The application API
           -------------------

           Some random text here.

           See also :topic:`this section <application API>`.

        (Of course, the element following the ``topic`` directive needn't be a
        section.)

        If *override* is True, the given crossref_type is forcedly installed even if
        a crossref_type having the same name is already installed.

        .. versionchanged:: 1.8
           Add *override* keyword.
        """
        self.registry.add_crossref_type(directivename, rolename,
                                        indextemplate, ref_nodeclass, objname,
                                        override=override)

    def add_transform(self, transform: "Type[Transform]") -> None:
        """Register a Docutils transform to be applied after parsing.

        Add the standard docutils :class:`Transform` subclass *transform* to
        the list of transforms that are applied after Sphinx parses a reST
        document.

        .. list-table:: priority range categories for Sphinx transforms
           :widths: 20,80

           * - Priority
             - Main purpose in Sphinx
           * - 0-99
             - Fix invalid nodes by docutils. Translate a doctree.
           * - 100-299
             - Preparation
           * - 300-399
             - early
           * - 400-699
             - main
           * - 700-799
             - Post processing. Deadline to modify text and referencing.
           * - 800-899
             - Collect referencing and referenced nodes. Domain processing.
           * - 900-999
             - Finalize and clean up.

        refs: `Transform Priority Range Categories`__

        __ http://docutils.sourceforge.net/docs/ref/transforms.html#transform-priority-range-categories
        """  # NOQA
        self.registry.add_transform(transform)

    def add_post_transform(self, transform: "Type[Transform]") -> None:
        """Register a Docutils transform to be applied before writing.

        Add the standard docutils :class:`Transform` subclass *transform* to
        the list of transforms that are applied before Sphinx writes a
        document.
        """
        self.registry.add_post_transform(transform)

    def add_javascript(self, filename: str, **kwargs: str) -> None:
        """An alias of :meth:`add_js_file`."""
        warnings.warn('The app.add_javascript() is deprecated. '
                      'Please use app.add_js_file() instead.',
                      RemovedInSphinx40Warning, stacklevel=2)
        self.add_js_file(filename, **kwargs)

    def add_js_file(self, filename: str, **kwargs: str) -> None:
        """Register a JavaScript file to include in the HTML output.

        Add *filename* to the list of JavaScript files that the default HTML
        template will include.  The filename must be relative to the HTML
        static path , or a full URI with scheme.  If the keyword argument
        ``body`` is given, its value will be added between the
        ``<script>`` tags. Extra keyword arguments are included as
        attributes of the ``<script>`` tag.

        Example::

            app.add_js_file('example.js')
            # => <script src="_static/example.js"></script>

            app.add_js_file('example.js', async="async")
            # => <script src="_static/example.js" async="async"></script>

            app.add_js_file(None, body="var myVariable = 'foo';")
            # => <script>var myVariable = 'foo';</script>

        .. versionadded:: 0.5

        .. versionchanged:: 1.8
           Renamed from ``app.add_javascript()``.
           And it allows keyword arguments as attributes of script tag.
        """
        self.registry.add_js_file(filename, **kwargs)
        if hasattr(self.builder, 'add_js_file'):
            self.builder.add_js_file(filename, **kwargs)  # type: ignore

    def add_css_file(self, filename: str, **kwargs: str) -> None:
        """Register a stylesheet to include in the HTML output.

        Add *filename* to the list of CSS files that the default HTML template
        will include.  The filename must be relative to the HTML static path,
        or a full URI with scheme.  The keyword arguments are also accepted for
        attributes of ``<link>`` tag.

        Example::

            app.add_css_file('custom.css')
            # => <link rel="stylesheet" href="_static/custom.css" type="text/css" />

            app.add_css_file('print.css', media='print')
            # => <link rel="stylesheet" href="_static/print.css"
            #          type="text/css" media="print" />

            app.add_css_file('fancy.css', rel='alternate stylesheet', title='fancy')
            # => <link rel="alternate stylesheet" href="_static/fancy.css"
            #          type="text/css" title="fancy" />

        .. versionadded:: 1.0

        .. versionchanged:: 1.6
           Optional ``alternate`` and/or ``title`` attributes can be supplied
           with the *alternate* (of boolean type) and *title* (a string)
           arguments. The default is no title and *alternate* = ``False``. For
           more information, refer to the `documentation
           <https://mdn.io/Web/CSS/Alternative_style_sheets>`__.

        .. versionchanged:: 1.8
           Renamed from ``app.add_stylesheet()``.
           And it allows keyword arguments as attributes of link tag.
        """
        logger.debug('[app] adding stylesheet: %r', filename)
        self.registry.add_css_files(filename, **kwargs)
        if hasattr(self.builder, 'add_css_file'):
            self.builder.add_css_file(filename, **kwargs)  # type: ignore

    def add_stylesheet(self, filename: str, alternate: bool = False, title: str = None
                       ) -> None:
        """An alias of :meth:`add_css_file`."""
        warnings.warn('The app.add_stylesheet() is deprecated. '
                      'Please use app.add_css_file() instead.',
                      RemovedInSphinx40Warning, stacklevel=2)

        attributes = {}  # type: Dict[str, str]
        if alternate:
            attributes['rel'] = 'alternate stylesheet'
        else:
            attributes['rel'] = 'stylesheet'

        if title:
            attributes['title'] = title

        self.add_css_file(filename, **attributes)

    def add_latex_package(self, packagename: str, options: str = None,
                          after_hyperref: bool = False) -> None:
        r"""Register a package to include in the LaTeX source code.

        Add *packagename* to the list of packages that LaTeX source code will
        include.  If you provide *options*, it will be taken to `\usepackage`
        declaration.  If you set *after_hyperref* truthy, the package will be
        loaded after ``hyperref`` package.

        .. code-block:: python

           app.add_latex_package('mypackage')
           # => \usepackage{mypackage}
           app.add_latex_package('mypackage', 'foo,bar')
           # => \usepackage[foo,bar]{mypackage}

        .. versionadded:: 1.3
        .. versionadded:: 3.1

           *after_hyperref* option.
        """
        self.registry.add_latex_package(packagename, options, after_hyperref)

    def add_lexer(self, alias: str, lexer: Union[Lexer, "Type[Lexer]"]) -> None:
        """Register a new lexer for source code.

        Use *lexer* to highlight code blocks with the given language *alias*.

        .. versionadded:: 0.6
        .. versionchanged:: 2.1
           Take a lexer class as an argument.  An instance of lexers are
           still supported until Sphinx-3.x.
        """
        logger.debug('[app] adding lexer: %r', (alias, lexer))
        if isinstance(lexer, Lexer):
            warnings.warn('app.add_lexer() API changed; '
                          'Please give lexer class instead of instance',
                          RemovedInSphinx40Warning, stacklevel=2)
            lexers[alias] = lexer
        else:
            lexer_classes[alias] = lexer

    def add_autodocumenter(self, cls: Any, override: bool = False) -> None:
        """Register a new documenter class for the autodoc extension.

        Add *cls* as a new documenter class for the :mod:`sphinx.ext.autodoc`
        extension.  It must be a subclass of
        :class:`sphinx.ext.autodoc.Documenter`.  This allows to auto-document
        new types of objects.  See the source of the autodoc module for
        examples on how to subclass :class:`Documenter`.

        If *override* is True, the given *cls* is forcedly installed even if
        a documenter having the same name is already installed.

        .. todo:: Add real docs for Documenter and subclassing

        .. versionadded:: 0.6
        .. versionchanged:: 2.2
           Add *override* keyword.
        """
        logger.debug('[app] adding autodocumenter: %r', cls)
        from sphinx.ext.autodoc.directive import AutodocDirective
        self.registry.add_documenter(cls.objtype, cls)
        self.add_directive('auto' + cls.objtype, AutodocDirective, override=override)

    def add_autodoc_attrgetter(self, typ: "Type", getter: Callable[[Any, str, Any], Any]
                               ) -> None:
        """Register a new ``getattr``-like function for the autodoc extension.

        Add *getter*, which must be a function with an interface compatible to
        the :func:`getattr` builtin, as the autodoc attribute getter for
        objects that are instances of *typ*.  All cases where autodoc needs to
        get an attribute of a type are then handled by this function instead of
        :func:`getattr`.

        .. versionadded:: 0.6
        """
        logger.debug('[app] adding autodoc attrgetter: %r', (typ, getter))
        self.registry.add_autodoc_attrgetter(typ, getter)

    def add_search_language(self, cls: Any) -> None:
        """Register a new language for the HTML search index.

        Add *cls*, which must be a subclass of
        :class:`sphinx.search.SearchLanguage`, as a support language for
        building the HTML full-text search index.  The class must have a *lang*
        attribute that indicates the language it should be used for.  See
        :confval:`html_search_language`.

        .. versionadded:: 1.1
        """
        logger.debug('[app] adding search language: %r', cls)
        from sphinx.search import languages, SearchLanguage
        assert issubclass(cls, SearchLanguage)
        languages[cls.lang] = cls

    def add_source_suffix(self, suffix: str, filetype: str, override: bool = False) -> None:
        """Register a suffix of source files.

        Same as :confval:`source_suffix`.  The users can override this
        using the setting.

        If *override* is True, the given *suffix* is forcedly installed even if
        a same suffix is already installed.

        .. versionadded:: 1.8
        """
        self.registry.add_source_suffix(suffix, filetype, override=override)

    def add_source_parser(self, parser: "Type[Parser]", override: bool = False) -> None:
        """Register a parser class.

        If *override* is True, the given *parser* is forcedly installed even if
        a parser for the same suffix is already installed.

        .. versionadded:: 1.4
        .. versionchanged:: 1.8
           *suffix* argument is deprecated.  It only accepts *parser* argument.
           Use :meth:`add_source_suffix` API to register suffix instead.
        .. versionchanged:: 1.8
           Add *override* keyword.
        """
        self.registry.add_source_parser(parser, override=override)

    def add_env_collector(self, collector: "Type[EnvironmentCollector]") -> None:
        """Register an environment collector class.

        Refer to :ref:`collector-api`.

        .. versionadded:: 1.6
        """
        logger.debug('[app] adding environment collector: %r', collector)
        collector().enable(self)

    def add_html_theme(self, name: str, theme_path: str) -> None:
        """Register a HTML Theme.

        The *name* is a name of theme, and *path* is a full path to the theme
        (refs: :ref:`distribute-your-theme`).

        .. versionadded:: 1.6
        """
        logger.debug('[app] adding HTML theme: %r, %r', name, theme_path)
        self.html_themes[name] = theme_path

    def add_html_math_renderer(self, name: str,
                               inline_renderers: Tuple[Callable, Callable] = None,
                               block_renderers: Tuple[Callable, Callable] = None) -> None:
        """Register a math renderer for HTML.

        The *name* is a name of math renderer.  Both *inline_renderers* and
        *block_renderers* are used as visitor functions for the HTML writer:
        the former for inline math node (``nodes.math``), the latter for
        block math node (``nodes.math_block``).  Regarding visitor functions,
        see :meth:`add_node` for details.

        .. versionadded:: 1.8

        """
        self.registry.add_html_math_renderer(name, inline_renderers, block_renderers)

    def add_message_catalog(self, catalog: str, locale_dir: str) -> None:
        """Register a message catalog.

        The *catalog* is a name of catalog, and *locale_dir* is a base path
        of message catalog.  For more details, see
        :func:`sphinx.locale.get_translation()`.

        .. versionadded:: 1.8
        """
        locale.init([locale_dir], self.config.language, catalog)
        locale.init_console(locale_dir, catalog)

    # ---- other methods -------------------------------------------------
    def is_parallel_allowed(self, typ: str) -> bool:
        """Check parallel processing is allowed or not.

        ``typ`` is a type of processing; ``'read'`` or ``'write'``.
        """
        if typ == 'read':
            attrname = 'parallel_read_safe'
            message_not_declared = __("the %s extension does not declare if it "
                                      "is safe for parallel reading, assuming "
                                      "it isn't - please ask the extension author "
                                      "to check and make it explicit")
            message_not_safe = __("the %s extension is not safe for parallel reading")
        elif typ == 'write':
            attrname = 'parallel_write_safe'
            message_not_declared = __("the %s extension does not declare if it "
                                      "is safe for parallel writing, assuming "
                                      "it isn't - please ask the extension author "
                                      "to check and make it explicit")
            message_not_safe = __("the %s extension is not safe for parallel writing")
        else:
            raise ValueError('parallel type %s is not supported' % typ)

        for ext in self.extensions.values():
            allowed = getattr(ext, attrname, None)
            if allowed is None:
                logger.warning(message_not_declared, ext.name)
                logger.warning(__('doing serial %s'), typ)
                return False
            elif not allowed:
                logger.warning(message_not_safe, ext.name)
                logger.warning(__('doing serial %s'), typ)
                return False

        return True


class TemplateBridge:
    """
    This class defines the interface for a "template bridge", that is, a class
    that renders templates given a template name and a context.
    """

    def init(self, builder: "Builder", theme: Theme = None, dirs: List[str] = None) -> None:
        """Called by the builder to initialize the template system.

        *builder* is the builder object; you'll probably want to look at the
        value of ``builder.config.templates_path``.

        *theme* is a :class:`sphinx.theming.Theme` object or None; in the latter
        case, *dirs* can be list of fixed directories to look for templates.
        """
        raise NotImplementedError('must be implemented in subclasses')

    def newest_template_mtime(self) -> float:
        """Called by the builder to determine if output files are outdated
        because of template changes.  Return the mtime of the newest template
        file that was changed.  The default implementation returns ``0``.
        """
        return 0

    def render(self, template: str, context: Dict) -> None:
        """Called by the builder to render a template given as a filename with
        a specified context (a Python dictionary).
        """
        raise NotImplementedError('must be implemented in subclasses')

    def render_string(self, template: str, context: Dict) -> str:
        """Called by the builder to render a template given as a string with a
        specified context (a Python dictionary).
        """
        raise NotImplementedError('must be implemented in subclasses')
