use core:io;
use parser;

/**
 * A HTTP response.
 */
class Response {
	// HTTP version.
	Version version = Version:HTTP_1_0;

	// HTTP status.
	Status status = Status:OK;

	// Headers to send. Keys are assumed to be all lowercase.
	private Str->Str headers;

	// Cookies to send. These are added automatically as headers if present.
	Cookie[] cookies;

	// Body of the response.
	Buffer body;

	// Default constructor.
	init() {
		init {}
	}

	// Create a 200 response that contains text/html.
	init(Str response) {
		init {
			body = response.toUtf8();
		}
		contentType = "text/html; charset=utf-8";
	}

	// Create a response with a particular status and an associated message.
	init(Str response, Status status) {
		self(response);
		status = status;
	}

	// Create with version and status.
	init(Version version, Status status) {
		init {
			version = version;
			status = status;
		}
	}

	// Explicit set for content-type.
	assign contentType(Str type) {
		headers.put("content-type", type);
	}

	// Set header.
	assign header(Str key, Str val) {
		headers.put(key.toAsciiLowercase, val);
	}

	// Set header.
	assign header(Buffer key, Buffer val) {
		headers.put(key.toAsciiLowercase, val.fromUtf8);
	}

	// Get header.
	Str? header(Str key) {
		return headers.at(key.toAsciiLowercase);
	}

	// Get header, assume it exists.
	Str getHeader(Str key) {
		unless (r = header(key))
			throw HttpError("Header ${key} does not exist.");
		r;
	}

	// To string.
	void toS(StrBuf to) {
		to << "response {";
		{
			Indent z(to);
			to << "\nversion: " << version;
			to << "\nstatus: " << status;
			to << "\nheaders: " << headers;
			to << "\ncookies: " << cookies;
			to << "\nbody: " << body;
		}
		to << "\n}";
	}

	// Serialize the request into a binary object.
	// Returns 'true' if the connection should be closed afterwards.
	Bool write(OStream to) {
		Utf8Output text(to, windowsTextInfo(false));
		text.autoFlush = false;

		if (version == Version:HTTP_0_9) {
			// HTTP 0.9 is special.
			to.write(body);
			return true;
		} else if (version == Version:HTTP_1_0) {
			text.write("HTTP/1.0 ");
		} else if (version == Version:HTTP_1_1) {
			text.write("HTTP/1.1 ");
		} else {
			throw HttpError("Invalid protocol version!");
		}

		text.write(status.v.toS);
		text.write(" ");
		text.write(status.toS);
		text.write("\n");

		for (k, v in headers) {
			if (k == "content-length")
				continue;
			if (k == "content-encoding")
				continue;
			text.write(k);
			text.write(": ");
			text.write(v);
			text.write("\n");
		}

		for (l in cookies) {
			if (!l.cookieValid)
				continue;

			text.write("Set-Cookie: ");
			text.write(l.toS);
			text.write("\n");
		}

		text.write("Content-Length: ");
		text.write(body.filled.toS);
		text.write("\n\n");
		text.flush();

		to.write(body);

		return false;
	}
}

parseResponseHeader : parser(recursive descent, binary) {
	start = SHttpResponse;

	optional delimiter = SOptDelimiter;
	required delimiter = SReqDelimiter;

	void SOptDelimiter();
	SOptDelimiter : " *";

	void SReqDelimiter();
	SReqDelimiter : " +";

	void SNewline();
	SNewline : "\r\n";

	Response SHttpResponse();
	SHttpResponse => Response(version, status)
		: SVersion version ~ SStatus status ~ "[^\r\n]+" - SNewline - SHeaders(me);

	Version SVersion();
	SVersion => Version.HTTP_1_0() : "HTTP/1.0";
	SVersion => Version.HTTP_1_1() : "HTTP/1.1";

	Status SStatus();
	SStatus => toStatus(code) : "[0-9]+" code;

	void SHeaders(Response r);
	SHeaders : SNewline;
	SHeaders => header(r, k, v) : "[^:\r\n]+" k - ":" ~ "[^\r\n]+" v - SNewline - SHeaders(r);
}

private Status toStatus(Buffer buffer) {
	Status(buffer.fromUtf8().toNat());
}
