/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Maskbox    masklonlatbox   Mask lon/lat box
      Maskbox    maskindexbox    Mask index box
      Maskbox    maskregion      Mask regions
*/

#include <cdi.h>

#include <utility>

#include "cdo_options.h"
#include "process_int.h"
#include "readline.h"
#include <mpim_grid.h>
#include "selboxinfo.h"

static int
ReadCoords(double *xvals, double *yvals, const char *polyfile, FILE *fp)
{
  int z = 0, number = 0, jumpedlines = 0;
  constexpr size_t MAX_LINE = 256;
  char line[MAX_LINE];

  while (cdo::readline(fp, line, MAX_LINE))
    {
      int i = 0;
      if (line[0] == '#' || line[0] == '\0')
        {
          jumpedlines++;
          continue;
        }
      if (line[0] == '&') break;

      char *linep = &line[0];

      const auto xcoord = strtod(linep, &linep);

      if (std::fabs(xcoord) <= 0) jumpedlines++;

      while (((isdigit((int) *linep) == false) && (*linep != '-')) && (i < 64))
        {
          if (*linep == 0)
            {
              cdoAbort(" y value missing in file %s at  line %d", polyfile, (number + jumpedlines + 1));
              break;
            }
          if ((isspace((int) *linep) == false && (*linep != '-')) && (linep != nullptr))
            cdoWarning("unknown character in file %s at line %d", polyfile, (number + jumpedlines + 1));

          linep++;
          i++;
        }
      if ((i >= 63) && (number != 0)) cdoAbort("Wrong value format in file %s at line %d", polyfile, (number + jumpedlines + 1));

      const auto ycoord = strtod(linep, nullptr);
      xvals[number] = xcoord;
      yvals[number] = ycoord;
      number++;
    }

  if ((number != 0) && (!(IS_EQUAL(xvals[0], xvals[number - 1]) && IS_EQUAL(yvals[0], yvals[number - 1]))))
    {
      xvals[number] = xvals[0];
      yvals[number] = yvals[0];
      number++;
    }

  if (Options::cdoVerbose)
    {
      for (z = 0; z < number; z++) fprintf(stderr, "%d %g %g\n", (z + 1), xvals[z], yvals[z]);
    }

  return number;
}

static void
maskbox(std::vector<bool> &mask, const int gridID, const SelboxInfo &sbox)
{
  const auto &lat1 = sbox.lat1;
  const auto &lat2 = sbox.lat2;
  const auto &lon11 = sbox.lon11;
  const auto &lon12 = sbox.lon12;
  const auto &lon21 = sbox.lon21;
  const auto &lon22 = sbox.lon22;
  const long nlon = gridInqXsize(gridID);
  const long nlat = gridInqYsize(gridID);

  for (long ilat = 0; ilat < nlat; ilat++)
    for (long ilon = 0; ilon < nlon; ilon++)
      if ((lat1 <= ilat && ilat <= lat2 && ((lon11 <= ilon && ilon <= lon12) || (lon21 <= ilon && ilon <= lon22))))
        mask[nlon * ilat + ilon] = false;
}

void getlonlatparams(int argc_offset, double &xlon1, double &xlon2, double &xlat1, double &xlat2);

static void
maskbox_cell(std::vector<bool> &mask, const int gridID)
{
  double xlon1 = 0, xlon2 = 0, xlat1 = 0, xlat2 = 0;
  getlonlatparams(0, xlon1, xlon2, xlat1, xlat2);

  const auto gridsize = gridInqSize(gridID);

  Varray<double> xvals(gridsize), yvals(gridsize);
  gridInqXvals(gridID, xvals.data());
  gridInqYvals(gridID, yvals.data());

  // Convert lat/lon units if required
  cdo_grid_to_degree(gridID, CDI_XAXIS, gridsize, xvals.data(), "grid center lon");
  cdo_grid_to_degree(gridID, CDI_YAXIS, gridsize, yvals.data(), "grid center lat");

  if (xlon1 > xlon2) cdoAbort("The second longitude have to be greater than the first one!");

  if (xlat1 > xlat2) std::swap(xlat1, xlat2);

  for (size_t i = 0; i < gridsize; ++i)
    {
      mask[i] = true;

      const auto xval = xvals[i];
      const auto yval = yvals[i];
      if (yval >= xlat1 && yval <= xlat2)
        {
          if (((xval >= xlon1 && xval <= xlon2) || (xval - 360 >= xlon1 && xval - 360 <= xlon2)
               || (xval + 360 >= xlon1 && xval + 360 <= xlon2)))
            {
              mask[i] = false;
            }
        }
    }
}

static void
maskregion(std::vector<bool> &mask, const int gridID, const Varray<double> &xcoords, const Varray<double> &ycoords,
           const size_t nofcoords)
{
  const auto nlon = gridInqXsize(gridID);
  const auto nlat = gridInqYsize(gridID);

  Varray<double> xvals(nlon), yvals(nlat);
  gridInqXvals(gridID, xvals.data());
  gridInqYvals(gridID, yvals.data());

  // Convert lat/lon units if required
  cdo_grid_to_degree(gridID, CDI_XAXIS, nlon, xvals.data(), "grid center lon");
  cdo_grid_to_degree(gridID, CDI_YAXIS, nlat, yvals.data(), "grid center lat");

  const auto xmm = varrayMinMax(xvals);
  const auto ymm = varrayMinMax(yvals);

  for (size_t ilat = 0; ilat < nlat; ilat++)
    {
      const auto yval = yvals[ilat];
      for (size_t ilon = 0; ilon < nlon; ilon++)
        {
          int c = 0;
          const auto xval = xvals[ilon];
          if (!(((xval > xmm.min) || (xval < xmm.max)) || ((yval > ymm.min) || (yval < ymm.max)))) c = !c;

          if (c == 0)
            {
              for (size_t i = 0, j = nofcoords - 1; i < nofcoords; j = i++)
                {
                  if ((((ycoords[i] <= yval) && (yval < ycoords[j])) || ((ycoords[j] <= yval) && (yval < ycoords[i])))
                      && ((xval) < (xcoords[j] - (xcoords[i])) * (yval - ycoords[i]) / (ycoords[j] - ycoords[i]) + (xcoords[i])))
                    c = !c;
                }
            }

          if (c == 0)
            {
              for (size_t i = 0, j = nofcoords - 1; i < nofcoords; j = i++)
                {
                  if (xvals[ilon] > 180)
                    {
                      if ((((ycoords[i] <= yval) && (yval < ycoords[j])) || ((ycoords[j] <= yval) && (yval < ycoords[i])))
                          && ((xval - 360)
                              < (xcoords[j] - (xcoords[i])) * (yval - ycoords[i]) / (ycoords[j] - ycoords[i]) + (xcoords[i])))
                        c = !c;
                    }
                }
            }

          if (c == 0)
            {
              for (size_t i = 0, j = nofcoords - 1; i < nofcoords; j = i++)
                {
                  if (xval < 0)
                    {
                      if ((((ycoords[i] <= yval) && (yval < ycoords[j])) || ((ycoords[j] <= yval) && (yval < ycoords[i])))
                          && ((xval + 360)
                              < (xcoords[j] - (xcoords[i])) * (yval - ycoords[i]) / (ycoords[j] - ycoords[i]) + (xcoords[i])))
                        c = !c;
                    }
                }
            }

          if (c) mask[nlon * ilat + ilon] = false;
        }
    }
}

void *
Maskbox(void *process)
{
  int nrecs;
  int varID, levelID;
  int index, gridtype = -1;
  SelboxInfo sbox;

  cdoInitialize(process);

  // clang-format off
  const auto MASKLONLATBOX = cdoOperatorAdd("masklonlatbox", 0, 0, "western and eastern longitude and southern and northern latitude");
  const auto MASKINDEXBOX  = cdoOperatorAdd("maskindexbox",  0, 0, "index of first and last longitude and index of first and last latitude");
  const auto MASKREGION    = cdoOperatorAdd("maskregion",    0, 0, "limiting coordinates of the region");
  // clang-format on

  const auto operatorID = cdoOperatorID();

  operatorInputArg(cdoOperatorEnter(operatorID));

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto nvars = vlistNvars(vlistID1);
  std::vector<bool> vars(nvars, false);

  const auto ngrids = vlistNgrids(vlistID1);
  int ndiffgrids = 0;
  for (index = 1; index < ngrids; index++)
    if (vlistGrid(vlistID1, 0) != vlistGrid(vlistID1, index)) ndiffgrids++;

  int gridID = -1;
  for (index = 0; index < ngrids; index++)
    {
      gridID = vlistGrid(vlistID1, index);
      gridtype = gridInqType(gridID);

      if (gridtype == GRID_LONLAT || gridtype == GRID_GAUSSIAN) break;
      if (operatorID != MASKREGION && gridtype == GRID_CURVILINEAR) break;
      if (operatorID != MASKREGION && gridtype == GRID_UNSTRUCTURED) break;
      if (operatorID == MASKINDEXBOX && gridtype == GRID_GENERIC && gridInqXsize(gridID) > 0 && gridInqYsize(gridID) > 0) break;
    }

  if (gridtype == GRID_GAUSSIAN_REDUCED) cdoAbort("Gaussian reduced grid found. Use option -R to convert it to a regular grid!");

  if (index == ngrids) cdoAbort("No regular lon/lat grid found!");
  if (ndiffgrids > 0) cdoAbort("Too many different grids!");

  operatorInputArg(cdoOperatorEnter(operatorID));

  for (varID = 0; varID < nvars; varID++) vars[varID] = (gridID == vlistInqVarGrid(vlistID1, varID));

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  const auto gridsize = gridInqSize(gridID);
  Varray<double> array(gridsize);
  std::vector<bool> mask(gridsize, true);

  if (operatorID == MASKLONLATBOX)
    {
      if (gridtype == GRID_CURVILINEAR || gridtype == GRID_UNSTRUCTURED)
        {
          maskbox_cell(mask, gridID);
        }
      else
        {
          genlonlatbox(0, gridID, sbox);
          maskbox(mask, gridID, sbox);
        }
    }
  else if (operatorID == MASKINDEXBOX)
    {
      genindexbox(0, gridID, sbox);
      maskbox(mask, gridID, sbox);
    }
  else if (operatorID == MASKREGION)
    {
      constexpr size_t maxVals = 1048576;
      Varray<double> xcoords(maxVals), ycoords(maxVals);
      const auto nfiles = operatorArgc();

      for (int i = 0; i < nfiles; i++)
        {
          const auto polyfile = cdoOperatorArgv(i).c_str();
          auto fp = fopen(polyfile, "r");
          if (fp == nullptr) cdoAbort("Open failed on %s", polyfile);

          while (true)
            {
              const auto number = ReadCoords(xcoords.data(), ycoords.data(), polyfile, fp);
              if (number == 0) break;
              if (number < 3) cdoAbort("Too less values in file %s", polyfile);
              maskregion(mask, gridID, xcoords, ycoords, number);
            }
          fclose(fp);
        }
    }

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          if (vars[varID])
            {
              size_t nmiss;
              cdoReadRecord(streamID1, &array[0], &nmiss);

              const auto missval = vlistInqVarMissval(vlistID1, varID);
              for (size_t i = 0; i < gridsize; i++)
                if (mask[i]) array[i] = missval;

              nmiss = varrayNumMV(gridsize, array, missval);
              cdoDefRecord(streamID2, varID, levelID);
              cdoWriteRecord(streamID2, array.data(), nmiss);
            }
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
