#ifndef H_CDW_LIST_DISPLAY
#define H_CDW_LIST_DISPLAY

#include <stdbool.h>
#include <stddef.h> /* size_t */

#include "cdw_dll.h"
#include "cdw_colors.h"
#include "cdw_ncurses.h"
#include "cdw_widgets.h" /* N_RETURN_KEYS_MAX */


/* Maximal horizontal offset of contents of list display. The list may
   be scrolled to the right no more than this (I'm not willing to test
   more than this). */
#define CDW_LIST_DISPLAY_H_OFFSET_LIMIT   128


typedef struct {
	WINDOW *window;        /* parent window of display's subwindow */
	WINDOW *subwindow;     /* window, in which content of display's list is drawn */

	int display_format;   /* see enum below */

	/* sizes and offsets (vertical and horizontal) of
	   subwindow created by a display; */
	int n_lines;
	int n_cols;
	int begin_y;
        int begin_x;

	cdw_colors_t colors; /* color scheme used by a display */

	/* "item display function" */
	void (*display_item)(void *display, void *data, size_t row, size_t h_offset, bool isreverse);

	/* Horizontal offset of list displayed in display's subwindow.
	   The list may be scrolled left and right, but not all users
	   of the display need to support horizontal scrolling.

	   This is useful when displayed item is represented by line
	   with more chars than there is columns in display's
	   subwindow - the line should be moved (shifted) to left to
	   display content truncated by display's border. This value
	   affects all lines displayed in the display.*/
	size_t h_offset;

	/* list of items displayed in a display */
	cdw_dll_item_t *list;
	/* always points to current (currently highlighted) list item,
	   or is null for empty list */
	cdw_dll_item_t *current_item;
	/* index of current (currently highlighted) list item,
	   undefined if list is empty */
	size_t current_item_ind;
	/* number of items on the list, user of display should make
	   sure that it is always correct */
	size_t n_items;

	int n_return_chars;
	int return_chars[N_RETURN_KEYS_MAX];

} CDW_LIST_DISPLAY;


/* Possible values assigned to display->display_format field.
   Assigned by code creating a display, read and acted upon by
   "item display function". */
enum {
	CDW_LIST_DISPLAY_FORMAT_SHORT,
	CDW_LIST_DISPLAY_FORMAT_MIDDLE,
	CDW_LIST_DISPLAY_FORMAT_LONG
};



CDW_LIST_DISPLAY *cdw_list_display_new(WINDOW *parent, int n_lines, int n_cols, int begin_y, int begin_x, cdw_colors_t colors);
void              cdw_list_display_delete(CDW_LIST_DISPLAY **display);
int               cdw_list_display_driver(CDW_LIST_DISPLAY *display);
void              cdw_list_display_scroll_to(CDW_LIST_DISPLAY *display, size_t current_item_ind, size_t h_offset, bool highlight);

cdw_rv_t cdw_list_display_remove_item(CDW_LIST_DISPLAY *display, size_t item_ind);
cdw_rv_t cdw_list_display_add_item(CDW_LIST_DISPLAY *display, void *data, bool (*pred)(const void *, const void *));
void    *cdw_list_display_get_current_item_data(CDW_LIST_DISPLAY *display);
/* basically a wrapper around cdw_dll_is_member() */
bool     cdw_list_display_is_member(CDW_LIST_DISPLAY *display, void const * data, bool (*pred)(void const *, void const *));

void cdw_list_display_add_return_key(CDW_LIST_DISPLAY *display, int c);
void cdw_list_display_add_return_keys(CDW_LIST_DISPLAY *display, ...);
bool cdw_list_display_is_return_key(CDW_LIST_DISPLAY *display, int c);

void cdw_list_display_refresh(CDW_LIST_DISPLAY *display);

#endif /* #ifndef H_CDW_LIST_DISPLAY */
