/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "cdw_widgets.h"
#include "cdw_main_window.h"
#include "cdw_string.h"
#include "cdw_verify_wizard.h"
#include "cdw_debug.h"
#include "gettext.h"
#include "cdw_calculate_digest.h"
#include "cdw_form.h"
#include "cdw_task.h"
#include "cdw_window.h"

/**
   \file cdw_verify_wizard.c

   Simple, one page wizard: a dialog window with dropdown
   in which user can select mode of verification, i.e. what he wants
   to verify.
*/




static void     cdw_verify_wizard_init(void);
static cdw_rv_t cdw_verify_wizard_build(void);
static cdw_rv_t cdw_verify_wizard_driver(void);
static void     cdw_verify_wizard_destroy(void);
static cdw_rv_t cdw_verify_wizard_build_fields(void);

static CDW_DROPDOWN *cdw_verify_wizard_make_verify_mode_dropdown(WINDOW *window, int begin_y, int begin_x, int width);

static char *cdw_verify_wizard_make_message(int n_cols);

static struct {
	cdw_id_t mode;
	cdw_form_t *cdw_form;
} wizard;



static cdw_form_dropdown_maker_t dropdown_makers[] = {
	cdw_verify_wizard_make_verify_mode_dropdown
};


static cdw_form_text_maker_t text_makers[] = {
	cdw_verify_wizard_make_message
};



/* constants for layout of elements in wizard window */
#define window_n_cols             52
#define window_n_lines            20
#define subwindow_n_cols           (window_n_cols - 2)
#define subwindow_n_lines          (window_n_lines - 2)
#define label_n_cols               (subwindow_n_cols - 2)
#define dropdown_n_cols            (subwindow_n_cols - 2)

#define first_col                  1 /* main message and some labels start in leftmost column */
#define top_message_row            1
#define mode_row                   8
#define buttons_row                (subwindow_n_lines - 3) /* Verify and Cancel buttons are at the bottom of window */




#define CDW_VERIFY_WIZARD_N_FIELDS 4 /* does not include ending NULL element */
static FIELD *wizard_fields[CDW_VERIFY_WIZARD_N_FIELDS + 1];

enum {
	f_top_message_t = 0,

	f_mode_dd,

	f_ok_b,
	f_cancel_b
};




/**
   \brief Window displayed just before erasing is performed

   Wizard window displayed before actual erasing. User can select
   here some erasing / formatting options.

   In case of erasing a CD user can select erasing mode and erasing speed.
   In case of erasing a DVD user can select erasing mode, formatting mode and
   erasing / formatting speed.

   In case of erasing of DVD+/-RW user is also warned that multiple erasing
   of DVD can quickly render it unusable.

   \return CDW_OK if user pressed accepts erasing
   \return CDW_CANCEL if user cancels erasing
   \return CDW_GEN_ERROR on errors
*/
cdw_id_t cdw_verify_wizard(void)
{
	/* TODO: there is a problem with implicit initialization of
	   dropdowns in the wizard: dropdowns are initialized with
	   some deeply hidden values in functions creating dropdown,
	   and then (below) states of the dropdowns are explicitly
	   used to set values of fields in task variable;
	   this mix of implicit/explicit assignments is confusing */

	cdw_verify_wizard_init();

	cdw_rv_t crv = cdw_verify_wizard_build();
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to create wizard\n");
		cdw_verify_wizard_destroy();
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("ERROR"),
				   /* 2TRANS: this is message in dialog window */
				   _("Failed to create erasing wizard. Please restart cdw and try again."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_ERROR;
	}

	crv = cdw_verify_wizard_driver();
	if (crv == CDW_OK) {
		CDW_DROPDOWN *dd = cdw_form_get_dropdown(wizard.cdw_form, f_mode_dd);
		wizard.mode = cdw_dropdown_get_current_item_id(dd);
	} else {
		cdw_vdm ("INFO: not attempting erasing, not displaying settings\n");
		wizard.mode = CDW_CALCULATE_DIGEST_MODE_NONE;
	}

	cdw_verify_wizard_destroy();

	/* redraw parent */
	cdw_main_window_wrefresh();

	return wizard.mode;
}





cdw_rv_t cdw_verify_wizard_driver(void)
{
	int key = 'a'; /* safe initial value */
	int fi = f_mode_dd; /* Initial focus on "Mode" dropdown. */
	cdw_form_driver_focus_on(wizard.cdw_form, fi);

	while (key != CDW_KEY_ESCAPE && key != 'q' && key != 'Q') {
		key = cdw_form_driver_new(wizard.cdw_form, &fi);

		/* in this wizard driver we are interested only in
		   ENTER being pressed on either "verify" or "cancel"
		   buttons */
		if (key == CDW_KEY_ENTER) {
			if (fi == f_ok_b) {

				/* flush */
				form_driver(wizard.cdw_form->form, REQ_VALIDATION);

				return CDW_OK;
			} else if (fi == f_cancel_b) {
				return CDW_CANCEL;
			} else {
				;
			}
		} else {
			;
		}
	}

	return CDW_CANCEL;
}





/**
   \brief Initialize wizard data structures
*/
void cdw_verify_wizard_init(void)
{
	cdw_assert (LINES > window_n_lines, "ERROR: LINES is too small: %d\n", LINES);
	cdw_assert (COLS > window_n_cols, "ERROR: COLS is too small: %d\n", COLS);

	return;
}





/**
   \brief Cleanup function for verify wizard

   Function deallocates all resources used by wizard window.
   Call this function when you close wizard in normal mode or when
   you want to clean up after unsuccessful creation of wizard.
*/
void cdw_verify_wizard_destroy(void)
{
	/* This order of these four function calls ensures minimum
	   problems reported by valgrind. */
	cdw_form_delete_form_objects(wizard.cdw_form);
	cdw_window_delete(&wizard.cdw_form->subwindow);
	cdw_window_delete(&wizard.cdw_form->window);
	cdw_form_delete(&(wizard.cdw_form));

	return;
}






/**
   \brief Create all UI elements in wizard window

   \param task - variable describing current task
   \param disc - variable describing disc currently in drive

   \return CDW_OK on success
   \return CDW_ERROR on failure
*/
cdw_rv_t cdw_verify_wizard_build(void)
{
	wizard.cdw_form = cdw_form_new(CDW_VERIFY_WIZARD_N_FIELDS);
	if (!wizard.cdw_form) {
		cdw_vdm ("ERROR: failed to create cdw form\n");
		return CDW_ERROR;
	}

	int begin_y = ((LINES - window_n_lines) / 2) - 2;
	int begin_x = (COLS - window_n_cols) / 2;
	wizard.cdw_form->window = cdw_window_new((WINDOW *) NULL,
						 window_n_lines, window_n_cols,
						 begin_y, begin_x,
						 CDW_COLORS_DIALOG,
						 /* 2TRANS: this is title of wizard window; "verify" as in
						    "Erasing an optical disc" */
						 _("Verify wizard"),
						 /* 2TRANS: this is tip at the bottom of window - user can
						    switch between window elements using tab key */
						 _("Use 'Tab' key to move"));
	if (!wizard.cdw_form->window) {
		cdw_vdm ("ERROR: failed to create window\n");
		return CDW_ERROR;
	}

	wizard.cdw_form->subwindow = cdw_window_new(wizard.cdw_form->window,
						    subwindow_n_lines, subwindow_n_cols,
						    1, 1,
						    CDW_COLORS_DIALOG, (char *) NULL, (char *) NULL);

	if (!wizard.cdw_form->subwindow) {
		cdw_vdm ("ERROR: failed to create subwindow\n");
		return CDW_ERROR;
	}

	cdw_verify_wizard_build_fields();

	wizard.cdw_form->form = cdw_ncurses_new_form(wizard.cdw_form->window,
						     wizard.cdw_form->subwindow,
						     wizard.cdw_form->fields);
	if (!wizard.cdw_form->form) {
		cdw_vdm ("ERROR: failed to create form\n");
		return CDW_ERROR;
	}


	cdw_form_redraw_widgets(wizard.cdw_form);

	cdw_form_add_return_chars(wizard.cdw_form, CDW_KEY_ENTER, CDW_KEY_ESCAPE, 'q', 'Q', 0);

	cdw_form_set_button_return_key(wizard.cdw_form, f_ok_b, CDW_KEY_ENTER);
	cdw_form_set_button_return_key(wizard.cdw_form, f_cancel_b, CDW_KEY_ESCAPE);

	wrefresh(wizard.cdw_form->subwindow);
	wrefresh(wizard.cdw_form->window);

	return CDW_OK;
}





cdw_rv_t cdw_verify_wizard_build_fields(void)
{
	cdw_form_descr_t descr[] = {
		/*    widget type           begin_y            begin_x         n_cols         n_lines   field enum        data1                      data2 */

		/* 2TRANS: this is label in verify wizard */
		{ CDW_WIDGET_ID_TEXT,       top_message_row,   first_col,      label_n_cols,        4,  f_top_message_t,  text_makers,                  0 },

		{ CDW_WIDGET_ID_DROPDOWN,   mode_row,          first_col + 1,  dropdown_n_cols - 2, 1,  f_mode_dd,        dropdown_makers,              0 },

		/* 2TRANS: button label, "verify" means "verify data" */
		{ CDW_WIDGET_ID_BUTTON,     buttons_row,       3,              2,                   1,  f_ok_b,           _("Verify"),  CDW_COLORS_DIALOG },
		/* 2TRANS: button label */
		{ CDW_WIDGET_ID_BUTTON,     buttons_row,      15,              2,                   1,  f_cancel_b,       _("Cancel"),  CDW_COLORS_DIALOG },

		/* guard */
		{ -1,                       0,                 0,              0,                   0,  0,                (void *) NULL,                0 }};


	wizard.cdw_form->n_fields = CDW_VERIFY_WIZARD_N_FIELDS;
	wizard.cdw_form->fields = wizard_fields;

	cdw_rv_t crv = cdw_form_description_to_fields(descr, wizard.cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





CDW_DROPDOWN *cdw_verify_wizard_make_verify_mode_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	CDW_DROPDOWN *dropdown = cdw_dropdown_new(window, begin_y, begin_x, width,
						  3, /* n_items_max */
						  CDW_COLORS_DIALOG);
	if (!dropdown) {
		cdw_vdm ("ERROR: failed to create new dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}

	/* 2TRANS: this is a label in dropdown list; digest is an output of tools such as md5sum or sha512 sum */
	cdw_rv_t crv = cdw_dropdown_add_item(dropdown, CDW_CALCULATE_DIGEST_MODE_FILE, _("Calculate digest of a file"));
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to create verify mode dropdown label 0\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	}
#if 0 /* I don't have 100% certain method of obtaining correct size
	 of data on disc, so this functionality might provide invalid
	 results. I'm disabling it for now. */
	/* 2TRANS: this is a label in dropdown list; "disc" is an optical disc */
	crv = cdw_dropdown_add_item(dropdown, CDW_CALCULATE_DIGEST_MODE_DISC, _("Verify first track of a disc"));
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to create verify mode dropdown label 1\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	}
#endif
	/* 2TRANS: this is a label in dropdown list; "disc" is an optical disc */
	crv = cdw_dropdown_add_item(dropdown, CDW_CALCULATE_DIGEST_MODE_DISC_FILE, _("Compare a disc against a file (experimental)"));
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to create verify mode dropdown label 2\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	}

	crv = cdw_dropdown_finalize(dropdown);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to finalize dropdown\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	} else {
		cdw_dropdown_set_current_item_by_id(dropdown, CDW_CALCULATE_DIGEST_MODE_FILE);
		return dropdown;
	}
}





char *cdw_verify_wizard_make_message(int n_cols)
{
	/* 2TRANS: this is message printed in a wizard; */
	return cdw_string_wrap(_("Select what you want to verify. Verification algorithm (md5sum, sha1sum etc.) can be changed in main window -> Configuration -> Tools -> Digest tool."), (size_t) n_cols, CDW_ALIGN_LEFT);
}
