<?php

namespace Civi\Api4\Action\Afform;

use Civi\Api4\CustomField;
use Civi\Api4\CustomGroup;
use Civi\Core\Event\GenericHookEvent;
use CRM_Afform_ExtensionUtil as E;

/**
 * @inheritDoc
 * @package Civi\Api4\Action\Afform
 */
class Get extends \Civi\Api4\Generic\BasicGetAction {

  use \Civi\Api4\Utils\AfformFormatTrait;

  public function getRecords() {
    /** @var \CRM_Afform_AfformScanner $scanner */
    $scanner = \Civi::service('afform_scanner');
    $getComputed = $this->_isFieldSelected('has_local', 'has_base', 'base_module');
    $getLayout = $this->_isFieldSelected('layout');
    $getSearchDisplays = $this->_isFieldSelected('search_displays');
    $afforms = [];

    // This helps optimize lookups by file/module/directive name
    $getNames = array_filter([
      'name' => $this->_itemsToGet('name'),
      'module_name' => $this->_itemsToGet('module_name'),
      'directive_name' => $this->_itemsToGet('directive_name'),
    ]);
    $getTypes = $this->_itemsToGet('type');

    $names = $getNames['name'] ?? array_keys($scanner->findFilePaths());

    // Allow hooks to provide autoGenerated forms
    $hookForms = [];
    $hookParams = ['afforms' => &$hookForms, 'getNames' => $getNames, 'getTypes' => $getTypes, 'getLayout' => $getLayout];
    $event = GenericHookEvent::create($hookParams);
    \Civi::dispatcher()->dispatch('civi.afform.get', $event);
    // Set defaults for Afforms supplied by hook
    foreach ($hookForms as $afform) {
      $name = $afform['name'];
      $afform += [
        'has_base' => TRUE,
        'type' => 'form',
      ];
      // afCore and af would normally get required by AngularDependencyMapper but that only works on file-based afforms
      $afform['requires'] = array_unique(array_merge(['afCore', 'af'], $afform['requires'] ?? []));
      if (!in_array($name, $names)) {
        $names[] = $name;
      }
      $afforms[$name] = $afform;
    }

    if ($this->checkPermissions) {
      $names = array_filter($names, [$this, 'checkPermission']);
    }

    foreach ($names as $name) {
      $info = [
        'name' => $name,
        'module_name' => _afform_angular_module_name($name, 'camel'),
        'directive_name' => _afform_angular_module_name($name, 'dash'),
      ];
      // Skip if afform does not match requested name
      foreach ($getNames as $key => $names) {
        if (!in_array($info[$key], $names)) {
          continue 2;
        }
      }
      $record = $scanner->getMeta($name);
      // Skip if afform does not exist or is not of requested type(s)
      if (
        (!$record && !isset($afforms[$name])) ||
        ($getTypes && isset($record['type']) && !in_array($record['type'], $getTypes, TRUE))
      ) {
        continue;
      }
      $afforms[$name] = array_merge($afforms[$name] ?? [], $record ?? [], $info);
      if ($getComputed) {
        $scanner->addComputedFields($afforms[$name]);
      }
      if ($getLayout || $getSearchDisplays) {
        // Autogenerated layouts will already be in values but can be overridden; scanner takes priority
        $afforms[$name]['layout'] = $scanner->getLayout($name) ?? $afforms[$name]['layout'] ?? '';
      }
      if ($getSearchDisplays) {
        $afforms[$name]['search_displays'] = $this->getSearchDisplays($afforms[$name]['layout']);
      }
    }

    if ($getLayout && $this->layoutFormat !== 'html') {
      foreach ($afforms as $name => $record) {
        $afforms[$name]['layout'] = $this->convertHtmlToOutput($record['layout']);
      }
    }

    return $afforms;
  }

  /**
   * Assert that a form is authorized.
   *
   * @return bool
   */
  protected function checkPermission($name) {
    return \CRM_Core_Permission::check("@afform:$name");
  }

  /**
   * Generates afform blocks from custom field sets.
   *
   * @param \Civi\Core\Event\GenericHookEvent $event
   * @throws \API_Exception
   */
  public static function getCustomGroupBlocks($event) {
    // Early return if blocks are not requested
    if ($event->getTypes && !in_array('block', $event->getTypes, TRUE)) {
      return;
    }

    $getNames = $event->getNames;
    $getLayout = $event->getLayout;
    $groupNames = [];
    $afforms =& $event->afforms;
    foreach ($getNames['name'] ?? [] as $name) {
      if (strpos($name, 'afblockCustom_') === 0 && strlen($name) > 13) {
        $groupNames[] = substr($name, 14);
      }
    }
    // Early return if this api call is fetching afforms by name and those names are not custom-related
    if ((!empty($getNames['name']) && !$groupNames)
      || (!empty($getNames['module_name']) && !strstr(implode(' ', $getNames['module_name']), 'afblockCustom'))
      || (!empty($getNames['directive_name']) && !strstr(implode(' ', $getNames['directive_name']), 'afblock-custom'))
    ) {
      return;
    }
    $customApi = CustomGroup::get(FALSE)
      ->addSelect('name', 'title', 'help_pre', 'help_post', 'extends', 'max_multiple')
      ->addWhere('is_multiple', '=', 1)
      ->addWhere('is_active', '=', 1);
    if ($groupNames) {
      $customApi->addWhere('name', 'IN', $groupNames);
    }
    if ($getLayout) {
      $customApi->addSelect('help_pre', 'help_post');
      $customApi->addChain('fields', CustomField::get(FALSE)
        ->addSelect('name')
        ->addWhere('custom_group_id', '=', '$id')
        ->addWhere('is_active', '=', 1)
        ->addOrderBy('weight', 'ASC')
      );
    }
    foreach ($customApi->execute() as $custom) {
      $name = 'afblockCustom_' . $custom['name'];
      $item = [
        'name' => $name,
        'type' => 'block',
        'requires' => [],
        'title' => E::ts('%1 block', [1 => $custom['title']]),
        'description' => '',
        'is_dashlet' => FALSE,
        'is_public' => FALSE,
        'is_token' => FALSE,
        'permission' => 'access CiviCRM',
        'join_entity' => 'Custom_' . $custom['name'],
        'entity_type' => $custom['extends'],
      ];
      if ($getLayout) {
        $item['layout'] = ($custom['help_pre'] ? '<div class="af-markup">' . $custom['help_pre'] . "</div>\n" : '');
        foreach ($custom['fields'] as $field) {
          $item['layout'] .= "<af-field name=\"{$field['name']}\" />\n";
        }
        $item['layout'] .= ($custom['help_post'] ? '<div class="af-markup">' . $custom['help_post'] . "</div>\n" : '');
      }
      $afforms[$name] = $item;
    }
  }

  /**
   * Find search display tags in afform markup
   *
   * @param string $html
   * @return string[]
   */
  private function getSearchDisplays(string $html) {
    $tags = $searchDisplays = [];
    preg_match_all('/<\\s*crm-search-display[^>]+>/', $html, $tags);
    foreach ($tags[0] ?? [] as $tag) {
      $searchName = $displayName = [];
      preg_match('/search-name\\s*=\\s*[\'"]([^\'"]+)[\'"]/', $tag, $searchName);
      // Note: display name will be blank when using the default (autogenerated) display
      preg_match('/display-name\\s*=\\s*[\'"]([^\'"]+)[\'"]/', $tag, $displayName);
      if (!empty($searchName[1])) {
        $searchDisplays[] = $searchName[1] . (empty($displayName[1]) ? '' : '.' . $displayName[1]);
      }
    }
    return $searchDisplays;
  }

}
