#!/usr/bin/ruby -w
# -*- ruby -*-

# -------------------------------------------------------
# Logging
# -------------------------------------------------------

# Very minimal logging output. If verbose is set, this displays the method and
# line number whence called. It can be a mixin to a class, which displays the
# class and method from where it called. If not in a class, it displays only the
# method.

# All kids love log.
class Log

  VERSION = "1.0.1"

  attr_accessor :quiet
  
  module Severity
    DEBUG = 0
    INFO  = 1
    WARN  = 2
    ERROR = 3
    FATAL = 4
  end

  include Log::Severity

  WHENCE_RE = Regexp.new('(.*):(\d+)(?::in\s+\`(.*)\\\')?', Regexp::EXTENDED)

  def initialize
    @width   = 0
    @output  = $stdout
    @fmt     = "[%s:%04d] {%s}"
    @level   = FATAL
    @quiet   = false
  end
    
  def verbose=(v)
    @level = case v
             when TrueClass 
               DEBUG
             when FalseClass 
               FATAL
             when Integer
               v
             end
  end

  def verbose
    @level <= DEBUG
  end

  def level=(lvl)
    @level = lvl
  end

  # Assigns output to the given stream.
  def output=(io)
    @output = io
  end

  # Assigns output to a file with the given name. Returns the file; client
  # is responsible for closing it.
  def outfile=(f)
    @output = if f.kind_of?(IO) then f else File.new(f, "w") end
  end

  # Creates a printf format for the given widths, for aligning output.
  def set_widths(file_width, line_width, cls_width, meth_width)
    @fmt = "[%#{file_width}s:%#{line_width}d] {%#{cls_width}s\#%#{meth_width}s}"
  end

  def get_whence(c, classname)
    md   = WHENCE_RE.match(c)

    file = md[1].sub(%r{ .*/ }msx, "")
    line = md[2]
    cls  = classname || ""
    meth = md[3] || "???"
    [ file, line, cls, meth ]
  end

  # Logs the given message.
  def log(msg = "", level = DEBUG, depth = 1, cname = nil, &blk)
    if level >= @level
      c = caller(depth)[0]
      file, line, cls, meth = get_whence(c, cname)
      print_formatted(file, line, cls, meth, msg, level, &blk)
    end
    true
  end

  # Shows the current stack.
  def stack(msg = "", level = DEBUG, depth = 1, cname = nil, &blk)
    if level >= @level
      stk = caller(depth)
      stk.shift
      for c in stk
        file, line, cls, meth = get_whence(c, cname)
        print_formatted(file, line, cls, meth, msg, level, &blk)
        msg = '"'
        blk = nil
      end
    end
    true
  end

  def print_formatted(file, line, cls, meth, msg, level, &blk)
    hdr = sprintf @fmt, file, line, cls, meth

    if blk
      x = blk.call
      if x.kind_of?(String)
        msg = x
      else
        return
      end
    end
    @output.puts hdr + " " + msg.to_s.chomp
  end

  # by default, class methods delegate to a single app-wide log.

  @@log = Log.new

  def self.quiet
    @@log.quiet
  end

  def self.quiet=(q)
    @@log.quiet = q
  end

  def self.verbose
    @@log.verbose
  end

  def self.verbose=(v)
    @@log.verbose = v && v != 0 ? DEBUG : FATAL
  end

  def self.level=(lvl)
    @@log.level = lvl
  end

  def self.set_widths(file_width, line_width, cls_width, meth_width)
    @@log.set_widths(file_width, line_width, cls_width, meth_width)
  end

  def self.log(msg = "", level = DEBUG, depth = 1, cname = nil, &blk)
    @@log.log(msg, level, depth + 1, cname, &blk)
  end

  def self.stack(msg = "", level = DEBUG, depth = 1, cname = nil, &blk)
    @@log.stack(msg, level, depth, cname, &blk)
  end

  def self.warn(msg, depth = 1, cname = nil, &blk)
    write("WARNING: " + msg, depth + 1, cname, &blk)
  end

  def self.error(msg, depth = 1, cname = nil, &blk)
    if verbose
      stack(msg, Log::ERROR, depth + 1, cname, &blk)
    else
      $stderr.puts "ERROR: " + msg
    end
  end

  def self.write(msg, depth = 1, cname = nil, &blk)
    if verbose
      stack(msg, Log::WARN, depth + 1, cname, &blk)
    elsif quiet
      # nothing
    else
      $stderr.puts msg
    end
  end

end


module Loggable

  # Logs the given message, including the class whence invoked.
  def log(msg = "", level = Log::DEBUG, depth = 1, &blk)
    Log.log(msg, level, depth + 1, self.class.to_s, &blk)
  end
  
  def stack(msg = "", level = Log::DEBUG, depth = 1, &blk)
    Log.stack(msg, level, depth + 1, self.class.to_s, &blk)
  end

  def warn(msg = "", depth = 1, &blk)
    Log.warn(msg, depth + 1, self.class.to_s, &blk)
  end

  def error(msg = "", depth = 1, &blk)
    Log.error(msg, depth + 1, self.class.to_s, &blk)
  end

  def write(msg = "", depth = 1, &blk)
    Log.write(msg, depth + 1, self.class.to_s, &blk)
  end

end


