/*************************************************************************
* Copyright (C) 2021 Intel Corporation
*
* Licensed under the Apache License,  Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* 	http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law  or agreed  to  in  writing,  software
* distributed under  the License  is  distributed  on  an  "AS IS"  BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the  specific  language  governing  permissions  and
* limitations under the License.
*************************************************************************/

#ifndef EC_SM2_H
#define EC_SM2_H

#include <crypto_mb/defs.h>
#include <crypto_mb/status.h>

#ifndef BN_OPENSSL_DISABLE
      #include <openssl/bn.h>
      #include <openssl/ec.h>
#endif // BN_OPENSSL_DISABLE

/*
// Computes public key
// pa_pubx[]   array of pointers to the public keys X-coordinates
// pa_puby[]   array of pointers to the public keys Y-coordinates
// pa_pubz[]   array of pointers to the public keys Z-coordinates (or NULL, if affine coordinate requested)
// pa_skey[]   array of pointers to the private keys
// pBuffer     pointer to the scratch buffer
*/
EXTERN_C mbx_status mbx_sm2_ecpublic_key_mb8(int64u* pa_pubx[8],
                                             int64u* pa_puby[8],
                                             int64u* pa_pubz[8],
                                 const int64u* const pa_skey[8],
                                              int8u* pBuffer);

/*
// Computes shared key
// pa_shared_key[]   array of pointers to the shared keys
// pa_skey[]         array of pointers to the own (ephemeral) private keys
// pa_pubx[]         array of pointers to the party's public keys X-coordinates
// pa_puby[]         array of pointers to the party's public keys Y-coordinates
// pa_pubz[]         array of pointers to the party's public keys Z-coordinates (or NULL, if affine coordinate requested)
// pBuffer           pointer to the scratch buffer
//
// Note:
// This function implements ECDHE over SM2 curve according to IEEE 1363-2000 standard.
*/
EXTERN_C mbx_status mbx_sm2_ecdh_mb8(int8u* pa_shared_key[8],
                        const int64u* const pa_skey[8],
                        const int64u* const pa_pubx[8],
                        const int64u* const pa_puby[8],
                        const int64u* const pa_pubz[8],
                                     int8u* pBuffer);

/*
// Computes SM2 ECDSA signature
// pa_sign_r[]       array of pointers to the computed r-components of the signatures
// pa_sign_s[]       array of pointers to the computed s-components of the signatures
// pa_user_id[]      array of pointers to the users ID
// user_id_len[]     array of users ID length
// pa_msg[]          array of pointers to the messages are being signed
// msg_len[]         array of messages length
// pa_eph_skey[]     array of pointers to the signer's ephemeral private keys
// pa_reg_skey[]     array of pointers to the signer's regular private keys
// pa_pubx[]         array of pointers to the party's public keys X-coordinates
// pa_puby[]         array of pointers to the party's public keys Y-coordinates
// pa_pubz[]         array of pointers to the party's public keys Z-coordinates (or NULL, if affine coordinate requested)
// pBuffer           pointer to the scratch buffer
*/
EXTERN_C mbx_status mbx_sm2_ecdsa_sign_mb8(int8u* pa_sign_r[8],
                                           int8u* pa_sign_s[8],
                               const int8u* const pa_user_id[8],
                                        const int user_id_len[8],
                               const int8u* const pa_msg[8],
                                        const int msg_len[8],
                              const int64u* const pa_eph_skey[8],
                              const int64u* const pa_reg_skey[8],
                              const int64u* const pa_pubx[8],
                              const int64u* const pa_puby[8],
                              const int64u* const pa_pubz[8],
                                           int8u* pBuffer);

/*
// Verifies SM2 ECDSA signature
// pa_sign_r[]       array of pointers to the computed r-components of the signatures
// pa_sign_s[]       array of pointers to the computed s-components of the signatures
// pa_user_id[]      array of pointers to the users ID
// user_id_len[]     array of users ID length
// pa_msg[]          array of pointers to the messages are being signed
// msg_len[]         array of messages length
// pa_pubx[]         array of pointers to the signer's public keys X-coordinates
// pa_puby[]         array of pointers to the signer's public keys Y-coordinates
// pa_pubz[]         array of pointers to the signer's public keys Z-coordinates  (or NULL, if affine coordinate requested)
// pBuffer           pointer to the scratch buffer
*/
EXTERN_C mbx_status mbx_sm2_ecdsa_verify_mb8(const int8u* const pa_sign_r[8],
                                             const int8u* const pa_sign_s[8],
                                             const int8u* const pa_user_id[8],
                                                      const int user_id_len[8],
                                             const int8u* const pa_msg[8],
                                                      const int msg_len[8],
                                            const int64u* const pa_pubx[8],
                                            const int64u* const pa_puby[8],
                                            const int64u* const pa_pubz[8],
                                                         int8u* pBuffer);

/*
// OpenSSL's specific similar APIs
*/
#ifndef BN_OPENSSL_DISABLE
EXTERN_C mbx_status mbx_sm2_ecpublic_key_ssl_mb8(BIGNUM* pa_pubx[8],
                                                 BIGNUM* pa_puby[8],
                                                 BIGNUM* pa_pubz[8],
                                     const BIGNUM* const pa_skey[8],
                                                  int8u* pBuffer);

EXTERN_C mbx_status mbx_sm2_ecdh_ssl_mb8(int8u* pa_shared_key[8],
                            const BIGNUM* const pa_skey[8],
                            const BIGNUM* const pa_pubx[8],
                            const BIGNUM* const pa_puby[8],
                            const BIGNUM* const pa_pubz[8],
                                         int8u* pBuffer);

EXTERN_C mbx_status mbx_sm2_ecdsa_sign_ssl_mb8(int8u* pa_sign_r[8],
                                               int8u* pa_sign_s[8],
                                   const int8u* const pa_user_id[8],
                                            const int user_id_len[8],
                                   const int8u* const pa_msg[8],
                                            const int msg_len[8],
                                  const BIGNUM* const pa_eph_skey[8],
                                  const BIGNUM* const pa_reg_skey[8],
                                  const BIGNUM* const pa_pubx[8],
                                  const BIGNUM* const pa_puby[8],
                                  const BIGNUM* const pa_pubz[8],
                                               int8u* pBuffer);

EXTERN_C mbx_status mbx_sm2_ecdsa_verify_ssl_mb8(const ECDSA_SIG* const pa_sig[8],
                                                     const int8u* const pa_user_id[8],
                                                              const int user_id_len[8],
                                                     const int8u* const pa_msg[8],
                                                              const int msg_len[8],
                                                    const BIGNUM* const pa_pubx[8],
                                                    const BIGNUM* const pa_puby[8],
                                                    const BIGNUM* const pa_pubz[8],
                                                                 int8u* pBuffer);
#endif // BN_OPENSSL_DISABLE
#endif /* EC_SM2_H */
