/*
 * Configurable ps-like program.
 * High-level display device routines.
 * These global routines call the currently selected display routines.
 *
 * Copyright (c) 2010 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include "ips.h"


/*
 * The current display device.
 */
static	DISPLAY *	display;


/*
 * Set the display type to use for later dpy calls.
 * A NULL or illegal type sets the display type to a dumb terminal.
 * Returns TRUE if the display type was NULL or was known.
 */
BOOL
DpySetDisplay(const char * type)
{
	if ((type == NULL) || (strcmp(type, DISPLAY_TYPE_TTY) == 0))
		display = GetTtyDisplay();
	else if (strcmp(type, DISPLAY_TYPE_CURSES) == 0)
		display = GetCursesDisplay();
#ifdef X11
	else if (strcmp(type, DISPLAY_TYPE_X11) == 0)
		display = GetX11Display();
#endif
	else
	{
		display = GetTtyDisplay();

		return FALSE;
	}

	return TRUE;
}


BOOL
DpyOpen(void)
{
	return display->open(display);
}


BOOL
DpyDefineColor(int colorId, const char * foreground, const char * background, int flags)
{
	return display->defineColor(display, colorId, foreground, background, flags);
}


void
DpyCreateWindow()
{
	display->createWindow(display);
}


void
DpyClose(void)
{
	display->close(display);
}


void
DpySetColor(int colorId)
{
	display->setColor(display, colorId);
}


void
DpyRefresh(void)
{
	display->refresh(display);
}


void
DpyBeginPage(void)
{
	display->beginPage(display);
}


void
DpyChar(int ch)
{
	display->putChar(display, ch);
}


void
DpyString(const char * str)
{
	display->putString(display, str);
}


void
DpyBuffer(const char * buffer, int length)
{
	display->putBuffer(display, buffer, length);
}


void
DpyEndPage(void)
{
	display->endPage(display);
}


/*
 * Handle any events from the display device while waiting for the
 * specified number of milliSeconds.  This call returns early if an
 * input character is available for reading or if a refresh is required.
 * Returns TRUE if a refresh is required due to a window resize.
 */
BOOL
DpyEventWait(int milliSeconds)
{
	return display->eventWait(display, milliSeconds);
}


BOOL
DpyInputReady(void)
{
	return display->inputReady(display);
}


int
DpyReadChar(void)
{
	return display->readChar(display);
}


void
DpyRingBell(void)
{
	display->ringBell(display);
}


int
DpyGetRows(void)
{
	return display->getRows(display);
}


int
DpyGetCols(void)
{
	return display->getCols(display);
}


BOOL
DpyDoesScroll(void)
{
	return display->doesScroll(display);
}

/* END CODE */
