package shared

import (
	"os"
	"testing"

	"golang.org/x/sys/unix"
)

func TestGetAllXattr(t *testing.T) {
	var (
		err       error
		testxattr = map[string]string{
			"user.checksum": "asdfsf13434qwf1324",
			"user.random":   "This is a test",
			"user.empty":    "",
		}
	)
	xattrFile, err := os.CreateTemp("", "")
	if err != nil {
		t.Error(err)
		return
	}

	defer func() { _ = os.Remove(xattrFile.Name()) }()
	_ = xattrFile.Close()

	xattrDir, err := os.MkdirTemp("", "")
	if err != nil {
		t.Error(err)
		return
	}

	defer func() { _ = os.Remove(xattrDir) }()

	for k, v := range testxattr {
		err = unix.Setxattr(xattrFile.Name(), k, []byte(v), 0)
		if err == unix.ENOTSUP {
			t.Log(err)
			return
		}

		if err != nil {
			t.Error(err)
			return
		}

		err = unix.Setxattr(xattrDir, k, []byte(v), 0)
		if err == unix.ENOTSUP {
			t.Log(err)
			return
		}

		if err != nil {
			t.Error(err)
			return
		}
	}

	// Test retrieval of extended attributes for regular files.
	h, err := GetAllXattr(xattrFile.Name())
	if err != nil {
		t.Error(err)
		return
	}

	if h == nil {
		t.Errorf("Expected to find extended attributes but did not find any.")
		return
	}

	for k, v := range h {
		found, ok := h[k]
		if !ok || found != testxattr[k] {
			t.Errorf("Expected to find extended attribute %s with a value of %s on regular file but did not find it.", k, v)
			return
		}
	}

	// Test retrieval of extended attributes for directories.
	h, err = GetAllXattr(xattrDir)
	if err != nil {
		t.Error(err)
		return
	}

	if h == nil {
		t.Errorf("Expected to find extended attributes but did not find any.")
		return
	}

	for k, v := range h {
		found, ok := h[k]
		if !ok || found != testxattr[k] {
			t.Errorf("Expected to find extended attribute %s with a value of %s on directory but did not find it.", k, v)
			return
		}
	}
}
