/* 
 * pam_rsa PAM-module for local authentication with RSA keypairs
 * copyright (c) 2006 Vesa-Matti Kari <hyperllama@laamanaama.helsinki.fi>
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 * 
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <openssl/crypto.h>
#include <openssl/err.h>
#include <openssl/evp.h>
#include <openssl/rand.h>
#include <openssl/pem.h>
#include <sys/types.h>
#include "pam_rsa.h"
#include "util.h"
#include "conf.h"
#include "rsa.h"

/* We must define PAM_SM_AUTH before including security/pam_modules.h */
#define PAM_SM_AUTH
#include <security/pam_appl.h> 
#include <security/pam_modules.h>

#if HAVE_SECURITY__PAM_MACROS_H
#include <security/_pam_macros.h>
#endif


/* On Linux PAM the third argument to pam_get_item() is
 * const qualified and the same is 1 of the second 
 * argument to pam_get_user(). At least the PAM on Solaris 8 
 * lacks these const qualifiers. This is not nice, but I
 * cannot stand the C compiler whining, so bring in XCONST. 
 */
#if HAVE_CONST_IN_PAM_FUNCS
#define XCONST const
#else
#define XCONST /* empty */
#endif /* HAVE_CONST_IN_PAM_FUNCS */


/* Linux vs Solaris issue again. PAM-module hook functions' 
 * return values are PAM_EXTERN on Linux, but not at least
 * on Solaris 8. So here we go.
 */ 
#ifndef PAM_EXTERN
#define PAM_EXTERN	/* empty */
#endif


#define PAM_RSA_NAME "pam_rsa"
#define PAM_RSA_VERSION "version 0.8-9"

const char *pam_rsa_name = PAM_RSA_NAME;
static const char *version = PAM_RSA_NAME ".so " PAM_RSA_VERSION;

static int converse(pam_handle_t *pamh, int nargs,
		XCONST struct pam_message **message, struct pam_response **resp);
static int read_password(pam_handle_t *pamh, char *prompt, char **pass);


static int converse(pam_handle_t *pamh, int nargs,
				XCONST struct pam_message **message, struct pam_response **resp)
{
	struct pam_conv *conv;
	int r;
	
	*resp = NULL;
	if ((r = pam_get_item(pamh, PAM_CONV, (XCONST void **) &conv)) == PAM_SUCCESS) {
		r = conv->conv(nargs, message, resp, conv->appdata_ptr);
	} else {	
		pamrsa_log(LOG_ERR, "could not get conversation: %s", pam_strerror(pamh, r));
		return r;
	}

	/* pam_get_item returned PAM_SUCCESS and conv->conv got called. It may have 
     * returned PAM_SUCCESS as well, but even so, we do not know whether the
     * the application's response is non-NULL and must check it out.
	 */

	if (*resp == NULL) {
		r = PAM_AUTH_ERR;
	}

	return r;
}


static int read_password(pam_handle_t *pamh, char *prompt, char **pass)
{
	struct pam_message msg;
	XCONST struct pam_message *pmsg = &msg;
	struct pam_response *resp;
	int r;

	msg.msg_style = PAM_PROMPT_ECHO_OFF;
	msg.msg = prompt;

	resp = NULL;
	if ((r = converse(pamh, 1, &pmsg, &resp)) == PAM_SUCCESS) {
		if (resp->resp != NULL) {
			if ((*pass = strdup(resp->resp)) == NULL) {
					pamrsa_log(LOG_CRIT, "memory allocation failure");
					r = PAM_AUTH_ERR;
			}
		} else {
			pamrsa_log(LOG_CRIT, "resp->resp == NULL");
			r = PAM_AUTH_ERR;
		}
	}

#if HAVE_SECURITY__PAM_MACROS_H
	if (resp != NULL) {
		_pam_drop_reply(resp, 1);
	}
#else
	/* At least Solaris 8 lacks security/_pam_macros.h so 
       we must get rid of the pam_response using our own means */
	if (resp != NULL) {
		if (resp->resp != NULL) {	
			char *p;
			p = resp->resp;
			while (*p != '\0') {	
				*p = '\0';
				++p;
			}
			free(resp->resp);
		}
		free(resp);
	}
#endif /* HAVE_SECURITY__PAM_MACROS_H */
	
	return r;
}


/* PAM module functions start here */

PAM_EXTERN int pam_sm_authenticate(pam_handle_t *pamh, int flags, int argc, const char **argv)
{
	static const char *conffile = "/etc/security/" PAM_RSA_NAME ".conf";
	struct pamrsaconf *pconf = NULL;
	struct pamrsaarg *parg = NULL;
	XCONST char *service = NULL;
	XCONST char *user = NULL;
	const char *result = NULL;
	char *authtok = NULL;
	char *passphrase = NULL;
	int authres = PAM_AUTH_ERR;	
	int authokay = 0;
	char buf[1024];
	int r;

	if ((r = pam_get_user(pamh, &user, NULL)) != PAM_SUCCESS) {
#if defined PAM_CONV_AGAIN && defined PAM_INCOMPLETE
		if (r == PAM_CONV_AGAIN) {
			return PAM_INCOMPLETE;
		}
#endif /* PAM_CONV_AGAIN  && PAM_INCOMPLETE */
		pamrsa_log(LOG_ERR, "%s: error retrieving PAM user: %s", pam_strerror(pamh, r));
		return PAM_USER_UNKNOWN;
	}

	if (user == NULL || *user == '\0') {
		return PAM_USER_UNKNOWN;
	}

	if ((r = pam_get_item(pamh, PAM_SERVICE, (XCONST void **)&service)) != PAM_SUCCESS) {
		pamrsa_log(LOG_ERR, "error retrieving PAM service: %s", pam_strerror(pamh, r));
		return r;
	}

	if (service == NULL || *service == '\0') {
		return PAM_SERVICE_ERR;
	}

	if ((parg = pamrsaarg_alloc()) == NULL) {
		pamrsa_log(LOG_CRIT, "memory allocation failure");
		return PAM_SERVICE_ERR;	
	}

	if (pamrsaarg_read(parg, argc, argv) != 0) {
		authres = PAM_SERVICE_ERR;	
		goto endfree;	
	}

	if ((pconf = pamrsaconf_alloc()) == NULL) {
		pamrsa_log(LOG_CRIT, "memory allocation failure");
		authres = PAM_SERVICE_ERR;	
		goto endfree;	
	}

	/* Directories and regular files allowed,
     * group readable directories and files allowed,
     * world readable directories and files allowed 
     */
	if (!is_safepath(conffile, "dr", "RrFf")) {
		authres = PAM_SERVICE_ERR;	
		goto endfree;	
	}

	if (pamrsaconf_read(pconf, conffile) != 0) {
		authres = PAM_SERVICE_ERR;	
		goto endfree;	
	}

	if (parg->debug) {
		pamrsa_log(LOG_DEBUG, "configuration file %s processed successfully", conffile);
	}

	/* after reading configuration, fill missing values with defaults */
	if (!pamrsaconf_set_defaults(pconf)) {
		pamrsa_log(LOG_CRIT, "memory allocation failure");
		authres = PAM_SERVICE_ERR;	
		goto endfree;	
	}

	if (parg->ask_pass || parg->ask_passphrase) {

		if (parg->debug) {
			pamrsa_log(LOG_DEBUG, "trying to get authentication token via PAM conversation");
		}

		if ((r = read_password(pamh, pconf->pam_prompt, &authtok)) != PAM_SUCCESS) {
			pamrsa_log(LOG_ERR, "%s: error getting authentication token: %s",
				pam_strerror(pamh, r));
#if defined PAM_CONV_AGAIN && defined PAM_INCOMPLETE
			if (r == PAM_CONV_AGAIN) {
				authres = PAM_INCOMPLETE;
				goto endfree;	
			}
#endif /* PAM_CONV_AGAIN && PAM_INCOMPLETE */
			authres = r;
			goto endfree;	
		}

		if ((flags & PAM_DISALLOW_NULL_AUTHTOK) &&
			(authtok == NULL || *authtok == '\0')) {
			authres = PAM_AUTH_ERR;
			goto endfree;
		}

		if (parg->ask_pass) {
			if ((r = pam_set_item(pamh, PAM_AUTHTOK, authtok)) != PAM_SUCCESS) {
				pamrsa_log(LOG_ERR, "error saving authentication token: %s",
					pam_strerror(pamh, r));
				authres = PAM_SYSTEM_ERR;
				goto endfree;
			}

			/* Wipe out the memory contents containing authentication token */
			memset(authtok, '\0', strlen(authtok));
			free(authtok); /* strdup():ed in read_password() */

			if (parg->debug) {
				pamrsa_log(LOG_DEBUG, "authentication token got and saved for other PAM modules");
			}
		} else { /* ask_passphrase */
			if (parg->debug) {
				pamrsa_log(LOG_DEBUG, "authentication token got and will be used as PEM-passphrase");
			}
		}
	}

	if (parg->ask_passphrase) {
		passphrase = authtok;
	} else {
		passphrase = NULL;
	}

	if (((r = rsa_auth_okay(pconf, parg, user, &authokay, passphrase)) == 0) &&
		(authokay)) {
		authres = PAM_SUCCESS;
		result = "success";
	} else {
		authres = PAM_AUTH_ERR;
		result = "failure";
	}
	
	/* Wipe out the memory contents containing passphrase */
	if (parg->ask_passphrase) {
		memset(authtok, '\0', strlen(authtok));
		free(authtok); /* strdup():ed in read_password() */
	}

	if ((pconf->log_auth_result) || (authokay == 0)) {
		const char *atok;
		if (parg->ask_pass) {
			atok = "[ask_pass]"; /* to show that we read some kind of password */
		} else if (parg->ask_passphrase) {
			atok = "[ask_passphrase]"; /* passphrase was used when opening RSA private key */
		} else {
			atok = "[none]";
		}

		memset(buf, '\0', sizeof(buf));
		snprintf(buf, sizeof(buf) -1, "uid=%d euid=%d user=%s authtok=%s service=%s result=",
			(int)getuid(),
			(int)geteuid(),
			user?user:"[unknown_user]",
			atok, 
			service?service:"[unknown_service]");

		if ((strlen(buf) + strlen(result)) < sizeof(buf)) {
			strcat(buf, result);
		}
		pamrsa_log(LOG_NOTICE, "%s", buf);
		memset(buf, '\0', sizeof(buf));
	}

endfree:
	pamrsaarg_free(parg);
	pamrsaconf_free(pconf);
	return authres;
}

PAM_EXTERN int pam_sm_setcred(pam_handle_t *pamh, int flags, int argc, const char **argv)
{
	return PAM_SUCCESS;
}

#ifdef PAM_STATIC 

struct pam_module _pam_rsa_modstruct = {
	PAM_RSA_NAME,
	pam_sm_authenticate,
	pam_sm_setcred,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};
	
#endif /* PAM_STATIC */
