---
# Exploiting Symmetry
---

Meep allows you to take advantage of mirror and rotational symmetries in your problem to reduce the amount of computation and storage required for simulation. The use of symmetry is designed to be **transparent**: once you specify the symmetry, Meep exploits it as an internal optimization, while everything else about the simulation *appears* the same as if you didn't specify the symmetry. That is, when you output the structure/geometry or currents/fields it still outputs the entire non-symmetrized cell; you can still call `get_field_point` to obtain the fields at any point, etcetera &mdash; internally, Meep transforms everything as necessary according to the symmetries to get whatever data is needed.

An important point to understand is that, when you specify a symmetry, it must be obeyed by **both** the structure/geometry **and** by the current sources and thus the fields. In particular, the symmetry of the currents/fields will typically require you to specify *phase factors* associated with the symmetry operations &mdash; for example, a mirror plane can either be used for *even* sources/fields (phase +1) or for *odd* sources/fields (phase −1).

**Meep does not check whether the symmetry is obeyed**. If you specify a symmetry that does not preserve your structure/sources, then the results are undefined.

Assuming your structure is symmetric, you might still have discretization effects that slightly break the symmetry.   For example, if the width of the computational cell in the direction of a mirror symmetry is not an integer number of pixels, Meep will round the cell width to the nearest pixel, which may slightly break the mirror symmetry.  (In such cases, Meep prints: *Warning: grid volume is not an integer number of pixels; cell size will be rounded to nearest pixel.*)  The results will still converge to the correct results as you increase resolution, but at finite resolutions you may observe discretization errors, such as even-symmetry sources coupling slightly to odd-symmetry modes.

For the Python syntax to specify a symmetry, see [Python Interface](Python_User_Interface.md#symmetry). There are also examples in [Tutorial/Basics](Python_Tutorials/Basics.md#exploiting-symmetry). For a description of how symmetries are implemented, see [Chunks and Symmetry](Chunks_and_Symmetry.md).

[TOC]

Kinds of Symmetries
-------------------

Meep supports exploiting several kinds of symmetries:

### Rotations and Reflections

-   Mirror planes through the origin, and normal to the $x$/$y$/$z$ axes.
-   180° rotational symmetry about the origin, around the $x$/$y$/$z$ axes. This is also known as a $C_2$ symmetry, in group theory. This is different from a mirror plane: e.g. as shown in the figure below, the letter "S" has $C_2$ but not mirror symmetry.
-   90° (fourfold) rotational symmetry about the origin, around the $x$/$y$/$z$ axes. This is also known as a $C_4$ symmetry.


![](images/C2_symmetry.png#center)


The first two kinds of symmetry each reduce the computational cell (internally) by a factor of two, and the third by a factor of four. If your structure has multiple symmetries, you can combine them arbitrarily. For example, your cell may have two orthogonal mirror planes (e.g. one even and one odd), or it may have four-fold rotational symmetry about the $z$ axis and a mirror plane through $z=0$. Thus, in 3d you might be able to reduce your cell by at most a factor of eight, or a factor of four in 2d. This is why the interface lets you specify a **list** `symmetries` of `symmetry` objects.

There are also two other kinds of symmetry you can exploit.

### Polarizations in 2d

When you have a two-dimensional simulation with a wavevector $\mathbf{k}$ (if any) lying in the plane, then the $xy$ plane is itself a mirror plane. The consequence of this is that the fields can be chosen purely polarized: either **$H_z$ polarized** (electric field in the plane and magnetic field in the $z$ direction) or **$E_z$ polarized** (magnetic field in the plane and electric field in the $z$ direction). Meep can exploit this to save a factor of two in storage and time.

To exploit $E_z$ or $H_z$ symmetry, you don't have to do anything special: if your sources are only $E_z$ polarized, then Meep will only propagate the fields $(E_z,H_x,H_y)$, and similarly for the $H_z$ polarization with fields $(H_z,E_x,E_y)$.

### Cylindrical Symmetry

If your structure has cylindrical symmetry (also known as the $C_{\infty\mathrm{v}}$ group), i.e. continuous rotational symmetry around the $z$ axis, then Meep can exploit this by running the simulation in cylindrical/polar coordinates.

To do this, you must set `dimensions = mp.CYLINDRICAL` in the Python script, and then the vectors are interpreted as $(r,\phi,z)$ triplets instead of $(x,y,z)$.

In a cylindrical simulation, fields can be written in the form of a function of $(r,z)$ multiplied by a function $\exp(im\phi)$ for the $\phi$ dependence, where $m$ is a user-specified parameter (related to the angular momentum of the field). If the cell includes the $r=0$ origin, then $m$ must be an integer to be single-valued at the origin. Otherwise, $m$ can be arbitrary (this is useful e.g. for bend simulations).

For examples, see [Tutorial/Cylindrical Coordinates](Python_Tutorials/Cylindrical_Coordinates.md).

### Translational Symmetry

If the structure has discrete or continuous translational symmetry in some direction, then this is handled in Meep by specifying periodic boundary conditions in that direction. Here, the currents/fields take the form of $\exp(i\mathbf{k}\cdot\mathbf{x})$ multiplied by a periodic envelope, so one specifies **Bloch-periodic** boundary conditions with a given Bloch wavevector $\mathbf{k}$.

For example, to simulate the modes of a uniform waveguide (continuous translational symmetry) in the $x$ direction, one would specify a structure with 0 cell-size (Python) or `no-size` (Scheme) in the $x$ direction and a given $k_x$ corresponding to the wavenumber of the desired mode. For discrete periodicity, see [Tutorial/Resonant Modes and Transmission in a Waveguide Cavity/Band Diagram](Python_Tutorials/Resonant_Modes_and_Transmission_in_a_Waveguide_Cavity.md#band-diagram).

Symmetry Phase Factors
----------------------

In general, it is not sufficient to simply specify the symmetry operations that preserve the structure. When applied to the fields and sources, the symmetry operations in general transform the source/field into itself multiplied by some *phase factor*.

For example, if you have a mirror plane, then the phase factor is +1 for even fields and −1 for odd fields. For a 90° rotation, values of +1, −1, +*i*, and −*i* are typical (where the latter two correspond to "circularly polarized" fields).

As a specific example, suppose the structure is even in $y$. In this case, an $E_y$ (dipole) point source at $y=0$ is *odd*. Imagine an arrow pointing in the $y$ direction. If you mirror flip $y$ to $-y$, then the arrow flips direction, which means that E points in the opposite direction (because E is a vector).  That is, the mirror flip of the source (and field) is the original source (and field) multiplied by -1, and hence it is odd.

The phase factors are ignored when applying the symmetry to the structure (e.g. to $\varepsilon$.)

Technically, the symmetry operations that preserve the structure form the [symmetry group](https://en.wikipedia.org/wiki/Symmetry_group). In particular, this is a [point group](https://en.wikipedia.org/wiki/Point_group) or, if you include translations, the [space group](https://en.wikipedia.org/wiki/Space_group) of the structure. The phase factors specify the *representation* of the group that the fields/sources transform as. See also any book on group theory in physics, e.g. [Group Theory and Its Applications in Physics](http://www.springer.com/us/book/9783540604457) by Inui, Tanabe, and Onodera.

Vectors and Pseudovectors
-------------------------

One subtlety that arises in specifying the symmetry of the system is that you have to transform the fields appropriately according to their vector nature. In particular, the *electric* fields and currents transform as vectors, while the *magnetic* fields and currents transforms as [pseudovectors](https://en.wikipedia.org/wiki/pseudovector). If you don't keep these things straight, you will be endlessly confused because otherwise it will seem that every field component has a different symmetry.

For example, suppose we have an *even* (phase=+1) mirror plane through $x=0$. Then, if we look at components of the electric field individually, $E_y$ and $E_z$ are *even* while $E_x$ is *odd*. Overall, the whole field considered as a *vector* is still even although one of its components is not.

The magnetic field is less intuitive because pseudovectors are multiplied by an additional factor of −1 under mirror flips or any [improper rotation](https://en.wikipedia.org/wiki/improper_rotation). Thus, for our even $x=0$ mirror plane, if we look at the components of the *magnetic* field individually, then $H_y$ and $H_z$ are *odd* while $H_x$ is *even*. That is, the magnetic field as a whole *looks* odd when the electric field is even, but the two really have the *same* symmetry if one realizes that they are different sorts of object (pseudovectors vs. vectors).
