/*	Parameter_View

PIRL CVS ID: Parameter_View.java,v 1.25 2012/04/16 06:22:59 castalia Exp

Copyright (C) 2001-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Viewers;

import	PIRL.PVL.*;
import	PIRL.TreeTable.*;
import	PIRL.Utilities.Streams;

import	javax.swing.*;
import	javax.swing.border.*;
import	javax.swing.event.*;
import	javax.swing.table.*;
import	javax.swing.tree.*;
import	java.awt.*;
import	java.awt.event.*;
import	java.io.InputStream;
import	java.io.File;
import	java.io.FileOutputStream;
import	java.io.BufferedOutputStream;
import	java.io.FileNotFoundException;
import	java.io.IOException;
import	java.net.URL;
import	java.net.MalformedURLException;
import	java.util.Hashtable;
import	java.util.Vector;
import	java.util.Enumeration;

/**	A <I>Parameter_View</I> provides a graphical view of a Parameter.
<P>
	The view contains a Parameter_Pane to display the Parameter data.
<P>
	@see		Parameter_Pane
	@see		Parameter

	@author		Bradford Castalia, UA/PIRL
	@version	1.25
*/
public class Parameter_View
	extends JFrame
{
private static final String
	ID = "PIRL.Viewers.Parameter_View (1.25 2012/04/16 06:22:59)";

/**	Started from main?
*/
private static boolean		_Application_ = false;

/**	The initial screen location.
*/
private static final Point	DEFAULT_INITIAL_LOCATION
								= new Point (175, 75);

/**	Locate a Parameter_View for an opened file.
*/
private View_Locator		Parameter_View_Locator
								= new View_Locator ();

/** The total number of Parameter_View displays being viewed.
	When this drops to zero the application will exit.
*/
private static int			_Total_Displays_ = 0;

/**	Workaround hack for OS X TreeTable folder open handle bug.
*/
private static boolean		Use_LAF_Hack
	= System.getProperty ("os.name").equals ("Mac OS X");

/**	For file selection.
*/
private JFileChooser		File_Chooser
								= new JFileChooser
									(System.getProperty ("user.dir"));

/**	For URL selection.
*/
private URL_Dialog			URL_Chooser
								= new URL_Dialog
									("Open URL", null, this, true);

/**	The display pane for the paramater.
*/
private	Parameter_Pane		The_Parameter_Pane;

/*==============================================================================
	Constructors
*/
/**	Creates a Parameter_View from the specified Parameter.
<P>
	@param	parameter	The Parameter to be displayed.
*/
public Parameter_View
	(
	String		title,
	Parameter	parameter
	)
{
if (title == null)
	{
	title = parameter.Name ();
	if (title == null ||
		title.length () == 0)
		title = "Parameter_View";
	}
setTitle (title);

if (Use_LAF_Hack ())
	{
	try
		{
		UIManager.setLookAndFeel
			(UIManager.getCrossPlatformLookAndFeelClassName ());
		}
	catch (Exception exception) {/* Just leave the existing L&F. */}
	}

//	Assemble the Parameter table pane.
The_Parameter_Pane = new Parameter_Pane (parameter);

//	Assemble the menus.
JMenuBar
	menu_bar = Menu_Bar ();
if (menu_bar != null)
	setJMenuBar (menu_bar);

getContentPane ().add (The_Parameter_Pane);

//	File chooser modes.
File_Chooser.setFileSelectionMode (JFileChooser.FILES_ONLY);
File_Chooser.setFileHidingEnabled (false);

//	On window close...
addWindowListener (new WindowAdapter ()
	{public void windowClosing (WindowEvent event)
	{
	if (--_Total_Displays_ == 0 && _Application_)
		System.exit (0);
	}});
pack ();
_Total_Displays_++;
}

/**	Creates a Parameter_View from the specified Parameter and
	gives it the Parameter's name as the title.
<P>
	@param	parameter	The Parameter to be displayed.
*/
public Parameter_View
	(
	Parameter	parameter
	)
{this (null, parameter);}

/*==============================================================================
	Accessors
*/
/**	Gets the Parameter_Pane used by this Parameter_View.
<P>
	@return	The Parameter_Pane used by this Parameter_View.
*/
public Parameter_Pane Parameter_Pane ()
{return The_Parameter_Pane;}

/**	Gets the Parameter being viewed by this Parameter_View.
<P>
	@return	The Parameter viewed by this Parameter_View.
*/
public Parameter Parameter ()
{return The_Parameter_Pane.Parameter ();}

/**	Sets the Parameter to be viewed by this Parameter_View.
<P>
	@param	parameter	The Parameter to be viewed by this Parameter_View.
	@see	Parameter_Pane#Parameter(Parameter)
*/
public void Parameter
	(
	Parameter parameter
	)
{The_Parameter_Pane.Parameter (parameter);}

/**	Gets the JTreeTable being used to represent the Parameters in the
	display.
<P>
	@return	The JTreeTable representing this Parameter_View.
*/
public JTreeTable TreeTable ()
{return The_Parameter_Pane.TreeTable ();}


/**	Gets the current working directory for finding parameter files.
*/
public String CWD ()
{return File_Chooser.getCurrentDirectory ().getAbsolutePath ();}

/**	Sets the current working directory for finding parameter files.
<P>
	@param	pathname	A host filesystem pathname. If null, the
		"user.dir" System property will be used.
	@return	This Parameter_View.
	@throws	FileNotFoundException	If neither the file nor the file's
		parent refers to an existing directory.
	@see	#CWD(File)
*/
public Parameter_View CWD
	(
	String	pathname
	)
	throws FileNotFoundException
{
if (pathname == null)
	pathname = System.getProperty ("user.dir");
return CWD (new File (pathname));
}

/**	Sets the current working directory for finding parameter files.
<P>
	@param	file	A File to which to set the CWD. If null, the
		"user.dir" System property will be used. If the file does not
		refer to an existing directory, the file's parent will be used.
	@return	This Parameter_View.
	@throws	FileNotFoundException	If neither the file nor the file's
		parent refers to an existing directory.
*/
public Parameter_View CWD
	(
	File	file
	)
	throws FileNotFoundException
{
if (file == null)
	file = new File (System.getProperty ("user.dir"));
if (! file.isAbsolute ())
	file = file.getAbsoluteFile ();
if (! file.isDirectory ())
	//	Can only set directories.
	file = file.getParentFile ();
if (file.exists ())
	File_Chooser.setCurrentDirectory (file);
else
	throw new FileNotFoundException
		(ID + '\n'
		+"Can't set working directory to " + file.getAbsolutePath () + '\n'
		+"No such directory.");
return this;
}


/**	Tests if the Java (cross platform) Look and Feel is to be used.
<P>
	As of Java 1.4 the native Look and Feel for Apple OS X does not
	function properly with the JTreeTable. The Java Look and Feel does
	not have this problem. Thus this hack to force its use.
<P>
	@return true if the Java Look and Feel hack will be used; false otherwise.
*/
public static boolean Use_LAF_Hack ()
{return Use_LAF_Hack;}

/**	Enables or disables the Java (cross platform) Look and Feel hack.
<P>
	As of Java 1.4 the native Look and Feel for Apple Mac OS X does not
	function properly with the JTreeTable used to display Parameters and
	their Values. The Java Look and Feel does not have this problem. Thus
	this hack to force its use. If the hack is disabled the hack will not
	be used.
<P>
	This hack is enabled by default if the "os.name" system property
	is "Mac OS X"; otherwise it is disabled.
<P>
	@param enable	true if the Look and Feel hack is to be used;
		false otherwise.
*/
public static void Use_LAF_Hack
	(
	boolean	enable
	)
{Use_LAF_Hack = enable;}

/*==============================================================================
	Display Elements
*/
/**	Sets the window title to the filename portion of the name.
<p>
	The source may be a file pathname or URL reference. The filename
	portion of either will be used as the window title.
<p>
	@param	source	The source reference from which to obtain the
		filename. If null the tile will be blank.
	@return	This Parameter_View.
*/
public Parameter_View Title
	(
	String	source
	)
{
if (source != null &&
	source.length () != 0)
	{
	try
		{
		URL
			url = new URL (source);
		source = url.getPath ();
		}
	catch (MalformedURLException except) {}
	source = new File (source).getName ();
	}
else
	source = "";
setTitle (source);
return this;
}

/**	Sets the window title to the filename portion of the URL.
<p>
	@param	url	The URL from which to obtain the filename. If null
		the tile will be blank.
	@return	This Parameter_View.
*/
public Parameter_View Title
	(
	URL		url
	)
{
String
	name = "";
if (url != null)
	name = new File (url.getPath ()).getName ();
setTitle (name);
return this;
}

/**	Creates a menu bar for the viewer.
<P>
<DL>
<DT><B>File</B>
	<DT><B>New</B>
		<DD>A JFileChooser is used to select a another file for display
		of its PVL parameters with a new Parameter_View. The
		JFileChooser directory is initially the user.dir System
		property, but it is reset after each file is opened to the
		directory that contained that file.
	<DT><B>Open</B>
		<DD>A JFileChooser is used to select a another file for display
		of its PVL parameters in the current Parameter_View.
	<DT><B>Close</B>
		<DD>Will cause the current window to close. If being run from
		main and this is the last window the application will exit.
	<DT><B>Exit</B>
		<DD>Will cause the application to exit.
	</DL>
</DL>
	@return	The viewer's JMenuBar.
*/
protected JMenuBar Menu_Bar ()
{
JMenuBar
	menu_bar = new JMenuBar ();
JMenu
	menu,
	submenu,
	subsubmenu;
JMenuItem
	menu_item;

menu = new JMenu ("File");

if (_Application_)
	{
	menu_item = new JMenuItem ("New");
	menu_item.setMnemonic ('N');
	menu_item.setAccelerator (KeyStroke.getKeyStroke ('N', Event.CTRL_MASK));
	menu_item.addActionListener (new ActionListener ()
		{public void actionPerformed (ActionEvent event)
		{New ();}});
	menu.add (menu_item);

	menu_item = new JMenuItem ("Open File ...");
	menu_item.setMnemonic ('O');
	menu_item.setAccelerator (KeyStroke.getKeyStroke ('O', Event.CTRL_MASK));
	menu_item.addActionListener (new ActionListener ()
		{public void actionPerformed (ActionEvent event)
		{Open_File ();}});
	menu.add (menu_item);

	menu_item = new JMenuItem ("Open URL ...");
	menu_item.setMnemonic ('U');
	menu_item.setAccelerator (KeyStroke.getKeyStroke ('U', Event.CTRL_MASK));
	menu_item.addActionListener (new ActionListener ()
		{public void actionPerformed (ActionEvent event)
		{Open_URL ();}});
	menu.add (menu_item);

	menu.addSeparator ();
	}

menu_item = new JMenuItem ("Save As ...");
menu_item.setMnemonic ('S');
menu_item.setAccelerator (KeyStroke.getKeyStroke ('S', Event.CTRL_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Save_Parameters ();}});
menu.add (menu_item);

menu_item = new JMenuItem ("Close");
menu_item.setMnemonic ('C');
menu_item.setAccelerator (KeyStroke.getKeyStroke ('W', Event.CTRL_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{
	if (_Total_Displays_ == 0 && _Application_)
		System.exit (0);
	dispose ();
	}});
menu.add (menu_item);

if (_Application_)
	{
	menu_item = new JMenuItem ("Exit"); 
	menu_item.setMnemonic ('X');
	menu_item.setAccelerator (KeyStroke.getKeyStroke ('Q', Event.CTRL_MASK));
	menu_item.addActionListener (new ActionListener ()
		{public void actionPerformed (ActionEvent event)
		{System.exit (0);}});
	menu.add (menu_item);
	}

menu_bar.add (menu);

menu = new JMenu ("View");
menu.add (The_Parameter_Pane.Array_Values_Menu ());

menu_bar.add (menu);

return menu_bar;
}


private void New ()
{
Parameter_View
	view = new Parameter_View (new Parameter ());
Parameter_View_Locator.Relocate (view, this);
view.setVisible (true);
}


private void Open_File ()
{
if (File_Chooser.showOpenDialog (rootPane) == JFileChooser.APPROVE_OPTION)
	{
	File
		file = File_Chooser.getSelectedFile ();
	if (! file.isFile ())
		{
		Dialog_Box.Notice
			("Can't open file " + file.getAbsolutePath () + '\n'
			+"No such file.",
			this);
		return;
		}
	try {CWD (file);}
	catch (FileNotFoundException exception) {}
	String
		canonical_pathname = null;
	try {canonical_pathname = file.getCanonicalPath ();}
	catch (Exception exception) {}
	Parameter
		parameters = null;
	try {parameters = new Parameter (file.getPath (), new Parser (file));}
	catch (PVL_Exception exception)
		{
		Dialog_Box.Error ("Unable to load Parameters.\n\n"
			+ ID + '\n'
			+ exception.Message ());
		return;
		}
	if (canonical_pathname != null &&
		! canonical_pathname.equals (parameters.Name ()))
		parameters.Comments (canonical_pathname);
	The_Parameter_Pane.Parameter (parameters);
	setTitle (file.getName ());
	}
}


private void Open_URL ()
{
URL_Chooser.setVisible (true);
URL
	url = URL_Chooser.Current_URL ();
if (url != null)
	{
	InputStream
		stream = null;
	try {stream = url.openStream ();}
	catch (IOException exception)
		{
		Dialog_Box.Error
			("Unable to open a connection to the URL\n"
			+ url + "\n\n"
			+ exception.getMessage (),
			this);
		return;
		}
	Parameter
		parameters = null;
	try {parameters = new Parameter (url.toString (), new Parser (stream));}
	catch (PVL_Exception exception)
		{
		Dialog_Box.Error ("Unable to load Parameters.\n\n"
			+ ID + '\n'
			+ exception.Message ());
		return;
		}
	The_Parameter_Pane.Parameter (parameters);
	Title (url);
	}
}


private void Save_Parameters ()
{
if (File_Chooser.showSaveDialog (this)
		!= JFileChooser.APPROVE_OPTION)
	return;
File
	file = File_Chooser.getSelectedFile ();
try {CWD (file);}
catch (FileNotFoundException exception)
	{
	Dialog_Box.Notice
		("The pathname to " + file.getPath () + '\n'
		+"does not exist.",
		this);
	return;
	}
if (file.exists ())
	{
	if (file.isDirectory ())
		{
		Dialog_Box.Notice ("Can't replace directory " + file.getAbsolutePath (),
			this);
		return;
		}
	if (! Dialog_Box.Confirm ("Replace " + file.getAbsolutePath () + '?', this))
		return;
	}

BufferedOutputStream
	output = null;
try
	{
	output = new BufferedOutputStream (new FileOutputStream (file), 4096);
	The_Parameter_Pane.Parameter ().Name (Parser.CONTAINER_NAME).Write (output);
	}
catch (Exception exception)
	{
	Dialog_Box.Error
		("Couldn't write file " + file.getAbsolutePath () + "\n\n"
		+ exception.getMessage ());
	}
}

/*=*****************************************************************************
	Application
*/
/**	Creates a Parameter_View displaying the Parameters found in each file
	listed on the command line.
<p>
	@param	arguments	The command line arguments String array. If no
		arguments are provided an empty viewer will be created.
*/
public static void main (String[] arguments)
{
_Application_ = true;
Parameter_View
	view;
if (arguments.length == 0)
	{
	view = new Parameter_View (new Parameter ());
	view.setLocation (DEFAULT_INITIAL_LOCATION);
	view.setVisible (true);
	}
else
	{
	try
		{
		File
			source_file = null;
		String
			canonical_pathname = null;
		InputStream
			source_stream;
		Parameter_View
			previous_view = null;
		View_Locator
			locator = new View_Locator ();

		for (int count = 0;
				 count < arguments.length;
				 count++)
			{
			canonical_pathname = null;
			if (arguments[count].equals ("-"))
				{
				source_stream = System.in;
				//	Prevent the display of the root node.
				source_file =
					new File (arguments[count] = Parser.CONTAINER_NAME);
				}
			else if ((source_stream = Streams.Get_Stream (arguments[count]))
					!= null)
				{
				//	Identify the source file.
				try
					{
					URL
						url = new URL (arguments[count]);
					source_file = new File (url.getPath ());
					if (url.getProtocol ().equals ("file"))
						{
						try
							{
							canonical_pathname =
								source_file.getCanonicalPath ();
							source_file = new File (canonical_pathname);
							}
						catch (Exception e) {}
						}
					}
				catch (MalformedURLException except)
					{
					source_file = new File (arguments[count]);
					try
						{
						canonical_pathname =
							source_file.getCanonicalPath ();
						source_file = new File (canonical_pathname);
						}
					catch (Exception e) {}
					}
				}
			else
				{
				System.err.println
					("Unable to access source: " + arguments[count]);
				continue;
				}

			//	Load the parameters.
			Parameter
				parameters = null;
			try {parameters = new Parameter
						(arguments[count], new Parser (source_stream))
					.Comments (canonical_pathname);}
			catch (PVL_Exception exception)
				{
				System.err.println
					("Unable to parse the PVL in "
						+ arguments[count] + '\n'
					+ exception.getMessage () + '\n');
				continue;
				}

			//	Construct the viewer.
			view = new Parameter_View (source_file.getName (), parameters);
			if (canonical_pathname != null)
				view.CWD (source_file.getParent ());
			if (previous_view == null)
				view.setLocation (DEFAULT_INITIAL_LOCATION);
			else
				locator.Relocate (view, previous_view);
			previous_view = view;
			view.setVisible (true);
			}
		}
	catch (Exception exception)
		{
		System.err.println (exception.getMessage ());
		System.exit (-1);
		}
	}
if (_Total_Displays_ == 0)
	System.exit (1);
}

}	//	End of class Parameter_View.
