# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright 2013 VMware, Inc.  All rights reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
#
# @author: Kaiwei Fan, VMware, Inc

import sqlalchemy as sa
from sqlalchemy import event
from sqlalchemy.orm import exc

from quantum.common import exceptions as qexception
from quantum.db import model_base
from quantum.extensions import routedserviceinsertion as rsi


class ServiceRouterBinding(model_base.BASEV2):
    resource_id = sa.Column(sa.String(36),
                            primary_key=True)
    resource_type = sa.Column(sa.String(36),
                              primary_key=True)
    router_id = sa.Column(sa.String(36),
                          sa.ForeignKey('routers.id'),
                          nullable=False)


class AttributeException(qexception.QuantumException):
    message = _("Resource type '%(resource_type)s' is longer "
                "than %(maxlen)d characters")


@event.listens_for(ServiceRouterBinding.resource_type, 'set', retval=True)
def validate_resource_type(target, value, oldvalue, initiator):
    """Make sure the resource type fit the resource_type column."""
    maxlen = ServiceRouterBinding.resource_type.property.columns[0].type.length
    if len(value) > maxlen:
        raise AttributeException(resource_type=value, maxlen=maxlen)
    return value


class RoutedServiceInsertionDbMixin(object):
    """Mixin class to add router service insertion."""

    def _process_create_resource_router_id(self, context, resource, model):
        with context.session.begin(subtransactions=True):
            db = ServiceRouterBinding(
                resource_id=resource['id'],
                resource_type=model.__tablename__,
                router_id=resource[rsi.ROUTER_ID])
            context.session.add(db)
        return self._make_resource_router_id_dict(db, model)

    def _extend_resource_router_id_dict(self, context, resource, model):
        binding = self._get_resource_router_id_binding(
            context, resource['resource_id'], model)
        resource[rsi.ROUTER_ID] = binding['router_id']

    def _get_resource_router_id_binding(self, context, resource_id, model):
        query = self._model_query(context, ServiceRouterBinding)
        query = query.filter(
            ServiceRouterBinding.resource_id == resource_id,
            ServiceRouterBinding.resource_type == model.__tablename__)
        return query.first()

    def _make_resource_router_id_dict(self, resource_router_binding, model,
                                      fields=None):
        resource = {'resource_id': resource_router_binding['resource_id'],
                    'resource_type': model.__tablename__,
                    rsi.ROUTER_ID: resource_router_binding[rsi.ROUTER_ID]}
        return self._fields(resource, fields)

    def _delete_resource_router_id_binding(self, context, resource_id, model):
        with context.session.begin(subtransactions=True):
            binding = self._get_resource_router_id_binding(
                context, resource_id, model)
            if binding:
                context.session.delete(binding)
