

<!-- README.md is generated from README.Rmd. Please edit that file -->

# callr

> Call R from R

<!-- badges: start -->
[![lifecycle](https://lifecycle.r-lib.org/articles/figures/lifecycle-stable.svg)](https://lifecycle.r-lib.org/articles/stages.html)
[![R-CMD-check](https://github.com/r-lib/callr/actions/workflows/R-CMD-check.yaml/badge.svg)](https://github.com/r-lib/callr/actions/workflows/R-CMD-check.yaml)
[![](https://www.r-pkg.org/badges/version/callr)](https://www.r-pkg.org/pkg/callr)
[![CRAN Posit mirror downloads](https://cranlogs.r-pkg.org/badges/callr)](https://www.r-pkg.org/pkg/callr)
[![Codecov test coverage](https://codecov.io/gh/r-lib/callr/branch/main/graph/badge.svg)](https://app.codecov.io/gh/r-lib/callr?branch=main)
<!-- badges: end -->


It is sometimes useful to perform a computation in a separate R process,
without affecting the current R process at all. This packages does exactly
that.

---

- [Features](#features)
- [Installation](#installation)
- [Synchronous, one-off R processes](#synchronous-one-off-r-processes)
  - [Passing arguments](#passing-arguments)
  - [Using packages](#using-packages)
  - [Error handling](#error-handling)
  - [Standard output and error](#standard-output-and-error)
- [Background R processes](#background-r-processes)
- [Multiple background R processes and
  `poll()`](#multiple-background-r-processes-and-poll)
- [Persistent R sessions](#persistent-r-sessions)
- [Running `R CMD` commands](#running-r-cmd-commands)
- [Code of Conduct](#code-of-conduct)

## Features

- Calls an R function, with arguments, in a subprocess.
- Copies function arguments to the subprocess and copies the return
  value of the function back, seamlessly.
- Copies error objects back from the subprocess, including a stack
  trace.
- Shows and/or collects the standard output and standard error of the
  subprocess.
- Supports both one-off and persistent R subprocesses.
- Calls the function synchronously or asynchronously (in the
  background).
- Can call `R CMD` commands, synchronously or asynchronously.
- Can call R scripts, synchronously or asynchronously.
- Provides extensible `r_process`, `rcmd_process` and `rscript_process`
  R6 classes, based on `processx::process`.

## Installation

Install the stable version from CRAN:

``` r
install.packages("callr")
```

Install the development version from GitHub:

``` r
pak::pak("r-lib/callr")
```

## Synchronous, one-off R processes

Use `r()` to run an R function in a new R process. The results are
passed back seamlessly:

``` r
callr::r(function() var(iris[, 1:4]))
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/simple-dark.svg">
<img src="man/figures/simple.svg" /> </picture>

### Passing arguments

You can pass arguments to the function by setting `args` to the list of
arguments. This is often necessary as these arguments are explicitly
copied to the child process, whereas the evaluated function cannot refer
to variables in the parent. For example, the following does not work:

``` r
mycars <- cars
callr::r(function() summary(mycars))
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/passargsfail-dark.svg">
<img src="man/figures/passargsfail.svg" /> </picture>

But this does:

``` r
mycars <- cars
callr::r(function(x) summary(x), args = list(mycars))
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/passargsok-dark.svg">
<img src="man/figures/passargsok.svg" /> </picture>

Note that the arguments will be serialized and saved to a file, so if
they are large R objects, it might take a long time for the child
process to start up.

### Using packages

You can use any R package in the child process, just make sure to refer
to it explicitly with the `::` operator. For example, the following code
creates an [igraph](https://github.com/igraph/rigraph) graph in the
child, and calculates some metrics of it.

``` r
callr::r(function() { g <- igraph::sample_gnp(1000, 4/1000); igraph::diameter(g) })
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/packages-dark.svg">
<img src="man/figures/packages.svg" /> </picture>

### Error handling

callr copies errors from the child process back to the main R session:

``` r
callr::r(function() 1 + "A")
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/error1-dark.svg">
<img src="man/figures/error1.svg" /> </picture> callr sets the
`.Last.error` variable, and after an error you can inspect this for more
details about the error, including stack traces both from the main R
process and the subprocess.

``` r
.Last.error
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/error2-2-dark.svg">
<img src="man/figures/error2-2.svg" /> </picture>

The error objects has two parts. The first belongs to the main process,
and the second belongs to the subprocess.

`.Last.error` also includes a stack trace, that includes both the main R
process and the subprocess:

The top part of the trace contains the frames in the main process, and
the bottom part contains the frames in the subprocess, starting with the
anonymous function.

### Standard output and error

By default, the standard output and error of the child is lost, but you
can request callr to redirect them to files, and then inspect the files
in the parent:

``` r
x <- callr::r(function() { print("hello world!"); message("hello again!") },
  stdout = "/tmp/out", stderr = "/tmp/err"
)
readLines("/tmp/out")
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/io-dark.svg">
<img src="man/figures/io.svg" /> </picture>

``` r
readLines("/tmp/err")
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/io-2-dark.svg">
<img src="man/figures/io-2.svg" /> </picture>

With the `stdout` option, the standard output is collected and can be
examined once the child process finished. The `show = TRUE` options will
also show the output of the child, as it is printed, on the console of
the parent.

## Background R processes

`r_bg()` is similar to `r()` but it starts the R process in the
background. It returns an `r_process` R6 object, that provides a rich
API:

``` r
rp <- callr::r_bg(function() Sys.sleep(.2))
rp
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/bg-dark.svg">
<img src="man/figures/bg.svg" /> </picture>

This is a list of all `r_process` methods:

``` r
ls(rp)
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/bg-methods-dark.svg">
<img src="man/figures/bg-methods.svg" /> </picture>

These include all methods of the `processx::process` superclass and the
new `get_result()` method, to retrieve the R object returned by the
function call. Some of the handiest methods are:

- `get_exit_status()` to query the exit status of a finished process.
- `get_result()` to collect the return value of the R function call.
- `interrupt()` to send an interrupt to the process. This is equivalent
  to a `CTRL+C` key press, and the R process might ignore it.
- `is_alive()` to check if the process is alive.
- `kill()` to terminate the process.
- `poll_io()` to wait for any standard output, standard error, or the
  completion of the process, with a timeout.
- `read_*()` to read the standard output or error.
- `suspend()` and `resume()` to stop and continue a process.
- `wait()` to wait for the completion of the process, with a timeout.

## Multiple background R processes and `poll()`

Multiple background R processes are best managed with the
`processx::poll()` function that waits for events (standard output/error
or termination) from multiple processes. It returns as soon as one
process has generated an event, or if its timeout has expired. The
timeout is in milliseconds.

``` r
rp1 <- callr::r_bg(function() { Sys.sleep(1/2); "1 done" })
rp2 <- callr::r_bg(function() { Sys.sleep(1/1000); "2 done" })
processx::poll(list(rp1, rp2), 1000)
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/poll-dark.svg">
<img src="man/figures/poll.svg" /> </picture>

``` r
rp2$get_result()
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/poll-2-dark.svg">
<img src="man/figures/poll-2.svg" /> </picture>

``` r
processx::poll(list(rp1), 1000)
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/poll-3-dark.svg">
<img src="man/figures/poll-3.svg" /> </picture>

``` r
rp1$get_result()
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/poll-4-dark.svg">
<img src="man/figures/poll-4.svg" /> </picture>

## Persistent R sessions

`r_session` is another `processx::process` subclass that represents a
persistent background R session:

``` r
rs <- callr::r_session$new()
rs
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/rsession-dark.svg">
<img src="man/figures/rsession.svg" /> </picture>

`r_session$run()` is a synchronous call, that works similarly to `r()`,
but uses the persistent session. `r_session$call()` starts the function
call and returns immediately. The `r_session$poll_process()` method or
`processx::poll()` can then be used to wait for the completion or other
events from one or more R sessions, R processes or other
`processx::process` objects.

Once an R session is done with an asynchronous computation, its
`poll_process()` method returns `"ready"` and the `r_session$read()`
method can read out the result.

``` r
rs <- callr::r_session$new()
rs$run(function() runif(10))
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/rsession2-dark.svg">
<img src="man/figures/rsession2.svg" /> </picture>

``` r
rs$call(function() rnorm(10))
rs
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/rsession2-2-dark.svg">
<img src="man/figures/rsession2-2.svg" /> </picture>

``` r
rs$poll_process(2000)
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/rsession-4-dark.svg">
<img src="man/figures/rsession-4.svg" /> </picture>

``` r
rs$read()
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/rsession-5-dark.svg">
<img src="man/figures/rsession-5.svg" /> </picture>

## Running `R CMD` commands

The `rcmd()` function calls an `R CMD` command. For example, you can
call `R CMD INSTALL`, `R CMD check` or `R CMD config` this way:

``` r
callr::rcmd("config", "CC")
```

<picture>
<source media="(prefers-color-scheme: dark)" srcset="man/figures/rcmd-dark.svg">
<img src="man/figures/rcmd.svg" /> </picture>

This returns a list with three components: the standard output, the
standard error, and the exit (status) code of the `R CMD` command.

## Configuration

### Environment variables

* `CALLR_NO_TEMP_DLLS`: If `true`, then callr does not use a temporary
  directory to copy the client DLL files from, in the subprocess. By
  default callr copies the DLL file that drives the callr subprocess into
  a temporary directory and loads it from there. This is mainly to avoid
  locking a DLL file in the package library, on Windows. If this default
  causes issues for you, set it to `true`, and then callr will use the DLL
  file from the installed processx package. See also
  https://github.com/r-lib/callr/issues/273.

## Code of Conduct

Please note that the callr project is released with a [Contributor Code
of Conduct](https://callr.r-lib.org/CODE_OF_CONDUCT.html). By
contributing to this project, you agree to abide by its terms.
