<?php
/**
 * Forwards_Driver:: defines an API for implementing forwarding backends
 * for Forwards.
 *
 * $Horde: forwards/lib/Driver.php,v 1.29.2.4 2009/01/06 15:22:46 jan Exp $
 *
 * Copyright 2001-2009 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file LICENSE for license information (BSDL). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Mike Cochrane <mike@graftonhall.co.nz>
 * @author  Jan Schneider <jan@horde.org>
 * @since   Forwards 2.1
 * @package Forwards
 */
class Forwards_Driver {

    /**
     * The current user name.
     *
     * @var string
     */
    var $_user;

    /**
     * The user's realm.
     *
     * @var string
     */
    var $_realm;

    /**
     * Hash containing configuration data.
     *
     * @var array
     */
    var $_params;

    /**
     * Constructor.
     *
     * @param string $user   A user name.
     * @param string $realm  The realm of the user.
     * @param array $params  A hash containing connection parameters.
     */
    function Forwards_Driver($user, $realm, $params = null)
    {
        if (is_null($params)) {
            $params = Horde::getDriverConfig('server', null);
        }
        $this->_params = $params;
        $this->_user = $user;
        if (!isset($params[$realm])) {
            $realm = 'default';
        }
        $this->_realm = $realm;
    }

    /**
     * Returns the current user.
     *
     * @return string  The current user name.
     */
    function getUser()
    {
        return $this->_user;
    }

    /**
     * Begins forwarding of mail for a user.
     *
     * @abstract
     *
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     */
    function enableForwarding($password, $target, $keeplocal = false)
    {
        return PEAR::raiseError('Abstract method enableForwarding() not implemented');
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @abstract
     *
     * @param string $password  The password of the user.
     */
    function disableForwarding($password)
    {
        return PEAR::raiseError('Abstract method disableForwarding() not implemented');
    }

    /**
     * Retrieves current state of mail redirection for a user.
     *
     * @abstract
     *
     * @param string $password  The password of the user.
     *
     * @return mixed  Returns 'Y' if forwarding is enabled for the user, 'N' if
     *                forwarding is currently disabled, false if the status
     *                cannot be determined, and PEAR_Error on error.
     */
    function isEnabledForwarding($password)
    {
        return PEAR::raiseError('Abstract method disableForwarding() not implemented');
    }

    /**
     * Checks if user is keeping a local copy of forwarded mail.
     *
     * @abstract
     *
     * @param string $password  The password of the user.
     *
     * @return boolean  True if user is keeping a local copy of mail,
     *                  otherwise false.
     */
    function isKeepLocal($password)
    {
        return PEAR::raiseError('Abstract method isKeepLocal() not implemented');
    }

    /**
     * Retrieves current target of mail redirection for a user.
     *
     * @abstract
     *
     * @param string $password  The password of the user.
     *
     * @return string  The current forwarding mail address, or PEAR_Error on
     *                 error.
     */
    function currentTarget($password)
    {
        return PEAR::raiseError('Abstract method currentTarget() not implemented');
    }

    /**
     * Parses an email address list and returns it in a known standard form.
     *
     * This will attempt to add the domain (realm) to unqualified addresses if
     * the realm is non-blank and not 'default'.
     *
     * @param string $user   The email address.
     * @param string $realm  The domain/realm to add if none is present.
     *
     * @return string  The email address(es) on success, false on error.
     */
    function _makeEmailAddress($user)
    {
        $domain = $this->_realm != 'default' ? $this->_realm : '';
        $email = '';

        require_once 'Mail/RFC822.php';
        $parser = new Mail_RFC822();
        $parsed_email = $parser->parseAddressList($user, $domain, false, false);
        if (!is_array($parsed_email) || !count($parsed_email)) {
            return PEAR::raiseError(_("Cannot parse your email address"));
        }

        for ($i = 0; $i < count($parsed_email); $i++) {
            if (!empty($email)) {
                $email .= ',';
            }
            if (is_object($parsed_email[$i])) {
                $email .= $parsed_email[$i]->mailbox;
                if (!empty($parsed_email[$i]->host)) {
                    $email .=  '@' . $parsed_email[$i]->host;
                }
            } else {
                $email .= $parsed_email[$i];
            }
        }

        return $email;
    }

    /**
     * Gets a concrete Forwards_Driver instance.
     *
     * @param string $user    A user name.
     * @param string $realm   The realm of the user.
     * @param string $driver  The type of concrete Forwards_Driver subclass to
     *                        return.  The code for the driver is dynamically
     *                        included.
     * @param array $params   A hash containing any additional configuration or
     *                        connection parameters a subclass might need.
     *
     * @return Forwards_Driver  The newly created concrete Forwards_Driver
     *                          instance, or false on error.
     */
    function factory($user, $realm, $driver = null, $params = null)
    {
        if (is_null($driver)) {
            $driver = $GLOBALS['conf']['server']['driver'];
        }
        $driver = basename($driver);

        include_once dirname(__FILE__) . '/Driver/' . $driver . '.php';
        $class = 'Forwards_Driver_' . $driver;
        if (class_exists($class)) {
            return new $class($user, $realm, $params);
        }

        return false;
    }

}
