<?php

require_once dirname(__FILE__) . '/customsql.php';

/**
 * Forwards_Driver_sql:: implements the Forwards_Driver API for SQL servers.
 *
 * $Horde: forwards/lib/Driver/sql.php,v 1.29.2.4 2009/01/06 15:22:46 jan Exp $
 *
 * Copyright 2001-2009 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file LICENSE for license information (BSDL). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Ilya Krel <mail@krel.org>
 * @author  Mike Cochrane <mike@graftonhall.co.nz>
 * @author  Jan Schneider <jan@horde.org>
 * @since   Forwards 2.1
 * @package Forwards
 */
class Forwards_Driver_sql extends Forwards_Driver_customsql {

    /**
     * List of required parameters.
     *
     * @var array
     */
    var $_requiredParams = array('phptype', 'table');

    /**
     * Constructor.
     *
     * @param string $user   A user name.
     * @param string $realm  The realm of the user.
     * @param array $params  A hash containing connection parameters.
     */
    function Forwards_Driver_sql($user, $realm, $params = null)
    {
        parent::Forwards_Driver_customsql($user, $realm, $params);
        $this->_params['query_select'] = 'SELECT * FROM '
            . $this->_params[$this->_realm]['table']
            . ' WHERE ' . $this->_params[$this->_realm]['user_col'] . ' = \U'
            . ' AND ' . $this->_params[$this->_realm]['pass_col'] . ' = \P';
    }

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     */
    function enableForwarding($password, $target, $keeplocal)
    {
        // Build the SQL query.
        $query = 'UPDATE ' . $this->_params[$this->_realm]['table']
            . ' SET ' . $this->_params[$this->_realm]['forward'] . ' = \'Y\', '
            . $this->_params[$this->_realm]['altemail'] . ' = \T'
            . ' WHERE ' . $this->_params[$this->_realm]['user_col'] . ' = \U'
            . ' AND ' . $this->_params[$this->_realm]['pass_col'] . ' = \P';

        // Build username.
        $myuser = $this->_buildUsername();
        if ($keeplocal === 'on' && !strstr($target, $myuser)) {
            $target .= ',' . $myuser;
        }

        return $this->_enableForwarding($query, md5($password), $target);
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string $password  The password of the user.
     */
    function disableForwarding($password)
    {
        // Build the SQL query.
        $query = 'UPDATE ' . $this->_params[$this->_realm]['table'] .
                 ' SET ' . $this->_params[$this->_realm]['forward'] . ' = \'N\',' .
                 ' ' . $this->_params[$this->_realm]['altemail'] . ' = \'\'' .
                 ' WHERE ' . $this->_params[$this->_realm]['user_col'] . ' = \U' .
                 ' AND ' . $this->_params[$this->_realm]['pass_col'] . ' = \P';

        return $this->_disableForwarding($query, md5($password));
    }

    /**
     * Retrieves current state of mail redirection for a user.
     *
     * @param string $password  The password of the user.
     *
     * @return mixed  Returns 'Y' if forwarding is enabled for the user, 'N' if
     *                forwarding is currently disabled, false if the status
     *                cannot be determined, and PEAR_Error on error.
     */
    function isEnabledForwarding($password)
    {
        // Get current details.
        $current_details = $this->_getUserDetails(md5($password));
        if ($current_details === false ||
            is_a($current_details, 'PEAR_Error')) {
            return $current_details;
        }

        // Check forwarding flag.
        if ($current_details[$this->_params[$this->_realm]['forward']] === 'y' ||
            $current_details[$this->_params[$this->_realm]['forward']] === 'Y') {
            return 'Y';
        }
        if ($current_details[$this->_params[$this->_realm]['forward']] === 'n' ||
            $current_details[$this->_params[$this->_realm]['forward']] === 'N') {
            return 'N';
        }
        return false;
    }

    /**
     * Checks if user is keeping a local copy of forwarded mail.
     *
     * @param string $password  The password of the user.
     *
     * @return boolean  True if user is keeping a local copy of mail,
     *                  otherwise false.
     */
    function isKeepLocal($password)
    {
        // Get current details.
        $current_details = $this->_getUserDetails(md5($password));
        if ($current_details === false ||
            is_a($current_details, 'PEAR_Error')) {
            return $current_details;
        }

        // Check retain copy flag.
        return false !== strpos($current_details[$this->_params[$this->_realm]['altemail']],
                                $this->_buildUsername());
    }

    /**
     * Retrieves current target of mail redirection for a user.
     *
     * @param string $password  The password of the user.
     *
     * @return string  The current forwarding mail address, false if no
     *                 forwarding is set, or PEAR_Error on error.
     */
    function currentTarget($password)
    {
        $current_details = $this->_getUserDetails(md5($password));
        if ($current_details === false ||
            is_a($current_details, 'PEAR_Error')) {
            return $current_details;
        }

        // Check current forwarding mail address.
        return str_replace(',' . $this->_buildUsername(), '',
                           $current_details[$this->_params[$this->_realm]['altemail']]);
    }

}
