/*
* ===========================
* VDK Visual Development Kit
* Version 1.2.3
* October 1998, August 2000
* ===========================
*
* Copyright (C) 1998, Mario Motta
* Developed by Mario Motta <mmotta@guest.net>
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
* 02111-130
*/ 

#ifndef VDKSTRING_H
#define VDKSTRING_H

#define MAXPRINTFLEN	65535	// max size for Sprintf and Concatf buffer
#define INT_DATE		0		// for FormatDate
#define ENG_DATE		1
#define EUR_DATE		2

struct STRING 
{          
	char* s;              
	unsigned int ref ;    
};

/*!
\class VDKString
\brief Implements famous cont referenced string objects
*/
class VDKString 
{       
protected:
	STRING* p;         
public:
/*!
Constructor, makes an empty string
\code
VDKString s;
\endcode
*/
	VDKString();
/*!
Constructor
\param s a null terminated C string
\code
VDKString s = "uncle bill";
\endcode
*/

	VDKString (const char*s);
/*!
Constructor
\param c a single character
\code
VDKString s(c);
\endcode
*/

	VDKString (const char& c);
/*!
Copy-initializer
\param s a VDKString reference
\code
VDKString s = "uncle bill";
VDKString s1 = s;
\endcode
*/

	VDKString(const VDKString& s);

/*!
Assignement operator
\param s a VDKString reference
\code
VDKString s = "uncle bill";
VDKString s1 = s;
VDKString s2 = "uncle sam";
s = s2;
\endcode
*/
	VDKString& operator= (const VDKString& s);

/*!
Assignement operator
\param s a null terminated C string
\code
VDKString s = "uncle bill";
s = "uncle sam";
\endcode
*/
	VDKString& operator= (const char* s);

/*!
Destructor
*/
~VDKString();
/*!
VDKString to char* casting
** warning ** can violate data hiding OO concept
*/
	operator char*() { return p->s; }

/*!
Equality operator
*/
	int operator == (const VDKString& s) const ;
/*!
less than operator
*/
	int operator <( const VDKString& s) const ;
/*!
greater than operator
*/
	int operator>(const VDKString& s)  const ;
/*!
less-equal operator
*/
	int operator <=(const VDKString& s) const ;
/*!
greater-equal operator
*/
	int operator >=(const VDKString& s) const ;
/*!
disequality operator
*/
	int operator !=(const VDKString& s) const ;
/*!
cat to this
\param s a null terminated string
\code
VDKString s = "uncle bill";
s += " is a smart boy";
\endcode
*/
	VDKString& operator +=(const char* s);
/*!
cat to this
\param s a VDKString
\code
VDKString s = "uncle bill";
VDKString s1 = " is a smart boy";
s += s1;
\endcode
*/
	VDKString& operator +=(const VDKString& s);
/*!
Returns a VDKString concatenated\param s a null terminated string
\code
VDKString s = "uncle bill";
VDKString s1 = s + " is a smart boy";
\endcode
*/
	VDKString  operator + (const char* s) const;
	friend VDKString operator + (const char* s, const VDKString& vdks);
/*!
Returns a VDKString concatenated
\param s a VDKString
*/
	VDKString  operator +(const VDKString& s) const;
/*!
Returns true if this is an empty string
*/
	bool isNull() const;
/*!
as strlen()
*/
	int size() const;
/*!
index operator for const instances returns NULL if ix >= size
*/
char operator[](unsigned int ix) const;
/*! 
string pointer access for const instances
*/
const char* c_str() const;
/*!
Removes a part of the string, beginning at 'begin' on 'len' length.
Modifies and returns the resulting VDKString.
\param begin char number where begins the selection (0 based)
\param len   selection length
*/
	VDKString& DelSelection(unsigned int begin, unsigned int len);
/*!
Removes all trailing spaces.
Modifies and returns the resulting VDKString.
*/
	VDKString& RTrim();
/*!
Removes all leading spaces.
Modifies and returns the resulting VDKString.
*/
	VDKString& LTrim();
/*!
Removes all leading and trailing spaces.
Modifies and returns the resulting VDKString.
*/
	VDKString& Trim();
/*!
Returns the number of the specified char 'car' contained in the string.
\param car char to be counted
\code
VDKString s = "uncle bill";
int NumCar = s.CharCount('l');	// NumCar value is 3
\endcode
*/
	unsigned int CharCount(const char car) const;
/*!
Returns the upper case VDKString after having modify it.
\warning Does not modify unknown characters.
\warning Upper case characters are assumed without accents.
*/
	VDKString& UpperCase();
/*!
Returns the lower case VDKString after having modify it.
\warning Upper case characters are assumed without accents.
*/
	VDKString& LowerCase();
/*!
Returns true if this is an empty string meaning
NULL buffer or strlen() == 0.
*/
	bool isEmpty() const;
/*!
Strcat() to the existing string (printf style).
Modifies and returns the resulting VDKString.
\warning Final string is 65534 chars max.
\warning Returns the previous string in case of memory overflow
or buffer overflow.
\param format	a NULL terminated string
\param ...		a list of parameters
\code
VDKString s = "uncle bill";
s.Concatf("%s", " is a smart boy");	// s value is "uncle bill is a smart boy"
\endcode
*/
	VDKString& Concatf(const char* format, ...);
/*!
Assignment to string (printf style).
Modifies and returns the resulting VDKString.
\warning Final string is 65534 chars max.
\warning Returns the previous string in case of memory overflow
or buffer overflow.
\param format	a NULL terminated string
\param ...		a list of parameters
\code
VDKString s;
s.Sprintf("%s is %d years old", "uncle bill", 40);	// s value is "uncle bill is 40 years old"
\endcode
*/
	VDKString& Sprintf(const char* format, ...);
/*!
Extract the specified part of a formatted string.
Modifies and returns the resulting VDKString.
\warning Returns an isNull() string if the specified part not found.
\param i	the desired part position (starting at 1)
\param sep	the parts separator, '|' by default
\code
VDKString s = "one|two|three|four";
VDKString p = s;
p.GetPart(2);		// p value is "two"
\endcode
*/
	VDKString& GetPart(unsigned int i, const char sep = '|');
/*!
Returns the first occurrence position of the specified char 'car' (0 based)
or -1 if 'car ' not found.
\param car char to be searched for
*/
	int GetFCharPos(const char car) const;
/*!
Returns the last occurrence position of the specified char 'car' (0 based)
or -1 if 'car ' not found.
\param car char to be searched for
*/
	int GetLCharPos(const char car) const;
/*!
Returns the converted string to double.
See atof() for details.
*/
	double StrtoDouble() const;
/*!
Returns the converted string to int.
See atoi() for details.
*/
	int StrtoInt() const;
/*!
Extract a part of the string beginning at 'start' upon 'len' length.
Modifies and returns the resulting VDKString.
\param start	first char position  (0 based)
\param len		maximum length of the resulting string
*/
	VDKString& SubStr(unsigned int start, unsigned int len);
/*!
Cut the string at 'len' length.
Modifies and returns the resulting VDKString.
\param len		length of the resulting string
*/
	VDKString& Cut(unsigned int len);
/*!
Pad left of string with a specified char 'car' upon 'len' length.
Modifies and returns the resulting VDKString.
\param len		length of the resulting string
\param car		char to be padded
*/
	VDKString& LPad(unsigned int len, const char car);
/*!
Pad right of string with a specified char 'car' upon 'len' length.
Modifies and returns the resulting VDKString.
\param len		length of the resulting string
\param car		char to be padded
*/
	VDKString& RPad(unsigned int len, const char car);
/*!
Double all 'car' chars in the string (for SQL purpose).
Modifies and returns the resulting VDKString.
\param car		char to be doubled, '\'' (cote) by default
\code
VDKString s = "Don't do that";
VDKString p = s;
p.DoublaChar(); // p value is "Don''t do that"
\endcode
*/
	VDKString& DoubleChar(const char car = '\'');
/*!
Returns a VDKString containing a formatted date according to
parameters settings.
Modifies and returns the resulting VDKString.
\warning Only complete dates are supported.
That's to say days and months on two digits
and years on 4 digits. For ex. : 02/03/2000.
\param sep	desired separator. If 0, no separator left
\param orig	date style staying in VDKString buffer
\param ret	date style to return
\code
VDKString s = "12/25/2000";
VDKString p = s;
p.FormatDate(0, ENG_DATE, INT_DATE); // p value is "20001225"

VDKString s = "12/25/2000";
VDKString p = s;
p.FormatDate('-', ENG_DATE, EUR_DATE); // p value is "25-12-2000"
\endcode
*/
	VDKString& FormatDate(const char sep, int orig, int ret);
};

#endif




