/* Copyright (C) 2001, 2007 United States Government as represented by
   the Administrator of the National Aeronautics and Space Administration.
   All Rights Reserved.
*/
package gov.nasa.worldwind.servers.wms;

import com.sun.xml.bind.v2.util.ByteArrayOutputStreamEx;
import gov.nasa.worldwind.servers.wms.formats.ImageTypes;
import gov.nasa.worldwind.servers.wms.formats.ImageFormatter;
import gov.nasa.worldwind.servers.wms.xml.ObjectFactory;
import gov.nasa.worldwind.servers.wms.xml.ServiceExceptionReport;
import gov.nasa.worldwind.servers.wms.xml.ServiceExceptionType;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;
import java.io.*;
import java.util.Iterator;
import java.util.List;
import java.util.zip.DeflaterOutputStream;
import java.util.zip.GZIPOutputStream;

/**
 *
 * @author brownrigg
 * @version $Id: WMSServlet.java 5011 2008-04-10 16:53:54Z rick $
 */
public class WMSServlet {

    public WMSServlet() {
    }
    
    public void init(WMSServletConfig config) {
        mapRegistry = (MapRegistry) config.getServletContext().getAttribute("MAPREGISTRY");
        // TODO -- check if null and throw WMSServletException if so!
    }
    
    public void doGet(WMSHttpServletRequest req, WMSHttpServletResponse resp) throws IOException {
        try {
            WMSRequest wmsReq = WMSRequest.createWMSRequest(req);
            if (!wmsReq.isValid())
                throw new WMSServiceException("Invalid WMS Request: " + wmsReq.getParseReport());
            SysLog.inst().info(wmsReq.toString());  // log this *after* check for validity
            
            if (wmsReq instanceof WMSGetCapabilitiesRequest)
                serviceGetCapabilities( (WMSGetCapabilitiesRequest) wmsReq, resp);
            
            else if (wmsReq instanceof WMSGetMapRequest)
                serviceGetMapRequest( (WMSGetMapRequest) wmsReq, resp);
            
            else if (wmsReq instanceof WMSGetFeatureInfoRequest)
                serviceGetFeatureInfoRequest( (WMSGetFeatureInfoRequest) wmsReq, resp);

            else if (wmsReq instanceof WMSGetImageryListRequest)
                serviceGetImageryListRequest( (WMSGetImageryListRequest) wmsReq, resp);
            
            else
                throw new WMSServiceException("Unknown WMS Request");
        }
        catch (WMSServiceException ex) {
            // ill-formed or unconforming request...
            writeWMSServiceException(resp, ex.toString());
        }
        catch (Exception ex) {
            writeServletError(resp, ex);
        }
    }
    
    public void doPost(WMSHttpServletRequest req, WMSHttpServletResponse resp) throws IOException {
        doGet(req, resp);
    }
    
    private void serviceGetCapabilities(WMSGetCapabilitiesRequest req, WMSHttpServletResponse resp) 
        throws WMSServiceException, IOException 
    {
        String caps = getCapabilitiesString();
        byte[] bytes = caps.getBytes();
        resp.setContentLength(bytes.length);
        resp.setContentType("text/xml");
        resp.setStatus(WMSHttpServletResponse.OK);
        
        OutputStream os = resp.getOutputStream();
        os.write(bytes);
        resp.flushBuffer();
    }
    
    
    private void serviceGetMapRequest(WMSGetMapRequest req, WMSHttpServletResponse resp) 
        throws WMSServiceException, IOException
    {
        // do we know about this map layer?
        // TODO - for the moment ignore all but first one, until we've implemented map-compositing
        String[] layers = req.getLayers();
        MapSource map = mapRegistry.getNamedMapSource(layers[0]);
        if (map == null)
            throw new WMSServiceException("unsupported/unknown map layer: " + layers[0]);
        
        if (!ImageFormatter.isSupportedType(req.getFormat()))
            throw new WMSServiceException("unsupported format: " + req.getFormat());

        // create an instance of the MapGenerator.ServiceInstance and hand request to it...
        ImageFormatter image = null;
        MapGenerator.ServiceInstance mapService = null;
        try {
            MapGenerator mapGen = map.getMapGenerator();
            mapService = mapGen.getServiceInstance();
            image = mapService.serviceRequest(req);
        }
        catch (Exception ex) {
            throw new WMSServiceException("Failed to instance map-generator: " + ex.toString());
        }

        resp.setStatus(WMSHttpServletResponse.OK);
        resp.setContentType(req.getFormat());
        
        InputStream inp = encodePayload(image.getStreamFromMimeType(req.getFormat()), req, resp);
        resp.setContentLength(inp.available());
        OutputStream out = resp.getOutputStream();
        byte[] buff = new byte[4096];
        int len;
        while ((len=inp.read(buff)) > 0)
            out.write(buff, 0, len);
        
        resp.flushBuffer();
        inp.close();
        mapService.freeResources();
    }        
            
    private void serviceGetFeatureInfoRequest(WMSGetFeatureInfoRequest req, WMSHttpServletResponse resp) {
        // TODO
    }

    // An extended service added to support querying of RPF files.
    private void serviceGetImageryListRequest(WMSGetImageryListRequest req, WMSHttpServletResponse resp)
        throws WMSServiceException, IOException
    {
        // do we know about this map layer?
        // TODO - for the moment ignore all but first one, until we've implemented map-compositing
        String[] layers = req.getLayers();
        Iterator<String> maps = mapRegistry.getMapNames();
        if (maps == null)
            throw new WMSServiceException("No registered map sources!");

        StringBuffer xmlResp = new StringBuffer();
        xmlResp.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
        xmlResp.append("<ImageryList bbox=\"");
        xmlResp.append(Double.toString(req.getBBoxXMin()));  xmlResp.append(",");
        xmlResp.append(Double.toString(req.getBBoxYMin()));  xmlResp.append(",");
        xmlResp.append(Double.toString(req.getBBoxXMax()));  xmlResp.append(",");
        xmlResp.append(Double.toString(req.getBBoxYMax()));
        xmlResp.append("\">");

        while (maps.hasNext()) {
            try {
                MapSource mapSource = mapRegistry.getNamedMapSource(maps.next());
                String mapLayer = mapSource.getFullName();
                for (String layer : req.getLayers()) {
                    if (!"all".equalsIgnoreCase(layer) && !layer.equalsIgnoreCase(mapLayer)) continue;

                    MapGenerator mapGen = mapSource.getMapGenerator();
                    MapGenerator.ServiceInstance mapService = mapGen.getServiceInstance();
                    List<File> mapFiles = mapService.serviceRequest(req);
                    if (mapFiles == null) continue;

                    for (File f : mapFiles) {
                        xmlResp.append("<file path=\"");
                        xmlResp.append(f.getAbsolutePath());
                        xmlResp.append("\" layer=\"");
                        xmlResp.append(mapLayer);
                        xmlResp.append("\" filesize=\"");
                        xmlResp.append(Long.toString(f.length()));
                        xmlResp.append("\" />");
                    }
                }
            }
            catch (Exception ex) {
                throw new WMSServiceException("Failed to instance map-generator: " + ex.toString());
            }

        }

        xmlResp.append("</ImageryList>");
        byte[] bytes = xmlResp.toString().getBytes();
        resp.setStatus(WMSHttpServletResponse.OK);
        resp.setContentType("text/xml");
        resp.setContentLength(bytes.length);

        OutputStream os = resp.getOutputStream();
        os.write(bytes);
        resp.flushBuffer();
    }

    //
    //  Read in capabilities_template.xml
    //  replace <Insert></Insert> with information for the layers defined in config.xml
    //
    private String getCapabilitiesString() throws WMSServiceException {
        // Load capabilities file once, lazily...
        if (capabilitiesXML == null) {
            try {
                StringBuffer strBuff = new StringBuffer();
                InputStream is = this.getClass().getClassLoader().getResourceAsStream(CAPSFILE);
                if (is == null)
                    throw new WMSServiceException("Could not located Capabilites File: " + CAPSFILE);

                InputStreamReader inp = new InputStreamReader(is);
                char[] charBuff = new char[1024];
                int len;

                while ((len = inp.read(charBuff)) > 0)
                    strBuff.append(charBuff, 0, len);

                capabilitiesXML = strBuff.toString();

                //insert dynamic layers here
                StringBuffer sb=new StringBuffer();
                try{
                    Iterator<MapSource> iter = mapRegistry.getAllMapSources();
                    while(iter.hasNext()) {
                        MapSource source = iter.next();
                        sb.append(source.getCapsLayerXML());
                    }
                } catch(Exception ex)
                {
                    throw new WMSServiceException("Error retrieving layer information fulfilling getCapabilities request: " + ex.toString());                    
                }


                capabilitiesXML=capabilitiesXML.replaceFirst("<Insert></Insert>", sb.toString());
            } catch (IOException ex) {
                throw new WMSServiceException("Error loading capabilities file: " + ex.toString());
            }
        }

        return capabilitiesXML;
    }


    private void writeWMSServiceException(WMSHttpServletResponse resp, String msg) throws IOException {
        try {
            SysLog.inst().info("WMSException: " + msg);
            resp.setStatus(WMSHttpServletResponse.BAD_REQUEST);
            resp.setContentType("text/xml");
            
            // Create the WMS ServiceException msg...
            ObjectFactory of = new ObjectFactory();
            ServiceExceptionReport exr = of.createServiceExceptionReport();
            ServiceExceptionType ex = of.createServiceExceptionType();
            ex.setValue(msg);
            exr.getServiceException().add(ex);
            
            // Marshall the ServiceException msg out to the response...
            JAXBContext jaxbContext = JAXBContext.newInstance(getClass().getPackage().getName() + ".xml");
            Marshaller marshaller = jaxbContext.createMarshaller();
            marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, new Boolean(true));
            marshaller.marshal(exr, resp.getOutputStream());
        } catch (Exception ex) {
            writeServletError(resp, ex);
        }
    }

    private void writeServletError(WMSHttpServletResponse resp, Exception ex) throws IOException {
        String errMsg = ex.getMessage();
        SysLog.inst().error("ServletError: " + errMsg);
        SysLog.inst().stackTrace(ex);
        
        byte[] msgBytes = errMsg.getBytes();
        resp.setStatus(WMSHttpServletResponse.SERVER_ERROR);
        resp.setContentLength(msgBytes.length);
        resp.setContentType("text");
        
        OutputStream out = resp.getOutputStream();
        out.write(msgBytes);        
    }
    
    //
    // Returns an appropriate InputStream representing the response payload bytes.
    // Either returns the given InputStream or one that contains a compressed encoding
    // of the original.
    // As a heuristic, we don't bother to compress PNGs nor JPEGs, as they don't compress well, 
    // and not worth the computational expense.
    //
    private InputStream encodePayload(InputStream payload, WMSGetMapRequest req, WMSHttpServletResponse resp) 
        throws IOException
    {
        String reqFormat = req.getFormat();
        if (reqFormat.equalsIgnoreCase(ImageTypes.PNG.mimeType) || reqFormat.equalsIgnoreCase(ImageTypes.JPEG.mimeType))
            return payload;

        String hdr = req.getHttpRequest().getHeader(ACCEPT_ENCODING);
        if (hdr == null)
            return payload;
        
        // For now, we ignore the presence of any q-values; just pick whichever compression scheme we happen to 
        // find first...
        DeflaterOutputStream out = null;
        ByteArrayOutputStream bytes = null;
        if (hdr.contains(COMPRESSED_ENCODING)) {
            bytes = new ByteArrayOutputStream(50000);  // empirically determined size for typical DDS requests.
            out = new DeflaterOutputStream(bytes);
            resp.addHeader(CONTENT_ENCODING, COMPRESSED_ENCODING);
        } else if (hdr.contains(GZIPPED_ENCODING)) {
            bytes = new ByteArrayOutputStream(50000);
            out = new GZIPOutputStream(bytes);
            resp.addHeader(CONTENT_ENCODING, GZIPPED_ENCODING);
        }
        
        if (out == null)
            return payload;
        
        // otherwise, copy (filtered) bytes...
        byte[] buff = new byte[4096];
        int len;
        while ((len=payload.read(buff)) > 0)
            out.write(buff, 0, len);
        out.close();
        
        return new ByteArrayInputStream(bytes.toByteArray());
    }
    
    private MapRegistry mapRegistry;

    // @TODO:  This should eventually come from configuration.  Stub in a hand-jammed file for now.
    private static String capabilitiesXML = null;
    private static final String CAPSFILE = "./WEB-INF/capabilities_template.xml";
    private static final String ACCEPT_ENCODING = "Accept-Encoding";
    private static final String CONTENT_ENCODING = "Content-Encoding";
    private static final String COMPRESSED_ENCODING = "compress";
    private static final String GZIPPED_ENCODING = "gzip";
}
