/*
 *  Bitwise operators (E5 Sections 11.4.8, 11.7.1, 11.7.2, 11.7.3,
 *  11.10).
 */

/* XXX: add more coercion tests */

function hex(v) {
    return v.toString(16);
}

function printHex(v) {
    print(hex(v));
}

function valWithSideEffect(v) {
    return {
        valueOf: function () { print('coerced:', hex(v)); return v; }
    };
}

/*===
minimal and-or-xor
64
192
191
===*/

function minimalAndOrXorTest() {
    print(255 & 64);
    print(128 | 64);
    print(255 ^ 64);
}

try {
    print('minimal and-or-xor');
    minimalAndOrXorTest();
} catch (e) {
    print(e.stack || e);
}

/*===
minimal not
-124
123
-1410065409
1410065408
===*/

function minimalNotTest() {
    print(~123);
    print(~~123);
    print(~1e10);
    print(~~1e10);
}

try {
    print('minimal not');
    minimalNotTest();
} catch (e) {
    print(e.stack || e);
}

/*===
precedence and-or-xor
- XOR binds more strongly than OR
5678
0
5678
0
- AND binds more strongly than XOR
0
5678
5678
0
- AND binds more strongly than OR
-6dcc0000
12340000
-6dcc0000
12340000
- Bitwise NOT bings more strongly than AND
-12350000
-12340001
- Bitwise NOT bings more strongly than XOR
1234a987
1234a987
- Bitwise NOT bings more strongly than OR
-5679
a986
===*/

function precedenceAndOrXorTest() {
    print('- XOR binds more strongly than OR');
    printHex(0x12345678 ^ 0x12340000 | 0x00005678);
    printHex(0x12345678 ^ (0x12340000 | 0x00005678));
    printHex(0x00005678 | 0x12345678 ^ 0x12345678);
    printHex((0x00005678 | 0x12345678) ^ 0x12345678);

    print('- AND binds more strongly than XOR');
    printHex((0x12345678 ^ 0x12345678) & 0xffff0000);
    printHex(0x12345678 ^ 0x12345678 & 0xffff0000);
    printHex(0xffff0000 & 0x12345678 ^ 0x12345678);
    printHex(0xffff0000 & (0x12345678 ^ 0x12345678));

    print('- AND binds more strongly than OR');
    printHex(0x12345678 & 0xffff0000 | 0x80000000);
    printHex(0x12345678 & (0xffff0000 | 0x80000000));
    printHex(0x80000000 | 0xffff0000 & 0x12345678);
    printHex((0x80000000 | 0xffff0000) & 0x12345678);

    print('- Bitwise NOT bings more strongly than AND');
    printHex(~0x12345678 & 0xffff0000);
    printHex(~(0x12345678 & 0xffff0000));

    print('- Bitwise NOT bings more strongly than XOR');
    printHex(~0x12345678 ^ 0xffff0000);
    printHex(~(0x12345678 ^ 0xffff0000));

    print('- Bitwise NOT bings more strongly than OR');
    printHex(~0x12345678 | 0xffff0000);
    printHex(~(0x12345679 | 0xffff0000));
}

try {
    print('precedence and-or-xor');
    precedenceAndOrXorTest();
} catch (e) {
    print(e.stack || e);
}

/*===
associativity and-or-xor
- Bitwise OR
coerced: 12345678
coerced: 23456789
coerced: 3456789a
37777ffb
coerced: 23456789
coerced: 3456789a
coerced: 12345678
37777ffb
- Bitwise XOR
coerced: 12345678
coerced: 23456789
coerced: 3456789a
527496b
coerced: 23456789
coerced: 3456789a
coerced: 12345678
527496b
- Bitwise AND
coerced: 12345678
coerced: 23456789
coerced: 3456789a
44008
coerced: 23456789
coerced: 3456789a
coerced: 12345678
44008
===*/

function associativityAndOrXorTest() {
    /* Associativity is not always easy to see, e.g. bitwise OR is left
     * associative but the result would be the same.  Using coercion side
     * effects is not entirely reliable either because implementations
     * differ in their evaluation order.
     *
     * For instance, if evaluating val(A) | (val(B) | val(C)), the coercion
     * order is different on various engines:
     *
     *    Duktape 0.12.0: B, C, A
     *    V8: B, C, A
     *    Rhino: A, B, C
     *
     * The current code here relies on Duktape evaluation order.  This order
     * may not actually be fully compliant: E5.1 Section 11.10 algorithm
     * states that left side (val(A)) should be evaluated first.
     */

    var val = valWithSideEffect;

    print('- Bitwise OR');
    printHex(val(0x12345678) | val(0x23456789) | val(0x3456789a));
    printHex(val(0x12345678) | (val(0x23456789) | val(0x3456789a)));

    print('- Bitwise XOR');
    printHex(val(0x12345678) ^ val(0x23456789) ^ val(0x3456789a));
    printHex(val(0x12345678) ^ (val(0x23456789) ^ val(0x3456789a)));

    print('- Bitwise AND');
    printHex(val(0x12345678) & val(0x23456789) & val(0x3456789a));
    printHex(val(0x12345678) & (val(0x23456789) & val(0x3456789a)));
}

try {
    print('associativity and-or-xor');
    associativityAndOrXorTest();
} catch (e) {
    print(e.stack || e);
}

/*===
target modify not
-124
123
-2
1
===*/

function targetModifyNotTest() {
    /* Ensure that target is not modified (this would happen if expression is
     * not allocated a real temporary register but operates directly on a register
     * bound value).
     */

    var x = 123;
    print(~x);
    print(x);

    var obj = {x:1};
    print(~obj.x);
    print(obj.x);
}

try {
    print('target modify not');
    targetModifyNotTest();
} catch (e) {
    print(e.stack || e);
}

/*===
matrix and-or-xor-not
0 0 -Infinity -Infinity 0 0 0
0 1 -Infinity -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0
0 2 -Infinity -123456789ab 0 -456789ab -456789ab
0 3 -Infinity -ffffffff 0 1 1
0 4 -Infinity -80000000 0 -80000000 -80000000
0 5 -Infinity -7fffffff 0 -7fffffff -7fffffff
0 6 -Infinity -12345678.c 0 -12345678 -12345678
0 7 -Infinity -12345678.8 0 -12345678 -12345678
0 8 -Infinity -12345678.4 0 -12345678 -12345678
0 9 -Infinity -12345678 0 -12345678 -12345678
0 10 -Infinity -1 0 -1 -1
0 11 -Infinity 0 0 0 0
0 12 -Infinity 0 0 0 0
0 13 -Infinity 1 0 1 1
0 14 -Infinity 12345678 0 12345678 12345678
0 15 -Infinity 12345678.4 0 12345678 12345678
0 16 -Infinity 12345678.8 0 12345678 12345678
0 17 -Infinity 12345678.c 0 12345678 12345678
0 18 -Infinity 7fffffff 0 7fffffff 7fffffff
0 19 -Infinity 80000000 0 -80000000 -80000000
0 20 -Infinity ffffffff 0 -1 -1
0 21 -Infinity 123456789ab 0 456789ab 456789ab
0 22 -Infinity 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0
0 23 -Infinity Infinity 0 0 0
0 24 -Infinity NaN 0 0 0
1 0 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -Infinity 0 0 0
1 1 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0
1 2 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -123456789ab 0 -456789ab -456789ab
1 3 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -ffffffff 0 1 1
1 4 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -80000000 0 -80000000 -80000000
1 5 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -7fffffff 0 -7fffffff -7fffffff
1 6 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -12345678.c 0 -12345678 -12345678
1 7 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -12345678.8 0 -12345678 -12345678
1 8 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -12345678.4 0 -12345678 -12345678
1 9 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -12345678 0 -12345678 -12345678
1 10 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -1 0 -1 -1
1 11 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0 0
1 12 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0 0
1 13 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 1 0 1 1
1 14 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 12345678 0 12345678 12345678
1 15 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 12345678.4 0 12345678 12345678
1 16 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 12345678.8 0 12345678 12345678
1 17 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 12345678.c 0 12345678 12345678
1 18 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 7fffffff 0 7fffffff 7fffffff
1 19 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 80000000 0 -80000000 -80000000
1 20 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 ffffffff 0 -1 -1
1 21 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 123456789ab 0 456789ab 456789ab
1 22 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0
1 23 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 Infinity 0 0 0
1 24 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 NaN 0 0 0
2 0 -123456789ab -Infinity 0 -456789ab -456789ab
2 1 -123456789ab -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -456789ab -456789ab
2 2 -123456789ab -123456789ab -456789ab -456789ab 0
2 3 -123456789ab -ffffffff 1 -456789ab -456789ac
2 4 -123456789ab -80000000 -80000000 -456789ab 3a987655
2 5 -123456789ab -7fffffff -7fffffff -456789ab 3a987654
2 6 -123456789ab -12345678.c -5777e000 -240023 5753dfdd
2 7 -123456789ab -12345678.8 -5777e000 -240023 5753dfdd
2 8 -123456789ab -12345678.4 -5777e000 -240023 5753dfdd
2 9 -123456789ab -12345678 -5777e000 -240023 5753dfdd
2 10 -123456789ab -1 -456789ab -1 456789aa
2 11 -123456789ab 0 0 -456789ab -456789ab
2 12 -123456789ab 0 0 -456789ab -456789ab
2 13 -123456789ab 1 1 -456789ab -456789ac
2 14 -123456789ab 12345678 12105650 -45438983 -5753dfd3
2 15 -123456789ab 12345678.4 12105650 -45438983 -5753dfd3
2 16 -123456789ab 12345678.8 12105650 -45438983 -5753dfd3
2 17 -123456789ab 12345678.c 12105650 -45438983 -5753dfd3
2 18 -123456789ab 7fffffff 3a987655 -1 -3a987656
2 19 -123456789ab 80000000 -80000000 -456789ab 3a987655
2 20 -123456789ab ffffffff -456789ab -1 456789aa
2 21 -123456789ab 123456789ab 1 -1 -2
2 22 -123456789ab 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -456789ab -456789ab
2 23 -123456789ab Infinity 0 -456789ab -456789ab
2 24 -123456789ab NaN 0 -456789ab -456789ab
3 0 -ffffffff -Infinity 0 1 1
3 1 -ffffffff -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 1 1
3 2 -ffffffff -123456789ab 1 -456789ab -456789ac
3 3 -ffffffff -ffffffff 1 1 0
3 4 -ffffffff -80000000 0 -7fffffff -7fffffff
3 5 -ffffffff -7fffffff 1 -7fffffff -80000000
3 6 -ffffffff -12345678.c 0 -12345677 -12345677
3 7 -ffffffff -12345678.8 0 -12345677 -12345677
3 8 -ffffffff -12345678.4 0 -12345677 -12345677
3 9 -ffffffff -12345678 0 -12345677 -12345677
3 10 -ffffffff -1 1 -1 -2
3 11 -ffffffff 0 0 1 1
3 12 -ffffffff 0 0 1 1
3 13 -ffffffff 1 1 1 0
3 14 -ffffffff 12345678 0 12345679 12345679
3 15 -ffffffff 12345678.4 0 12345679 12345679
3 16 -ffffffff 12345678.8 0 12345679 12345679
3 17 -ffffffff 12345678.c 0 12345679 12345679
3 18 -ffffffff 7fffffff 1 7fffffff 7ffffffe
3 19 -ffffffff 80000000 0 -7fffffff -7fffffff
3 20 -ffffffff ffffffff 1 -1 -2
3 21 -ffffffff 123456789ab 1 456789ab 456789aa
3 22 -ffffffff 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 1 1
3 23 -ffffffff Infinity 0 1 1
3 24 -ffffffff NaN 0 1 1
4 0 -80000000 -Infinity 0 -80000000 -80000000
4 1 -80000000 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -80000000 -80000000
4 2 -80000000 -123456789ab -80000000 -456789ab 3a987655
4 3 -80000000 -ffffffff 0 -7fffffff -7fffffff
4 4 -80000000 -80000000 -80000000 -80000000 0
4 5 -80000000 -7fffffff -80000000 -7fffffff 1
4 6 -80000000 -12345678.c -80000000 -12345678 6dcba988
4 7 -80000000 -12345678.8 -80000000 -12345678 6dcba988
4 8 -80000000 -12345678.4 -80000000 -12345678 6dcba988
4 9 -80000000 -12345678 -80000000 -12345678 6dcba988
4 10 -80000000 -1 -80000000 -1 7fffffff
4 11 -80000000 0 0 -80000000 -80000000
4 12 -80000000 0 0 -80000000 -80000000
4 13 -80000000 1 0 -7fffffff -7fffffff
4 14 -80000000 12345678 0 -6dcba988 -6dcba988
4 15 -80000000 12345678.4 0 -6dcba988 -6dcba988
4 16 -80000000 12345678.8 0 -6dcba988 -6dcba988
4 17 -80000000 12345678.c 0 -6dcba988 -6dcba988
4 18 -80000000 7fffffff 0 -1 -1
4 19 -80000000 80000000 -80000000 -80000000 0
4 20 -80000000 ffffffff -80000000 -1 7fffffff
4 21 -80000000 123456789ab 0 -3a987655 -3a987655
4 22 -80000000 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -80000000 -80000000
4 23 -80000000 Infinity 0 -80000000 -80000000
4 24 -80000000 NaN 0 -80000000 -80000000
5 0 -7fffffff -Infinity 0 -7fffffff -7fffffff
5 1 -7fffffff -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -7fffffff -7fffffff
5 2 -7fffffff -123456789ab -7fffffff -456789ab 3a987654
5 3 -7fffffff -ffffffff 1 -7fffffff -80000000
5 4 -7fffffff -80000000 -80000000 -7fffffff 1
5 5 -7fffffff -7fffffff -7fffffff -7fffffff 0
5 6 -7fffffff -12345678.c -80000000 -12345677 6dcba989
5 7 -7fffffff -12345678.8 -80000000 -12345677 6dcba989
5 8 -7fffffff -12345678.4 -80000000 -12345677 6dcba989
5 9 -7fffffff -12345678 -80000000 -12345677 6dcba989
5 10 -7fffffff -1 -7fffffff -1 7ffffffe
5 11 -7fffffff 0 0 -7fffffff -7fffffff
5 12 -7fffffff 0 0 -7fffffff -7fffffff
5 13 -7fffffff 1 1 -7fffffff -80000000
5 14 -7fffffff 12345678 0 -6dcba987 -6dcba987
5 15 -7fffffff 12345678.4 0 -6dcba987 -6dcba987
5 16 -7fffffff 12345678.8 0 -6dcba987 -6dcba987
5 17 -7fffffff 12345678.c 0 -6dcba987 -6dcba987
5 18 -7fffffff 7fffffff 1 -1 -2
5 19 -7fffffff 80000000 -80000000 -7fffffff 1
5 20 -7fffffff ffffffff -7fffffff -1 7ffffffe
5 21 -7fffffff 123456789ab 1 -3a987655 -3a987656
5 22 -7fffffff 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -7fffffff -7fffffff
5 23 -7fffffff Infinity 0 -7fffffff -7fffffff
5 24 -7fffffff NaN 0 -7fffffff -7fffffff
6 0 -12345678.c -Infinity 0 -12345678 -12345678
6 1 -12345678.c -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -12345678 -12345678
6 2 -12345678.c -123456789ab -5777e000 -240023 5753dfdd
6 3 -12345678.c -ffffffff 0 -12345677 -12345677
6 4 -12345678.c -80000000 -80000000 -12345678 6dcba988
6 5 -12345678.c -7fffffff -80000000 -12345677 6dcba989
6 6 -12345678.c -12345678.c -12345678 -12345678 0
6 7 -12345678.c -12345678.8 -12345678 -12345678 0
6 8 -12345678.c -12345678.4 -12345678 -12345678 0
6 9 -12345678.c -12345678 -12345678 -12345678 0
6 10 -12345678.c -1 -12345678 -1 12345677
6 11 -12345678.c 0 0 -12345678 -12345678
6 12 -12345678.c 0 0 -12345678 -12345678
6 13 -12345678.c 1 0 -12345677 -12345677
6 14 -12345678.c 12345678 8 -8 -10
6 15 -12345678.c 12345678.4 8 -8 -10
6 16 -12345678.c 12345678.8 8 -8 -10
6 17 -12345678.c 12345678.c 8 -8 -10
6 18 -12345678.c 7fffffff 6dcba988 -1 -6dcba989
6 19 -12345678.c 80000000 -80000000 -12345678 6dcba988
6 20 -12345678.c ffffffff -12345678 -1 12345677
6 21 -12345678.c 123456789ab 45438988 -12105655 -5753dfdd
6 22 -12345678.c 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -12345678 -12345678
6 23 -12345678.c Infinity 0 -12345678 -12345678
6 24 -12345678.c NaN 0 -12345678 -12345678
7 0 -12345678.8 -Infinity 0 -12345678 -12345678
7 1 -12345678.8 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -12345678 -12345678
7 2 -12345678.8 -123456789ab -5777e000 -240023 5753dfdd
7 3 -12345678.8 -ffffffff 0 -12345677 -12345677
7 4 -12345678.8 -80000000 -80000000 -12345678 6dcba988
7 5 -12345678.8 -7fffffff -80000000 -12345677 6dcba989
7 6 -12345678.8 -12345678.c -12345678 -12345678 0
7 7 -12345678.8 -12345678.8 -12345678 -12345678 0
7 8 -12345678.8 -12345678.4 -12345678 -12345678 0
7 9 -12345678.8 -12345678 -12345678 -12345678 0
7 10 -12345678.8 -1 -12345678 -1 12345677
7 11 -12345678.8 0 0 -12345678 -12345678
7 12 -12345678.8 0 0 -12345678 -12345678
7 13 -12345678.8 1 0 -12345677 -12345677
7 14 -12345678.8 12345678 8 -8 -10
7 15 -12345678.8 12345678.4 8 -8 -10
7 16 -12345678.8 12345678.8 8 -8 -10
7 17 -12345678.8 12345678.c 8 -8 -10
7 18 -12345678.8 7fffffff 6dcba988 -1 -6dcba989
7 19 -12345678.8 80000000 -80000000 -12345678 6dcba988
7 20 -12345678.8 ffffffff -12345678 -1 12345677
7 21 -12345678.8 123456789ab 45438988 -12105655 -5753dfdd
7 22 -12345678.8 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -12345678 -12345678
7 23 -12345678.8 Infinity 0 -12345678 -12345678
7 24 -12345678.8 NaN 0 -12345678 -12345678
8 0 -12345678.4 -Infinity 0 -12345678 -12345678
8 1 -12345678.4 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -12345678 -12345678
8 2 -12345678.4 -123456789ab -5777e000 -240023 5753dfdd
8 3 -12345678.4 -ffffffff 0 -12345677 -12345677
8 4 -12345678.4 -80000000 -80000000 -12345678 6dcba988
8 5 -12345678.4 -7fffffff -80000000 -12345677 6dcba989
8 6 -12345678.4 -12345678.c -12345678 -12345678 0
8 7 -12345678.4 -12345678.8 -12345678 -12345678 0
8 8 -12345678.4 -12345678.4 -12345678 -12345678 0
8 9 -12345678.4 -12345678 -12345678 -12345678 0
8 10 -12345678.4 -1 -12345678 -1 12345677
8 11 -12345678.4 0 0 -12345678 -12345678
8 12 -12345678.4 0 0 -12345678 -12345678
8 13 -12345678.4 1 0 -12345677 -12345677
8 14 -12345678.4 12345678 8 -8 -10
8 15 -12345678.4 12345678.4 8 -8 -10
8 16 -12345678.4 12345678.8 8 -8 -10
8 17 -12345678.4 12345678.c 8 -8 -10
8 18 -12345678.4 7fffffff 6dcba988 -1 -6dcba989
8 19 -12345678.4 80000000 -80000000 -12345678 6dcba988
8 20 -12345678.4 ffffffff -12345678 -1 12345677
8 21 -12345678.4 123456789ab 45438988 -12105655 -5753dfdd
8 22 -12345678.4 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -12345678 -12345678
8 23 -12345678.4 Infinity 0 -12345678 -12345678
8 24 -12345678.4 NaN 0 -12345678 -12345678
9 0 -12345678 -Infinity 0 -12345678 -12345678
9 1 -12345678 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -12345678 -12345678
9 2 -12345678 -123456789ab -5777e000 -240023 5753dfdd
9 3 -12345678 -ffffffff 0 -12345677 -12345677
9 4 -12345678 -80000000 -80000000 -12345678 6dcba988
9 5 -12345678 -7fffffff -80000000 -12345677 6dcba989
9 6 -12345678 -12345678.c -12345678 -12345678 0
9 7 -12345678 -12345678.8 -12345678 -12345678 0
9 8 -12345678 -12345678.4 -12345678 -12345678 0
9 9 -12345678 -12345678 -12345678 -12345678 0
9 10 -12345678 -1 -12345678 -1 12345677
9 11 -12345678 0 0 -12345678 -12345678
9 12 -12345678 0 0 -12345678 -12345678
9 13 -12345678 1 0 -12345677 -12345677
9 14 -12345678 12345678 8 -8 -10
9 15 -12345678 12345678.4 8 -8 -10
9 16 -12345678 12345678.8 8 -8 -10
9 17 -12345678 12345678.c 8 -8 -10
9 18 -12345678 7fffffff 6dcba988 -1 -6dcba989
9 19 -12345678 80000000 -80000000 -12345678 6dcba988
9 20 -12345678 ffffffff -12345678 -1 12345677
9 21 -12345678 123456789ab 45438988 -12105655 -5753dfdd
9 22 -12345678 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -12345678 -12345678
9 23 -12345678 Infinity 0 -12345678 -12345678
9 24 -12345678 NaN 0 -12345678 -12345678
10 0 -1 -Infinity 0 -1 -1
10 1 -1 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -1 -1
10 2 -1 -123456789ab -456789ab -1 456789aa
10 3 -1 -ffffffff 1 -1 -2
10 4 -1 -80000000 -80000000 -1 7fffffff
10 5 -1 -7fffffff -7fffffff -1 7ffffffe
10 6 -1 -12345678.c -12345678 -1 12345677
10 7 -1 -12345678.8 -12345678 -1 12345677
10 8 -1 -12345678.4 -12345678 -1 12345677
10 9 -1 -12345678 -12345678 -1 12345677
10 10 -1 -1 -1 -1 0
10 11 -1 0 0 -1 -1
10 12 -1 0 0 -1 -1
10 13 -1 1 1 -1 -2
10 14 -1 12345678 12345678 -1 -12345679
10 15 -1 12345678.4 12345678 -1 -12345679
10 16 -1 12345678.8 12345678 -1 -12345679
10 17 -1 12345678.c 12345678 -1 -12345679
10 18 -1 7fffffff 7fffffff -1 -80000000
10 19 -1 80000000 -80000000 -1 7fffffff
10 20 -1 ffffffff -1 -1 0
10 21 -1 123456789ab 456789ab -1 -456789ac
10 22 -1 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -1 -1
10 23 -1 Infinity 0 -1 -1
10 24 -1 NaN 0 -1 -1
11 0 0 -Infinity 0 0 0
11 1 0 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0
11 2 0 -123456789ab 0 -456789ab -456789ab
11 3 0 -ffffffff 0 1 1
11 4 0 -80000000 0 -80000000 -80000000
11 5 0 -7fffffff 0 -7fffffff -7fffffff
11 6 0 -12345678.c 0 -12345678 -12345678
11 7 0 -12345678.8 0 -12345678 -12345678
11 8 0 -12345678.4 0 -12345678 -12345678
11 9 0 -12345678 0 -12345678 -12345678
11 10 0 -1 0 -1 -1
11 11 0 0 0 0 0
11 12 0 0 0 0 0
11 13 0 1 0 1 1
11 14 0 12345678 0 12345678 12345678
11 15 0 12345678.4 0 12345678 12345678
11 16 0 12345678.8 0 12345678 12345678
11 17 0 12345678.c 0 12345678 12345678
11 18 0 7fffffff 0 7fffffff 7fffffff
11 19 0 80000000 0 -80000000 -80000000
11 20 0 ffffffff 0 -1 -1
11 21 0 123456789ab 0 456789ab 456789ab
11 22 0 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0
11 23 0 Infinity 0 0 0
11 24 0 NaN 0 0 0
12 0 0 -Infinity 0 0 0
12 1 0 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0
12 2 0 -123456789ab 0 -456789ab -456789ab
12 3 0 -ffffffff 0 1 1
12 4 0 -80000000 0 -80000000 -80000000
12 5 0 -7fffffff 0 -7fffffff -7fffffff
12 6 0 -12345678.c 0 -12345678 -12345678
12 7 0 -12345678.8 0 -12345678 -12345678
12 8 0 -12345678.4 0 -12345678 -12345678
12 9 0 -12345678 0 -12345678 -12345678
12 10 0 -1 0 -1 -1
12 11 0 0 0 0 0
12 12 0 0 0 0 0
12 13 0 1 0 1 1
12 14 0 12345678 0 12345678 12345678
12 15 0 12345678.4 0 12345678 12345678
12 16 0 12345678.8 0 12345678 12345678
12 17 0 12345678.c 0 12345678 12345678
12 18 0 7fffffff 0 7fffffff 7fffffff
12 19 0 80000000 0 -80000000 -80000000
12 20 0 ffffffff 0 -1 -1
12 21 0 123456789ab 0 456789ab 456789ab
12 22 0 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0
12 23 0 Infinity 0 0 0
12 24 0 NaN 0 0 0
13 0 1 -Infinity 0 1 1
13 1 1 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 1 1
13 2 1 -123456789ab 1 -456789ab -456789ac
13 3 1 -ffffffff 1 1 0
13 4 1 -80000000 0 -7fffffff -7fffffff
13 5 1 -7fffffff 1 -7fffffff -80000000
13 6 1 -12345678.c 0 -12345677 -12345677
13 7 1 -12345678.8 0 -12345677 -12345677
13 8 1 -12345678.4 0 -12345677 -12345677
13 9 1 -12345678 0 -12345677 -12345677
13 10 1 -1 1 -1 -2
13 11 1 0 0 1 1
13 12 1 0 0 1 1
13 13 1 1 1 1 0
13 14 1 12345678 0 12345679 12345679
13 15 1 12345678.4 0 12345679 12345679
13 16 1 12345678.8 0 12345679 12345679
13 17 1 12345678.c 0 12345679 12345679
13 18 1 7fffffff 1 7fffffff 7ffffffe
13 19 1 80000000 0 -7fffffff -7fffffff
13 20 1 ffffffff 1 -1 -2
13 21 1 123456789ab 1 456789ab 456789aa
13 22 1 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 1 1
13 23 1 Infinity 0 1 1
13 24 1 NaN 0 1 1
14 0 12345678 -Infinity 0 12345678 12345678
14 1 12345678 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 12345678 12345678
14 2 12345678 -123456789ab 12105650 -45438983 -5753dfd3
14 3 12345678 -ffffffff 0 12345679 12345679
14 4 12345678 -80000000 0 -6dcba988 -6dcba988
14 5 12345678 -7fffffff 0 -6dcba987 -6dcba987
14 6 12345678 -12345678.c 8 -8 -10
14 7 12345678 -12345678.8 8 -8 -10
14 8 12345678 -12345678.4 8 -8 -10
14 9 12345678 -12345678 8 -8 -10
14 10 12345678 -1 12345678 -1 -12345679
14 11 12345678 0 0 12345678 12345678
14 12 12345678 0 0 12345678 12345678
14 13 12345678 1 0 12345679 12345679
14 14 12345678 12345678 12345678 12345678 0
14 15 12345678 12345678.4 12345678 12345678 0
14 16 12345678 12345678.8 12345678 12345678 0
14 17 12345678 12345678.c 12345678 12345678 0
14 18 12345678 7fffffff 12345678 7fffffff 6dcba987
14 19 12345678 80000000 0 -6dcba988 -6dcba988
14 20 12345678 ffffffff 12345678 -1 -12345679
14 21 12345678 123456789ab 240028 5777dffb 5753dfd3
14 22 12345678 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 12345678 12345678
14 23 12345678 Infinity 0 12345678 12345678
14 24 12345678 NaN 0 12345678 12345678
15 0 12345678.4 -Infinity 0 12345678 12345678
15 1 12345678.4 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 12345678 12345678
15 2 12345678.4 -123456789ab 12105650 -45438983 -5753dfd3
15 3 12345678.4 -ffffffff 0 12345679 12345679
15 4 12345678.4 -80000000 0 -6dcba988 -6dcba988
15 5 12345678.4 -7fffffff 0 -6dcba987 -6dcba987
15 6 12345678.4 -12345678.c 8 -8 -10
15 7 12345678.4 -12345678.8 8 -8 -10
15 8 12345678.4 -12345678.4 8 -8 -10
15 9 12345678.4 -12345678 8 -8 -10
15 10 12345678.4 -1 12345678 -1 -12345679
15 11 12345678.4 0 0 12345678 12345678
15 12 12345678.4 0 0 12345678 12345678
15 13 12345678.4 1 0 12345679 12345679
15 14 12345678.4 12345678 12345678 12345678 0
15 15 12345678.4 12345678.4 12345678 12345678 0
15 16 12345678.4 12345678.8 12345678 12345678 0
15 17 12345678.4 12345678.c 12345678 12345678 0
15 18 12345678.4 7fffffff 12345678 7fffffff 6dcba987
15 19 12345678.4 80000000 0 -6dcba988 -6dcba988
15 20 12345678.4 ffffffff 12345678 -1 -12345679
15 21 12345678.4 123456789ab 240028 5777dffb 5753dfd3
15 22 12345678.4 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 12345678 12345678
15 23 12345678.4 Infinity 0 12345678 12345678
15 24 12345678.4 NaN 0 12345678 12345678
16 0 12345678.8 -Infinity 0 12345678 12345678
16 1 12345678.8 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 12345678 12345678
16 2 12345678.8 -123456789ab 12105650 -45438983 -5753dfd3
16 3 12345678.8 -ffffffff 0 12345679 12345679
16 4 12345678.8 -80000000 0 -6dcba988 -6dcba988
16 5 12345678.8 -7fffffff 0 -6dcba987 -6dcba987
16 6 12345678.8 -12345678.c 8 -8 -10
16 7 12345678.8 -12345678.8 8 -8 -10
16 8 12345678.8 -12345678.4 8 -8 -10
16 9 12345678.8 -12345678 8 -8 -10
16 10 12345678.8 -1 12345678 -1 -12345679
16 11 12345678.8 0 0 12345678 12345678
16 12 12345678.8 0 0 12345678 12345678
16 13 12345678.8 1 0 12345679 12345679
16 14 12345678.8 12345678 12345678 12345678 0
16 15 12345678.8 12345678.4 12345678 12345678 0
16 16 12345678.8 12345678.8 12345678 12345678 0
16 17 12345678.8 12345678.c 12345678 12345678 0
16 18 12345678.8 7fffffff 12345678 7fffffff 6dcba987
16 19 12345678.8 80000000 0 -6dcba988 -6dcba988
16 20 12345678.8 ffffffff 12345678 -1 -12345679
16 21 12345678.8 123456789ab 240028 5777dffb 5753dfd3
16 22 12345678.8 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 12345678 12345678
16 23 12345678.8 Infinity 0 12345678 12345678
16 24 12345678.8 NaN 0 12345678 12345678
17 0 12345678.c -Infinity 0 12345678 12345678
17 1 12345678.c -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 12345678 12345678
17 2 12345678.c -123456789ab 12105650 -45438983 -5753dfd3
17 3 12345678.c -ffffffff 0 12345679 12345679
17 4 12345678.c -80000000 0 -6dcba988 -6dcba988
17 5 12345678.c -7fffffff 0 -6dcba987 -6dcba987
17 6 12345678.c -12345678.c 8 -8 -10
17 7 12345678.c -12345678.8 8 -8 -10
17 8 12345678.c -12345678.4 8 -8 -10
17 9 12345678.c -12345678 8 -8 -10
17 10 12345678.c -1 12345678 -1 -12345679
17 11 12345678.c 0 0 12345678 12345678
17 12 12345678.c 0 0 12345678 12345678
17 13 12345678.c 1 0 12345679 12345679
17 14 12345678.c 12345678 12345678 12345678 0
17 15 12345678.c 12345678.4 12345678 12345678 0
17 16 12345678.c 12345678.8 12345678 12345678 0
17 17 12345678.c 12345678.c 12345678 12345678 0
17 18 12345678.c 7fffffff 12345678 7fffffff 6dcba987
17 19 12345678.c 80000000 0 -6dcba988 -6dcba988
17 20 12345678.c ffffffff 12345678 -1 -12345679
17 21 12345678.c 123456789ab 240028 5777dffb 5753dfd3
17 22 12345678.c 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 12345678 12345678
17 23 12345678.c Infinity 0 12345678 12345678
17 24 12345678.c NaN 0 12345678 12345678
18 0 7fffffff -Infinity 0 7fffffff 7fffffff
18 1 7fffffff -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 7fffffff 7fffffff
18 2 7fffffff -123456789ab 3a987655 -1 -3a987656
18 3 7fffffff -ffffffff 1 7fffffff 7ffffffe
18 4 7fffffff -80000000 0 -1 -1
18 5 7fffffff -7fffffff 1 -1 -2
18 6 7fffffff -12345678.c 6dcba988 -1 -6dcba989
18 7 7fffffff -12345678.8 6dcba988 -1 -6dcba989
18 8 7fffffff -12345678.4 6dcba988 -1 -6dcba989
18 9 7fffffff -12345678 6dcba988 -1 -6dcba989
18 10 7fffffff -1 7fffffff -1 -80000000
18 11 7fffffff 0 0 7fffffff 7fffffff
18 12 7fffffff 0 0 7fffffff 7fffffff
18 13 7fffffff 1 1 7fffffff 7ffffffe
18 14 7fffffff 12345678 12345678 7fffffff 6dcba987
18 15 7fffffff 12345678.4 12345678 7fffffff 6dcba987
18 16 7fffffff 12345678.8 12345678 7fffffff 6dcba987
18 17 7fffffff 12345678.c 12345678 7fffffff 6dcba987
18 18 7fffffff 7fffffff 7fffffff 7fffffff 0
18 19 7fffffff 80000000 0 -1 -1
18 20 7fffffff ffffffff 7fffffff -1 -80000000
18 21 7fffffff 123456789ab 456789ab 7fffffff 3a987654
18 22 7fffffff 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 7fffffff 7fffffff
18 23 7fffffff Infinity 0 7fffffff 7fffffff
18 24 7fffffff NaN 0 7fffffff 7fffffff
19 0 80000000 -Infinity 0 -80000000 -80000000
19 1 80000000 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -80000000 -80000000
19 2 80000000 -123456789ab -80000000 -456789ab 3a987655
19 3 80000000 -ffffffff 0 -7fffffff -7fffffff
19 4 80000000 -80000000 -80000000 -80000000 0
19 5 80000000 -7fffffff -80000000 -7fffffff 1
19 6 80000000 -12345678.c -80000000 -12345678 6dcba988
19 7 80000000 -12345678.8 -80000000 -12345678 6dcba988
19 8 80000000 -12345678.4 -80000000 -12345678 6dcba988
19 9 80000000 -12345678 -80000000 -12345678 6dcba988
19 10 80000000 -1 -80000000 -1 7fffffff
19 11 80000000 0 0 -80000000 -80000000
19 12 80000000 0 0 -80000000 -80000000
19 13 80000000 1 0 -7fffffff -7fffffff
19 14 80000000 12345678 0 -6dcba988 -6dcba988
19 15 80000000 12345678.4 0 -6dcba988 -6dcba988
19 16 80000000 12345678.8 0 -6dcba988 -6dcba988
19 17 80000000 12345678.c 0 -6dcba988 -6dcba988
19 18 80000000 7fffffff 0 -1 -1
19 19 80000000 80000000 -80000000 -80000000 0
19 20 80000000 ffffffff -80000000 -1 7fffffff
19 21 80000000 123456789ab 0 -3a987655 -3a987655
19 22 80000000 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -80000000 -80000000
19 23 80000000 Infinity 0 -80000000 -80000000
19 24 80000000 NaN 0 -80000000 -80000000
20 0 ffffffff -Infinity 0 -1 -1
20 1 ffffffff -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -1 -1
20 2 ffffffff -123456789ab -456789ab -1 456789aa
20 3 ffffffff -ffffffff 1 -1 -2
20 4 ffffffff -80000000 -80000000 -1 7fffffff
20 5 ffffffff -7fffffff -7fffffff -1 7ffffffe
20 6 ffffffff -12345678.c -12345678 -1 12345677
20 7 ffffffff -12345678.8 -12345678 -1 12345677
20 8 ffffffff -12345678.4 -12345678 -1 12345677
20 9 ffffffff -12345678 -12345678 -1 12345677
20 10 ffffffff -1 -1 -1 0
20 11 ffffffff 0 0 -1 -1
20 12 ffffffff 0 0 -1 -1
20 13 ffffffff 1 1 -1 -2
20 14 ffffffff 12345678 12345678 -1 -12345679
20 15 ffffffff 12345678.4 12345678 -1 -12345679
20 16 ffffffff 12345678.8 12345678 -1 -12345679
20 17 ffffffff 12345678.c 12345678 -1 -12345679
20 18 ffffffff 7fffffff 7fffffff -1 -80000000
20 19 ffffffff 80000000 -80000000 -1 7fffffff
20 20 ffffffff ffffffff -1 -1 0
20 21 ffffffff 123456789ab 456789ab -1 -456789ac
20 22 ffffffff 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 -1 -1
20 23 ffffffff Infinity 0 -1 -1
20 24 ffffffff NaN 0 -1 -1
21 0 123456789ab -Infinity 0 456789ab 456789ab
21 1 123456789ab -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 456789ab 456789ab
21 2 123456789ab -123456789ab 1 -1 -2
21 3 123456789ab -ffffffff 1 456789ab 456789aa
21 4 123456789ab -80000000 0 -3a987655 -3a987655
21 5 123456789ab -7fffffff 1 -3a987655 -3a987656
21 6 123456789ab -12345678.c 45438988 -12105655 -5753dfdd
21 7 123456789ab -12345678.8 45438988 -12105655 -5753dfdd
21 8 123456789ab -12345678.4 45438988 -12105655 -5753dfdd
21 9 123456789ab -12345678 45438988 -12105655 -5753dfdd
21 10 123456789ab -1 456789ab -1 -456789ac
21 11 123456789ab 0 0 456789ab 456789ab
21 12 123456789ab 0 0 456789ab 456789ab
21 13 123456789ab 1 1 456789ab 456789aa
21 14 123456789ab 12345678 240028 5777dffb 5753dfd3
21 15 123456789ab 12345678.4 240028 5777dffb 5753dfd3
21 16 123456789ab 12345678.8 240028 5777dffb 5753dfd3
21 17 123456789ab 12345678.c 240028 5777dffb 5753dfd3
21 18 123456789ab 7fffffff 456789ab 7fffffff 3a987654
21 19 123456789ab 80000000 0 -3a987655 -3a987655
21 20 123456789ab ffffffff 456789ab -1 -456789ac
21 21 123456789ab 123456789ab 456789ab 456789ab 0
21 22 123456789ab 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 456789ab 456789ab
21 23 123456789ab Infinity 0 456789ab 456789ab
21 24 123456789ab NaN 0 456789ab 456789ab
22 0 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -Infinity 0 0 0
22 1 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0
22 2 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -123456789ab 0 -456789ab -456789ab
22 3 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -ffffffff 0 1 1
22 4 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -80000000 0 -80000000 -80000000
22 5 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -7fffffff 0 -7fffffff -7fffffff
22 6 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -12345678.c 0 -12345678 -12345678
22 7 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -12345678.8 0 -12345678 -12345678
22 8 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -12345678.4 0 -12345678 -12345678
22 9 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -12345678 0 -12345678 -12345678
22 10 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -1 0 -1 -1
22 11 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0 0
22 12 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0 0
22 13 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 1 0 1 1
22 14 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 12345678 0 12345678 12345678
22 15 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 12345678.4 0 12345678 12345678
22 16 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 12345678.8 0 12345678 12345678
22 17 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 12345678.c 0 12345678 12345678
22 18 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 7fffffff 0 7fffffff 7fffffff
22 19 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 80000000 0 -80000000 -80000000
22 20 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 ffffffff 0 -1 -1
22 21 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 123456789ab 0 456789ab 456789ab
22 22 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0
22 23 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 Infinity 0 0 0
22 24 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 NaN 0 0 0
23 0 Infinity -Infinity 0 0 0
23 1 Infinity -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0
23 2 Infinity -123456789ab 0 -456789ab -456789ab
23 3 Infinity -ffffffff 0 1 1
23 4 Infinity -80000000 0 -80000000 -80000000
23 5 Infinity -7fffffff 0 -7fffffff -7fffffff
23 6 Infinity -12345678.c 0 -12345678 -12345678
23 7 Infinity -12345678.8 0 -12345678 -12345678
23 8 Infinity -12345678.4 0 -12345678 -12345678
23 9 Infinity -12345678 0 -12345678 -12345678
23 10 Infinity -1 0 -1 -1
23 11 Infinity 0 0 0 0
23 12 Infinity 0 0 0 0
23 13 Infinity 1 0 1 1
23 14 Infinity 12345678 0 12345678 12345678
23 15 Infinity 12345678.4 0 12345678 12345678
23 16 Infinity 12345678.8 0 12345678 12345678
23 17 Infinity 12345678.c 0 12345678 12345678
23 18 Infinity 7fffffff 0 7fffffff 7fffffff
23 19 Infinity 80000000 0 -80000000 -80000000
23 20 Infinity ffffffff 0 -1 -1
23 21 Infinity 123456789ab 0 456789ab 456789ab
23 22 Infinity 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0
23 23 Infinity Infinity 0 0 0
23 24 Infinity NaN 0 0 0
24 0 NaN -Infinity 0 0 0
24 1 NaN -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0
24 2 NaN -123456789ab 0 -456789ab -456789ab
24 3 NaN -ffffffff 0 1 1
24 4 NaN -80000000 0 -80000000 -80000000
24 5 NaN -7fffffff 0 -7fffffff -7fffffff
24 6 NaN -12345678.c 0 -12345678 -12345678
24 7 NaN -12345678.8 0 -12345678 -12345678
24 8 NaN -12345678.4 0 -12345678 -12345678
24 9 NaN -12345678 0 -12345678 -12345678
24 10 NaN -1 0 -1 -1
24 11 NaN 0 0 0 0
24 12 NaN 0 0 0 0
24 13 NaN 1 0 1 1
24 14 NaN 12345678 0 12345678 12345678
24 15 NaN 12345678.4 0 12345678 12345678
24 16 NaN 12345678.8 0 12345678 12345678
24 17 NaN 12345678.c 0 12345678 12345678
24 18 NaN 7fffffff 0 7fffffff 7fffffff
24 19 NaN 80000000 0 -80000000 -80000000
24 20 NaN ffffffff 0 -1 -1
24 21 NaN 123456789ab 0 456789ab 456789ab
24 22 NaN 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 0 0 0
24 23 NaN Infinity 0 0 0
24 24 NaN NaN 0 0 0
0 -Infinity -1
1 -1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -1
2 -123456789ab 456789aa
3 -ffffffff -2
4 -80000000 7fffffff
5 -7fffffff 7ffffffe
6 -12345678.c 12345677
7 -12345678.8 12345677
8 -12345678.4 12345677
9 -12345678 12345677
10 -1 0
11 0 -1
12 0 -1
13 1 -2
14 12345678 -12345679
15 12345678.4 -12345679
16 12345678.8 -12345679
17 12345678.c -12345679
18 7fffffff -80000000
19 80000000 7fffffff
20 ffffffff 0
21 123456789ab -456789ac
22 1249ad2594c37d0000000000000000000000000000000000000000000000000000000000000000000000 -1
23 Infinity -1
24 NaN -1
===*/

function matrixAndOrXorNotTest() {
    var vals = [
        Number.NEGATIVE_INFINITY,
        -1e100,
        -0x123456789ab,
        -0xffffffff, -0x80000000, -0x7fffffff,
        -0x12345678 - 0.75, -0x12345678 - 0.50, -0x12345678 - 0.25, -0x12345678,
        -1, -0, +0, +1,
        0x12345678, 0x12345678 + 0.25, 0x12345678 + 0.50, 0x12345678 + 0.75,
        0x7fffffff, 0x80000000, 0xffffffff,
        0x123456789ab,
        1e100,
        Number.POSITIVE_INFINITY,
        Number.NaN
    ];
    var i, j;

    for (i = 0; i < vals.length; i++) {
        for (j = 0; j < vals.length; j++) {
            print(i, j, hex(vals[i]), hex(vals[j]),
                  hex(vals[i] & vals[j]),
                  hex(vals[i] | vals[j]),
                  hex(vals[i] ^ vals[j]));
        }
    }

    for (i = 0; i < vals.length; i++) {
        print(i, hex(vals[i]), hex(~vals[i]));
    }
}

try {
    print('matrix and-or-xor-not');
    matrixAndOrXorNotTest();
} catch (e) {
    print(e.stack || e);
}
