/*-------------------------------------------------------------------------
 * drawElements Quality Program OpenGL ES 2.0 Module
 * -------------------------------------------------
 *
 * Copyright 2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *//*!
 * \file
 * \brief Negative Shader API tests.
 *//*--------------------------------------------------------------------*/

#include "es2fNegativeShaderApiTests.hpp"
#include "es2fApiCase.hpp"
#include "gluShaderProgram.hpp"
#include "gluContextInfo.hpp"

#include "glwDefs.hpp"
#include "glwEnums.hpp"

using namespace glw; // GL types

namespace deqp
{
namespace gles2
{
namespace Functional
{

static const char* vertexShaderSource	= "void main (void) { gl_Position = vec4(0.0); }\n\0";
static const char* fragmentShaderSource	= "void main (void) { gl_FragColor = vec4(0.0); }\n\0";

static const char* uniformTestVertSource	=	"uniform mediump vec4 vTest;\n"
												"uniform mediump mat4 vMatrix;\n"
												"void main (void)\n"
												"{\n"
												"	gl_Position = vMatrix * vTest;\n"
												"}\n\0";
static const char* uniformTestFragSource	=	"uniform mediump ivec4 fTest;\n"
												"uniform sampler2D fSampler;\n"
												"void main (void)\n"
												"{\n"
												"	gl_FragColor.xy = vec4(fTest).xy;\n"
												"	gl_FragColor.zw = texture2D(fSampler, vec2(0.0, 0.0)).zw;\n"
												"}\n\0";

using tcu::TestLog;

NegativeShaderApiTests::NegativeShaderApiTests (Context& context)
	: TestCaseGroup(context, "shader", "Negative Shader API Cases")
{
}

NegativeShaderApiTests::~NegativeShaderApiTests (void)
{
}

void NegativeShaderApiTests::init (void)
{
	ES2F_ADD_API_CASE(create_shader, "Invalid glCreateShader() usage",
		{
			m_log << TestLog::Section("", "GL_INVALID_ENUM is generated if shaderType is not an accepted value.");
			glCreateShader(-1);
			expectError(GL_INVALID_ENUM);
			m_log << TestLog::EndSection;
		});
	ES2F_ADD_API_CASE(shader_source, "Invalid glShaderSource() usage",
		{
			GLboolean shaderCompilerSupported;
			glGetBooleanv(GL_SHADER_COMPILER, &shaderCompilerSupported);
			if (!shaderCompilerSupported)
				m_log << TestLog::Message << "// Shader compiler not supported, always expect GL_INVALID_OPERATION" << TestLog::EndMessage;
			else
				m_log << TestLog::Message << "// Shader compiler supported" << TestLog::EndMessage;

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if shader is not a value generated by OpenGL.");
			glShaderSource(1, 0, 0, 0);
			expectError(shaderCompilerSupported ? GL_INVALID_VALUE : GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if count is less than 0.");
			GLuint shader = glCreateShader(GL_VERTEX_SHADER);
			glShaderSource(shader, -1, 0, 0);
			expectError(shaderCompilerSupported ? GL_INVALID_VALUE : GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if shader is not a shader object.");
			GLuint program = glCreateProgram();
			glShaderSource(program, 0, 0, 0);
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			glDeleteProgram(program);
			glDeleteShader(shader);
		});
	ES2F_ADD_API_CASE(compile_shader, "Invalid glCompileShader() usage",
		{
			GLboolean shaderCompilerSupported;
			glGetBooleanv(GL_SHADER_COMPILER, &shaderCompilerSupported);
			if (!shaderCompilerSupported)
				m_log << TestLog::Message << "// Shader compiler not supported, always expect GL_INVALID_OPERATION" << TestLog::EndMessage;
			else
				m_log << TestLog::Message << "// Shader compiler supported" << TestLog::EndMessage;

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if shader is not a value generated by OpenGL.");
			glCompileShader(9);
			expectError(shaderCompilerSupported ? GL_INVALID_VALUE : GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if shader is not a shader object.");
			GLuint program = glCreateProgram();
			glCompileShader(program);
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			glDeleteProgram(program);
		});
	ES2F_ADD_API_CASE(delete_shader, "Invalid glDeleteShader() usage",
		{
			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if shader is not a value generated by OpenGL.");
			glDeleteShader(9);
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;
		});
	ES2F_ADD_API_CASE(shader_binary, "Invalid glShaderBinary() usage",
		{
			std::vector<deInt32> binaryFormats;
			getSupportedExtensions(GL_NUM_SHADER_BINARY_FORMATS, GL_SHADER_BINARY_FORMATS, binaryFormats);
			deBool shaderBinarySupported = !binaryFormats.empty();
			if (!shaderBinarySupported)
				m_log << TestLog::Message << "// Shader binaries not supported." << TestLog::EndMessage;
			else
				m_log << TestLog::Message << "// Shader binaries supported" << TestLog::EndMessage;

			GLuint shaders[2];

			shaders[0] = glCreateShader(GL_VERTEX_SHADER);
			shaders[1] = glCreateShader(GL_VERTEX_SHADER);
			GLuint program = glCreateProgram();

			m_log << TestLog::Section("", "GL_INVALID_ENUM is generated if binaryformat is not a supported format returned in GL_SHADER_BINARY_FORMATS.");
			glShaderBinary(1, &shaders[0], -1, 0, 0);
			expectError(GL_INVALID_ENUM);
			m_log << TestLog::EndSection;

			if (shaderBinarySupported)
			{
				m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if any value in shaders is not a value generated by OpenGL.");
				shaders[0] = 137;
				glShaderBinary(1, &shaders[0], binaryFormats[0], 0, 0);
				expectError(shaderBinarySupported ? GL_INVALID_VALUE : GL_INVALID_OPERATION);
				m_log << TestLog::EndSection;

				m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if n or length is negative.");
				shaders[0] = glCreateShader(GL_VERTEX_SHADER);
				glShaderBinary(-1, &shaders[0], binaryFormats[0], 0, 0);
				expectError(GL_INVALID_VALUE);
				glShaderBinary(1, &shaders[0], binaryFormats[0], 0, -1);
				expectError(GL_INVALID_VALUE);
				m_log << TestLog::EndSection;

				m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if any value in shaders is not a shader object.");
				glShaderBinary(1, &program, binaryFormats[0], 0, 0);
				expectError(GL_INVALID_OPERATION);
				m_log << TestLog::EndSection;

				m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if there is more than one vertex shader object handle or more than one fragment shader object handle in shaders.");
				shaders[0] = glCreateShader(GL_VERTEX_SHADER);
				shaders[1] = glCreateShader(GL_VERTEX_SHADER);
				glShaderBinary(2, &shaders[0], binaryFormats[0], 0, 1);
				expectError(GL_INVALID_OPERATION);
				m_log << TestLog::EndSection;
			}

			glDeleteShader(shaders[0]);
			glDeleteShader(shaders[1]);
			glDeleteProgram(program);

			// \note: The format of the data pointed to by binary does not match binaryformat.
		});
	ES2F_ADD_API_CASE(attach_shader, "Invalid glAttachShader() usage",
		{
			GLuint shader1 = glCreateShader(GL_VERTEX_SHADER);
			GLuint shader2 = glCreateShader(GL_VERTEX_SHADER);
			GLuint program = glCreateProgram();

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if program is not a program object.");
			glAttachShader(shader1, shader1);
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if shader is not a shader object.");
			glAttachShader(program, program);
			expectError(GL_INVALID_OPERATION);
			glAttachShader(shader1, program);
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if either program or shader is not a value generated by OpenGL.");
			glAttachShader(program, -1);
			expectError(GL_INVALID_VALUE);
			glAttachShader(-1, shader1);
			expectError(GL_INVALID_VALUE);
			glAttachShader(-1, -1);
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if shader is already attached to program, or if another shader object of the same type as shader is already attached to program.");
			glAttachShader(program, shader1);
			expectError(GL_NO_ERROR);
			glAttachShader(program, shader1);
			expectError(GL_INVALID_OPERATION);
			glAttachShader(program, shader2);
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			glDeleteProgram(program);
			glDeleteShader(shader1);
			glDeleteShader(shader2);
		});
	ES2F_ADD_API_CASE(detach_shader, "Invalid glDetachShader() usage",
		{
			GLuint shader = glCreateShader(GL_VERTEX_SHADER);
			GLuint program = glCreateProgram();

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if either program or shader is not a value generated by OpenGL.");
			glDetachShader(-1, shader);
			expectError(GL_INVALID_VALUE);
			glDetachShader(program, -1);
			expectError(GL_INVALID_VALUE);
			glDetachShader(-1, -1);
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if program is not a program object.");
			glDetachShader(shader, shader);
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if shader is not a shader object.");
			glDetachShader(program, program);
			expectError(GL_INVALID_OPERATION);
			glDetachShader(shader, program);
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if shader is not attached to program.");
			glDetachShader(program, shader);
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			glDeleteProgram(program);
			glDeleteShader(shader);
		});
	ES2F_ADD_API_CASE(link_program, "Invalid glLinkProgram() usage",
		{
			GLuint shader = glCreateShader(GL_VERTEX_SHADER);

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if program is not a value generated by OpenGL.");
			glLinkProgram(-1);
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if program is not a program object.");
			glLinkProgram(shader);
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			glDeleteShader(shader);
		});
	ES2F_ADD_API_CASE(use_program, "Invalid glUseProgram() usage",
		{
			GLuint shader = glCreateShader(GL_VERTEX_SHADER);

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if program is neither 0 nor a value generated by OpenGL.");
			glUseProgram(-1);
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if program is not a program object.");
			glUseProgram(shader);
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			glUseProgram(0);
			glDeleteShader(shader);
		});
	ES2F_ADD_API_CASE(delete_program, "Invalid glDeleteProgram() usage",
		{
			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if program is not a value generated by OpenGL.");
			glDeleteProgram(-1);
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;
		});
	ES2F_ADD_API_CASE(get_active_attrib, "Invalid glGetActiveAttrib() usage",
		{
			GLuint shader = glCreateShader(GL_VERTEX_SHADER);
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(vertexShaderSource, fragmentShaderSource));
			glUseProgram(program.getProgram());

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if program is not a value generated by OpenGL.");
			glGetActiveAttrib(-1, 0, 0, 0, 0, 0, 0);
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if program is not a program object.");
			glGetActiveAttrib(shader, 0, 0, 0, 0, 0, 0);
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if index is greater than or equal to the number of active attribute variables in program.");
			glGetActiveAttrib(program.getProgram(), 0, 0, 0, 0, 0, 0);
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if bufSize is less than 0.");
			glGetActiveAttrib(program.getProgram(), 0, -1, 0, 0, 0, 0);
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;

			glUseProgram(0);
			glDeleteShader(shader);
		});
	ES2F_ADD_API_CASE(get_attrib_location, "Invalid glGetAttribLocation() usage",
		{
			GLuint programEmpty = glCreateProgram();
			GLuint shader = glCreateShader(GL_VERTEX_SHADER);
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(vertexShaderSource, fragmentShaderSource));

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if program has not been successfully linked.");
			glBindAttribLocation(programEmpty, 0, "test");
			glGetAttribLocation(programEmpty, "test");
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if program is not a program or shader object.");
			glUseProgram(program.getProgram());
			glBindAttribLocation(program.getProgram(), 0, "test");
			expectError(GL_NO_ERROR);
			glGetAttribLocation(program.getProgram(), "test");
			expectError(GL_NO_ERROR);
			glGetAttribLocation(-2, "test");
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if program is not a program object.");
			glGetAttribLocation(shader, "test");
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			glUseProgram(0);
			glDeleteShader(shader);
			glDeleteProgram(programEmpty);
		});
	ES2F_ADD_API_CASE(get_uniform_location, "Invalid glGetUniformLocation() usage",
		{
			GLuint programEmpty = glCreateProgram();
			GLuint shader = glCreateShader(GL_VERTEX_SHADER);
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(vertexShaderSource, fragmentShaderSource));

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if program has not been successfully linked.");
			glGetUniformLocation(programEmpty, "test");
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if program is not a value generated by OpenGL.");
			glUseProgram(program.getProgram());
			glGetUniformLocation(-2, "test");
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if program is not a program object.");
			glGetAttribLocation(shader, "test");
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			glUseProgram(0);
			glDeleteProgram(programEmpty);
			glDeleteShader(shader);
		});
	ES2F_ADD_API_CASE(bind_attrib_location, "Invalid glBindAttribLocation() usage",
		{
			GLuint program = glCreateProgram();
			GLuint maxIndex = m_context.getContextInfo().getInt(GL_MAX_VERTEX_ATTRIBS);
			GLuint shader = glCreateShader(GL_VERTEX_SHADER);

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if index is greater than or equal to GL_MAX_VERTEX_ATTRIBS.");
			glBindAttribLocation(program, maxIndex, "test");
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if name starts with the reserved prefix \"gl_\".");
			glBindAttribLocation(program, maxIndex-1, "gl_test");
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if program is not a value generated by OpenGL.");
			glBindAttribLocation(-1, maxIndex-1, "test");
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if program is not a program object.");
			glBindAttribLocation(shader, maxIndex-1, "test");
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			glDeleteProgram(program);
			glDeleteShader(shader);
		});
	ES2F_ADD_API_CASE(get_active_uniform, "Invalid glGetActiveUniform() usage",
		{
			GLuint shader = glCreateShader(GL_VERTEX_SHADER);
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(vertexShaderSource, fragmentShaderSource));

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if program is not a value generated by OpenGL.");
			glGetActiveUniform(-1, 0, 0, 0, 0, 0, 0);
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if program is not a program object.");
			glGetActiveUniform(shader, 0, 0, 0, 0, 0, 0);
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if index is greater than or equal to the number of active attribute variables in program.");
			glUseProgram(program.getProgram());
			glGetActiveUniform(program.getProgram(), 5, 0, 0, 0, 0, 0);
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if bufSize is less than 0.");
			glGetActiveUniform(program.getProgram(), 0, -1, 0, 0, 0, 0);
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;

			glUseProgram(0);
			glDeleteShader(shader);
		});
	ES2F_ADD_API_CASE(validate_program, "Invalid glValidateProgram() usage",
		{
			GLuint shader = glCreateShader(GL_VERTEX_SHADER);

			m_log << TestLog::Section("", "GL_INVALID_VALUE is generated if program is not a value generated by OpenGL.");
			glValidateProgram(-1);
			expectError(GL_INVALID_VALUE);
			m_log << TestLog::EndSection;

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if program is not a program object.");
			glValidateProgram(shader);
			expectError(GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;

			glDeleteShader(shader);
		});

	ES2F_ADD_API_CASE(release_shader_compiler, "Invalid glReleaseShaderCompiler() usage",
		{
			GLboolean shaderCompilerSupported;
			glGetBooleanv(GL_SHADER_COMPILER, &shaderCompilerSupported);

			m_log << TestLog::Section("", "GL_INVALID_OPERATION is generated if a shader compiler is not supported.");
			glReleaseShaderCompiler();
			expectError(shaderCompilerSupported ? GL_NONE : GL_INVALID_OPERATION);
			m_log << TestLog::EndSection;
		});

	// glUniform*f

	ES2F_ADD_API_CASE(uniformf_invalid_program, "Invalid glUniform{1234}f() usage",
		{
			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if there is no current program object.");
			glUseProgram(0);
			glUniform1f(-1, 0.0f);
			expectError(GL_INVALID_OPERATION);
			glUniform2f(-1, 0.0f, 0.0f);
			expectError(GL_INVALID_OPERATION);
			glUniform3f(-1, 0.0f, 0.0f, 0.0f);
			expectError(GL_INVALID_OPERATION);
			glUniform4f(-1, 0.0f, 0.0f, 0.0f, 0.0f);
			expectError(GL_INVALID_OPERATION);
			m_log << tcu::TestLog::EndSection;
		});
	ES2F_ADD_API_CASE(uniformf_incompatible_type, "Invalid glUniform{1234}f() usage",
		{
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(uniformTestVertSource, uniformTestFragSource));
			glUseProgram(program.getProgram());
			GLint vUnif		= glGetUniformLocation(program.getProgram(), "vTest");		// vec4
			GLint fUnif		= glGetUniformLocation(program.getProgram(), "fTest");		// ivec4
			GLint fSampler	= glGetUniformLocation(program.getProgram(), "fSampler");	// sampler2D

			if (vUnif == -1 || fUnif == -1 || fSampler == -1)
				{
				m_log << TestLog::Message << "// ERROR: Failed to retrieve uniform location" << TestLog::EndMessage;
				m_testCtx.setTestResult(QP_TEST_RESULT_FAIL, "Failed to retrieve uniform location");
			}

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if the size of the uniform variable declared in the shader does not match the size indicated by the glUniform command.");
			glUseProgram(program.getProgram());
			glUniform1f(vUnif, 0.0f);
			expectError(GL_INVALID_OPERATION);
			glUniform2f(vUnif, 0.0f, 0.0f);
			expectError(GL_INVALID_OPERATION);
			glUniform3f(vUnif, 0.0f, 0.0f, 0.0f);
			expectError(GL_INVALID_OPERATION);
			glUniform4f(vUnif, 0.0f, 0.0f, 0.0f, 0.0f);
			expectError(GL_NO_ERROR);
			m_log << tcu::TestLog::EndSection;

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if one of the floating-point variants of this function is used to load a uniform variable of type int, ivec2, ivec3, or ivec4.");
			glUseProgram(program.getProgram());
			glUniform4f(fUnif, 0.0f, 0.0f, 0.0f, 0.0f);
			expectError(GL_INVALID_OPERATION);
			m_log << tcu::TestLog::EndSection;

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if a sampler is loaded using a command other than glUniform1i and glUniform1iv.");
			glUseProgram(program.getProgram());
			glUniform1f(fSampler, 0.0f);
			expectError(GL_INVALID_OPERATION);
			m_log << tcu::TestLog::EndSection;

			glUseProgram(0);
		});
	ES2F_ADD_API_CASE(uniformf_invalid_location, "Invalid glUniform{1234}f() usage",
		{
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(uniformTestVertSource, uniformTestFragSource));
			glUseProgram(program.getProgram());

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if location is an invalid uniform location for the current program object and location is not equal to -1.");
			glUseProgram(program.getProgram());
			glUniform1f(-2, 0.0f);
			expectError(GL_INVALID_OPERATION);
			glUniform2f(-2, 0.0f, 0.0f);
			expectError(GL_INVALID_OPERATION);
			glUniform3f(-2, 0.0f, 0.0f, 0.0f);
			expectError(GL_INVALID_OPERATION);
			glUniform4f(-2, 0.0f, 0.0f, 0.0f, 0.0f);
			expectError(GL_INVALID_OPERATION);

			glUseProgram(program.getProgram());
			glUniform1f(-1, 0.0f);
			expectError(GL_NO_ERROR);
			glUniform2f(-1, 0.0f, 0.0f);
			expectError(GL_NO_ERROR);
			glUniform3f(-1, 0.0f, 0.0f, 0.0f);
			expectError(GL_NO_ERROR);
			glUniform4f(-1, 0.0f, 0.0f, 0.0f, 0.0f);
			expectError(GL_NO_ERROR);
			m_log << tcu::TestLog::EndSection;

			glUseProgram(0);
		});

	// glUniform*fv

	ES2F_ADD_API_CASE(uniformfv_invalid_program, "Invalid glUniform{1234}fv() usage",
		{
			std::vector<GLfloat> data(4);

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if there is no current program object.");
			glUseProgram(0);
			glUniform1fv(-1, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform2fv(-1, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform3fv(-1, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform4fv(-1, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			m_log << tcu::TestLog::EndSection;
		});
	ES2F_ADD_API_CASE(uniformfv_incompatible_type, "Invalid glUniform{1234}fv() usage",
		{
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(uniformTestVertSource, uniformTestFragSource));
			glUseProgram(program.getProgram());
			GLint vUnif		= glGetUniformLocation(program.getProgram(), "vTest");		// vec4
			GLint fUnif		= glGetUniformLocation(program.getProgram(), "fTest");		// ivec4
			GLint fSampler	= glGetUniformLocation(program.getProgram(), "fSampler");	// sampler2D

			if (vUnif == -1 || fUnif == -1 || fSampler == -1)
			{
				m_log << TestLog::Message << "// ERROR: Failed to retrieve uniform location" << TestLog::EndMessage;
				m_testCtx.setTestResult(QP_TEST_RESULT_FAIL, "Failed to retrieve uniform location");
			}

			std::vector<GLfloat> data(4);

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if the size of the uniform variable declared in the shader does not match the size indicated by the glUniform command.");
			glUseProgram(program.getProgram());
			glUniform1fv(vUnif, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform2fv(vUnif, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform3fv(vUnif, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform4fv(vUnif, 1, &data[0]);
			expectError(GL_NO_ERROR);
			m_log << tcu::TestLog::EndSection;

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if one of the floating-point variants of this function is used to load a uniform variable of type int, ivec2, ivec3, or ivec4.");
			glUseProgram(program.getProgram());
			glUniform4fv(fUnif, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			m_log << tcu::TestLog::EndSection;

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if a sampler is loaded using a command other than glUniform1i and glUniform1iv.");
			glUseProgram(program.getProgram());
			glUniform1fv(fSampler, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			m_log << tcu::TestLog::EndSection;

			glUseProgram(0);
		});
	ES2F_ADD_API_CASE(uniformfv_invalid_location, "Invalid glUniform{1234}fv() usage",
		{
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(uniformTestVertSource, uniformTestFragSource));
			glUseProgram(program.getProgram());

			std::vector<GLfloat> data(4);

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if location is an invalid uniform location for the current program object and location is not equal to -1.");
			glUseProgram(program.getProgram());
			glUniform1fv(-2, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform2fv(-2, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform3fv(-2, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform4fv(-2, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);

			glUseProgram(program.getProgram());
			glUniform1fv(-1, 1, &data[0]);
			expectError(GL_NO_ERROR);
			glUniform2fv(-1, 1, &data[0]);
			expectError(GL_NO_ERROR);
			glUniform3fv(-1, 1, &data[0]);
			expectError(GL_NO_ERROR);
			glUniform4fv(-1, 1, &data[0]);
			expectError(GL_NO_ERROR);
			m_log << tcu::TestLog::EndSection;

			glUseProgram(0);
		});
	ES2F_ADD_API_CASE(uniformfv_invalid_count, "Invalid glUniform{1234}fv() usage",
		{
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(uniformTestVertSource, uniformTestFragSource));
			glUseProgram(program.getProgram());
			GLint vUnif		= glGetUniformLocation(program.getProgram(), "vTest");		// vec4

			if (vUnif == -1)
			{
				m_log << TestLog::Message << "// ERROR: Failed to retrieve uniform location" << TestLog::EndMessage;
				m_testCtx.setTestResult(QP_TEST_RESULT_FAIL, "Failed to retrieve uniform location");
			}

			std::vector<GLfloat> data(8);

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if count is greater than 1 and the indicated uniform variable is not an array variable.");
			glUseProgram(program.getProgram());
			glUniform1fv(vUnif, 2, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform2fv(vUnif, 2, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform3fv(vUnif, 2, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform4fv(vUnif, 2, &data[0]);
			expectError(GL_INVALID_OPERATION);
			m_log << tcu::TestLog::EndSection;

			glUseProgram(0);
		});

	// glUniform*i

	ES2F_ADD_API_CASE(uniformi_invalid_program, "Invalid glUniform{1234}i() usage",
		{
			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if there is no current program object.");
			glUseProgram(0);
			glUniform1i(-1, 0);
			expectError(GL_INVALID_OPERATION);
			glUniform2i(-1, 0, 0);
			expectError(GL_INVALID_OPERATION);
			glUniform3i(-1, 0, 0, 0);
			expectError(GL_INVALID_OPERATION);
			glUniform4i(-1, 0, 0, 0, 0);
			expectError(GL_INVALID_OPERATION);
			m_log << tcu::TestLog::EndSection;
		});
	ES2F_ADD_API_CASE(uniformi_incompatible_type, "Invalid glUniform{1234}i() usage",
		{
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(uniformTestVertSource, uniformTestFragSource));
			glUseProgram(program.getProgram());
			GLint vUnif		= glGetUniformLocation(program.getProgram(), "vTest");		// vec4
			GLint fUnif		= glGetUniformLocation(program.getProgram(), "fTest");		// ivec4
			GLint fSampler	= glGetUniformLocation(program.getProgram(), "fSampler");	// sampler2D

			if (vUnif == -1 || fUnif == -1 || fSampler == -1)
			{
				m_log << TestLog::Message << "// ERROR: Failed to retrieve uniform location" << TestLog::EndMessage;
				m_testCtx.setTestResult(QP_TEST_RESULT_FAIL, "Failed to retrieve uniform location");
			}

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if the size of the uniform variable declared in the shader does not match the size indicated by the glUniform command.");
			glUseProgram(program.getProgram());
			glUniform1i(fUnif, 0);
			expectError(GL_INVALID_OPERATION);
			glUniform2i(fUnif, 0, 0);
			expectError(GL_INVALID_OPERATION);
			glUniform3i(fUnif, 0, 0, 0);
			expectError(GL_INVALID_OPERATION);
			glUniform4i(fUnif, 0, 0, 0, 0);
			expectError(GL_NO_ERROR);
			m_log << tcu::TestLog::EndSection;

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if one of the integer variants of this function is used to load a uniform variable of type float, vec2, vec3, or vec4.");
			glUseProgram(program.getProgram());
			glUniform4i(vUnif, 0, 0, 0, 0);
			expectError(GL_INVALID_OPERATION);
			m_log << tcu::TestLog::EndSection;

			glUseProgram(0);
		});
	ES2F_ADD_API_CASE(uniformi_invalid_location, "Invalid glUniform{1234}i() usage",
		{
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(uniformTestVertSource, uniformTestFragSource));
			glUseProgram(program.getProgram());

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if location is an invalid uniform location for the current program object and location is not equal to -1.");
			glUseProgram(program.getProgram());
			glUniform1i(-2, 0);
			expectError(GL_INVALID_OPERATION);
			glUniform2i(-2, 0, 0);
			expectError(GL_INVALID_OPERATION);
			glUniform3i(-2, 0, 0, 0);
			expectError(GL_INVALID_OPERATION);
			glUniform4i(-2, 0, 0, 0, 0);
			expectError(GL_INVALID_OPERATION);

			glUseProgram(program.getProgram());
			glUniform1i(-1, 0);
			expectError(GL_NO_ERROR);
			glUniform2i(-1, 0, 0);
			expectError(GL_NO_ERROR);
			glUniform3i(-1, 0, 0, 0);
			expectError(GL_NO_ERROR);
			glUniform4i(-1, 0, 0, 0, 0);
			expectError(GL_NO_ERROR);
			m_log << tcu::TestLog::EndSection;

			glUseProgram(0);
		});

	// glUniform*iv

	ES2F_ADD_API_CASE(uniformiv_invalid_program, "Invalid glUniform{1234}iv() usage",
		{
			std::vector<GLint> data(4);

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if there is no current program object.");
			glUseProgram(0);
			glUniform1iv(-1, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform2iv(-1, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform3iv(-1, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform4iv(-1, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			m_log << tcu::TestLog::EndSection;
		});
	ES2F_ADD_API_CASE(uniformiv_incompatible_type, "Invalid glUniform{1234}iv() usage",
		{
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(uniformTestVertSource, uniformTestFragSource));
			glUseProgram(program.getProgram());
			GLint vUnif		= glGetUniformLocation(program.getProgram(), "vTest");		// vec4
			GLint fUnif		= glGetUniformLocation(program.getProgram(), "fTest");		// ivec4
			GLint fSampler	= glGetUniformLocation(program.getProgram(), "fSampler");	// sampler2D

			if (vUnif == -1 || fUnif == -1 || fSampler == -1)
			{
				m_log << TestLog::Message << "// ERROR: Failed to retrieve uniform location" << TestLog::EndMessage;
				m_testCtx.setTestResult(QP_TEST_RESULT_FAIL, "Failed to retrieve uniform location");
			}

			std::vector<GLint> data(4);

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if the size of the uniform variable declared in the shader does not match the size indicated by the glUniform command.");
			glUseProgram(program.getProgram());
			glUniform1iv(fUnif, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform2iv(fUnif, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform3iv(fUnif, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform4iv(fUnif, 1, &data[0]);
			expectError(GL_NO_ERROR);
			m_log << tcu::TestLog::EndSection;

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if one of the integer variants of this function is used to load a uniform variable of type float, vec2, vec3, or vec4.");
			glUseProgram(program.getProgram());
			glUniform4iv(vUnif, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			m_log << tcu::TestLog::EndSection;

			glUseProgram(0);
		});
	ES2F_ADD_API_CASE(uniformiv_invalid_location, "Invalid glUniform{1234}iv() usage",
		{
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(uniformTestVertSource, uniformTestFragSource));
			glUseProgram(program.getProgram());

			std::vector<GLint> data(4);

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if location is an invalid uniform location for the current program object and location is not equal to -1.");
			glUseProgram(program.getProgram());
			glUniform1iv(-2, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform2iv(-2, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform3iv(-2, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform4iv(-2, 1, &data[0]);
			expectError(GL_INVALID_OPERATION);

			glUseProgram(program.getProgram());
			glUniform1iv(-1, 1, &data[0]);
			expectError(GL_NO_ERROR);
			glUniform2iv(-1, 1, &data[0]);
			expectError(GL_NO_ERROR);
			glUniform3iv(-1, 1, &data[0]);
			expectError(GL_NO_ERROR);
			glUniform4iv(-1, 1, &data[0]);
			expectError(GL_NO_ERROR);
			m_log << tcu::TestLog::EndSection;

			glUseProgram(0);
		});
	ES2F_ADD_API_CASE(uniformiv_invalid_count, "Invalid glUniform{1234}iv() usage",
		{
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(uniformTestVertSource, uniformTestFragSource));
			glUseProgram(program.getProgram());
			GLint fUnif		= glGetUniformLocation(program.getProgram(), "fTest");		// ivec4

			if (fUnif == -1)
			{
				m_log << TestLog::Message << "// ERROR: Failed to retrieve uniform location" << TestLog::EndMessage;
				m_testCtx.setTestResult(QP_TEST_RESULT_FAIL, "Failed to retrieve uniform location");
			}

			std::vector<GLint> data(8);

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if count is greater than 1 and the indicated uniform variable is not an array variable.");
			glUseProgram(program.getProgram());
			glUniform1iv(fUnif, 2, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform2iv(fUnif, 2, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform3iv(fUnif, 2, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniform4iv(fUnif, 2, &data[0]);
			expectError(GL_INVALID_OPERATION);
			m_log << tcu::TestLog::EndSection;

			glUseProgram(0);
		});

	// glUniformMatrix*fv

	ES2F_ADD_API_CASE(uniform_matrixfv_invalid_program, "Invalid glUniformMatrix{234}fv() usage",
		{
			std::vector<GLfloat> data(16);

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if there is no current program object.");
			glUseProgram(0);
			glUniformMatrix2fv(-1, 1, GL_FALSE, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniformMatrix3fv(-1, 1, GL_FALSE, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniformMatrix4fv(-1, 1, GL_FALSE, &data[0]);
			expectError(GL_INVALID_OPERATION);
			m_log << tcu::TestLog::EndSection;
		});
	ES2F_ADD_API_CASE(uniform_matrixfv_incompatible_type, "Invalid glUniformMatrix{234}fv() usage",
		{
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(uniformTestVertSource, uniformTestFragSource));
			glUseProgram(program.getProgram());
			GLint vMatUnif		= glGetUniformLocation(program.getProgram(), "vMatrix");	// mat4
			GLint fSamplerUnif	= glGetUniformLocation(program.getProgram(), "fSampler");	// sampler2D

			m_log << program;

			if (vMatUnif == -1 || fSamplerUnif == -1)
			{
				m_log << TestLog::Message << "// ERROR: Failed to retrieve uniform location" << TestLog::EndMessage;
				m_testCtx.setTestResult(QP_TEST_RESULT_FAIL, "Failed to retrieve uniform location");
			}

			std::vector<GLfloat> data(16);

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if the size of the uniform variable declared in the shader does not match the size indicated by the glUniform command.");
			glUseProgram(program.getProgram());
			glUniformMatrix2fv(vMatUnif, 1, GL_FALSE, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniformMatrix3fv(vMatUnif, 1, GL_FALSE, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniformMatrix4fv(vMatUnif, 1, GL_FALSE, &data[0]);
			expectError(GL_NO_ERROR);
			m_log << tcu::TestLog::EndSection;

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if a sampler is loaded using a command other than glUniform1i and glUniform1iv.");
			glUseProgram(program.getProgram());
			glUniformMatrix4fv(fSamplerUnif, 1, GL_FALSE, &data[0]);
			expectError(GL_INVALID_OPERATION);
			m_log << tcu::TestLog::EndSection;

			glUseProgram(0);
		});
	ES2F_ADD_API_CASE(uniform_matrixfv_invalid_location, "Invalid glUniformMatrix{234}fv() usage",
		{
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(uniformTestVertSource, uniformTestFragSource));
			glUseProgram(program.getProgram());

			m_log << program;

			std::vector<GLfloat> data(16);

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if location is an invalid uniform location for the current program object and location is not equal to -1.");
			glUseProgram(program.getProgram());
			glUniformMatrix2fv(-2, 1, GL_FALSE, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniformMatrix3fv(-2, 1, GL_FALSE, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniformMatrix4fv(-2, 1, GL_FALSE, &data[0]);
			expectError(GL_INVALID_OPERATION);

			glUseProgram(program.getProgram());
			glUniformMatrix2fv(-1, 1, GL_FALSE, &data[0]);
			expectError(GL_NO_ERROR);
			glUniformMatrix3fv(-1, 1, GL_FALSE, &data[0]);
			expectError(GL_NO_ERROR);
			glUniformMatrix4fv(-1, 1, GL_FALSE, &data[0]);
			expectError(GL_NO_ERROR);
			m_log << tcu::TestLog::EndSection;

			glUseProgram(0);
		});
	ES2F_ADD_API_CASE(uniform_matrixfv_invalid_count, "Invalid glUniformMatrix{234}fv() usage",
		{
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(uniformTestVertSource, uniformTestFragSource));
			glUseProgram(program.getProgram());
			GLint vMatUnif		= glGetUniformLocation(program.getProgram(), "vMatrix");		// mat4

			m_log << program;

			if (vMatUnif == -1)
			{
				m_log << TestLog::Message << "// ERROR: Failed to retrieve uniform location" << TestLog::EndMessage;
				m_testCtx.setTestResult(QP_TEST_RESULT_FAIL, "Failed to retrieve uniform location");
			}


			std::vector<GLfloat> data(32);

			m_log << tcu::TestLog::Section("", "GL_INVALID_OPERATION is generated if count is greater than 1 and the indicated uniform variable is not an array variable.");
			glUseProgram(program.getProgram());
			glUniformMatrix2fv(vMatUnif, 2, GL_FALSE, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniformMatrix3fv(vMatUnif, 2, GL_FALSE, &data[0]);
			expectError(GL_INVALID_OPERATION);
			glUniformMatrix4fv(vMatUnif, 2, GL_FALSE, &data[0]);
			expectError(GL_INVALID_OPERATION);
			m_log << tcu::TestLog::EndSection;

			glUseProgram(0);
		});
	ES2F_ADD_API_CASE(uniform_matrixfv_invalid_transpose, "Invalid glUniformMatrix{234}fv() usage",
		{
			glu::ShaderProgram program(m_context.getRenderContext(), glu::makeVtxFragSources(uniformTestVertSource, uniformTestFragSource));
			glUseProgram(program.getProgram());

			m_log << program;

			std::vector<GLfloat> data(16);

			m_log << tcu::TestLog::Section("", "GL_INVALID_VALUE is generated if transpose is not GL_FALSE.");
			glUseProgram(program.getProgram());
			glUniformMatrix2fv(-1, 1, GL_TRUE, &data[0]);
			expectError(GL_INVALID_VALUE);
			glUniformMatrix3fv(-1, 1, GL_TRUE, &data[0]);
			expectError(GL_INVALID_VALUE);
			glUniformMatrix4fv(-1, 1, GL_TRUE, &data[0]);
			expectError(GL_INVALID_VALUE);
			m_log << tcu::TestLog::EndSection;

			glUseProgram(0);
		});
}

} // Functional
} // gles2
} // deqp
