/*
 * Copyright 2015 Google Inc.
 *
 * Use of this source code is governed by a BSD-style license that can be
 * found in the LICENSE file.
 */
#include "PathOpsTestCommon.h"
#include "SkIntersections.h"
#include "SkPathOpsConic.h"
#include "SkPathOpsCubic.h"
#include "SkReduceOrder.h"
#include "Test.h"

static struct cubicConic {
    SkDCubic cubic;
    SkDConic conic;
} cubicConicTests[] = {
    {{{{188.60000610351562, 2041.5999755859375}, {188.60000610351562, 2065.39990234375},
        {208, 2084.800048828125}, {231.80000305175781, 2084.800048828125}}},
    {{{{231.80000305175781, 2084.800048828125}, {188.60000610351562, 2084.800048828125},
        {188.60000610351562, 2041.5999755859375}}}, 0.707107008f}},

    {{{{231.80000305175781, 2084.800048828125}, {255.60000610351562, 2084.800048828125},
        {275, 2065.39990234375}, {275, 2041.5999755859375}}},
    {{{{275, 2041.5999755859375}, {275, 2084.800048828125},
        {231.80000305175781, 2084.800048828125}}}, 0.707107008f}},
};

static const int cubicConicTests_count = (int) SK_ARRAY_COUNT(cubicConicTests);

static void cubicConicIntersection(skiatest::Reporter* reporter, int index) {
    const SkDCubic& cubic = cubicConicTests[index].cubic;
    SkASSERT(ValidCubic(cubic));
    const SkDConic& conic = cubicConicTests[index].conic;
    SkASSERT(ValidConic(conic));
    SkReduceOrder reduce1;
    SkReduceOrder reduce2;
    int order1 = reduce1.reduce(cubic, SkReduceOrder::kNo_Quadratics);
    int order2 = reduce2.reduce(conic.fPts);
    if (order1 != 4) {
        SkDebugf("[%d] cubic order=%d\n", index, order1);
        REPORTER_ASSERT(reporter, 0);
    }
    if (order2 != 3) {
        SkDebugf("[%d] conic order=%d\n", index, order2);
        REPORTER_ASSERT(reporter, 0);
    }
    SkIntersections i;
    int roots = i.intersect(cubic, conic);
    for (int pt = 0; pt < roots; ++pt) {
        double tt1 = i[0][pt];
        SkDPoint xy1 = cubic.ptAtT(tt1);
        double tt2 = i[1][pt];
        SkDPoint xy2 = conic.ptAtT(tt2);
        if (!xy1.approximatelyEqual(xy2)) {
            SkDebugf("%s [%d,%d] x!= t1=%g (%g,%g) t2=%g (%g,%g)\n",
                __FUNCTION__, index, pt, tt1, xy1.fX, xy1.fY, tt2, xy2.fX, xy2.fY);
        }
        REPORTER_ASSERT(reporter, xy1.approximatelyEqual(xy2));
    }
    reporter->bumpTestCount();
}

DEF_TEST(PathOpsCubicConicIntersection, reporter) {
    for (int index = 0; index < cubicConicTests_count; ++index) {
        cubicConicIntersection(reporter, index);
        reporter->bumpTestCount();
    }
}

DEF_TEST(PathOpsCubicConicIntersectionOneOff, reporter) {
    cubicConicIntersection(reporter, 1);
}
