# vgp_access_ext samba group policy
# Copyright (C) David Mulder <dmulder@suse.com> 2020
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os, re
from samba.gp.gpclass import gp_xml_ext, gp_file_applier
from tempfile import NamedTemporaryFile
from samba.common import get_bytes
from samba.gp.util.logging import log

intro = '''
### autogenerated by samba
#
# This file is generated by the vgp_access_ext Group Policy
# Client Side Extension. To modify the contents of this file,
# modify the appropriate Group Policy objects which apply
# to this machine. DO NOT MODIFY THIS FILE DIRECTLY.
#

'''

# The deny all file is implicit any time an allow entry is used
DENY_BOUND = 9000000000
DENY_FILE = '_gp_DENY_ALL.conf'

# Each policy MUST create it's own DENY_ALL file if an allow entry exists,
# otherwise policies will conflict and one could remove a DENY_ALL when another
# one still requires it.
def deny_file(access):
    deny_filename = os.path.join(access,
                        '%d%s' % (select_next_deny(access), DENY_FILE))
    with NamedTemporaryFile(delete=False, dir=access) as f:
        with open(f.name, 'w') as w:
            w.write(intro)
            w.write('-:ALL:ALL')
        os.chmod(f.name, 0o644)
        os.rename(f.name, deny_filename)
    return deny_filename

def select_next_deny(directory):
    configs = [re.match(r'(\d+)', f) for f in os.listdir(directory) if DENY_FILE in f]
    return max([int(m.group(1)) for m in configs if m]+[DENY_BOUND])+1

# Access files in /etc/security/access.d are read in the order of the system
# locale. Here we number the conf files to ensure they are read in the correct
# order. Ignore the deny file, since allow entries should always come before
# the implicit deny ALL.
def select_next_conf(directory):
    configs = [re.match(r'(\d+)', f) for f in os.listdir(directory) if DENY_FILE not in f]
    return max([int(m.group(1)) for m in configs if m]+[0])+1

class vgp_access_ext(gp_xml_ext, gp_file_applier):
    def __str__(self):
        return 'VGP/Unix Settings/Host Access'

    def process_group_policy(self, deleted_gpo_list, changed_gpo_list,
                             access='/etc/security/access.d'):
        for guid, settings in deleted_gpo_list:
            if str(self) in settings:
                for attribute, policy_files in settings[str(self)].items():
                    self.unapply(guid, attribute, policy_files)

        for gpo in changed_gpo_list:
            if gpo.file_sys_path:
                allow = 'MACHINE/VGP/VTLA/VAS/HostAccessControl/Allow/manifest.xml'
                path = os.path.join(gpo.file_sys_path, allow)
                allow_conf = self.parse(path)
                deny = 'MACHINE/VGP/VTLA/VAS/HostAccessControl/Deny/manifest.xml'
                path = os.path.join(gpo.file_sys_path, deny)
                deny_conf = self.parse(path)
                entries = []
                policy_files = []
                winbind_sep = self.lp.get('winbind separator')
                if allow_conf:
                    policy = allow_conf.find('policysetting')
                    data = policy.find('data')
                    allow_listelements = data.findall('listelement')
                    for listelement in allow_listelements:
                        adobject = listelement.find('adobject')
                        name = adobject.find('name').text
                        domain = adobject.find('domain').text
                        entries.append('+:%s%s%s:ALL' % (domain,
                                                         winbind_sep,
                                                         name))
                    if len(allow_listelements) > 0:
                        log.info('Adding an implicit deny ALL because an allow'
                                 ' entry is present')
                        policy_files.append(deny_file(access))
                if deny_conf:
                    policy = deny_conf.find('policysetting')
                    data = policy.find('data')
                    for listelement in data.findall('listelement'):
                        adobject = listelement.find('adobject')
                        name = adobject.find('name').text
                        domain = adobject.find('domain').text
                        entries.append('-:%s%s%s:ALL' % (domain,
                                                         winbind_sep,
                                                         name))
                        if len(allow_listelements) > 0:
                            log.warn("Deny entry '%s' is meaningless with "
                                     "allow present" % entries[-1])
                if len(entries) == 0:
                    continue
                conf_id = select_next_conf(access)
                access_file = os.path.join(access, '%010d_gp.conf' % conf_id)
                policy_files.append(access_file)
                access_contents = '\n'.join(entries)
                # Each GPO applies only one set of access policies, so the
                # attribute does not need uniqueness.
                attribute = self.generate_attribute(gpo.name)
                # The value hash is generated from the access policy, ensuring
                # any changes to this GPO will cause the files to be rewritten.
                value_hash = self.generate_value_hash(access_contents)
                def applier_func(access, access_file, policy_files):
                    if not os.path.isdir(access):
                        os.mkdir(access, 0o644)
                    with NamedTemporaryFile(delete=False, dir=access) as f:
                        with open(f.name, 'w') as w:
                            w.write(intro)
                            w.write(access_contents)
                        os.chmod(f.name, 0o644)
                        os.rename(f.name, access_file)
                    return policy_files
                self.apply(gpo.name, attribute, value_hash, applier_func,
                           access, access_file, policy_files)
                # Cleanup any old entries that are no longer part of the policy
                self.clean(gpo.name, keep=[attribute])

    def rsop(self, gpo):
        output = {}
        if gpo.file_sys_path:
            allow = 'MACHINE/VGP/VTLA/VAS/HostAccessControl/Allow/manifest.xml'
            path = os.path.join(gpo.file_sys_path, allow)
            allow_conf = self.parse(path)
            deny = 'MACHINE/VGP/VTLA/VAS/HostAccessControl/Deny/manifest.xml'
            path = os.path.join(gpo.file_sys_path, deny)
            deny_conf = self.parse(path)
            entries = []
            winbind_sep = self.lp.get('winbind separator')
            if allow_conf:
                policy = allow_conf.find('policysetting')
                data = policy.find('data')
                allow_listelements = data.findall('listelement')
                for listelement in allow_listelements:
                    adobject = listelement.find('adobject')
                    name = adobject.find('name').text
                    domain = adobject.find('domain').text
                    if str(self) not in output.keys():
                        output[str(self)] = []
                    output[str(self)].append('+:%s%s%s:ALL' % (name,
                                                               winbind_sep,
                                                               domain))
                if len(allow_listelements) > 0:
                    output[str(self)].append('-:ALL:ALL')
            if deny_conf:
                policy = deny_conf.find('policysetting')
                data = policy.find('data')
                for listelement in data.findall('listelement'):
                    adobject = listelement.find('adobject')
                    name = adobject.find('name').text
                    domain = adobject.find('domain').text
                    if str(self) not in output.keys():
                        output[str(self)] = []
                    output[str(self)].append('-:%s%s%s:ALL' % (name,
                                                               winbind_sep,
                                                               domain))
        return output
