#pragma once

/***
  This file is part of systemd.

  Copyright 2014 Lennart Poettering

  systemd is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as published by
  the Free Software Foundation; either version 2.1 of the License, or
  (at your option) any later version.

  systemd is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License
  along with systemd; If not, see <http://www.gnu.org/licenses/>.
***/

#include "list.h"

typedef struct DnsScope DnsScope;

#include "resolved-dns-cache.h"
#include "resolved-dns-dnssec.h"
#include "resolved-dns-packet.h"
#include "resolved-dns-server.h"
#include "resolved-dns-zone.h"
#include "resolved-link.h"

typedef enum DnsScopeMatch {
        DNS_SCOPE_NO,
        DNS_SCOPE_MAYBE,
        DNS_SCOPE_YES,
        _DNS_SCOPE_MATCH_MAX,
        _DNS_SCOPE_INVALID = -1
} DnsScopeMatch;

struct DnsScope {
        Manager *manager;

        DnsProtocol protocol;
        int family;
        DnssecMode dnssec_mode;

        Link *link;

        DnsCache cache;
        DnsZone zone;

        OrderedHashmap *conflict_queue;
        sd_event_source *conflict_event_source;

        RateLimit ratelimit;

        usec_t resend_timeout;
        usec_t max_rtt;

        LIST_HEAD(DnsQueryCandidate, query_candidates);

        /* Note that we keep track of ongoing transactions in two
         * ways: once in a hashmap, indexed by the rr key, and once in
         * a linked list. We use the hashmap to quickly find
         * transactions we can reuse for a key. But note that there
         * might be multiple transactions for the same key (because
         * the zone probing can't reuse a transaction answered from
         * the zone or the cache), and the hashmap only tracks the
         * most recent entry. */
        Hashmap *transactions_by_key;
        LIST_HEAD(DnsTransaction, transactions);

        LIST_FIELDS(DnsScope, scopes);
};

int dns_scope_new(Manager *m, DnsScope **ret, Link *l, DnsProtocol p, int family);
DnsScope* dns_scope_free(DnsScope *s);

void dns_scope_packet_received(DnsScope *s, usec_t rtt);
void dns_scope_packet_lost(DnsScope *s, usec_t usec);

int dns_scope_emit_udp(DnsScope *s, int fd, DnsPacket *p);
int dns_scope_socket_tcp(DnsScope *s, int family, const union in_addr_union *address, DnsServer *server, uint16_t port);
int dns_scope_socket_udp(DnsScope *s, DnsServer *server, uint16_t port);

DnsScopeMatch dns_scope_good_domain(DnsScope *s, int ifindex, uint64_t flags, const char *domain);
bool dns_scope_good_key(DnsScope *s, const DnsResourceKey *key);

DnsServer *dns_scope_get_dns_server(DnsScope *s);
void dns_scope_next_dns_server(DnsScope *s);

int dns_scope_llmnr_membership(DnsScope *s, bool b);
int dns_scope_mdns_membership(DnsScope *s, bool b);

void dns_scope_process_query(DnsScope *s, DnsStream *stream, DnsPacket *p);

DnsTransaction *dns_scope_find_transaction(DnsScope *scope, DnsResourceKey *key, bool cache_ok);

int dns_scope_notify_conflict(DnsScope *scope, DnsResourceRecord *rr);
void dns_scope_check_conflicts(DnsScope *scope, DnsPacket *p);

void dns_scope_dump(DnsScope *s, FILE *f);

DnsSearchDomain *dns_scope_get_search_domains(DnsScope *s);

bool dns_scope_name_needs_search_domain(DnsScope *s, const char *name);

bool dns_scope_network_good(DnsScope *s);
