/*
 * Copyright (c) 2018-2020 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __ARM_FEATURE_SVE

#include <algorithm>

#include "arm_gemm.hpp"
#include "../../bfloat.hpp"
#include "../../asmlib.hpp"
#include "../../utils.hpp"

namespace arm_gemm {

void sve_hybrid_bf16fp32_dot_4VLx4(const bfloat16 *A, int lda, const bfloat16 *B, float *C, int ldc, int M, int N, int K, const float *bias, Activation act, bool accumulate) {
    const int K_stride = ((K + 1) / 2) * 2;
    const long loops_count = ((K + 8) / 16) - 1;
    K -= loops_count * 16;
    const long regs_count = (K / 8) - 1;
    K -= (regs_count + 1) * 8;
    const long leftovers = K;
    const long blocks_count = (K + 1) / 2;
    float nullbias[256];
    if (!accumulate && !bias) {
        memset(nullbias, 0, (4 * get_vector_length<float>() * sizeof(float)));
    }
    float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
    float maxval =   static_cast<float>(std::numeric_limits<float>::infinity());
    const float * const minptr = &minval;
    const float * const maxptr = &maxval;

    switch(act.type)
    {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            minval = 0.0f;
            break;
    }

    int rows_to_compute;

    for (int y=0; y<M; y+=rows_to_compute) {
        const bfloat16 * const a_ptr0_base = A + (y * lda);
        const unsigned long ldab = lda * sizeof(bfloat16);

        float *c_ptr0 = C + (y * ldc);

        rows_to_compute = M-y;
        if (rows_to_compute > 4) {
            if (rows_to_compute % 4) {
                rows_to_compute = 4 - 1;
            } else {
                rows_to_compute = 4;
            }
        }

        for (int x0=0; x0<N; x0+=(4 * get_vector_length<float>())) {
            const long width = std::min((unsigned long)N-x0, (4 * get_vector_length<float>()));
            long loops = loops_count;
            long regs = regs_count;
            long temp = 0;
            long blocks = blocks_count;
            const bfloat16 *a_ptr0 = a_ptr0_base;
            const bfloat16 *b_ptr0 = B + (K_stride * x0);
            const unsigned long ldcb = ldc * sizeof(float);
            const float *biasptr = bias ? bias+x0 : nullbias;

            switch(rows_to_compute) {
                case 1:
                    __asm __volatile (
                        "whilelt p6.h, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.h\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p2.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p3.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "ld1w z16.s, p0/z, [%[biasptr]]\n"
                        "ld1w z17.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[biasptr], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[biasptr], #3, MUL VL]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z16.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z17.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[c_ptr0], #3, MUL VL]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        ".inst 0x64604110 // bfdot z16.s, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64604131 // bfdot z17.s, z9.h, z0.h[0]\n"
                        "ld1rqh z4.h, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x64604152 // bfdot z18.s, z10.h, z0.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64604173 // bfdot z19.s, z11.h, z0.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64684190 // bfdot z16.s, z12.h, z0.h[1]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x646841b1 // bfdot z17.s, z13.h, z0.h[1]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x646841d2 // bfdot z18.s, z14.h, z0.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646841f3 // bfdot z19.s, z15.h, z0.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64704110 // bfdot z16.s, z8.h, z0.h[2]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64704131 // bfdot z17.s, z9.h, z0.h[2]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64704152 // bfdot z18.s, z10.h, z0.h[2]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        ".inst 0x64704173 // bfdot z19.s, z11.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x64784190 // bfdot z16.s, z12.h, z0.h[3]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x647841b1 // bfdot z17.s, z13.h, z0.h[3]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x647841d2 // bfdot z18.s, z14.h, z0.h[3]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x647841f3 // bfdot z19.s, z15.h, z0.h[3]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64644110 // bfdot z16.s, z8.h, z4.h[0]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x64644131 // bfdot z17.s, z9.h, z4.h[0]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x64644152 // bfdot z18.s, z10.h, z4.h[0]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x64644173 // bfdot z19.s, z11.h, z4.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x646c4190 // bfdot z16.s, z12.h, z4.h[1]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x646c41b1 // bfdot z17.s, z13.h, z4.h[1]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x646c41d2 // bfdot z18.s, z14.h, z4.h[1]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x646c41f3 // bfdot z19.s, z15.h, z4.h[1]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64744110 // bfdot z16.s, z8.h, z4.h[2]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64744131 // bfdot z17.s, z9.h, z4.h[2]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64744152 // bfdot z18.s, z10.h, z4.h[2]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64744173 // bfdot z19.s, z11.h, z4.h[2]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x647c4190 // bfdot z16.s, z12.h, z4.h[3]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x647c41b1 // bfdot z17.s, z13.h, z4.h[3]\n"
                        ".inst 0x647c41d2 // bfdot z18.s, z14.h, z4.h[3]\n"
                        ".inst 0x647c41f3 // bfdot z19.s, z15.h, z4.h[3]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        ".inst 0x64604110 // bfdot z16.s, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64604131 // bfdot z17.s, z9.h, z0.h[0]\n"
                        "ld1rqh z4.h, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x64604152 // bfdot z18.s, z10.h, z0.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64604173 // bfdot z19.s, z11.h, z0.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64684190 // bfdot z16.s, z12.h, z0.h[1]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x646841b1 // bfdot z17.s, z13.h, z0.h[1]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x646841d2 // bfdot z18.s, z14.h, z0.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646841f3 // bfdot z19.s, z15.h, z0.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64704110 // bfdot z16.s, z8.h, z0.h[2]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64704131 // bfdot z17.s, z9.h, z0.h[2]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64704152 // bfdot z18.s, z10.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x64704173 // bfdot z19.s, z11.h, z0.h[2]\n"
                        ".inst 0x64784190 // bfdot z16.s, z12.h, z0.h[3]\n"
                        ".inst 0x647841b1 // bfdot z17.s, z13.h, z0.h[3]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x647841d2 // bfdot z18.s, z14.h, z0.h[3]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x647841f3 // bfdot z19.s, z15.h, z0.h[3]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64644110 // bfdot z16.s, z8.h, z4.h[0]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x64644131 // bfdot z17.s, z9.h, z4.h[0]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x64644152 // bfdot z18.s, z10.h, z4.h[0]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x64644173 // bfdot z19.s, z11.h, z4.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x646c4190 // bfdot z16.s, z12.h, z4.h[1]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x646c41b1 // bfdot z17.s, z13.h, z4.h[1]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x646c41d2 // bfdot z18.s, z14.h, z4.h[1]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x646c41f3 // bfdot z19.s, z15.h, z4.h[1]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64744110 // bfdot z16.s, z8.h, z4.h[2]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64744131 // bfdot z17.s, z9.h, z4.h[2]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64744152 // bfdot z18.s, z10.h, z4.h[2]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64744173 // bfdot z19.s, z11.h, z4.h[2]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x647c4190 // bfdot z16.s, z12.h, z4.h[3]\n"
                        "ld1rqh z0.h, p6/z, [%[a_ptr0], #0x10]\n"
                        ".inst 0x647c41b1 // bfdot z17.s, z13.h, z4.h[3]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        ".inst 0x647c41d2 // bfdot z18.s, z14.h, z4.h[3]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        ".inst 0x647c41f3 // bfdot z19.s, z15.h, z4.h[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64604110 // bfdot z16.s, z8.h, z0.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64604131 // bfdot z17.s, z9.h, z0.h[0]\n"
                        ".inst 0x64604152 // bfdot z18.s, z10.h, z0.h[0]\n"
                        ".inst 0x64604173 // bfdot z19.s, z11.h, z0.h[0]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64684190 // bfdot z16.s, z12.h, z0.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x646841b1 // bfdot z17.s, z13.h, z0.h[1]\n"
                        ".inst 0x646841d2 // bfdot z18.s, z14.h, z0.h[1]\n"
                        ".inst 0x646841f3 // bfdot z19.s, z15.h, z0.h[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64704110 // bfdot z16.s, z8.h, z0.h[2]\n"
                        ".inst 0x64704131 // bfdot z17.s, z9.h, z0.h[2]\n"
                        ".inst 0x64704152 // bfdot z18.s, z10.h, z0.h[2]\n"
                        ".inst 0x64704173 // bfdot z19.s, z11.h, z0.h[2]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64784190 // bfdot z16.s, z12.h, z0.h[3]\n"
                        ".inst 0x647841b1 // bfdot z17.s, z13.h, z0.h[3]\n"
                        ".inst 0x647841d2 // bfdot z18.s, z14.h, z0.h[3]\n"
                        ".inst 0x647841f3 // bfdot z19.s, z15.h, z0.h[3]\n"
                        "b 5f\n"
                        "4:\n"
                        ".inst 0x64604110 // bfdot z16.s, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64604131 // bfdot z17.s, z9.h, z0.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64604152 // bfdot z18.s, z10.h, z0.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64604173 // bfdot z19.s, z11.h, z0.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64684190 // bfdot z16.s, z12.h, z0.h[1]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x646841b1 // bfdot z17.s, z13.h, z0.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646841d2 // bfdot z18.s, z14.h, z0.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x646841f3 // bfdot z19.s, z15.h, z0.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64704110 // bfdot z16.s, z8.h, z0.h[2]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64704131 // bfdot z17.s, z9.h, z0.h[2]\n"
                        "ld1rqh z4.h, p6/z, [%[a_ptr0]]\n"
                        ".inst 0x64704152 // bfdot z18.s, z10.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        ".inst 0x64704173 // bfdot z19.s, z11.h, z0.h[2]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        ".inst 0x64784190 // bfdot z16.s, z12.h, z0.h[3]\n"
                        ".inst 0x647841b1 // bfdot z17.s, z13.h, z0.h[3]\n"
                        ".inst 0x647841d2 // bfdot z18.s, z14.h, z0.h[3]\n"
                        ".inst 0x647841f3 // bfdot z19.s, z15.h, z0.h[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64644110 // bfdot z16.s, z8.h, z4.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64644131 // bfdot z17.s, z9.h, z4.h[0]\n"
                        ".inst 0x64644152 // bfdot z18.s, z10.h, z4.h[0]\n"
                        ".inst 0x64644173 // bfdot z19.s, z11.h, z4.h[0]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646c4190 // bfdot z16.s, z12.h, z4.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x646c41b1 // bfdot z17.s, z13.h, z4.h[1]\n"
                        ".inst 0x646c41d2 // bfdot z18.s, z14.h, z4.h[1]\n"
                        ".inst 0x646c41f3 // bfdot z19.s, z15.h, z4.h[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64744110 // bfdot z16.s, z8.h, z4.h[2]\n"
                        ".inst 0x64744131 // bfdot z17.s, z9.h, z4.h[2]\n"
                        ".inst 0x64744152 // bfdot z18.s, z10.h, z4.h[2]\n"
                        ".inst 0x64744173 // bfdot z19.s, z11.h, z4.h[2]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x647c4190 // bfdot z16.s, z12.h, z4.h[3]\n"
                        ".inst 0x647c41b1 // bfdot z17.s, z13.h, z4.h[3]\n"
                        ".inst 0x647c41d2 // bfdot z18.s, z14.h, z4.h[3]\n"
                        ".inst 0x647c41f3 // bfdot z19.s, z15.h, z4.h[3]\n"
                        "5:\n"
                        "ld1rw z14.s, p7/z, [%[minptr]]\n"
                        "ld1rw z15.s, p7/z, [%[maxptr]]\n"
                        "fmax z16.s, p7/m, z16.s, z14.s\n"
                        "fmax z17.s, p7/m, z17.s, z14.s\n"
                        "fmax z18.s, p7/m, z18.s, z14.s\n"
                        "fmax z19.s, p7/m, z19.s, z14.s\n"
                        "fmin z16.s, p7/m, z16.s, z15.s\n"
                        "fmin z17.s, p7/m, z17.s, z15.s\n"
                        "fmin z18.s, p7/m, z18.s, z15.s\n"
                        "fmin z19.s, p7/m, z19.s, z15.s\n"
                        "st1w z16.s, p0, [%[c_ptr0]]\n"
                        "st1w z17.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "st1w z18.s, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "st1w z19.s, p3, [%[c_ptr0], #3, MUL VL]\n"
                        "addvl %[c_ptr0], %[c_ptr0], #4\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                    );
                    break;
                case 2:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "c_ptr1 .req X1\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "whilelt p6.h, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.h\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p2.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p3.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "ld1w z16.s, p0/z, [%[biasptr]]\n"
                        "ld1w z17.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[biasptr], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[biasptr], #3, MUL VL]\n"
                        "mov z20.d, z16.d\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "mov z21.d, z17.d\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "mov z22.d, z18.d\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "mov z23.d, z19.d\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z16.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z17.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[c_ptr0], #3, MUL VL]\n"
                        "ld1w z20.s, p0/z, [c_ptr1]\n"
                        "ld1w z21.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1w z22.s, p2/z, [c_ptr1, #2, MUL VL]\n"
                        "ld1w z23.s, p3/z, [c_ptr1, #3, MUL VL]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        ".inst 0x64604110 // bfdot z16.s, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64614114 // bfdot z20.s, z8.h, z1.h[0]\n"
                        "ld1rqh z4.h, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x64604131 // bfdot z17.s, z9.h, z0.h[0]\n"
                        "ld1rqh z5.h, p7/z, [a_ptr1]\n"
                        ".inst 0x64614135 // bfdot z21.s, z9.h, z1.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64604152 // bfdot z18.s, z10.h, z0.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64614156 // bfdot z22.s, z10.h, z1.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64604173 // bfdot z19.s, z11.h, z0.h[0]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        ".inst 0x64614177 // bfdot z23.s, z11.h, z1.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64684190 // bfdot z16.s, z12.h, z0.h[1]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x64694194 // bfdot z20.s, z12.h, z1.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646841b1 // bfdot z17.s, z13.h, z0.h[1]\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        ".inst 0x646941b5 // bfdot z21.s, z13.h, z1.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x646841d2 // bfdot z18.s, z14.h, z0.h[1]\n"
                        ".inst 0x646941d6 // bfdot z22.s, z14.h, z1.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646841f3 // bfdot z19.s, z15.h, z0.h[1]\n"
                        ".inst 0x646941f7 // bfdot z23.s, z15.h, z1.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64704110 // bfdot z16.s, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x64714114 // bfdot z20.s, z8.h, z1.h[2]\n"
                        ".inst 0x64704131 // bfdot z17.s, z9.h, z0.h[2]\n"
                        ".inst 0x64714135 // bfdot z21.s, z9.h, z1.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64704152 // bfdot z18.s, z10.h, z0.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64714156 // bfdot z22.s, z10.h, z1.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64704173 // bfdot z19.s, z11.h, z0.h[2]\n"
                        ".inst 0x64714177 // bfdot z23.s, z11.h, z1.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64784190 // bfdot z16.s, z12.h, z0.h[3]\n"
                        ".inst 0x64794194 // bfdot z20.s, z12.h, z1.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x647841b1 // bfdot z17.s, z13.h, z0.h[3]\n"
                        ".inst 0x647941b5 // bfdot z21.s, z13.h, z1.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x647841d2 // bfdot z18.s, z14.h, z0.h[3]\n"
                        ".inst 0x647941d6 // bfdot z22.s, z14.h, z1.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x647841f3 // bfdot z19.s, z15.h, z0.h[3]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x647941f7 // bfdot z23.s, z15.h, z1.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64644110 // bfdot z16.s, z8.h, z4.h[0]\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1, #-0x10]\n"
                        ".inst 0x64654114 // bfdot z20.s, z8.h, z5.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64644131 // bfdot z17.s, z9.h, z4.h[0]\n"
                        ".inst 0x64654135 // bfdot z21.s, z9.h, z5.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64644152 // bfdot z18.s, z10.h, z4.h[0]\n"
                        ".inst 0x64654156 // bfdot z22.s, z10.h, z5.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64644173 // bfdot z19.s, z11.h, z4.h[0]\n"
                        ".inst 0x64654177 // bfdot z23.s, z11.h, z5.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x646c4190 // bfdot z16.s, z12.h, z4.h[1]\n"
                        ".inst 0x646d4194 // bfdot z20.s, z12.h, z5.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646c41b1 // bfdot z17.s, z13.h, z4.h[1]\n"
                        ".inst 0x646d41b5 // bfdot z21.s, z13.h, z5.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x646c41d2 // bfdot z18.s, z14.h, z4.h[1]\n"
                        ".inst 0x646d41d6 // bfdot z22.s, z14.h, z5.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646c41f3 // bfdot z19.s, z15.h, z4.h[1]\n"
                        ".inst 0x646d41f7 // bfdot z23.s, z15.h, z5.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64744110 // bfdot z16.s, z8.h, z4.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x64754114 // bfdot z20.s, z8.h, z5.h[2]\n"
                        ".inst 0x64744131 // bfdot z17.s, z9.h, z4.h[2]\n"
                        ".inst 0x64754135 // bfdot z21.s, z9.h, z5.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64744152 // bfdot z18.s, z10.h, z4.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64754156 // bfdot z22.s, z10.h, z5.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64744173 // bfdot z19.s, z11.h, z4.h[2]\n"
                        ".inst 0x64754177 // bfdot z23.s, z11.h, z5.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x647c4190 // bfdot z16.s, z12.h, z4.h[3]\n"
                        ".inst 0x647d4194 // bfdot z20.s, z12.h, z5.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x647c41b1 // bfdot z17.s, z13.h, z4.h[3]\n"
                        ".inst 0x647d41b5 // bfdot z21.s, z13.h, z5.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x647c41d2 // bfdot z18.s, z14.h, z4.h[3]\n"
                        ".inst 0x647d41d6 // bfdot z22.s, z14.h, z5.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x647c41f3 // bfdot z19.s, z15.h, z4.h[3]\n"
                        ".inst 0x647d41f7 // bfdot z23.s, z15.h, z5.h[3]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        ".inst 0x64604110 // bfdot z16.s, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64614114 // bfdot z20.s, z8.h, z1.h[0]\n"
                        "ld1rqh z4.h, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x64604131 // bfdot z17.s, z9.h, z0.h[0]\n"
                        "ld1rqh z5.h, p7/z, [a_ptr1]\n"
                        ".inst 0x64614135 // bfdot z21.s, z9.h, z1.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64604152 // bfdot z18.s, z10.h, z0.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64614156 // bfdot z22.s, z10.h, z1.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64604173 // bfdot z19.s, z11.h, z0.h[0]\n"
                        ".inst 0x64614177 // bfdot z23.s, z11.h, z1.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64684190 // bfdot z16.s, z12.h, z0.h[1]\n"
                        ".inst 0x64694194 // bfdot z20.s, z12.h, z1.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646841b1 // bfdot z17.s, z13.h, z0.h[1]\n"
                        ".inst 0x646941b5 // bfdot z21.s, z13.h, z1.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x646841d2 // bfdot z18.s, z14.h, z0.h[1]\n"
                        ".inst 0x646941d6 // bfdot z22.s, z14.h, z1.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646841f3 // bfdot z19.s, z15.h, z0.h[1]\n"
                        ".inst 0x646941f7 // bfdot z23.s, z15.h, z1.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64704110 // bfdot z16.s, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x64714114 // bfdot z20.s, z8.h, z1.h[2]\n"
                        ".inst 0x64704131 // bfdot z17.s, z9.h, z0.h[2]\n"
                        ".inst 0x64714135 // bfdot z21.s, z9.h, z1.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64704152 // bfdot z18.s, z10.h, z0.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64714156 // bfdot z22.s, z10.h, z1.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64704173 // bfdot z19.s, z11.h, z0.h[2]\n"
                        ".inst 0x64714177 // bfdot z23.s, z11.h, z1.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64784190 // bfdot z16.s, z12.h, z0.h[3]\n"
                        ".inst 0x64794194 // bfdot z20.s, z12.h, z1.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x647841b1 // bfdot z17.s, z13.h, z0.h[3]\n"
                        ".inst 0x647941b5 // bfdot z21.s, z13.h, z1.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x647841d2 // bfdot z18.s, z14.h, z0.h[3]\n"
                        ".inst 0x647941d6 // bfdot z22.s, z14.h, z1.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x647841f3 // bfdot z19.s, z15.h, z0.h[3]\n"
                        "ld1rqh z0.h, p6/z, [%[a_ptr0], #0x10]\n"
                        ".inst 0x647941f7 // bfdot z23.s, z15.h, z1.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64644110 // bfdot z16.s, z8.h, z4.h[0]\n"
                        "ld1rqh z1.h, p6/z, [a_ptr1, #0x10]\n"
                        ".inst 0x64654114 // bfdot z20.s, z8.h, z5.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64644131 // bfdot z17.s, z9.h, z4.h[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        ".inst 0x64654135 // bfdot z21.s, z9.h, z5.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64644152 // bfdot z18.s, z10.h, z4.h[0]\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        ".inst 0x64654156 // bfdot z22.s, z10.h, z5.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64644173 // bfdot z19.s, z11.h, z4.h[0]\n"
                        ".inst 0x64654177 // bfdot z23.s, z11.h, z5.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x646c4190 // bfdot z16.s, z12.h, z4.h[1]\n"
                        ".inst 0x646d4194 // bfdot z20.s, z12.h, z5.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646c41b1 // bfdot z17.s, z13.h, z4.h[1]\n"
                        ".inst 0x646d41b5 // bfdot z21.s, z13.h, z5.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x646c41d2 // bfdot z18.s, z14.h, z4.h[1]\n"
                        ".inst 0x646d41d6 // bfdot z22.s, z14.h, z5.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646c41f3 // bfdot z19.s, z15.h, z4.h[1]\n"
                        ".inst 0x646d41f7 // bfdot z23.s, z15.h, z5.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64744110 // bfdot z16.s, z8.h, z4.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        ".inst 0x64754114 // bfdot z20.s, z8.h, z5.h[2]\n"
                        ".inst 0x64744131 // bfdot z17.s, z9.h, z4.h[2]\n"
                        ".inst 0x64754135 // bfdot z21.s, z9.h, z5.h[2]\n"
                        ".inst 0x64744152 // bfdot z18.s, z10.h, z4.h[2]\n"
                        ".inst 0x64754156 // bfdot z22.s, z10.h, z5.h[2]\n"
                        ".inst 0x64744173 // bfdot z19.s, z11.h, z4.h[2]\n"
                        ".inst 0x64754177 // bfdot z23.s, z11.h, z5.h[2]\n"
                        ".inst 0x647c4190 // bfdot z16.s, z12.h, z4.h[3]\n"
                        ".inst 0x647d4194 // bfdot z20.s, z12.h, z5.h[3]\n"
                        ".inst 0x647c41b1 // bfdot z17.s, z13.h, z4.h[3]\n"
                        ".inst 0x647d41b5 // bfdot z21.s, z13.h, z5.h[3]\n"
                        ".inst 0x647c41d2 // bfdot z18.s, z14.h, z4.h[3]\n"
                        ".inst 0x647d41d6 // bfdot z22.s, z14.h, z5.h[3]\n"
                        ".inst 0x647c41f3 // bfdot z19.s, z15.h, z4.h[3]\n"
                        ".inst 0x647d41f7 // bfdot z23.s, z15.h, z5.h[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64604110 // bfdot z16.s, z8.h, z0.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64614114 // bfdot z20.s, z8.h, z1.h[0]\n"
                        ".inst 0x64604131 // bfdot z17.s, z9.h, z0.h[0]\n"
                        ".inst 0x64614135 // bfdot z21.s, z9.h, z1.h[0]\n"
                        ".inst 0x64604152 // bfdot z18.s, z10.h, z0.h[0]\n"
                        ".inst 0x64614156 // bfdot z22.s, z10.h, z1.h[0]\n"
                        ".inst 0x64604173 // bfdot z19.s, z11.h, z0.h[0]\n"
                        ".inst 0x64614177 // bfdot z23.s, z11.h, z1.h[0]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64684190 // bfdot z16.s, z12.h, z0.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64694194 // bfdot z20.s, z12.h, z1.h[1]\n"
                        ".inst 0x646841b1 // bfdot z17.s, z13.h, z0.h[1]\n"
                        ".inst 0x646941b5 // bfdot z21.s, z13.h, z1.h[1]\n"
                        ".inst 0x646841d2 // bfdot z18.s, z14.h, z0.h[1]\n"
                        ".inst 0x646941d6 // bfdot z22.s, z14.h, z1.h[1]\n"
                        ".inst 0x646841f3 // bfdot z19.s, z15.h, z0.h[1]\n"
                        ".inst 0x646941f7 // bfdot z23.s, z15.h, z1.h[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64704110 // bfdot z16.s, z8.h, z0.h[2]\n"
                        ".inst 0x64714114 // bfdot z20.s, z8.h, z1.h[2]\n"
                        ".inst 0x64704131 // bfdot z17.s, z9.h, z0.h[2]\n"
                        ".inst 0x64714135 // bfdot z21.s, z9.h, z1.h[2]\n"
                        ".inst 0x64704152 // bfdot z18.s, z10.h, z0.h[2]\n"
                        ".inst 0x64714156 // bfdot z22.s, z10.h, z1.h[2]\n"
                        ".inst 0x64704173 // bfdot z19.s, z11.h, z0.h[2]\n"
                        ".inst 0x64714177 // bfdot z23.s, z11.h, z1.h[2]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64784190 // bfdot z16.s, z12.h, z0.h[3]\n"
                        ".inst 0x64794194 // bfdot z20.s, z12.h, z1.h[3]\n"
                        ".inst 0x647841b1 // bfdot z17.s, z13.h, z0.h[3]\n"
                        ".inst 0x647941b5 // bfdot z21.s, z13.h, z1.h[3]\n"
                        ".inst 0x647841d2 // bfdot z18.s, z14.h, z0.h[3]\n"
                        ".inst 0x647941d6 // bfdot z22.s, z14.h, z1.h[3]\n"
                        ".inst 0x647841f3 // bfdot z19.s, z15.h, z0.h[3]\n"
                        ".inst 0x647941f7 // bfdot z23.s, z15.h, z1.h[3]\n"
                        "b 5f\n"
                        "4:\n"
                        ".inst 0x64604110 // bfdot z16.s, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64614114 // bfdot z20.s, z8.h, z1.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64604131 // bfdot z17.s, z9.h, z0.h[0]\n"
                        "ld1rqh z4.h, p6/z, [%[a_ptr0]]\n"
                        ".inst 0x64614135 // bfdot z21.s, z9.h, z1.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64604152 // bfdot z18.s, z10.h, z0.h[0]\n"
                        "ld1rqh z5.h, p6/z, [a_ptr1]\n"
                        ".inst 0x64614156 // bfdot z22.s, z10.h, z1.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64604173 // bfdot z19.s, z11.h, z0.h[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        ".inst 0x64614177 // bfdot z23.s, z11.h, z1.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64684190 // bfdot z16.s, z12.h, z0.h[1]\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        ".inst 0x64694194 // bfdot z20.s, z12.h, z1.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646841b1 // bfdot z17.s, z13.h, z0.h[1]\n"
                        ".inst 0x646941b5 // bfdot z21.s, z13.h, z1.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x646841d2 // bfdot z18.s, z14.h, z0.h[1]\n"
                        ".inst 0x646941d6 // bfdot z22.s, z14.h, z1.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646841f3 // bfdot z19.s, z15.h, z0.h[1]\n"
                        ".inst 0x646941f7 // bfdot z23.s, z15.h, z1.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64704110 // bfdot z16.s, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        ".inst 0x64714114 // bfdot z20.s, z8.h, z1.h[2]\n"
                        ".inst 0x64704131 // bfdot z17.s, z9.h, z0.h[2]\n"
                        ".inst 0x64714135 // bfdot z21.s, z9.h, z1.h[2]\n"
                        ".inst 0x64704152 // bfdot z18.s, z10.h, z0.h[2]\n"
                        ".inst 0x64714156 // bfdot z22.s, z10.h, z1.h[2]\n"
                        ".inst 0x64704173 // bfdot z19.s, z11.h, z0.h[2]\n"
                        ".inst 0x64714177 // bfdot z23.s, z11.h, z1.h[2]\n"
                        ".inst 0x64784190 // bfdot z16.s, z12.h, z0.h[3]\n"
                        ".inst 0x64794194 // bfdot z20.s, z12.h, z1.h[3]\n"
                        ".inst 0x647841b1 // bfdot z17.s, z13.h, z0.h[3]\n"
                        ".inst 0x647941b5 // bfdot z21.s, z13.h, z1.h[3]\n"
                        ".inst 0x647841d2 // bfdot z18.s, z14.h, z0.h[3]\n"
                        ".inst 0x647941d6 // bfdot z22.s, z14.h, z1.h[3]\n"
                        ".inst 0x647841f3 // bfdot z19.s, z15.h, z0.h[3]\n"
                        ".inst 0x647941f7 // bfdot z23.s, z15.h, z1.h[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64644110 // bfdot z16.s, z8.h, z4.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64654114 // bfdot z20.s, z8.h, z5.h[0]\n"
                        ".inst 0x64644131 // bfdot z17.s, z9.h, z4.h[0]\n"
                        ".inst 0x64654135 // bfdot z21.s, z9.h, z5.h[0]\n"
                        ".inst 0x64644152 // bfdot z18.s, z10.h, z4.h[0]\n"
                        ".inst 0x64654156 // bfdot z22.s, z10.h, z5.h[0]\n"
                        ".inst 0x64644173 // bfdot z19.s, z11.h, z4.h[0]\n"
                        ".inst 0x64654177 // bfdot z23.s, z11.h, z5.h[0]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646c4190 // bfdot z16.s, z12.h, z4.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x646d4194 // bfdot z20.s, z12.h, z5.h[1]\n"
                        ".inst 0x646c41b1 // bfdot z17.s, z13.h, z4.h[1]\n"
                        ".inst 0x646d41b5 // bfdot z21.s, z13.h, z5.h[1]\n"
                        ".inst 0x646c41d2 // bfdot z18.s, z14.h, z4.h[1]\n"
                        ".inst 0x646d41d6 // bfdot z22.s, z14.h, z5.h[1]\n"
                        ".inst 0x646c41f3 // bfdot z19.s, z15.h, z4.h[1]\n"
                        ".inst 0x646d41f7 // bfdot z23.s, z15.h, z5.h[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64744110 // bfdot z16.s, z8.h, z4.h[2]\n"
                        ".inst 0x64754114 // bfdot z20.s, z8.h, z5.h[2]\n"
                        ".inst 0x64744131 // bfdot z17.s, z9.h, z4.h[2]\n"
                        ".inst 0x64754135 // bfdot z21.s, z9.h, z5.h[2]\n"
                        ".inst 0x64744152 // bfdot z18.s, z10.h, z4.h[2]\n"
                        ".inst 0x64754156 // bfdot z22.s, z10.h, z5.h[2]\n"
                        ".inst 0x64744173 // bfdot z19.s, z11.h, z4.h[2]\n"
                        ".inst 0x64754177 // bfdot z23.s, z11.h, z5.h[2]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x647c4190 // bfdot z16.s, z12.h, z4.h[3]\n"
                        ".inst 0x647d4194 // bfdot z20.s, z12.h, z5.h[3]\n"
                        ".inst 0x647c41b1 // bfdot z17.s, z13.h, z4.h[3]\n"
                        ".inst 0x647d41b5 // bfdot z21.s, z13.h, z5.h[3]\n"
                        ".inst 0x647c41d2 // bfdot z18.s, z14.h, z4.h[3]\n"
                        ".inst 0x647d41d6 // bfdot z22.s, z14.h, z5.h[3]\n"
                        ".inst 0x647c41f3 // bfdot z19.s, z15.h, z4.h[3]\n"
                        ".inst 0x647d41f7 // bfdot z23.s, z15.h, z5.h[3]\n"
                        "5:\n"
                        "ld1rw z14.s, p7/z, [%[minptr]]\n"
                        "ld1rw z15.s, p7/z, [%[maxptr]]\n"
                        "fmax z16.s, p7/m, z16.s, z14.s\n"
                        "fmax z17.s, p7/m, z17.s, z14.s\n"
                        "fmax z18.s, p7/m, z18.s, z14.s\n"
                        "fmax z19.s, p7/m, z19.s, z14.s\n"
                        "fmin z16.s, p7/m, z16.s, z15.s\n"
                        "fmin z17.s, p7/m, z17.s, z15.s\n"
                        "fmin z18.s, p7/m, z18.s, z15.s\n"
                        "fmin z19.s, p7/m, z19.s, z15.s\n"
                        "st1w z16.s, p0, [%[c_ptr0]]\n"
                        "fmax z20.s, p7/m, z20.s, z14.s\n"
                        "fmax z21.s, p7/m, z21.s, z14.s\n"
                        "fmax z22.s, p7/m, z22.s, z14.s\n"
                        "st1w z17.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmax z23.s, p7/m, z23.s, z14.s\n"
                        "fmin z20.s, p7/m, z20.s, z15.s\n"
                        "fmin z21.s, p7/m, z21.s, z15.s\n"
                        "st1w z18.s, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "fmin z22.s, p7/m, z22.s, z15.s\n"
                        "fmin z23.s, p7/m, z23.s, z15.s\n"
                        "st1w z19.s, p3, [%[c_ptr0], #3, MUL VL]\n"
                        "addvl %[c_ptr0], %[c_ptr0], #4\n"
                        "st1w z20.s, p0, [c_ptr1]\n"
                        "st1w z21.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "st1w z22.s, p2, [c_ptr1, #2, MUL VL]\n"
                        "st1w z23.s, p3, [c_ptr1, #3, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq c_ptr1\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "cc", "memory"
                    );
                    break;
                case 3:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "c_ptr1 .req X2\n"
                        "c_ptr2 .req X3\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "whilelt p6.h, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.h\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p2.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p3.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "ld1w z16.s, p0/z, [%[biasptr]]\n"
                        "ld1w z17.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[biasptr], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[biasptr], #3, MUL VL]\n"
                        "mov z20.d, z16.d\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "mov z21.d, z17.d\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "mov z22.d, z18.d\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2]\n"
                        "mov z23.d, z19.d\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "mov z24.d, z16.d\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "mov z25.d, z17.d\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "mov z26.d, z18.d\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z27.d, z19.d\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z16.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z17.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[c_ptr0], #3, MUL VL]\n"
                        "ld1w z20.s, p0/z, [c_ptr1]\n"
                        "ld1w z21.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1w z22.s, p2/z, [c_ptr1, #2, MUL VL]\n"
                        "ld1w z23.s, p3/z, [c_ptr1, #3, MUL VL]\n"
                        "ld1w z24.s, p0/z, [c_ptr2]\n"
                        "ld1w z25.s, p1/z, [c_ptr2, #1, MUL VL]\n"
                        "ld1w z26.s, p2/z, [c_ptr2, #2, MUL VL]\n"
                        "ld1w z27.s, p3/z, [c_ptr2, #3, MUL VL]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        ".inst 0x64604110 // bfdot z16.s, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64614114 // bfdot z20.s, z8.h, z1.h[0]\n"
                        "ld1rqh z4.h, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x64624118 // bfdot z24.s, z8.h, z2.h[0]\n"
                        "ld1rqh z5.h, p7/z, [a_ptr1]\n"
                        ".inst 0x64604131 // bfdot z17.s, z9.h, z0.h[0]\n"
                        "ld1rqh z6.h, p7/z, [a_ptr2]\n"
                        ".inst 0x64614135 // bfdot z21.s, z9.h, z1.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64624139 // bfdot z25.s, z9.h, z2.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64604152 // bfdot z18.s, z10.h, z0.h[0]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        ".inst 0x64614156 // bfdot z22.s, z10.h, z1.h[0]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x6462415a // bfdot z26.s, z10.h, z2.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64604173 // bfdot z19.s, z11.h, z0.h[0]\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        ".inst 0x64614177 // bfdot z23.s, z11.h, z1.h[0]\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        ".inst 0x6462417b // bfdot z27.s, z11.h, z2.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64684190 // bfdot z16.s, z12.h, z0.h[1]\n"
                        ".inst 0x64694194 // bfdot z20.s, z12.h, z1.h[1]\n"
                        ".inst 0x646a4198 // bfdot z24.s, z12.h, z2.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646841b1 // bfdot z17.s, z13.h, z0.h[1]\n"
                        ".inst 0x646941b5 // bfdot z21.s, z13.h, z1.h[1]\n"
                        ".inst 0x646a41b9 // bfdot z25.s, z13.h, z2.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x646841d2 // bfdot z18.s, z14.h, z0.h[1]\n"
                        ".inst 0x646941d6 // bfdot z22.s, z14.h, z1.h[1]\n"
                        ".inst 0x646a41da // bfdot z26.s, z14.h, z2.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646841f3 // bfdot z19.s, z15.h, z0.h[1]\n"
                        ".inst 0x646941f7 // bfdot z23.s, z15.h, z1.h[1]\n"
                        ".inst 0x646a41fb // bfdot z27.s, z15.h, z2.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64704110 // bfdot z16.s, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x64714114 // bfdot z20.s, z8.h, z1.h[2]\n"
                        ".inst 0x64724118 // bfdot z24.s, z8.h, z2.h[2]\n"
                        ".inst 0x64704131 // bfdot z17.s, z9.h, z0.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64714135 // bfdot z21.s, z9.h, z1.h[2]\n"
                        ".inst 0x64724139 // bfdot z25.s, z9.h, z2.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64704152 // bfdot z18.s, z10.h, z0.h[2]\n"
                        ".inst 0x64714156 // bfdot z22.s, z10.h, z1.h[2]\n"
                        ".inst 0x6472415a // bfdot z26.s, z10.h, z2.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64704173 // bfdot z19.s, z11.h, z0.h[2]\n"
                        ".inst 0x64714177 // bfdot z23.s, z11.h, z1.h[2]\n"
                        ".inst 0x6472417b // bfdot z27.s, z11.h, z2.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64784190 // bfdot z16.s, z12.h, z0.h[3]\n"
                        ".inst 0x64794194 // bfdot z20.s, z12.h, z1.h[3]\n"
                        ".inst 0x647a4198 // bfdot z24.s, z12.h, z2.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x647841b1 // bfdot z17.s, z13.h, z0.h[3]\n"
                        ".inst 0x647941b5 // bfdot z21.s, z13.h, z1.h[3]\n"
                        ".inst 0x647a41b9 // bfdot z25.s, z13.h, z2.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x647841d2 // bfdot z18.s, z14.h, z0.h[3]\n"
                        ".inst 0x647941d6 // bfdot z22.s, z14.h, z1.h[3]\n"
                        ".inst 0x647a41da // bfdot z26.s, z14.h, z2.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x647841f3 // bfdot z19.s, z15.h, z0.h[3]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x647941f7 // bfdot z23.s, z15.h, z1.h[3]\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1, #-0x10]\n"
                        ".inst 0x647a41fb // bfdot z27.s, z15.h, z2.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64644110 // bfdot z16.s, z8.h, z4.h[0]\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2, #-0x10]\n"
                        ".inst 0x64654114 // bfdot z20.s, z8.h, z5.h[0]\n"
                        ".inst 0x64664118 // bfdot z24.s, z8.h, z6.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64644131 // bfdot z17.s, z9.h, z4.h[0]\n"
                        ".inst 0x64654135 // bfdot z21.s, z9.h, z5.h[0]\n"
                        ".inst 0x64664139 // bfdot z25.s, z9.h, z6.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64644152 // bfdot z18.s, z10.h, z4.h[0]\n"
                        ".inst 0x64654156 // bfdot z22.s, z10.h, z5.h[0]\n"
                        ".inst 0x6466415a // bfdot z26.s, z10.h, z6.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64644173 // bfdot z19.s, z11.h, z4.h[0]\n"
                        ".inst 0x64654177 // bfdot z23.s, z11.h, z5.h[0]\n"
                        ".inst 0x6466417b // bfdot z27.s, z11.h, z6.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x646c4190 // bfdot z16.s, z12.h, z4.h[1]\n"
                        ".inst 0x646d4194 // bfdot z20.s, z12.h, z5.h[1]\n"
                        ".inst 0x646e4198 // bfdot z24.s, z12.h, z6.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646c41b1 // bfdot z17.s, z13.h, z4.h[1]\n"
                        ".inst 0x646d41b5 // bfdot z21.s, z13.h, z5.h[1]\n"
                        ".inst 0x646e41b9 // bfdot z25.s, z13.h, z6.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x646c41d2 // bfdot z18.s, z14.h, z4.h[1]\n"
                        ".inst 0x646d41d6 // bfdot z22.s, z14.h, z5.h[1]\n"
                        ".inst 0x646e41da // bfdot z26.s, z14.h, z6.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646c41f3 // bfdot z19.s, z15.h, z4.h[1]\n"
                        ".inst 0x646d41f7 // bfdot z23.s, z15.h, z5.h[1]\n"
                        ".inst 0x646e41fb // bfdot z27.s, z15.h, z6.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64744110 // bfdot z16.s, z8.h, z4.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x64754114 // bfdot z20.s, z8.h, z5.h[2]\n"
                        ".inst 0x64764118 // bfdot z24.s, z8.h, z6.h[2]\n"
                        ".inst 0x64744131 // bfdot z17.s, z9.h, z4.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64754135 // bfdot z21.s, z9.h, z5.h[2]\n"
                        ".inst 0x64764139 // bfdot z25.s, z9.h, z6.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64744152 // bfdot z18.s, z10.h, z4.h[2]\n"
                        ".inst 0x64754156 // bfdot z22.s, z10.h, z5.h[2]\n"
                        ".inst 0x6476415a // bfdot z26.s, z10.h, z6.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64744173 // bfdot z19.s, z11.h, z4.h[2]\n"
                        ".inst 0x64754177 // bfdot z23.s, z11.h, z5.h[2]\n"
                        ".inst 0x6476417b // bfdot z27.s, z11.h, z6.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x647c4190 // bfdot z16.s, z12.h, z4.h[3]\n"
                        ".inst 0x647d4194 // bfdot z20.s, z12.h, z5.h[3]\n"
                        ".inst 0x647e4198 // bfdot z24.s, z12.h, z6.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x647c41b1 // bfdot z17.s, z13.h, z4.h[3]\n"
                        ".inst 0x647d41b5 // bfdot z21.s, z13.h, z5.h[3]\n"
                        ".inst 0x647e41b9 // bfdot z25.s, z13.h, z6.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x647c41d2 // bfdot z18.s, z14.h, z4.h[3]\n"
                        ".inst 0x647d41d6 // bfdot z22.s, z14.h, z5.h[3]\n"
                        ".inst 0x647e41da // bfdot z26.s, z14.h, z6.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x647c41f3 // bfdot z19.s, z15.h, z4.h[3]\n"
                        ".inst 0x647d41f7 // bfdot z23.s, z15.h, z5.h[3]\n"
                        ".inst 0x647e41fb // bfdot z27.s, z15.h, z6.h[3]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        ".inst 0x64604110 // bfdot z16.s, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64614114 // bfdot z20.s, z8.h, z1.h[0]\n"
                        "ld1rqh z4.h, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x64624118 // bfdot z24.s, z8.h, z2.h[0]\n"
                        "ld1rqh z5.h, p7/z, [a_ptr1]\n"
                        ".inst 0x64604131 // bfdot z17.s, z9.h, z0.h[0]\n"
                        "ld1rqh z6.h, p7/z, [a_ptr2]\n"
                        ".inst 0x64614135 // bfdot z21.s, z9.h, z1.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64624139 // bfdot z25.s, z9.h, z2.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64604152 // bfdot z18.s, z10.h, z0.h[0]\n"
                        ".inst 0x64614156 // bfdot z22.s, z10.h, z1.h[0]\n"
                        ".inst 0x6462415a // bfdot z26.s, z10.h, z2.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64604173 // bfdot z19.s, z11.h, z0.h[0]\n"
                        ".inst 0x64614177 // bfdot z23.s, z11.h, z1.h[0]\n"
                        ".inst 0x6462417b // bfdot z27.s, z11.h, z2.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64684190 // bfdot z16.s, z12.h, z0.h[1]\n"
                        ".inst 0x64694194 // bfdot z20.s, z12.h, z1.h[1]\n"
                        ".inst 0x646a4198 // bfdot z24.s, z12.h, z2.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646841b1 // bfdot z17.s, z13.h, z0.h[1]\n"
                        ".inst 0x646941b5 // bfdot z21.s, z13.h, z1.h[1]\n"
                        ".inst 0x646a41b9 // bfdot z25.s, z13.h, z2.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x646841d2 // bfdot z18.s, z14.h, z0.h[1]\n"
                        ".inst 0x646941d6 // bfdot z22.s, z14.h, z1.h[1]\n"
                        ".inst 0x646a41da // bfdot z26.s, z14.h, z2.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646841f3 // bfdot z19.s, z15.h, z0.h[1]\n"
                        ".inst 0x646941f7 // bfdot z23.s, z15.h, z1.h[1]\n"
                        ".inst 0x646a41fb // bfdot z27.s, z15.h, z2.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64704110 // bfdot z16.s, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x64714114 // bfdot z20.s, z8.h, z1.h[2]\n"
                        ".inst 0x64724118 // bfdot z24.s, z8.h, z2.h[2]\n"
                        ".inst 0x64704131 // bfdot z17.s, z9.h, z0.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64714135 // bfdot z21.s, z9.h, z1.h[2]\n"
                        ".inst 0x64724139 // bfdot z25.s, z9.h, z2.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64704152 // bfdot z18.s, z10.h, z0.h[2]\n"
                        ".inst 0x64714156 // bfdot z22.s, z10.h, z1.h[2]\n"
                        ".inst 0x6472415a // bfdot z26.s, z10.h, z2.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64704173 // bfdot z19.s, z11.h, z0.h[2]\n"
                        ".inst 0x64714177 // bfdot z23.s, z11.h, z1.h[2]\n"
                        ".inst 0x6472417b // bfdot z27.s, z11.h, z2.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64784190 // bfdot z16.s, z12.h, z0.h[3]\n"
                        ".inst 0x64794194 // bfdot z20.s, z12.h, z1.h[3]\n"
                        ".inst 0x647a4198 // bfdot z24.s, z12.h, z2.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x647841b1 // bfdot z17.s, z13.h, z0.h[3]\n"
                        ".inst 0x647941b5 // bfdot z21.s, z13.h, z1.h[3]\n"
                        ".inst 0x647a41b9 // bfdot z25.s, z13.h, z2.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x647841d2 // bfdot z18.s, z14.h, z0.h[3]\n"
                        ".inst 0x647941d6 // bfdot z22.s, z14.h, z1.h[3]\n"
                        ".inst 0x647a41da // bfdot z26.s, z14.h, z2.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x647841f3 // bfdot z19.s, z15.h, z0.h[3]\n"
                        "ld1rqh z0.h, p6/z, [%[a_ptr0], #0x10]\n"
                        ".inst 0x647941f7 // bfdot z23.s, z15.h, z1.h[3]\n"
                        "ld1rqh z1.h, p6/z, [a_ptr1, #0x10]\n"
                        ".inst 0x647a41fb // bfdot z27.s, z15.h, z2.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64644110 // bfdot z16.s, z8.h, z4.h[0]\n"
                        "ld1rqh z2.h, p6/z, [a_ptr2, #0x10]\n"
                        ".inst 0x64654114 // bfdot z20.s, z8.h, z5.h[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        ".inst 0x64664118 // bfdot z24.s, z8.h, z6.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64644131 // bfdot z17.s, z9.h, z4.h[0]\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        ".inst 0x64654135 // bfdot z21.s, z9.h, z5.h[0]\n"
                        "addvl a_ptr2, a_ptr2, #2\n"
                        ".inst 0x64664139 // bfdot z25.s, z9.h, z6.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64644152 // bfdot z18.s, z10.h, z4.h[0]\n"
                        ".inst 0x64654156 // bfdot z22.s, z10.h, z5.h[0]\n"
                        ".inst 0x6466415a // bfdot z26.s, z10.h, z6.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64644173 // bfdot z19.s, z11.h, z4.h[0]\n"
                        ".inst 0x64654177 // bfdot z23.s, z11.h, z5.h[0]\n"
                        ".inst 0x6466417b // bfdot z27.s, z11.h, z6.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x646c4190 // bfdot z16.s, z12.h, z4.h[1]\n"
                        ".inst 0x646d4194 // bfdot z20.s, z12.h, z5.h[1]\n"
                        ".inst 0x646e4198 // bfdot z24.s, z12.h, z6.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646c41b1 // bfdot z17.s, z13.h, z4.h[1]\n"
                        ".inst 0x646d41b5 // bfdot z21.s, z13.h, z5.h[1]\n"
                        ".inst 0x646e41b9 // bfdot z25.s, z13.h, z6.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x646c41d2 // bfdot z18.s, z14.h, z4.h[1]\n"
                        ".inst 0x646d41d6 // bfdot z22.s, z14.h, z5.h[1]\n"
                        ".inst 0x646e41da // bfdot z26.s, z14.h, z6.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646c41f3 // bfdot z19.s, z15.h, z4.h[1]\n"
                        ".inst 0x646d41f7 // bfdot z23.s, z15.h, z5.h[1]\n"
                        ".inst 0x646e41fb // bfdot z27.s, z15.h, z6.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64744110 // bfdot z16.s, z8.h, z4.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        ".inst 0x64754114 // bfdot z20.s, z8.h, z5.h[2]\n"
                        ".inst 0x64764118 // bfdot z24.s, z8.h, z6.h[2]\n"
                        ".inst 0x64744131 // bfdot z17.s, z9.h, z4.h[2]\n"
                        ".inst 0x64754135 // bfdot z21.s, z9.h, z5.h[2]\n"
                        ".inst 0x64764139 // bfdot z25.s, z9.h, z6.h[2]\n"
                        ".inst 0x64744152 // bfdot z18.s, z10.h, z4.h[2]\n"
                        ".inst 0x64754156 // bfdot z22.s, z10.h, z5.h[2]\n"
                        ".inst 0x6476415a // bfdot z26.s, z10.h, z6.h[2]\n"
                        ".inst 0x64744173 // bfdot z19.s, z11.h, z4.h[2]\n"
                        ".inst 0x64754177 // bfdot z23.s, z11.h, z5.h[2]\n"
                        ".inst 0x6476417b // bfdot z27.s, z11.h, z6.h[2]\n"
                        ".inst 0x647c4190 // bfdot z16.s, z12.h, z4.h[3]\n"
                        ".inst 0x647d4194 // bfdot z20.s, z12.h, z5.h[3]\n"
                        ".inst 0x647e4198 // bfdot z24.s, z12.h, z6.h[3]\n"
                        ".inst 0x647c41b1 // bfdot z17.s, z13.h, z4.h[3]\n"
                        ".inst 0x647d41b5 // bfdot z21.s, z13.h, z5.h[3]\n"
                        ".inst 0x647e41b9 // bfdot z25.s, z13.h, z6.h[3]\n"
                        ".inst 0x647c41d2 // bfdot z18.s, z14.h, z4.h[3]\n"
                        ".inst 0x647d41d6 // bfdot z22.s, z14.h, z5.h[3]\n"
                        ".inst 0x647e41da // bfdot z26.s, z14.h, z6.h[3]\n"
                        ".inst 0x647c41f3 // bfdot z19.s, z15.h, z4.h[3]\n"
                        ".inst 0x647d41f7 // bfdot z23.s, z15.h, z5.h[3]\n"
                        ".inst 0x647e41fb // bfdot z27.s, z15.h, z6.h[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64604110 // bfdot z16.s, z8.h, z0.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64614114 // bfdot z20.s, z8.h, z1.h[0]\n"
                        ".inst 0x64624118 // bfdot z24.s, z8.h, z2.h[0]\n"
                        ".inst 0x64604131 // bfdot z17.s, z9.h, z0.h[0]\n"
                        ".inst 0x64614135 // bfdot z21.s, z9.h, z1.h[0]\n"
                        ".inst 0x64624139 // bfdot z25.s, z9.h, z2.h[0]\n"
                        ".inst 0x64604152 // bfdot z18.s, z10.h, z0.h[0]\n"
                        ".inst 0x64614156 // bfdot z22.s, z10.h, z1.h[0]\n"
                        ".inst 0x6462415a // bfdot z26.s, z10.h, z2.h[0]\n"
                        ".inst 0x64604173 // bfdot z19.s, z11.h, z0.h[0]\n"
                        ".inst 0x64614177 // bfdot z23.s, z11.h, z1.h[0]\n"
                        ".inst 0x6462417b // bfdot z27.s, z11.h, z2.h[0]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64684190 // bfdot z16.s, z12.h, z0.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64694194 // bfdot z20.s, z12.h, z1.h[1]\n"
                        ".inst 0x646a4198 // bfdot z24.s, z12.h, z2.h[1]\n"
                        ".inst 0x646841b1 // bfdot z17.s, z13.h, z0.h[1]\n"
                        ".inst 0x646941b5 // bfdot z21.s, z13.h, z1.h[1]\n"
                        ".inst 0x646a41b9 // bfdot z25.s, z13.h, z2.h[1]\n"
                        ".inst 0x646841d2 // bfdot z18.s, z14.h, z0.h[1]\n"
                        ".inst 0x646941d6 // bfdot z22.s, z14.h, z1.h[1]\n"
                        ".inst 0x646a41da // bfdot z26.s, z14.h, z2.h[1]\n"
                        ".inst 0x646841f3 // bfdot z19.s, z15.h, z0.h[1]\n"
                        ".inst 0x646941f7 // bfdot z23.s, z15.h, z1.h[1]\n"
                        ".inst 0x646a41fb // bfdot z27.s, z15.h, z2.h[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64704110 // bfdot z16.s, z8.h, z0.h[2]\n"
                        ".inst 0x64714114 // bfdot z20.s, z8.h, z1.h[2]\n"
                        ".inst 0x64724118 // bfdot z24.s, z8.h, z2.h[2]\n"
                        ".inst 0x64704131 // bfdot z17.s, z9.h, z0.h[2]\n"
                        ".inst 0x64714135 // bfdot z21.s, z9.h, z1.h[2]\n"
                        ".inst 0x64724139 // bfdot z25.s, z9.h, z2.h[2]\n"
                        ".inst 0x64704152 // bfdot z18.s, z10.h, z0.h[2]\n"
                        ".inst 0x64714156 // bfdot z22.s, z10.h, z1.h[2]\n"
                        ".inst 0x6472415a // bfdot z26.s, z10.h, z2.h[2]\n"
                        ".inst 0x64704173 // bfdot z19.s, z11.h, z0.h[2]\n"
                        ".inst 0x64714177 // bfdot z23.s, z11.h, z1.h[2]\n"
                        ".inst 0x6472417b // bfdot z27.s, z11.h, z2.h[2]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64784190 // bfdot z16.s, z12.h, z0.h[3]\n"
                        ".inst 0x64794194 // bfdot z20.s, z12.h, z1.h[3]\n"
                        ".inst 0x647a4198 // bfdot z24.s, z12.h, z2.h[3]\n"
                        ".inst 0x647841b1 // bfdot z17.s, z13.h, z0.h[3]\n"
                        ".inst 0x647941b5 // bfdot z21.s, z13.h, z1.h[3]\n"
                        ".inst 0x647a41b9 // bfdot z25.s, z13.h, z2.h[3]\n"
                        ".inst 0x647841d2 // bfdot z18.s, z14.h, z0.h[3]\n"
                        ".inst 0x647941d6 // bfdot z22.s, z14.h, z1.h[3]\n"
                        ".inst 0x647a41da // bfdot z26.s, z14.h, z2.h[3]\n"
                        ".inst 0x647841f3 // bfdot z19.s, z15.h, z0.h[3]\n"
                        ".inst 0x647941f7 // bfdot z23.s, z15.h, z1.h[3]\n"
                        ".inst 0x647a41fb // bfdot z27.s, z15.h, z2.h[3]\n"
                        "b 5f\n"
                        "4:\n"
                        ".inst 0x64604110 // bfdot z16.s, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64614114 // bfdot z20.s, z8.h, z1.h[0]\n"
                        "ld1rqh z4.h, p6/z, [%[a_ptr0]]\n"
                        ".inst 0x64624118 // bfdot z24.s, z8.h, z2.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64604131 // bfdot z17.s, z9.h, z0.h[0]\n"
                        "ld1rqh z5.h, p6/z, [a_ptr1]\n"
                        ".inst 0x64614135 // bfdot z21.s, z9.h, z1.h[0]\n"
                        "ld1rqh z6.h, p6/z, [a_ptr2]\n"
                        ".inst 0x64624139 // bfdot z25.s, z9.h, z2.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64604152 // bfdot z18.s, z10.h, z0.h[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        ".inst 0x64614156 // bfdot z22.s, z10.h, z1.h[0]\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        ".inst 0x6462415a // bfdot z26.s, z10.h, z2.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64604173 // bfdot z19.s, z11.h, z0.h[0]\n"
                        "addvl a_ptr2, a_ptr2, #1\n"
                        ".inst 0x64614177 // bfdot z23.s, z11.h, z1.h[0]\n"
                        ".inst 0x6462417b // bfdot z27.s, z11.h, z2.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64684190 // bfdot z16.s, z12.h, z0.h[1]\n"
                        ".inst 0x64694194 // bfdot z20.s, z12.h, z1.h[1]\n"
                        ".inst 0x646a4198 // bfdot z24.s, z12.h, z2.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646841b1 // bfdot z17.s, z13.h, z0.h[1]\n"
                        ".inst 0x646941b5 // bfdot z21.s, z13.h, z1.h[1]\n"
                        ".inst 0x646a41b9 // bfdot z25.s, z13.h, z2.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x646841d2 // bfdot z18.s, z14.h, z0.h[1]\n"
                        ".inst 0x646941d6 // bfdot z22.s, z14.h, z1.h[1]\n"
                        ".inst 0x646a41da // bfdot z26.s, z14.h, z2.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646841f3 // bfdot z19.s, z15.h, z0.h[1]\n"
                        ".inst 0x646941f7 // bfdot z23.s, z15.h, z1.h[1]\n"
                        ".inst 0x646a41fb // bfdot z27.s, z15.h, z2.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64704110 // bfdot z16.s, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        ".inst 0x64714114 // bfdot z20.s, z8.h, z1.h[2]\n"
                        ".inst 0x64724118 // bfdot z24.s, z8.h, z2.h[2]\n"
                        ".inst 0x64704131 // bfdot z17.s, z9.h, z0.h[2]\n"
                        ".inst 0x64714135 // bfdot z21.s, z9.h, z1.h[2]\n"
                        ".inst 0x64724139 // bfdot z25.s, z9.h, z2.h[2]\n"
                        ".inst 0x64704152 // bfdot z18.s, z10.h, z0.h[2]\n"
                        ".inst 0x64714156 // bfdot z22.s, z10.h, z1.h[2]\n"
                        ".inst 0x6472415a // bfdot z26.s, z10.h, z2.h[2]\n"
                        ".inst 0x64704173 // bfdot z19.s, z11.h, z0.h[2]\n"
                        ".inst 0x64714177 // bfdot z23.s, z11.h, z1.h[2]\n"
                        ".inst 0x6472417b // bfdot z27.s, z11.h, z2.h[2]\n"
                        ".inst 0x64784190 // bfdot z16.s, z12.h, z0.h[3]\n"
                        ".inst 0x64794194 // bfdot z20.s, z12.h, z1.h[3]\n"
                        ".inst 0x647a4198 // bfdot z24.s, z12.h, z2.h[3]\n"
                        ".inst 0x647841b1 // bfdot z17.s, z13.h, z0.h[3]\n"
                        ".inst 0x647941b5 // bfdot z21.s, z13.h, z1.h[3]\n"
                        ".inst 0x647a41b9 // bfdot z25.s, z13.h, z2.h[3]\n"
                        ".inst 0x647841d2 // bfdot z18.s, z14.h, z0.h[3]\n"
                        ".inst 0x647941d6 // bfdot z22.s, z14.h, z1.h[3]\n"
                        ".inst 0x647a41da // bfdot z26.s, z14.h, z2.h[3]\n"
                        ".inst 0x647841f3 // bfdot z19.s, z15.h, z0.h[3]\n"
                        ".inst 0x647941f7 // bfdot z23.s, z15.h, z1.h[3]\n"
                        ".inst 0x647a41fb // bfdot z27.s, z15.h, z2.h[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64644110 // bfdot z16.s, z8.h, z4.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64654114 // bfdot z20.s, z8.h, z5.h[0]\n"
                        ".inst 0x64664118 // bfdot z24.s, z8.h, z6.h[0]\n"
                        ".inst 0x64644131 // bfdot z17.s, z9.h, z4.h[0]\n"
                        ".inst 0x64654135 // bfdot z21.s, z9.h, z5.h[0]\n"
                        ".inst 0x64664139 // bfdot z25.s, z9.h, z6.h[0]\n"
                        ".inst 0x64644152 // bfdot z18.s, z10.h, z4.h[0]\n"
                        ".inst 0x64654156 // bfdot z22.s, z10.h, z5.h[0]\n"
                        ".inst 0x6466415a // bfdot z26.s, z10.h, z6.h[0]\n"
                        ".inst 0x64644173 // bfdot z19.s, z11.h, z4.h[0]\n"
                        ".inst 0x64654177 // bfdot z23.s, z11.h, z5.h[0]\n"
                        ".inst 0x6466417b // bfdot z27.s, z11.h, z6.h[0]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646c4190 // bfdot z16.s, z12.h, z4.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x646d4194 // bfdot z20.s, z12.h, z5.h[1]\n"
                        ".inst 0x646e4198 // bfdot z24.s, z12.h, z6.h[1]\n"
                        ".inst 0x646c41b1 // bfdot z17.s, z13.h, z4.h[1]\n"
                        ".inst 0x646d41b5 // bfdot z21.s, z13.h, z5.h[1]\n"
                        ".inst 0x646e41b9 // bfdot z25.s, z13.h, z6.h[1]\n"
                        ".inst 0x646c41d2 // bfdot z18.s, z14.h, z4.h[1]\n"
                        ".inst 0x646d41d6 // bfdot z22.s, z14.h, z5.h[1]\n"
                        ".inst 0x646e41da // bfdot z26.s, z14.h, z6.h[1]\n"
                        ".inst 0x646c41f3 // bfdot z19.s, z15.h, z4.h[1]\n"
                        ".inst 0x646d41f7 // bfdot z23.s, z15.h, z5.h[1]\n"
                        ".inst 0x646e41fb // bfdot z27.s, z15.h, z6.h[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64744110 // bfdot z16.s, z8.h, z4.h[2]\n"
                        ".inst 0x64754114 // bfdot z20.s, z8.h, z5.h[2]\n"
                        ".inst 0x64764118 // bfdot z24.s, z8.h, z6.h[2]\n"
                        ".inst 0x64744131 // bfdot z17.s, z9.h, z4.h[2]\n"
                        ".inst 0x64754135 // bfdot z21.s, z9.h, z5.h[2]\n"
                        ".inst 0x64764139 // bfdot z25.s, z9.h, z6.h[2]\n"
                        ".inst 0x64744152 // bfdot z18.s, z10.h, z4.h[2]\n"
                        ".inst 0x64754156 // bfdot z22.s, z10.h, z5.h[2]\n"
                        ".inst 0x6476415a // bfdot z26.s, z10.h, z6.h[2]\n"
                        ".inst 0x64744173 // bfdot z19.s, z11.h, z4.h[2]\n"
                        ".inst 0x64754177 // bfdot z23.s, z11.h, z5.h[2]\n"
                        ".inst 0x6476417b // bfdot z27.s, z11.h, z6.h[2]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x647c4190 // bfdot z16.s, z12.h, z4.h[3]\n"
                        ".inst 0x647d4194 // bfdot z20.s, z12.h, z5.h[3]\n"
                        ".inst 0x647e4198 // bfdot z24.s, z12.h, z6.h[3]\n"
                        ".inst 0x647c41b1 // bfdot z17.s, z13.h, z4.h[3]\n"
                        ".inst 0x647d41b5 // bfdot z21.s, z13.h, z5.h[3]\n"
                        ".inst 0x647e41b9 // bfdot z25.s, z13.h, z6.h[3]\n"
                        ".inst 0x647c41d2 // bfdot z18.s, z14.h, z4.h[3]\n"
                        ".inst 0x647d41d6 // bfdot z22.s, z14.h, z5.h[3]\n"
                        ".inst 0x647e41da // bfdot z26.s, z14.h, z6.h[3]\n"
                        ".inst 0x647c41f3 // bfdot z19.s, z15.h, z4.h[3]\n"
                        ".inst 0x647d41f7 // bfdot z23.s, z15.h, z5.h[3]\n"
                        ".inst 0x647e41fb // bfdot z27.s, z15.h, z6.h[3]\n"
                        "5:\n"
                        "ld1rw z14.s, p7/z, [%[minptr]]\n"
                        "ld1rw z15.s, p7/z, [%[maxptr]]\n"
                        "fmax z16.s, p7/m, z16.s, z14.s\n"
                        "fmax z17.s, p7/m, z17.s, z14.s\n"
                        "fmax z18.s, p7/m, z18.s, z14.s\n"
                        "fmax z19.s, p7/m, z19.s, z14.s\n"
                        "fmin z16.s, p7/m, z16.s, z15.s\n"
                        "fmin z17.s, p7/m, z17.s, z15.s\n"
                        "fmin z18.s, p7/m, z18.s, z15.s\n"
                        "fmin z19.s, p7/m, z19.s, z15.s\n"
                        "st1w z16.s, p0, [%[c_ptr0]]\n"
                        "fmax z20.s, p7/m, z20.s, z14.s\n"
                        "fmax z21.s, p7/m, z21.s, z14.s\n"
                        "fmax z22.s, p7/m, z22.s, z14.s\n"
                        "st1w z17.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmax z23.s, p7/m, z23.s, z14.s\n"
                        "fmin z20.s, p7/m, z20.s, z15.s\n"
                        "fmin z21.s, p7/m, z21.s, z15.s\n"
                        "st1w z18.s, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "fmin z22.s, p7/m, z22.s, z15.s\n"
                        "fmin z23.s, p7/m, z23.s, z15.s\n"
                        "fmax z24.s, p7/m, z24.s, z14.s\n"
                        "st1w z19.s, p3, [%[c_ptr0], #3, MUL VL]\n"
                        "fmax z25.s, p7/m, z25.s, z14.s\n"
                        "addvl %[c_ptr0], %[c_ptr0], #4\n"
                        "fmax z26.s, p7/m, z26.s, z14.s\n"
                        "st1w z20.s, p0, [c_ptr1]\n"
                        "fmin z24.s, p7/m, z24.s, z15.s\n"
                        "fmin z25.s, p7/m, z25.s, z15.s\n"
                        "fmax z27.s, p7/m, z27.s, z14.s\n"
                        "st1w z21.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "fmin z26.s, p7/m, z26.s, z15.s\n"
                        "fmin z27.s, p7/m, z27.s, z15.s\n"
                        "st1w z22.s, p2, [c_ptr1, #2, MUL VL]\n"
                        "st1w z23.s, p3, [c_ptr1, #3, MUL VL]\n"
                        "st1w z24.s, p0, [c_ptr2]\n"
                        "st1w z25.s, p1, [c_ptr2, #1, MUL VL]\n"
                        "st1w z26.s, p2, [c_ptr2, #2, MUL VL]\n"
                        "st1w z27.s, p3, [c_ptr2, #3, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "x2", "x3", "cc", "memory"
                    );
                    break;
                default:
                case 4:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "a_ptr3 .req X2\n"
                        "c_ptr1 .req X3\n"
                        "c_ptr2 .req X4\n"
                        "c_ptr3 .req X5\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "add a_ptr3, a_ptr2, %[lda]\n"
                        "add c_ptr3, c_ptr2, %[ldc]\n"
                        "whilelt p6.h, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.h\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p2.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p3.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "ld1w z16.s, p0/z, [%[biasptr]]\n"
                        "ld1w z17.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[biasptr], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[biasptr], #3, MUL VL]\n"
                        "mov z20.d, z16.d\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "mov z21.d, z17.d\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "mov z22.d, z18.d\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2]\n"
                        "mov z23.d, z19.d\n"
                        "ld1rqh z3.h, p7/z, [a_ptr3]\n"
                        "mov z24.d, z16.d\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "mov z25.d, z17.d\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "mov z26.d, z18.d\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "mov z27.d, z19.d\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z28.d, z16.d\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "mov z29.d, z17.d\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "mov z30.d, z18.d\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "mov z31.d, z19.d\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z16.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z17.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[c_ptr0], #3, MUL VL]\n"
                        "ld1w z20.s, p0/z, [c_ptr1]\n"
                        "ld1w z21.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1w z22.s, p2/z, [c_ptr1, #2, MUL VL]\n"
                        "ld1w z23.s, p3/z, [c_ptr1, #3, MUL VL]\n"
                        "ld1w z24.s, p0/z, [c_ptr2]\n"
                        "ld1w z25.s, p1/z, [c_ptr2, #1, MUL VL]\n"
                        "ld1w z26.s, p2/z, [c_ptr2, #2, MUL VL]\n"
                        "ld1w z27.s, p3/z, [c_ptr2, #3, MUL VL]\n"
                        "ld1w z28.s, p0/z, [c_ptr3]\n"
                        "ld1w z29.s, p1/z, [c_ptr3, #1, MUL VL]\n"
                        "ld1w z30.s, p2/z, [c_ptr3, #2, MUL VL]\n"
                        "ld1w z31.s, p3/z, [c_ptr3, #3, MUL VL]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "ld1rqh z3.h, p7/z, [a_ptr3]\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        ".inst 0x64604110 // bfdot z16.s, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64614114 // bfdot z20.s, z8.h, z1.h[0]\n"
                        "ld1rqh z4.h, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x64624118 // bfdot z24.s, z8.h, z2.h[0]\n"
                        "ld1rqh z5.h, p7/z, [a_ptr1]\n"
                        ".inst 0x6463411c // bfdot z28.s, z8.h, z3.h[0]\n"
                        "ld1rqh z6.h, p7/z, [a_ptr2]\n"
                        ".inst 0x64604131 // bfdot z17.s, z9.h, z0.h[0]\n"
                        "ld1rqh z7.h, p7/z, [a_ptr3]\n"
                        ".inst 0x64614135 // bfdot z21.s, z9.h, z1.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64624139 // bfdot z25.s, z9.h, z2.h[0]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        ".inst 0x6463413d // bfdot z29.s, z9.h, z3.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64604152 // bfdot z18.s, z10.h, z0.h[0]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x64614156 // bfdot z22.s, z10.h, z1.h[0]\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        ".inst 0x6462415a // bfdot z26.s, z10.h, z2.h[0]\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        ".inst 0x6463415e // bfdot z30.s, z10.h, z3.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64604173 // bfdot z19.s, z11.h, z0.h[0]\n"
                        "add a_ptr3, a_ptr3, #0x20\n"
                        ".inst 0x64614177 // bfdot z23.s, z11.h, z1.h[0]\n"
                        ".inst 0x6462417b // bfdot z27.s, z11.h, z2.h[0]\n"
                        ".inst 0x6463417f // bfdot z31.s, z11.h, z3.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64684190 // bfdot z16.s, z12.h, z0.h[1]\n"
                        ".inst 0x64694194 // bfdot z20.s, z12.h, z1.h[1]\n"
                        ".inst 0x646a4198 // bfdot z24.s, z12.h, z2.h[1]\n"
                        ".inst 0x646b419c // bfdot z28.s, z12.h, z3.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646841b1 // bfdot z17.s, z13.h, z0.h[1]\n"
                        ".inst 0x646941b5 // bfdot z21.s, z13.h, z1.h[1]\n"
                        ".inst 0x646a41b9 // bfdot z25.s, z13.h, z2.h[1]\n"
                        ".inst 0x646b41bd // bfdot z29.s, z13.h, z3.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x646841d2 // bfdot z18.s, z14.h, z0.h[1]\n"
                        ".inst 0x646941d6 // bfdot z22.s, z14.h, z1.h[1]\n"
                        ".inst 0x646a41da // bfdot z26.s, z14.h, z2.h[1]\n"
                        ".inst 0x646b41de // bfdot z30.s, z14.h, z3.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646841f3 // bfdot z19.s, z15.h, z0.h[1]\n"
                        ".inst 0x646941f7 // bfdot z23.s, z15.h, z1.h[1]\n"
                        ".inst 0x646a41fb // bfdot z27.s, z15.h, z2.h[1]\n"
                        ".inst 0x646b41ff // bfdot z31.s, z15.h, z3.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64704110 // bfdot z16.s, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x64714114 // bfdot z20.s, z8.h, z1.h[2]\n"
                        ".inst 0x64724118 // bfdot z24.s, z8.h, z2.h[2]\n"
                        ".inst 0x6473411c // bfdot z28.s, z8.h, z3.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64704131 // bfdot z17.s, z9.h, z0.h[2]\n"
                        ".inst 0x64714135 // bfdot z21.s, z9.h, z1.h[2]\n"
                        ".inst 0x64724139 // bfdot z25.s, z9.h, z2.h[2]\n"
                        ".inst 0x6473413d // bfdot z29.s, z9.h, z3.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64704152 // bfdot z18.s, z10.h, z0.h[2]\n"
                        ".inst 0x64714156 // bfdot z22.s, z10.h, z1.h[2]\n"
                        ".inst 0x6472415a // bfdot z26.s, z10.h, z2.h[2]\n"
                        ".inst 0x6473415e // bfdot z30.s, z10.h, z3.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64704173 // bfdot z19.s, z11.h, z0.h[2]\n"
                        ".inst 0x64714177 // bfdot z23.s, z11.h, z1.h[2]\n"
                        ".inst 0x6472417b // bfdot z27.s, z11.h, z2.h[2]\n"
                        ".inst 0x6473417f // bfdot z31.s, z11.h, z3.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64784190 // bfdot z16.s, z12.h, z0.h[3]\n"
                        ".inst 0x64794194 // bfdot z20.s, z12.h, z1.h[3]\n"
                        ".inst 0x647a4198 // bfdot z24.s, z12.h, z2.h[3]\n"
                        ".inst 0x647b419c // bfdot z28.s, z12.h, z3.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x647841b1 // bfdot z17.s, z13.h, z0.h[3]\n"
                        ".inst 0x647941b5 // bfdot z21.s, z13.h, z1.h[3]\n"
                        ".inst 0x647a41b9 // bfdot z25.s, z13.h, z2.h[3]\n"
                        ".inst 0x647b41bd // bfdot z29.s, z13.h, z3.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x647841d2 // bfdot z18.s, z14.h, z0.h[3]\n"
                        ".inst 0x647941d6 // bfdot z22.s, z14.h, z1.h[3]\n"
                        ".inst 0x647a41da // bfdot z26.s, z14.h, z2.h[3]\n"
                        ".inst 0x647b41de // bfdot z30.s, z14.h, z3.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x647841f3 // bfdot z19.s, z15.h, z0.h[3]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x647941f7 // bfdot z23.s, z15.h, z1.h[3]\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1, #-0x10]\n"
                        ".inst 0x647a41fb // bfdot z27.s, z15.h, z2.h[3]\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2, #-0x10]\n"
                        ".inst 0x647b41ff // bfdot z31.s, z15.h, z3.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64644110 // bfdot z16.s, z8.h, z4.h[0]\n"
                        "ld1rqh z3.h, p7/z, [a_ptr3, #-0x10]\n"
                        ".inst 0x64654114 // bfdot z20.s, z8.h, z5.h[0]\n"
                        ".inst 0x64664118 // bfdot z24.s, z8.h, z6.h[0]\n"
                        ".inst 0x6467411c // bfdot z28.s, z8.h, z7.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64644131 // bfdot z17.s, z9.h, z4.h[0]\n"
                        ".inst 0x64654135 // bfdot z21.s, z9.h, z5.h[0]\n"
                        ".inst 0x64664139 // bfdot z25.s, z9.h, z6.h[0]\n"
                        ".inst 0x6467413d // bfdot z29.s, z9.h, z7.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64644152 // bfdot z18.s, z10.h, z4.h[0]\n"
                        ".inst 0x64654156 // bfdot z22.s, z10.h, z5.h[0]\n"
                        ".inst 0x6466415a // bfdot z26.s, z10.h, z6.h[0]\n"
                        ".inst 0x6467415e // bfdot z30.s, z10.h, z7.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64644173 // bfdot z19.s, z11.h, z4.h[0]\n"
                        ".inst 0x64654177 // bfdot z23.s, z11.h, z5.h[0]\n"
                        ".inst 0x6466417b // bfdot z27.s, z11.h, z6.h[0]\n"
                        ".inst 0x6467417f // bfdot z31.s, z11.h, z7.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x646c4190 // bfdot z16.s, z12.h, z4.h[1]\n"
                        ".inst 0x646d4194 // bfdot z20.s, z12.h, z5.h[1]\n"
                        ".inst 0x646e4198 // bfdot z24.s, z12.h, z6.h[1]\n"
                        ".inst 0x646f419c // bfdot z28.s, z12.h, z7.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646c41b1 // bfdot z17.s, z13.h, z4.h[1]\n"
                        ".inst 0x646d41b5 // bfdot z21.s, z13.h, z5.h[1]\n"
                        ".inst 0x646e41b9 // bfdot z25.s, z13.h, z6.h[1]\n"
                        ".inst 0x646f41bd // bfdot z29.s, z13.h, z7.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x646c41d2 // bfdot z18.s, z14.h, z4.h[1]\n"
                        ".inst 0x646d41d6 // bfdot z22.s, z14.h, z5.h[1]\n"
                        ".inst 0x646e41da // bfdot z26.s, z14.h, z6.h[1]\n"
                        ".inst 0x646f41de // bfdot z30.s, z14.h, z7.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646c41f3 // bfdot z19.s, z15.h, z4.h[1]\n"
                        ".inst 0x646d41f7 // bfdot z23.s, z15.h, z5.h[1]\n"
                        ".inst 0x646e41fb // bfdot z27.s, z15.h, z6.h[1]\n"
                        ".inst 0x646f41ff // bfdot z31.s, z15.h, z7.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64744110 // bfdot z16.s, z8.h, z4.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x64754114 // bfdot z20.s, z8.h, z5.h[2]\n"
                        ".inst 0x64764118 // bfdot z24.s, z8.h, z6.h[2]\n"
                        ".inst 0x6477411c // bfdot z28.s, z8.h, z7.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64744131 // bfdot z17.s, z9.h, z4.h[2]\n"
                        ".inst 0x64754135 // bfdot z21.s, z9.h, z5.h[2]\n"
                        ".inst 0x64764139 // bfdot z25.s, z9.h, z6.h[2]\n"
                        ".inst 0x6477413d // bfdot z29.s, z9.h, z7.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64744152 // bfdot z18.s, z10.h, z4.h[2]\n"
                        ".inst 0x64754156 // bfdot z22.s, z10.h, z5.h[2]\n"
                        ".inst 0x6476415a // bfdot z26.s, z10.h, z6.h[2]\n"
                        ".inst 0x6477415e // bfdot z30.s, z10.h, z7.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64744173 // bfdot z19.s, z11.h, z4.h[2]\n"
                        ".inst 0x64754177 // bfdot z23.s, z11.h, z5.h[2]\n"
                        ".inst 0x6476417b // bfdot z27.s, z11.h, z6.h[2]\n"
                        ".inst 0x6477417f // bfdot z31.s, z11.h, z7.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x647c4190 // bfdot z16.s, z12.h, z4.h[3]\n"
                        ".inst 0x647d4194 // bfdot z20.s, z12.h, z5.h[3]\n"
                        ".inst 0x647e4198 // bfdot z24.s, z12.h, z6.h[3]\n"
                        ".inst 0x647f419c // bfdot z28.s, z12.h, z7.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x647c41b1 // bfdot z17.s, z13.h, z4.h[3]\n"
                        ".inst 0x647d41b5 // bfdot z21.s, z13.h, z5.h[3]\n"
                        ".inst 0x647e41b9 // bfdot z25.s, z13.h, z6.h[3]\n"
                        ".inst 0x647f41bd // bfdot z29.s, z13.h, z7.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x647c41d2 // bfdot z18.s, z14.h, z4.h[3]\n"
                        ".inst 0x647d41d6 // bfdot z22.s, z14.h, z5.h[3]\n"
                        ".inst 0x647e41da // bfdot z26.s, z14.h, z6.h[3]\n"
                        ".inst 0x647f41de // bfdot z30.s, z14.h, z7.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x647c41f3 // bfdot z19.s, z15.h, z4.h[3]\n"
                        ".inst 0x647d41f7 // bfdot z23.s, z15.h, z5.h[3]\n"
                        ".inst 0x647e41fb // bfdot z27.s, z15.h, z6.h[3]\n"
                        ".inst 0x647f41ff // bfdot z31.s, z15.h, z7.h[3]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        ".inst 0x64604110 // bfdot z16.s, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64614114 // bfdot z20.s, z8.h, z1.h[0]\n"
                        "ld1rqh z4.h, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x64624118 // bfdot z24.s, z8.h, z2.h[0]\n"
                        "ld1rqh z5.h, p7/z, [a_ptr1]\n"
                        ".inst 0x6463411c // bfdot z28.s, z8.h, z3.h[0]\n"
                        "ld1rqh z6.h, p7/z, [a_ptr2]\n"
                        ".inst 0x64604131 // bfdot z17.s, z9.h, z0.h[0]\n"
                        "ld1rqh z7.h, p7/z, [a_ptr3]\n"
                        ".inst 0x64614135 // bfdot z21.s, z9.h, z1.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64624139 // bfdot z25.s, z9.h, z2.h[0]\n"
                        ".inst 0x6463413d // bfdot z29.s, z9.h, z3.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64604152 // bfdot z18.s, z10.h, z0.h[0]\n"
                        ".inst 0x64614156 // bfdot z22.s, z10.h, z1.h[0]\n"
                        ".inst 0x6462415a // bfdot z26.s, z10.h, z2.h[0]\n"
                        ".inst 0x6463415e // bfdot z30.s, z10.h, z3.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64604173 // bfdot z19.s, z11.h, z0.h[0]\n"
                        ".inst 0x64614177 // bfdot z23.s, z11.h, z1.h[0]\n"
                        ".inst 0x6462417b // bfdot z27.s, z11.h, z2.h[0]\n"
                        ".inst 0x6463417f // bfdot z31.s, z11.h, z3.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64684190 // bfdot z16.s, z12.h, z0.h[1]\n"
                        ".inst 0x64694194 // bfdot z20.s, z12.h, z1.h[1]\n"
                        ".inst 0x646a4198 // bfdot z24.s, z12.h, z2.h[1]\n"
                        ".inst 0x646b419c // bfdot z28.s, z12.h, z3.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646841b1 // bfdot z17.s, z13.h, z0.h[1]\n"
                        ".inst 0x646941b5 // bfdot z21.s, z13.h, z1.h[1]\n"
                        ".inst 0x646a41b9 // bfdot z25.s, z13.h, z2.h[1]\n"
                        ".inst 0x646b41bd // bfdot z29.s, z13.h, z3.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x646841d2 // bfdot z18.s, z14.h, z0.h[1]\n"
                        ".inst 0x646941d6 // bfdot z22.s, z14.h, z1.h[1]\n"
                        ".inst 0x646a41da // bfdot z26.s, z14.h, z2.h[1]\n"
                        ".inst 0x646b41de // bfdot z30.s, z14.h, z3.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646841f3 // bfdot z19.s, z15.h, z0.h[1]\n"
                        ".inst 0x646941f7 // bfdot z23.s, z15.h, z1.h[1]\n"
                        ".inst 0x646a41fb // bfdot z27.s, z15.h, z2.h[1]\n"
                        ".inst 0x646b41ff // bfdot z31.s, z15.h, z3.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64704110 // bfdot z16.s, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x64714114 // bfdot z20.s, z8.h, z1.h[2]\n"
                        ".inst 0x64724118 // bfdot z24.s, z8.h, z2.h[2]\n"
                        ".inst 0x6473411c // bfdot z28.s, z8.h, z3.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64704131 // bfdot z17.s, z9.h, z0.h[2]\n"
                        ".inst 0x64714135 // bfdot z21.s, z9.h, z1.h[2]\n"
                        ".inst 0x64724139 // bfdot z25.s, z9.h, z2.h[2]\n"
                        ".inst 0x6473413d // bfdot z29.s, z9.h, z3.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64704152 // bfdot z18.s, z10.h, z0.h[2]\n"
                        ".inst 0x64714156 // bfdot z22.s, z10.h, z1.h[2]\n"
                        ".inst 0x6472415a // bfdot z26.s, z10.h, z2.h[2]\n"
                        ".inst 0x6473415e // bfdot z30.s, z10.h, z3.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64704173 // bfdot z19.s, z11.h, z0.h[2]\n"
                        ".inst 0x64714177 // bfdot z23.s, z11.h, z1.h[2]\n"
                        ".inst 0x6472417b // bfdot z27.s, z11.h, z2.h[2]\n"
                        ".inst 0x6473417f // bfdot z31.s, z11.h, z3.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64784190 // bfdot z16.s, z12.h, z0.h[3]\n"
                        ".inst 0x64794194 // bfdot z20.s, z12.h, z1.h[3]\n"
                        ".inst 0x647a4198 // bfdot z24.s, z12.h, z2.h[3]\n"
                        ".inst 0x647b419c // bfdot z28.s, z12.h, z3.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x647841b1 // bfdot z17.s, z13.h, z0.h[3]\n"
                        ".inst 0x647941b5 // bfdot z21.s, z13.h, z1.h[3]\n"
                        ".inst 0x647a41b9 // bfdot z25.s, z13.h, z2.h[3]\n"
                        ".inst 0x647b41bd // bfdot z29.s, z13.h, z3.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x647841d2 // bfdot z18.s, z14.h, z0.h[3]\n"
                        ".inst 0x647941d6 // bfdot z22.s, z14.h, z1.h[3]\n"
                        ".inst 0x647a41da // bfdot z26.s, z14.h, z2.h[3]\n"
                        ".inst 0x647b41de // bfdot z30.s, z14.h, z3.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x647841f3 // bfdot z19.s, z15.h, z0.h[3]\n"
                        "ld1rqh z0.h, p6/z, [%[a_ptr0], #0x10]\n"
                        ".inst 0x647941f7 // bfdot z23.s, z15.h, z1.h[3]\n"
                        "ld1rqh z1.h, p6/z, [a_ptr1, #0x10]\n"
                        ".inst 0x647a41fb // bfdot z27.s, z15.h, z2.h[3]\n"
                        "ld1rqh z2.h, p6/z, [a_ptr2, #0x10]\n"
                        ".inst 0x647b41ff // bfdot z31.s, z15.h, z3.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64644110 // bfdot z16.s, z8.h, z4.h[0]\n"
                        "ld1rqh z3.h, p6/z, [a_ptr3, #0x10]\n"
                        ".inst 0x64654114 // bfdot z20.s, z8.h, z5.h[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        ".inst 0x64664118 // bfdot z24.s, z8.h, z6.h[0]\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        ".inst 0x6467411c // bfdot z28.s, z8.h, z7.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64644131 // bfdot z17.s, z9.h, z4.h[0]\n"
                        "addvl a_ptr2, a_ptr2, #2\n"
                        ".inst 0x64654135 // bfdot z21.s, z9.h, z5.h[0]\n"
                        "addvl a_ptr3, a_ptr3, #2\n"
                        ".inst 0x64664139 // bfdot z25.s, z9.h, z6.h[0]\n"
                        ".inst 0x6467413d // bfdot z29.s, z9.h, z7.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64644152 // bfdot z18.s, z10.h, z4.h[0]\n"
                        ".inst 0x64654156 // bfdot z22.s, z10.h, z5.h[0]\n"
                        ".inst 0x6466415a // bfdot z26.s, z10.h, z6.h[0]\n"
                        ".inst 0x6467415e // bfdot z30.s, z10.h, z7.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64644173 // bfdot z19.s, z11.h, z4.h[0]\n"
                        ".inst 0x64654177 // bfdot z23.s, z11.h, z5.h[0]\n"
                        ".inst 0x6466417b // bfdot z27.s, z11.h, z6.h[0]\n"
                        ".inst 0x6467417f // bfdot z31.s, z11.h, z7.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x646c4190 // bfdot z16.s, z12.h, z4.h[1]\n"
                        ".inst 0x646d4194 // bfdot z20.s, z12.h, z5.h[1]\n"
                        ".inst 0x646e4198 // bfdot z24.s, z12.h, z6.h[1]\n"
                        ".inst 0x646f419c // bfdot z28.s, z12.h, z7.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646c41b1 // bfdot z17.s, z13.h, z4.h[1]\n"
                        ".inst 0x646d41b5 // bfdot z21.s, z13.h, z5.h[1]\n"
                        ".inst 0x646e41b9 // bfdot z25.s, z13.h, z6.h[1]\n"
                        ".inst 0x646f41bd // bfdot z29.s, z13.h, z7.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x646c41d2 // bfdot z18.s, z14.h, z4.h[1]\n"
                        ".inst 0x646d41d6 // bfdot z22.s, z14.h, z5.h[1]\n"
                        ".inst 0x646e41da // bfdot z26.s, z14.h, z6.h[1]\n"
                        ".inst 0x646f41de // bfdot z30.s, z14.h, z7.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646c41f3 // bfdot z19.s, z15.h, z4.h[1]\n"
                        ".inst 0x646d41f7 // bfdot z23.s, z15.h, z5.h[1]\n"
                        ".inst 0x646e41fb // bfdot z27.s, z15.h, z6.h[1]\n"
                        ".inst 0x646f41ff // bfdot z31.s, z15.h, z7.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64744110 // bfdot z16.s, z8.h, z4.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        ".inst 0x64754114 // bfdot z20.s, z8.h, z5.h[2]\n"
                        ".inst 0x64764118 // bfdot z24.s, z8.h, z6.h[2]\n"
                        ".inst 0x6477411c // bfdot z28.s, z8.h, z7.h[2]\n"
                        ".inst 0x64744131 // bfdot z17.s, z9.h, z4.h[2]\n"
                        ".inst 0x64754135 // bfdot z21.s, z9.h, z5.h[2]\n"
                        ".inst 0x64764139 // bfdot z25.s, z9.h, z6.h[2]\n"
                        ".inst 0x6477413d // bfdot z29.s, z9.h, z7.h[2]\n"
                        ".inst 0x64744152 // bfdot z18.s, z10.h, z4.h[2]\n"
                        ".inst 0x64754156 // bfdot z22.s, z10.h, z5.h[2]\n"
                        ".inst 0x6476415a // bfdot z26.s, z10.h, z6.h[2]\n"
                        ".inst 0x6477415e // bfdot z30.s, z10.h, z7.h[2]\n"
                        ".inst 0x64744173 // bfdot z19.s, z11.h, z4.h[2]\n"
                        ".inst 0x64754177 // bfdot z23.s, z11.h, z5.h[2]\n"
                        ".inst 0x6476417b // bfdot z27.s, z11.h, z6.h[2]\n"
                        ".inst 0x6477417f // bfdot z31.s, z11.h, z7.h[2]\n"
                        ".inst 0x647c4190 // bfdot z16.s, z12.h, z4.h[3]\n"
                        ".inst 0x647d4194 // bfdot z20.s, z12.h, z5.h[3]\n"
                        ".inst 0x647e4198 // bfdot z24.s, z12.h, z6.h[3]\n"
                        ".inst 0x647f419c // bfdot z28.s, z12.h, z7.h[3]\n"
                        ".inst 0x647c41b1 // bfdot z17.s, z13.h, z4.h[3]\n"
                        ".inst 0x647d41b5 // bfdot z21.s, z13.h, z5.h[3]\n"
                        ".inst 0x647e41b9 // bfdot z25.s, z13.h, z6.h[3]\n"
                        ".inst 0x647f41bd // bfdot z29.s, z13.h, z7.h[3]\n"
                        ".inst 0x647c41d2 // bfdot z18.s, z14.h, z4.h[3]\n"
                        ".inst 0x647d41d6 // bfdot z22.s, z14.h, z5.h[3]\n"
                        ".inst 0x647e41da // bfdot z26.s, z14.h, z6.h[3]\n"
                        ".inst 0x647f41de // bfdot z30.s, z14.h, z7.h[3]\n"
                        ".inst 0x647c41f3 // bfdot z19.s, z15.h, z4.h[3]\n"
                        ".inst 0x647d41f7 // bfdot z23.s, z15.h, z5.h[3]\n"
                        ".inst 0x647e41fb // bfdot z27.s, z15.h, z6.h[3]\n"
                        ".inst 0x647f41ff // bfdot z31.s, z15.h, z7.h[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64604110 // bfdot z16.s, z8.h, z0.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64614114 // bfdot z20.s, z8.h, z1.h[0]\n"
                        ".inst 0x64624118 // bfdot z24.s, z8.h, z2.h[0]\n"
                        ".inst 0x6463411c // bfdot z28.s, z8.h, z3.h[0]\n"
                        ".inst 0x64604131 // bfdot z17.s, z9.h, z0.h[0]\n"
                        ".inst 0x64614135 // bfdot z21.s, z9.h, z1.h[0]\n"
                        ".inst 0x64624139 // bfdot z25.s, z9.h, z2.h[0]\n"
                        ".inst 0x6463413d // bfdot z29.s, z9.h, z3.h[0]\n"
                        ".inst 0x64604152 // bfdot z18.s, z10.h, z0.h[0]\n"
                        ".inst 0x64614156 // bfdot z22.s, z10.h, z1.h[0]\n"
                        ".inst 0x6462415a // bfdot z26.s, z10.h, z2.h[0]\n"
                        ".inst 0x6463415e // bfdot z30.s, z10.h, z3.h[0]\n"
                        ".inst 0x64604173 // bfdot z19.s, z11.h, z0.h[0]\n"
                        ".inst 0x64614177 // bfdot z23.s, z11.h, z1.h[0]\n"
                        ".inst 0x6462417b // bfdot z27.s, z11.h, z2.h[0]\n"
                        ".inst 0x6463417f // bfdot z31.s, z11.h, z3.h[0]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64684190 // bfdot z16.s, z12.h, z0.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64694194 // bfdot z20.s, z12.h, z1.h[1]\n"
                        ".inst 0x646a4198 // bfdot z24.s, z12.h, z2.h[1]\n"
                        ".inst 0x646b419c // bfdot z28.s, z12.h, z3.h[1]\n"
                        ".inst 0x646841b1 // bfdot z17.s, z13.h, z0.h[1]\n"
                        ".inst 0x646941b5 // bfdot z21.s, z13.h, z1.h[1]\n"
                        ".inst 0x646a41b9 // bfdot z25.s, z13.h, z2.h[1]\n"
                        ".inst 0x646b41bd // bfdot z29.s, z13.h, z3.h[1]\n"
                        ".inst 0x646841d2 // bfdot z18.s, z14.h, z0.h[1]\n"
                        ".inst 0x646941d6 // bfdot z22.s, z14.h, z1.h[1]\n"
                        ".inst 0x646a41da // bfdot z26.s, z14.h, z2.h[1]\n"
                        ".inst 0x646b41de // bfdot z30.s, z14.h, z3.h[1]\n"
                        ".inst 0x646841f3 // bfdot z19.s, z15.h, z0.h[1]\n"
                        ".inst 0x646941f7 // bfdot z23.s, z15.h, z1.h[1]\n"
                        ".inst 0x646a41fb // bfdot z27.s, z15.h, z2.h[1]\n"
                        ".inst 0x646b41ff // bfdot z31.s, z15.h, z3.h[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64704110 // bfdot z16.s, z8.h, z0.h[2]\n"
                        ".inst 0x64714114 // bfdot z20.s, z8.h, z1.h[2]\n"
                        ".inst 0x64724118 // bfdot z24.s, z8.h, z2.h[2]\n"
                        ".inst 0x6473411c // bfdot z28.s, z8.h, z3.h[2]\n"
                        ".inst 0x64704131 // bfdot z17.s, z9.h, z0.h[2]\n"
                        ".inst 0x64714135 // bfdot z21.s, z9.h, z1.h[2]\n"
                        ".inst 0x64724139 // bfdot z25.s, z9.h, z2.h[2]\n"
                        ".inst 0x6473413d // bfdot z29.s, z9.h, z3.h[2]\n"
                        ".inst 0x64704152 // bfdot z18.s, z10.h, z0.h[2]\n"
                        ".inst 0x64714156 // bfdot z22.s, z10.h, z1.h[2]\n"
                        ".inst 0x6472415a // bfdot z26.s, z10.h, z2.h[2]\n"
                        ".inst 0x6473415e // bfdot z30.s, z10.h, z3.h[2]\n"
                        ".inst 0x64704173 // bfdot z19.s, z11.h, z0.h[2]\n"
                        ".inst 0x64714177 // bfdot z23.s, z11.h, z1.h[2]\n"
                        ".inst 0x6472417b // bfdot z27.s, z11.h, z2.h[2]\n"
                        ".inst 0x6473417f // bfdot z31.s, z11.h, z3.h[2]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64784190 // bfdot z16.s, z12.h, z0.h[3]\n"
                        ".inst 0x64794194 // bfdot z20.s, z12.h, z1.h[3]\n"
                        ".inst 0x647a4198 // bfdot z24.s, z12.h, z2.h[3]\n"
                        ".inst 0x647b419c // bfdot z28.s, z12.h, z3.h[3]\n"
                        ".inst 0x647841b1 // bfdot z17.s, z13.h, z0.h[3]\n"
                        ".inst 0x647941b5 // bfdot z21.s, z13.h, z1.h[3]\n"
                        ".inst 0x647a41b9 // bfdot z25.s, z13.h, z2.h[3]\n"
                        ".inst 0x647b41bd // bfdot z29.s, z13.h, z3.h[3]\n"
                        ".inst 0x647841d2 // bfdot z18.s, z14.h, z0.h[3]\n"
                        ".inst 0x647941d6 // bfdot z22.s, z14.h, z1.h[3]\n"
                        ".inst 0x647a41da // bfdot z26.s, z14.h, z2.h[3]\n"
                        ".inst 0x647b41de // bfdot z30.s, z14.h, z3.h[3]\n"
                        ".inst 0x647841f3 // bfdot z19.s, z15.h, z0.h[3]\n"
                        ".inst 0x647941f7 // bfdot z23.s, z15.h, z1.h[3]\n"
                        ".inst 0x647a41fb // bfdot z27.s, z15.h, z2.h[3]\n"
                        ".inst 0x647b41ff // bfdot z31.s, z15.h, z3.h[3]\n"
                        "b 5f\n"
                        "4:\n"
                        ".inst 0x64604110 // bfdot z16.s, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x64614114 // bfdot z20.s, z8.h, z1.h[0]\n"
                        "ld1rqh z4.h, p6/z, [%[a_ptr0]]\n"
                        ".inst 0x64624118 // bfdot z24.s, z8.h, z2.h[0]\n"
                        "ld1rqh z5.h, p6/z, [a_ptr1]\n"
                        ".inst 0x6463411c // bfdot z28.s, z8.h, z3.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64604131 // bfdot z17.s, z9.h, z0.h[0]\n"
                        "ld1rqh z6.h, p6/z, [a_ptr2]\n"
                        ".inst 0x64614135 // bfdot z21.s, z9.h, z1.h[0]\n"
                        "ld1rqh z7.h, p6/z, [a_ptr3]\n"
                        ".inst 0x64624139 // bfdot z25.s, z9.h, z2.h[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        ".inst 0x6463413d // bfdot z29.s, z9.h, z3.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64604152 // bfdot z18.s, z10.h, z0.h[0]\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        ".inst 0x64614156 // bfdot z22.s, z10.h, z1.h[0]\n"
                        "addvl a_ptr2, a_ptr2, #1\n"
                        ".inst 0x6462415a // bfdot z26.s, z10.h, z2.h[0]\n"
                        "addvl a_ptr3, a_ptr3, #1\n"
                        ".inst 0x6463415e // bfdot z30.s, z10.h, z3.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64604173 // bfdot z19.s, z11.h, z0.h[0]\n"
                        ".inst 0x64614177 // bfdot z23.s, z11.h, z1.h[0]\n"
                        ".inst 0x6462417b // bfdot z27.s, z11.h, z2.h[0]\n"
                        ".inst 0x6463417f // bfdot z31.s, z11.h, z3.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64684190 // bfdot z16.s, z12.h, z0.h[1]\n"
                        ".inst 0x64694194 // bfdot z20.s, z12.h, z1.h[1]\n"
                        ".inst 0x646a4198 // bfdot z24.s, z12.h, z2.h[1]\n"
                        ".inst 0x646b419c // bfdot z28.s, z12.h, z3.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x646841b1 // bfdot z17.s, z13.h, z0.h[1]\n"
                        ".inst 0x646941b5 // bfdot z21.s, z13.h, z1.h[1]\n"
                        ".inst 0x646a41b9 // bfdot z25.s, z13.h, z2.h[1]\n"
                        ".inst 0x646b41bd // bfdot z29.s, z13.h, z3.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x646841d2 // bfdot z18.s, z14.h, z0.h[1]\n"
                        ".inst 0x646941d6 // bfdot z22.s, z14.h, z1.h[1]\n"
                        ".inst 0x646a41da // bfdot z26.s, z14.h, z2.h[1]\n"
                        ".inst 0x646b41de // bfdot z30.s, z14.h, z3.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646841f3 // bfdot z19.s, z15.h, z0.h[1]\n"
                        ".inst 0x646941f7 // bfdot z23.s, z15.h, z1.h[1]\n"
                        ".inst 0x646a41fb // bfdot z27.s, z15.h, z2.h[1]\n"
                        ".inst 0x646b41ff // bfdot z31.s, z15.h, z3.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64704110 // bfdot z16.s, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        ".inst 0x64714114 // bfdot z20.s, z8.h, z1.h[2]\n"
                        ".inst 0x64724118 // bfdot z24.s, z8.h, z2.h[2]\n"
                        ".inst 0x6473411c // bfdot z28.s, z8.h, z3.h[2]\n"
                        ".inst 0x64704131 // bfdot z17.s, z9.h, z0.h[2]\n"
                        ".inst 0x64714135 // bfdot z21.s, z9.h, z1.h[2]\n"
                        ".inst 0x64724139 // bfdot z25.s, z9.h, z2.h[2]\n"
                        ".inst 0x6473413d // bfdot z29.s, z9.h, z3.h[2]\n"
                        ".inst 0x64704152 // bfdot z18.s, z10.h, z0.h[2]\n"
                        ".inst 0x64714156 // bfdot z22.s, z10.h, z1.h[2]\n"
                        ".inst 0x6472415a // bfdot z26.s, z10.h, z2.h[2]\n"
                        ".inst 0x6473415e // bfdot z30.s, z10.h, z3.h[2]\n"
                        ".inst 0x64704173 // bfdot z19.s, z11.h, z0.h[2]\n"
                        ".inst 0x64714177 // bfdot z23.s, z11.h, z1.h[2]\n"
                        ".inst 0x6472417b // bfdot z27.s, z11.h, z2.h[2]\n"
                        ".inst 0x6473417f // bfdot z31.s, z11.h, z3.h[2]\n"
                        ".inst 0x64784190 // bfdot z16.s, z12.h, z0.h[3]\n"
                        ".inst 0x64794194 // bfdot z20.s, z12.h, z1.h[3]\n"
                        ".inst 0x647a4198 // bfdot z24.s, z12.h, z2.h[3]\n"
                        ".inst 0x647b419c // bfdot z28.s, z12.h, z3.h[3]\n"
                        ".inst 0x647841b1 // bfdot z17.s, z13.h, z0.h[3]\n"
                        ".inst 0x647941b5 // bfdot z21.s, z13.h, z1.h[3]\n"
                        ".inst 0x647a41b9 // bfdot z25.s, z13.h, z2.h[3]\n"
                        ".inst 0x647b41bd // bfdot z29.s, z13.h, z3.h[3]\n"
                        ".inst 0x647841d2 // bfdot z18.s, z14.h, z0.h[3]\n"
                        ".inst 0x647941d6 // bfdot z22.s, z14.h, z1.h[3]\n"
                        ".inst 0x647a41da // bfdot z26.s, z14.h, z2.h[3]\n"
                        ".inst 0x647b41de // bfdot z30.s, z14.h, z3.h[3]\n"
                        ".inst 0x647841f3 // bfdot z19.s, z15.h, z0.h[3]\n"
                        ".inst 0x647941f7 // bfdot z23.s, z15.h, z1.h[3]\n"
                        ".inst 0x647a41fb // bfdot z27.s, z15.h, z2.h[3]\n"
                        ".inst 0x647b41ff // bfdot z31.s, z15.h, z3.h[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64644110 // bfdot z16.s, z8.h, z4.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64654114 // bfdot z20.s, z8.h, z5.h[0]\n"
                        ".inst 0x64664118 // bfdot z24.s, z8.h, z6.h[0]\n"
                        ".inst 0x6467411c // bfdot z28.s, z8.h, z7.h[0]\n"
                        ".inst 0x64644131 // bfdot z17.s, z9.h, z4.h[0]\n"
                        ".inst 0x64654135 // bfdot z21.s, z9.h, z5.h[0]\n"
                        ".inst 0x64664139 // bfdot z25.s, z9.h, z6.h[0]\n"
                        ".inst 0x6467413d // bfdot z29.s, z9.h, z7.h[0]\n"
                        ".inst 0x64644152 // bfdot z18.s, z10.h, z4.h[0]\n"
                        ".inst 0x64654156 // bfdot z22.s, z10.h, z5.h[0]\n"
                        ".inst 0x6466415a // bfdot z26.s, z10.h, z6.h[0]\n"
                        ".inst 0x6467415e // bfdot z30.s, z10.h, z7.h[0]\n"
                        ".inst 0x64644173 // bfdot z19.s, z11.h, z4.h[0]\n"
                        ".inst 0x64654177 // bfdot z23.s, z11.h, z5.h[0]\n"
                        ".inst 0x6466417b // bfdot z27.s, z11.h, z6.h[0]\n"
                        ".inst 0x6467417f // bfdot z31.s, z11.h, z7.h[0]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x646c4190 // bfdot z16.s, z12.h, z4.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x646d4194 // bfdot z20.s, z12.h, z5.h[1]\n"
                        ".inst 0x646e4198 // bfdot z24.s, z12.h, z6.h[1]\n"
                        ".inst 0x646f419c // bfdot z28.s, z12.h, z7.h[1]\n"
                        ".inst 0x646c41b1 // bfdot z17.s, z13.h, z4.h[1]\n"
                        ".inst 0x646d41b5 // bfdot z21.s, z13.h, z5.h[1]\n"
                        ".inst 0x646e41b9 // bfdot z25.s, z13.h, z6.h[1]\n"
                        ".inst 0x646f41bd // bfdot z29.s, z13.h, z7.h[1]\n"
                        ".inst 0x646c41d2 // bfdot z18.s, z14.h, z4.h[1]\n"
                        ".inst 0x646d41d6 // bfdot z22.s, z14.h, z5.h[1]\n"
                        ".inst 0x646e41da // bfdot z26.s, z14.h, z6.h[1]\n"
                        ".inst 0x646f41de // bfdot z30.s, z14.h, z7.h[1]\n"
                        ".inst 0x646c41f3 // bfdot z19.s, z15.h, z4.h[1]\n"
                        ".inst 0x646d41f7 // bfdot z23.s, z15.h, z5.h[1]\n"
                        ".inst 0x646e41fb // bfdot z27.s, z15.h, z6.h[1]\n"
                        ".inst 0x646f41ff // bfdot z31.s, z15.h, z7.h[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64744110 // bfdot z16.s, z8.h, z4.h[2]\n"
                        ".inst 0x64754114 // bfdot z20.s, z8.h, z5.h[2]\n"
                        ".inst 0x64764118 // bfdot z24.s, z8.h, z6.h[2]\n"
                        ".inst 0x6477411c // bfdot z28.s, z8.h, z7.h[2]\n"
                        ".inst 0x64744131 // bfdot z17.s, z9.h, z4.h[2]\n"
                        ".inst 0x64754135 // bfdot z21.s, z9.h, z5.h[2]\n"
                        ".inst 0x64764139 // bfdot z25.s, z9.h, z6.h[2]\n"
                        ".inst 0x6477413d // bfdot z29.s, z9.h, z7.h[2]\n"
                        ".inst 0x64744152 // bfdot z18.s, z10.h, z4.h[2]\n"
                        ".inst 0x64754156 // bfdot z22.s, z10.h, z5.h[2]\n"
                        ".inst 0x6476415a // bfdot z26.s, z10.h, z6.h[2]\n"
                        ".inst 0x6477415e // bfdot z30.s, z10.h, z7.h[2]\n"
                        ".inst 0x64744173 // bfdot z19.s, z11.h, z4.h[2]\n"
                        ".inst 0x64754177 // bfdot z23.s, z11.h, z5.h[2]\n"
                        ".inst 0x6476417b // bfdot z27.s, z11.h, z6.h[2]\n"
                        ".inst 0x6477417f // bfdot z31.s, z11.h, z7.h[2]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x647c4190 // bfdot z16.s, z12.h, z4.h[3]\n"
                        ".inst 0x647d4194 // bfdot z20.s, z12.h, z5.h[3]\n"
                        ".inst 0x647e4198 // bfdot z24.s, z12.h, z6.h[3]\n"
                        ".inst 0x647f419c // bfdot z28.s, z12.h, z7.h[3]\n"
                        ".inst 0x647c41b1 // bfdot z17.s, z13.h, z4.h[3]\n"
                        ".inst 0x647d41b5 // bfdot z21.s, z13.h, z5.h[3]\n"
                        ".inst 0x647e41b9 // bfdot z25.s, z13.h, z6.h[3]\n"
                        ".inst 0x647f41bd // bfdot z29.s, z13.h, z7.h[3]\n"
                        ".inst 0x647c41d2 // bfdot z18.s, z14.h, z4.h[3]\n"
                        ".inst 0x647d41d6 // bfdot z22.s, z14.h, z5.h[3]\n"
                        ".inst 0x647e41da // bfdot z26.s, z14.h, z6.h[3]\n"
                        ".inst 0x647f41de // bfdot z30.s, z14.h, z7.h[3]\n"
                        ".inst 0x647c41f3 // bfdot z19.s, z15.h, z4.h[3]\n"
                        ".inst 0x647d41f7 // bfdot z23.s, z15.h, z5.h[3]\n"
                        ".inst 0x647e41fb // bfdot z27.s, z15.h, z6.h[3]\n"
                        ".inst 0x647f41ff // bfdot z31.s, z15.h, z7.h[3]\n"
                        "5:\n"
                        "ld1rw z14.s, p7/z, [%[minptr]]\n"
                        "ld1rw z15.s, p7/z, [%[maxptr]]\n"
                        "fmax z16.s, p7/m, z16.s, z14.s\n"
                        "fmax z17.s, p7/m, z17.s, z14.s\n"
                        "fmax z18.s, p7/m, z18.s, z14.s\n"
                        "fmax z19.s, p7/m, z19.s, z14.s\n"
                        "fmin z16.s, p7/m, z16.s, z15.s\n"
                        "fmin z17.s, p7/m, z17.s, z15.s\n"
                        "fmin z18.s, p7/m, z18.s, z15.s\n"
                        "fmin z19.s, p7/m, z19.s, z15.s\n"
                        "st1w z16.s, p0, [%[c_ptr0]]\n"
                        "fmax z20.s, p7/m, z20.s, z14.s\n"
                        "fmax z21.s, p7/m, z21.s, z14.s\n"
                        "fmax z22.s, p7/m, z22.s, z14.s\n"
                        "st1w z17.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmax z23.s, p7/m, z23.s, z14.s\n"
                        "fmin z20.s, p7/m, z20.s, z15.s\n"
                        "fmin z21.s, p7/m, z21.s, z15.s\n"
                        "st1w z18.s, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "fmin z22.s, p7/m, z22.s, z15.s\n"
                        "fmin z23.s, p7/m, z23.s, z15.s\n"
                        "fmax z24.s, p7/m, z24.s, z14.s\n"
                        "st1w z19.s, p3, [%[c_ptr0], #3, MUL VL]\n"
                        "fmax z25.s, p7/m, z25.s, z14.s\n"
                        "addvl %[c_ptr0], %[c_ptr0], #4\n"
                        "fmax z26.s, p7/m, z26.s, z14.s\n"
                        "st1w z20.s, p0, [c_ptr1]\n"
                        "fmin z24.s, p7/m, z24.s, z15.s\n"
                        "fmin z25.s, p7/m, z25.s, z15.s\n"
                        "fmax z27.s, p7/m, z27.s, z14.s\n"
                        "st1w z21.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "fmin z26.s, p7/m, z26.s, z15.s\n"
                        "fmax z28.s, p7/m, z28.s, z14.s\n"
                        "fmax z29.s, p7/m, z29.s, z14.s\n"
                        "st1w z22.s, p2, [c_ptr1, #2, MUL VL]\n"
                        "fmin z27.s, p7/m, z27.s, z15.s\n"
                        "fmax z30.s, p7/m, z30.s, z14.s\n"
                        "fmin z28.s, p7/m, z28.s, z15.s\n"
                        "st1w z23.s, p3, [c_ptr1, #3, MUL VL]\n"
                        "fmin z29.s, p7/m, z29.s, z15.s\n"
                        "fmax z31.s, p7/m, z31.s, z14.s\n"
                        "fmin z30.s, p7/m, z30.s, z15.s\n"
                        "st1w z24.s, p0, [c_ptr2]\n"
                        "fmin z31.s, p7/m, z31.s, z15.s\n"
                        "st1w z25.s, p1, [c_ptr2, #1, MUL VL]\n"
                        "st1w z26.s, p2, [c_ptr2, #2, MUL VL]\n"
                        "st1w z27.s, p3, [c_ptr2, #3, MUL VL]\n"
                        "st1w z28.s, p0, [c_ptr3]\n"
                        "st1w z29.s, p1, [c_ptr3, #1, MUL VL]\n"
                        "st1w z30.s, p2, [c_ptr3, #2, MUL VL]\n"
                        "st1w z31.s, p3, [c_ptr3, #3, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq a_ptr3\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        ".unreq c_ptr3\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "x2", "x3", "x4", "x5", "cc", "memory"
                    );
                    break;
            }

        }
    }
}

} // namespace arm_gemm

#endif // __ARM_FEATURE_SVE
