//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Plot2D/IntensityDataFFTPresenter.cpp
//! @brief     Defines class IntensityDataFFTPresenter
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Plot2D/IntensityDataFFTPresenter.h"
#include "Base/Util/Assert.h"
#include "Device/Data/DataUtil.h"
#include "Device/Data/Datafield.h"
#include "GUI/Model/Data/IntensityDataItem.h"
#include <QApplication>

IntensityDataFFTPresenter::IntensityDataFFTPresenter(QWidget* parent)
    : QObject(parent)
    , m_fftAction(nullptr)
    , m_fftItem(std::make_unique<IntensityDataItem>())
    , m_in_fft_mode(false)
{
    m_fftAction = new QAction(this);
    m_fftAction->setText("Fourier");
    m_fftAction->setIcon(QIcon(":/images/alpha-f-box.svg"));
    m_fftAction->setToolTip("Get the Fourier Transform of current intensity map");
    m_fftAction->setCheckable(true);
    connect(m_fftAction, &QAction::toggled, this, &IntensityDataFFTPresenter::onFFTActionToggled);
}

void IntensityDataFFTPresenter::reset()
{
    m_in_fft_mode = false;
}

IntensityDataItem* IntensityDataFFTPresenter::fftItem(IntensityDataItem* origItem)
{
    ASSERT(origItem);
    QApplication::setOverrideCursor(Qt::WaitCursor);

    m_fftItem->setDatafield(DataUtil::Data::createFFT(*origItem->c_field()).release());

    QApplication::restoreOverrideCursor();

    return m_fftItem.get();
}

QList<QAction*> IntensityDataFFTPresenter::actionList()
{
    return QList<QAction*>() << m_fftAction;
}

bool IntensityDataFFTPresenter::inFFTMode() const
{
    return m_in_fft_mode;
}

void IntensityDataFFTPresenter::onFFTActionToggled(bool toggled)
{
    m_in_fft_mode = toggled;
    fftActionRequest();
}
