// -*- indent-tabs-mode: nil -*-

#include <cppunit/extensions/HelperMacros.h>

#include <string>
#include <fstream>
#include <iostream>
#include <sys/stat.h>

#include "../canlxx.h"
#include "../fileutil.h"
#include "../nssutil.h"

class nssutilTest
  : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(nssutilTest);
  CPPUNIT_TEST(TestProxyCert);
  CPPUNIT_TEST_SUITE_END();

public:
  nssutilTest() : cadir("testCA_opensslutil"), trusted_cadir("openssl_trusted_certificates") {}
  void setUp();
  void tearDown();
  void TestProxyCert();

private:
  std::string cadir;
  std::string trusted_cadir;
};

void nssutilTest::TestProxyCert() {
  AuthN::Context ctx(AuthN::Context::EmptyContext);
  std::string config_dir = "./";
  std::string eec_cert_name = "eeccert";
  std::string eec_key_name = "eeckey";
  std::string proxy_cert_name = "proxy";
  std::string proxy_key_name = "myproxykey";
  std::string ca_cert_name = "testCA";
  std::string passwd = "secretpw";

  AuthN::NSS::CertContext certctx(ctx, config_dir);
  certctx.SetNSSDBPass(passwd);

  std::string trusts = "C,C,C";
  // Import the trusted CA certificates into nss db.
  std::string certdir;
  char* val = getenv("X509_CERT_DIR");
  if(val) certdir = val;
  if(!certdir.empty())
    certctx.ImportCertFromDirToDB(certdir.c_str(), trusts);

  std::string ca_certfile = "trusted_certificates/a04aa180.0";
  certctx.ImportCertFromFileToDB(ca_certfile.c_str(), ca_cert_name, trusts);

  // Retrieve the extension of certificate
  AuthN::Credentials::Extension ext;
  int pos = 0;
  for(; pos < 8; pos++)
    certctx.getCertExtension(pos, ext);

  // Retrieve the crl from crl url (if *.crl_url exists), and import it into nss db;
  // Import the crl from local file (if *.r0 exists) into nss db.
  AuthN::NSS::CRLContext crlctx(ctx, config_dir);
  std::string crldir;
  val = getenv("X509_CRL_DIR");
  if(val) crldir = val;
  else crldir = certdir;
  if(!crldir.empty())
    crlctx.ImportCRLFromDirToDB(crldir.c_str());

  std::string crl_file = "trusted_certificates/a04aa180.r0";
  crlctx.ImportCRLFromFileToDB(crl_file.c_str());

  // Create CSR
  AuthN::NSS::CSRContext proxy_request(ctx, config_dir, passwd);
  proxy_request.createRequest(proxy_key_name);
  std::string csr_str;
  std::string privk_str;
  proxy_request.getRequest(csr_str);
  proxy_request.getPrivKey(privk_str);

  // Signer, suppose the eec credential is 
  // from a file location, this operation practically
  // could only happen only once, when the EEC is imported
  // into the pkcs11 credential repository, e.g., mozilla softoken
  AuthN::NSS::CertContext eec_cert(ctx, config_dir);
 
  std::string eec_certfile = "testCA/certs/cert.pem"; 
  std::string eec_keyfile = "testCA/certs/key.pem"; 

  eec_cert.ImportKeyFromFileToDB(eec_keyfile.c_str(), eec_key_name); 
  eec_cert.ImportCertFromFileToDB(eec_certfile.c_str(), eec_cert_name);  

  std::string output_cert;
  eec_cert.certToPEM(output_cert);

  // Sign the proxy certificate
  int duration = 12;
  std::string proxy_str;
  eec_cert.signRequest(proxy_request, duration, proxy_str);
  CPPUNIT_ASSERT(!proxy_str.empty());

  // Verify the eec certificate
  AuthN::Validator::ValidationMode mode;
  AuthN::Status stat = eec_cert.validate(mode);
  CPPUNIT_ASSERT(stat == AuthN::Status(0));

  // Retrieve the extension of proxy certificate
  for(pos=0; pos < 8; pos++)
    eec_cert.getCertExtension(pos, ext);


  // Verify the proxy certificate
  std::string eec_str;
  eec_cert.certToPEM(eec_str);
  proxy_str.append(eec_str);
  AuthN::NSS::CertContext proxy_cert(ctx, config_dir);
  proxy_cert.certFromStr(proxy_str, proxy_cert_name, "u,u,u");
  stat = proxy_cert.validate(mode);
  CPPUNIT_ASSERT(stat == AuthN::Status(0));

  std::cout<<"Proxy:"<<std::endl<<proxy_str<<std::endl;
  std::cout<<"Cert:"<<std::endl<<eec_str<<std::endl;

  // Retrieve the extension of proxy certificate 
  for(pos=0; pos < 8; pos++)
    proxy_cert.getCertExtension(pos, ext);

}

void nssutilTest::setUp() {

}

void nssutilTest::tearDown() {
  //TODO: Remove the *.db files
}

CPPUNIT_TEST_SUITE_REGISTRATION(nssutilTest);

