/* 
   DB Mixer 
  ======== 
  Description: 
    interface controller for an external mixer to work with the DBMix system. 
 
    Copyright (c) 2000, 2001 Simon Mark Werner 
 
    DBMix Author: Robert Michael S Dean 
    exmixer Author: Simon Mark Werner 
    Version: 1.0 
 
 
   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version. 
 
   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of 
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
   GNU General Public License for more details. 
 
   You should have received a copy of the GNU General Public License 
   along with this program; if not, write to the Free Software 
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 
 */ 
 
 
#include <unistd.h> 
#include <string.h> 
#include <stdio.h> 
#include <stdlib.h> 
#include <errno.h> 
 
#include <sys/time.h> 
#include <sys/ioctl.h> 
#include <sys/types.h> 
#include <sys/stat.h> 
#include <sys/shm.h> 
#include <sys/ipc.h> 
#include <sys/msg.h> 
 
#include <gdk/gdkkeysyms.h> 
#include <gtk/gtk.h> 
#include <glib.h> 
 
/* required for js device */ 
#include <linux/joystick.h> 
#include <sys/ioctl.h> 
#include <sys/time.h> 
#include <fcntl.h> 
 
#include "dbmixer.h" 
#include <dbchannel.h> 
#include "exmixer.h" 
#include "save_prefs.h" 
#include "exmixer_prefs.h" 
#include <dbdebug.h> 
 
extern GtkAdjustment *main_volume_adj, *cue_volume_adj; 
extern GtkAdjustment *fader_adj, *autofade_adj, *balance_adj; 
extern GtkButton *punch_button_left, *punch_button_right; 
extern channel_widgets *widgets; 
extern dbfsd_data *sysdata; 
extern local_channel *local_channels; 
extern int *channel_indexes; 
extern int errno; 
extern int edit_prefs; 
 
//GtkWidget *exmixer_menu; 
int calibration = TRUE;     /* do we use on-the-fly calibration? */ 
int exmixer_enabled = FALSE; /* Enable the exmixer */ 
int num_channels;           /* number of channels available, derived from sysdata->num_channels */ 
 
/* Last VALue of the axis, used for axis movement detection */ 
float *axis_pitch_lval[MAX_DEV], *axis_vol_lval[MAX_DEV]; 
float cue_lval, main_volume_lval; 
float fade_speed_lval, cross_fader_lval, balance_lval; 
 
int first_time_round[MAX_DEV]; /* First time round don't accept any button presses */ 
 
/* Normalize to a number between 0.0 and 1.0 */ 
float normalize(float min, float cen, float max, float val, int flip) 
{ 
    float tmp_val; 
    
	/* this will detect if the .joystick or /etc/joystick.cal files were
       not found or were in the wrong format */
	if (min == cen == max == 0.0)
	{
		min = -32767;
		cen = 0.0;
		max = 32767;
	}

	if (flip) 
	{ 
		tmp_val = min; 
		min = max; 
		max = tmp_val; 
	} 
	
	if (val > cen) 
		val = 0.5 * ((val-cen)/(max-cen) + 1); 
	else 
		val = 0.5 * ((val-cen)/(cen-min) + 1); 
		
	if (val > 1.0) val = 1.0; 
	if (val < 0.0) val = 0.0; 	
	
    return val; 
} 
 
/* checks if a - b < radius */ 
int is_near( float a, float b, float r ) 
{ 
    if (a>b) 
    { 
       if ((a - b) < r) 
           return TRUE; 
       else 
           return FALSE; 
    } 
    else 
    { 
       if ((b - a) < r) 
           return TRUE; 
       else 
           return FALSE; 
    } 
} 
 
/* Recalibrates the mixer on-the-fly */ 
void calibrate(int dev_id, int axis_id) 
{ 
    if (!calibration) 
       return; 
    
    if (A_FLIP) 
    { 
       if (A_MIN > A_VAL) A_MIN = A_VAL; 
       if (A_MAX < A_VAL) A_MAX = A_VAL; 
    } 
    else 
    { 
       if (A_MIN < A_VAL) A_MIN = A_VAL; 
       if (A_MAX > A_VAL) A_MAX = A_VAL; 
    } 
    A_CEN = (A_MAX + A_MIN) / 2; 
} 
 
/* Change the given adjustment if necessary */ 
void em_set_adj(float *prev_value, float new_value, float inc_value, GtkAdjustment *wid) 
{ 
    /* Has the adjustment moved enough to care about it? */ 
    if( !is_near(*prev_value, new_value, inc_value) ) 
    { 
       /* Set the new previous value */ 
       *prev_value = new_value; 
        
       /* Set the new adjustment value */ 
       gtk_adjustment_set_value(wid, new_value);    
    } 
} 
 
/* Applies the movement of an axis to a GtkAdjustment */ 
void em_axis_moved( int dev_id, int axis_id ) 
{ 
    float value; 
    int ch_id = A_CHAN; 
           
    calibrate(dev_id, axis_id); 
    value = normalize((float)A_MIN, (float)A_CEN, (float)A_MAX, (float)A_VAL, A_FLIP); 
               
    switch (A_ASSIGN) 
    { 
    case EM_A_PITCH: 
       value *= PITCH_HIGH_BOUND - PITCH_LOW_BOUND; 
       value += PITCH_LOW_BOUND; 
       em_set_adj(&axis_pitch_lval[dev_id][axis_id], value, 0.1, widgets[ch_id].pitch_adj); 
       break; 
    case EM_A_VOLUME: 
       value *= 100.0; 
       em_set_adj(&axis_vol_lval[dev_id][axis_id], value, 1.0, widgets[ch_id].adj); 
       break; 
    case EM_A_CUE_VOLUME: 
       value *= 100.0; 
       em_set_adj(&cue_lval, value, 1.0, cue_volume_adj); 
       break; 
    case EM_A_MAIN_VOLUME: 
       value *= 100.0; 
       em_set_adj(&main_volume_lval, value, 1.0, main_volume_adj); 
       break; 
    case EM_A_CROSS_FADE: 
       value *= 200.0; 
       em_set_adj(&cross_fader_lval, value, 2.0, fader_adj); 
       break; 
    case EM_A_FADE_SPEED: 
       value *= 2.0; 
       em_set_adj(&fade_speed_lval, value, 0.02, autofade_adj); 
       break; 
    case EM_A_BALANCE: 
       value *= 200.0; 
       em_set_adj(&balance_lval, value, 2.0, balance_adj); 
       break; 
    case EM_A_NULL: 
    default: 
       break; 
    }    /* Switch */ 
} 
    
 
/* Applies a button press event to an action */ 
void em_button_pressed(int dev_id, int but_id) 
{    
    dbfsd_msg msg; 
    local_channel * ch; 
    int ch_id = B_CHAN; 
	int pressed_flag = 0;

	pressed_flag = B_PRESSED;

    /* Ignore a released button */ 
    if (B_PRESSED) 
    { 
		printf("button released\n");
       B_PRESSED = FALSE; 

	   if ((B_ASSIGN != EM_B_MUTE) && (B_ASSIGN != EM_B_CUE) 
		   && (B_ASSIGN != EM_B_PAUSE))
       return; 
    } 
    else 
	{
		printf("button pressed\n");
       B_PRESSED = TRUE; 
	}

    switch (B_ASSIGN) 
    { 
    case EM_B_NULL: 
       msg.msg_type = DBMSG_NONE; 
       Debug("em_read_device(): BUTTON NOT ASSIGNED"); 
       break; 
           
    case EM_B_MUTE: 
       msg.msg_type = DBMSG_NONE; 

	   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widgets[ch_id].mute_button)))
	   {
		   if (pressed_flag)
		   {
			   gtk_button_clicked( GTK_BUTTON(widgets[ch_id].mute_button) ); 
		   }
	   }
	   else
	   {
		   if (!pressed_flag)
		   {
			   gtk_button_clicked( GTK_BUTTON(widgets[ch_id].mute_button) ); 
		   }
	   }       break; 
           
    case EM_B_CUE: 
       msg.msg_type = DBMSG_NONE; 

	   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widgets[ch_id].cue_button)))
	   {
		   if (pressed_flag)
		   {
			   gtk_button_clicked( GTK_BUTTON(widgets[ch_id].cue_button) ); 
		   }
	   }
	   else
	   {
		   if (!pressed_flag)
		   {
			   gtk_button_clicked( GTK_BUTTON(widgets[ch_id].cue_button) ); 
		   }
	   }
	   
       break; 
 
    case EM_B_STOP: 
       msg.msg_type = DBMSG_STOP; 
       break; 
               
    case EM_B_PLAY: 
       msg.msg_type = DBMSG_PLAY; 
       break; 
               
    case EM_B_PAUSE: 
       msg.msg_type = DBMSG_NONE; 

	   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widgets[ch_id].pause_button)))
	   {
		   if (pressed_flag)
		   {
			   gtk_button_clicked( GTK_BUTTON(widgets[ch_id].pause_button) ); 
		   }
	   }
	   else
	   {
		   if (!pressed_flag)
		   {
			   gtk_button_clicked( GTK_BUTTON(widgets[ch_id].pause_button) ); 
		   }
	   }
       break; 
               
    case EM_B_PUNCH_LEFT: 
       msg.msg_type = DBMSG_NONE; 
       gtk_button_clicked( GTK_BUTTON(punch_button_left) ); 
       break; 
               
    case EM_B_PUNCH_RIGHT: 
       msg.msg_type = DBMSG_NONE; 
       gtk_button_clicked( GTK_BUTTON(punch_button_right) ); 
       break; 
               
    case EM_B_FAST_FORWARD: 
       msg.msg_type = DBMSG_FFORWARD; 
       msg.data = 5; 
       break; 
               
    case EM_B_REWIND: 
       msg.msg_type = DBMSG_REWIND; 
       msg.data = 5; 
       break; 
               
    case EM_B_EJECT: 
       msg.msg_type = DBMSG_EJECT; 
       break; 
                  
    case EM_B_NEXT: 
       msg.msg_type = DBMSG_NEXT; 
       break; 
                  
    case EM_B_PREV: 
       msg.msg_type = DBMSG_PREV; 
       break; 
    } /* switch */ 
               
    if (msg.msg_type != DBMSG_NONE) 
    { 
       ch = &(local_channels[channel_indexes[ch_id]]); 
       Debug("ch->msg_q_id = %d", ch->msg_q_id); 
        
       if(msgsnd(ch->msg_q_id, &msg, sizeof(dbfsd_msg) - sizeof(long int), IPC_NOWAIT) == 0) 
           Debug("dbmixer: Message sent."); 
       else 
           Error("dbmixer: Message failure."); 
    } 
} 
 
 
/* Read from the joystick device and set the corresponding adjustments */ 
void em_read_device( gpointer d_id, gint source_fd, GdkInputCondition condition ) 
{ 
    struct js_event em_event; 
    int dev_id = GPOINTER_TO_INT(d_id); 
    int axis_id, but_id; 
 
    if (read(D_FD, &em_event, sizeof(struct js_event)) != sizeof(struct js_event)) 
    { 
       Error("\nDBmix - em_read_device(): error reading device %d", dev_id); 
       exmixer_disable_device(dev_id); 
       Error("You have found a bug! :-)"); 
       exit(-1); 
//       return; 
    } 
    
    /* Check if button or axis event */ 
    switch(em_event.type & ~JS_EVENT_INIT) 
    { 
    case JS_EVENT_AXIS: /* Was an axis event */ 
       axis_id = em_event.number; 
       A_VAL = em_event.value; 
       if (N_AXES) em_axis_moved( dev_id, axis_id ); 
       break; 
    
    case JS_EVENT_BUTTON: /* Was a button event */ 
       but_id = em_event.number; 
       B_VAL = em_event.value; 
        
       /* when in edit preferences mode remove button press detection */ 
       if (edit_prefs) 
           gtk_widget_grab_focus(GTK_WIDGET(B_ASSIGN_ENT)); 
       else if (N_BUT && (first_time_round[dev_id] > N_BUT)) /* Forget the first N_BUT button presses */ 
           em_button_pressed( dev_id, but_id); 
        
       break; 
    } 
 
    if (first_time_round[dev_id]<8)    
       first_time_round[dev_id]++; 
} 
 
 
/* Detect the device */ 
gint exmixer_init_device(js_device *em_dev, char *device_filename) 
{ 
    ioctl(em_dev->fd, JSIOCGVERSION, &em_dev->version); 
    ioctl(em_dev->fd, JSIOCGAXES, &em_dev->num_axes); 
    ioctl(em_dev->fd, JSIOCGBUTTONS, &em_dev->num_buttons); 
    ioctl(em_dev->fd, JSIOCGNAME(NAME_LENGTH), em_dev->name); 
    strncpy(em_dev->filename, device_filename, FILENAME_LENGTH); 
 
    /* Did we find a joystick device? */ 
    if ((em_dev->num_axes > 0) && (em_dev->num_buttons > 0)) 
    { 
       Debug("exmixer %s (%s) has %d sliders and %d buttons. Driver version is %d.%d.%d.", 
              em_dev->filename, em_dev->name, 
              em_dev->num_axes, em_dev->num_buttons, 
              em_dev->version >> 16, 
              (em_dev->version >> 8) & 0xff, em_dev->version & 0xff); 
       return TRUE; 
    } 
    else 
    { 
       Error("Unable to open device %s.", device_filename); 
       return FALSE; 
    } 
} 
 
 
/* Tries to open the given device and initialize it 
 * returns TRUE if sucessful */ 
int exmixer_detect( int dev_id ) 
{ 
    Debug("Detecting exmixer."); 
    
    /* Try to open the device */ 
    if ( (D_FD = open(D_FILENAME, O_RDONLY) ) < 0 ) 
    { 
       Error("No external mixer found at %s.", D_FILENAME); 
       exmixer_disable_device(dev_id); 
       return FALSE; 
    } 
    
    /* Try to detect the device */ 
    if (!exmixer_init_device(emix[dev_id].dev, D_FILENAME)) 
    { 
       exmixer_disable_device(dev_id); 
       return FALSE; 
    } 
    
    first_time_round[dev_id] = 0; 
    return TRUE; 
} 
 
 
void print_js_settings( int dev_id ) 
{ 
    int axis_id, but_id; 
    char *s_assign = (char *)malloc(30); 
    
    if (!emix[dev_id].dev->enabled) 
       return; 
           
    Debug("\ndevice_id = %d", dev_id); 
    Debug("device_fd = %d", D_FD); 
    Debug("device_name = %s", D_NAME); 
    Debug("device_filename = %s", D_FILENAME); 
    Debug("  enabled = %d", D_ENABLED); 
    Debug("  num_axes = %d", N_AXES); 
    Debug("  num_buttons = %d\n", N_BUT); 
    for(axis_id=0; axis_id < N_AXES; axis_id++) 
    { 
       switch (A_ASSIGN) 
       { 
       case EM_A_NULL: 
           sprintf(s_assign, "nothing"); 
           break; 
       case EM_A_PITCH: 
           sprintf(s_assign, "pitch"); 
            break; 
        case EM_A_VOLUME: 
            sprintf(s_assign, "volume"); 
            break; 
        case EM_A_MAIN_VOLUME: 
            sprintf(s_assign, "main_volume"); 
            break; 
        case EM_A_CUE_VOLUME: 
            sprintf(s_assign, "cue_volume"); 
            break; 
        case EM_A_CROSS_FADE: 
            sprintf(s_assign, "cross_fade"); 
            break; 
        case EM_A_FADE_SPEED: 
            sprintf(s_assign, "fade_speed"); 
            break; 
        case EM_A_BALANCE: 
            sprintf(s_assign, "balance"); 
            break; 
        default: 
            sprintf(s_assign, "unknown, A_ASSIGN = %d", A_ASSIGN); 
            break; 
        } 
 
       Debug("  axis = %d", axis_id); 
       Debug("    axis_min = %d", A_MIN); 
       Debug("    axis_center = %d", A_CEN); 
       Debug("    axis_max = %d", A_MAX); 
       Debug("    axis_flip = %d", A_FLIP); 
       Debug("    axis_channel = %d", A_CHAN); 
       Debug("    axis_assigned = %s", s_assign); 
    } 
        
    for(but_id=0; but_id < N_BUT; but_id++) 
    { 
       switch (B_ASSIGN) 
       { 
       case EM_B_NULL: 
           sprintf(s_assign, "nothing"); 
           break; 
       case EM_B_MUTE: 
           sprintf(s_assign, "mute"); 
            break; 
        case EM_B_CUE: 
            sprintf(s_assign, "cue"); 
            break; 
        case EM_B_PLAY: 
            sprintf(s_assign, "play"); 
            break; 
        case EM_B_PAUSE: 
            sprintf(s_assign, "pause"); 
            break; 
        case EM_B_STOP: 
            sprintf(s_assign, "stop"); 
            break; 
        case EM_B_PUNCH_LEFT: 
            sprintf(s_assign, "punch left"); 
            break; 
        case EM_B_PUNCH_RIGHT: 
            sprintf(s_assign, "punch right"); 
            break; 
        case EM_B_FAST_FORWARD: 
            sprintf(s_assign, "fast forward"); 
            break; 
        case EM_B_REWIND: 
            sprintf(s_assign, "rewind"); 
            break; 
        case EM_B_EJECT: 
            sprintf(s_assign, "eject"); 
            break; 
        case EM_B_NEXT: 
            sprintf(s_assign, "next track"); 
            break; 
        case EM_B_PREV: 
            sprintf(s_assign, "previous track"); 
            break; 
        default: 
            sprintf(s_assign, "unknown, B_ASSIGN = %d", B_ASSIGN); 
            break; 
        } 
        
       Debug("  button = %d", but_id); 
       Debug("    button_channel = %d", B_CHAN); 
       Debug("    button_assigned = %s", s_assign); 
    } 
    free(s_assign); 
} 
 
void set_calibration( int c ) 
{ 
    calibration = c; 
} 
 
 
void exmixer_clear( void ) 
{ 
    int dev_id, axis_id, but_id; 
    
    Debug("Clearing exmixer."); 
    /* Set the mixer to none as default */    
    for (dev_id=0; dev_id<MAX_DEV; dev_id++) 
    { 
       emix[dev_id].dev = (js_device *)malloc( sizeof(js_device) ); 
       D_FD = -1; 
       D_GTAG = -1; 
       D_ENABLED = FALSE; 
       sprintf(D_NAME, "Unknown"); 
       sprintf(D_FILENAME, "/dev/js%d", dev_id); 
       N_AXES = 0; 
       N_BUT = 0; 
       emix[dev_id].axis = (js_axis *)malloc( sizeof(js_axis) * MAX_AXES ); 
       emix[dev_id].button = (js_button *)malloc( sizeof(js_button) * MAX_BUT); 
        
       for (axis_id=0; axis_id < N_AXES; axis_id++) 
       { 
           A_FLIP = FALSE; 
           A_ASSIGN = EM_A_NULL; 
           A_CHAN = EM_A_NULL; 
       } 
       for (but_id=0; but_id < N_BUT; but_id++) 
       { 
           B_ASSIGN = EM_B_NULL; 
           B_CHAN = EM_B_NULL; 
       } 
        
       /* Create the movement detection variables */ 
       axis_vol_lval[dev_id] = (float *)malloc(MAX_AXES); 
       axis_pitch_lval[dev_id] = (float *)malloc(MAX_AXES); 
    } 
    
} 
 
 
/* Disable the given deivce */ 
void exmixer_disable_device( int dev_id ) 
{ 
    Debug("Disabling device %d at %s", dev_id, D_FILENAME); 
    
    if (D_GTAG != -1) 
       gdk_input_remove( D_GTAG ); 
    
    D_GTAG = -1; 
    D_FD = -1; 
    D_ENABLED = FALSE; 
    N_BUT = 0; 
    N_AXES = 0; 
} 
 
 
/* Stop the exmixer */ 
void exmixer_stop( void ) 
{ 
    int dev_id; 
    
    Debug("Stopping exmixer."); 
    
    for (dev_id=0; dev_id<MAX_DEV; dev_id++) 
    { 
       exmixer_disable_device( dev_id ); 
    } 
} 
 
 
/* Detect and open a given exmixer device */ 
void exmixer_start_device( int dev_id ) 
{           
    exmixer_disable_device( dev_id ); 
    
    Debug("Starting exmixer device %d at %s.", dev_id, D_FILENAME); 
    
    D_ENABLED = TRUE; 
    
    /* Detect and open the device */ 
    if( exmixer_detect(dev_id) ) 
    { 
       D_GTAG = gdk_input_add( D_FD, GDK_INPUT_READ, em_read_device, GINT_TO_POINTER(dev_id) ); 
    } 
    else 
       D_GTAG = -1; 
} 
 
 
/* Start the main exmixer routine */ 
void exmixer_start( void ) 
{ 
    int dev_id; 
    
    if (!exmixer_enabled) 
       return; 
        
    num_channels = sysdata->num_channels; 
 
    /* Start the device reading thread whenever the input is ready */ 
    for (dev_id=0; dev_id<MAX_DEV; dev_id++) 
    { 
       /* Is the current device enabled? */ 
       if (D_ENABLED) 
       { 
           exmixer_start_device( dev_id ); 
       } 
       else 
           D_GTAG = -1; 
    } 
} 
