/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/
#include "ui_list.h"
#include "din.h"
#include "morse_code.h"
#include "console.h"
#include "curve_editor.h"
#include "tcl_interp.h"
#include "command.h"
#include "chrono.h"
#include "font_editor.h"
#include "key_consts.h"
#include "keyboard_keyboard.h"
#include "fft.h"
#include "viewwin.h"
#include "authors_note.h"

using namespace std;

extern console cons;
extern cmdlist cmdlst;
extern din din0;
extern morse_code mc;
extern curve_editor delayed;
extern string dotdin;
extern tcl_interp* p_interp;
extern font fnt;
extern curve_editor waved, comed, octed, veled, dmod_ed;

extern authors_note anote;

extern din din0;

extern keyboard_keyboard keybd2;
extern fft fft0;

extern ui_list uis;

extern audio_out aout;

extern int quit_now;

extern viewport view;

extern float ATTACK_TIME;
extern float DECAY_TIME;
extern float PITCH_BEND;
extern float NOTE_VOLUME;

ui_list::ui_list () :
fed (fnt, "font.ed"),
d_parameters ("parameters"),
lfs_attack_time (0, 50, "attack_time", 8, 70, 15, atv),
lfs_decay_time (0, 30, "decay_time", 8, 70, 15, dkv),
lfs_pitch_bend (0, 10, "pitch_bend", 8, 70, 15, pbv),
lfs_note_volume (0, 20, "note_volume", 8, 70, 15, nvv),
d_min_max ("min/max"),
mml_min ("min"), mml_max ("max"),
mmf_min (8), mmf_max(8),
lmm_attack (lfs_attack_time),
lmm_decay (lfs_decay_time),
lmm_pitch_bend (lfs_pitch_bend),
lmm_note_volume (lfs_note_volume) {

  current = 0;
  prev = 0;

  crved = 0;

  ui* scrs [] = {
    &din0,
    &din0.waved,
    &keybd2.waved,
    &din0.moded,
    &din0.gated,
    &delayed,
    &octed,
    &din0.droneed,
    &comed,
    &mc.ed,
    &dmod_ed,
    &fed,
    &keybd2.attacked,
    &keybd2.decayed,
    &keybd2.veled,
    0
  };

  for (ui** p = scrs; *p != 0; ++p) {
    uis.push_back (*p);
  }

  esct = -1;

  setup_widgets ();

}

void ui_list::set_current (ui* u, int ised) {

  if (current) {
    current->leave ();
    prev = current;
  }

  current = u;
  cons << console::yellow << "@ " << current->name << eol;
  cons.reset_tips ();
  current->enter ();

  if (ised) {
    crved = dynamic_cast<curve_editor*>(u);
    if (crved->sine_enabled) fft0.go (crved->curveinfo[0].curve);
  } else {
    crved = 0;
    uis[0] = current;
  }


}

bool ui_list::handle_input () {

  current->calc_win_mouse ();

  din0.bg (); // bg for handling phrasor on microtonal-keyboard
  keybd2.bg (); // bg for removing decayed notes on keyboard-keyboard
  cons.bg (); // bg for tips display

  for (vector<ui*>::size_type i = 1, j = uis.size(); i < j; ++i) uis[i]->bg (); // bg for editors

  // fade fx for voice, gater & delay
  //
  eval_fade (fdr_voice, w_voice);
  flash_gater (); // flashes in sync with gater value
  eval_fade (fdr_delay, w_delay);


  if (widget::focus == 0) { // find a widget that has focus
    for (vector<widget*>::size_type i = 0, j = widgets.size (); i < j; ++i) {
      widget* wi = widgets[i];
      if (wi->is_screen (current) && wi->visible () && wi->enabled ()) {
        if (wi->handle_input ()) {
          return true;
        }
      }
    }
  } else { // handle input of widget with focus
    widget* wf = widget::focus;
    if (wf->handle_input ()) return true;
  }

  current->handle_input ();

  if (keypressed (k_enter)) { // calc tap bpm
    extern nano_chrono ui_clk;
    extern double TAP_BPM;
    static double last_tapt = ui_clk.secs;
    double dt = ui_clk.secs - last_tapt;
    if (dt) TAP_BPM = 60.0 / dt;
    extern tcl_interp interpreter;
    if (keydown (k_lctrl)) cons ("set resetbeat 1");
    Tcl_UpdateLinkedVar (interpreter.interp, "tapbpm");
    last_tapt = ui_clk.secs;
  }

  extern string INSTRUMENT;

  if (keypressed (k_1)) { // switch instrument
    if (crved == 0) {
      if (current == &din0) {
        INSTRUMENT = "keyboard_keyboard";
        set_current (&keybd2, 0);
        cons ("setup-editors");
      } else {
        INSTRUMENT = "microtonal_keyboard";
        set_current (&din0, 0);
        cons ("setup-editors");
      }
    } else set_current (uis[0], 0);
  } else {

    for (int i = 0; i < MAX_EDITORS; ++i) {
      if (keypressed (key[i])) {
        if (ed[i]) set_current (ed[i]);
        break;
      }
    }
  }

  extern nano_chrono ui_clk;
  if (keypressed (k_esc)) {
    if (cons.command_mode || (current != uis[0])) {
      cons.command_mode = 0;
      set_current (uis[0], 0);
      esct = -1;
    } else {
      esc:
      if (esct == -1) {
        cons << console::red << "Press ESC again to quit" << eol;
        esct = ui_clk.secs;
      } else {
        static const float ESC_TIMEOUT = 1; // in seconds
        float dt = ui_clk.secs - esct;
        if (dt > ESC_TIMEOUT) {
          esct = -1;
          goto esc;
        } else {
          din0.save_scale ();
          din0.dinfo.save ();
          if (din0.delete_all_drones ()) {
            w_voice.turn_off ();
            w_delay.turn_off ();
            quit_now = 1;
            cons << console::green << "Goodbye!" << eol;
          }
        }
      }
    }
  }

  return true;
}

bool ui_list::set_editor (const string& name, int screen) {
  if (screen > 1) {
    ui* ei = 0;
    for (int i = 0, j = uis.size(); i < j; ++i) {
      if (uis[i]->name == name) {
        ei = uis[i];
        break;
      }
    }
    if (ei) {
      ed [screen - 2] = ei;
      return true;
    } else {
      cmdlst.result = "bad editor name";
      return false;
    }
  } else {
    cmdlst.result = "bad screen number (valid: 2 to 8)";
    return false;
  }
}

void ui_list::draw () {

  glClear (GL_COLOR_BUFFER_BIT);

  current->draw ();

  glMatrixMode (GL_PROJECTION);
  glPushMatrix ();
  glLoadIdentity ();
  glOrtho (0, view.xmax, 0, view.ymax, -1, 1);

  glMatrixMode (GL_MODELVIEW);
  glPushMatrix ();
  glLoadIdentity ();

  for (vector<widget*>::size_type i = 0, j = widgets.size (); i < j; ++i) {
    widget* wi = widgets[i];
    if (wi->is_screen (current) && wi->visible()) {
      wi->draw ();
    }
  }

  glPopMatrix ();
  glMatrixMode (GL_PROJECTION);
  glPopMatrix ();
  glMatrixMode (GL_MODELVIEW);

}

void ui_list::setup_widgets () {

  w_voice.add_screen (&din0);
  w_voice.set_listener (&vlis);

  w_gater.add_screen (&din0);
  w_gater.set_listener (&glis);

  w_delay.add_screen (&din0);
  w_delay.add_screen (&keybd2);
  w_delay.set_listener (&dlis);

  w_compress.add_screen (&din0);
  w_compress.add_screen (&keybd2);
  w_compress.set_listener (&clis);
  w_compress.color_blend (0);

  widgets.push_back (&w_voice);
  widgets.push_back (&w_gater);
  widgets.push_back (&w_delay);
  widgets.push_back (&w_compress);

  widget* chd [] = {&mml_min, &mml_max, &mmf_min, &mmf_max};
  for (int i = 0; i < 4; ++i) {
    widget& w = *chd[i];
    w.add_screen (&keybd2);
    d_min_max.add_child (&w);
  }

  d_min_max.add_screen (&keybd2);
  d_min_max.set_moveable (1);

  widgets.push_back (&d_min_max);
  for (int i = 0; i < 4; ++i) widgets.push_back (chd[i]);

  d_min_max.hide ();

  widget* pw1[5] = {&d_min_max, &mml_min, &mml_max, &mmf_min, &mmf_max};
  widget_load<widget> ("d_min_max", pw1, 5);
  mmf_min.set_text ("");
  mmf_max.set_text ("");

  lfs_attack_time.set_color (0.45, 0.85, 1);
  lfs_decay_time.set_color (1, 0.65, 0.45);
  lfs_pitch_bend.set_color (0.7, 1, 0.4);
  lfs_note_volume.set_color (1, 0.7, 1);

  lfs_attack_time.add_screen (&keybd2);
  lfs_decay_time.add_screen (&keybd2);
  lfs_pitch_bend.add_screen (&keybd2);
  lfs_note_volume.add_screen (&keybd2);

  label_field_slider<float>* lim[3] = {&lfs_attack_time, &lfs_decay_time, &lfs_note_volume};
  ifstream flim ((dotdin + "d_parameter_limits").c_str(), ios::in);
  float lo, hi;
  string name;
  for (int i = 0, j = 3; i < j; ++i) {
    flim >> name >> lo >> hi;
    label_field_slider<float>* lfs = lim[i];
    lfs->set_name (name);
    lfs->set_limits (lo, hi);
  }

  int ilo, ihi;
  flim >> name >> ilo >> ihi;
  lfs_pitch_bend.set_name (name);
  lfs_pitch_bend.set_limits (ilo, ihi);

  d_parameters.add_screen (&keybd2);
  d_parameters.add_child (&d_min_max);
  d_parameters.add_child (&lfs_attack_time);
  d_parameters.add_child (&lfs_decay_time);
  d_parameters.add_child (&lfs_pitch_bend);
  d_parameters.add_child (&lfs_note_volume);

  widget* pw2[5] = {&d_parameters, &lfs_attack_time, &lfs_decay_time, &lfs_pitch_bend, &lfs_note_volume};
  widget_load<widget> ("d_parameters", pw2, 5);

  d_parameters.set_moveable (1);

  widgets.push_back (&d_parameters);
  widgets.push_back (&lfs_attack_time);
  widgets.push_back (&lfs_decay_time);
  widgets.push_back (&lfs_pitch_bend);
  widgets.push_back (&lfs_note_volume);

  d_parameters.set_name ("parameters");
  lfs_attack_time.set_name ("attack_time");
  lfs_decay_time.set_name ("decay_time");
  lfs_pitch_bend.set_name ("pitch_bend");
  lfs_note_volume.set_name ("note_volume");

  cons.add_screen (0); // 0 for all
  widgets.push_back (&cons);

  w_voice.set_state (din0.dinfo.voice);
  w_gater.set_state (din0.dinfo.gater);
  w_delay.set_state (din0.dinfo.delay);
  w_compress.set_state (din0.dinfo.compress);

  anote.setup_widgets (widgets);

}

void ui_list::update_widgets () {

  w_voice.set_label ("Voice");
  w_gater.set_label ("Gater");
  w_delay.set_label ("Delay");
  w_compress.set_label ("Compressor");

  d_min_max.set_text ("min/max");
  mml_min.set_text ("min");
  mml_max.set_text("max");
  mmf_min.set_text (mmf_min.get_text());
  mmf_max.set_text (mmf_max.get_text());

  button* wd [] = {&w_voice, &w_gater, &w_delay, &w_compress};
  int n = 4, dx = 50, w = 0;
  for (int i = 0; i < n; ++i) w += wd[i]->get_extents ().width;
  w += n * dx;

  int lh = get_line_height ();
  int x = (view.xmax - w) / 2, y = lh + 2;
  for (int i = 0; i < n; ++i) {
    button& b = *wd[i];
    const box<int>& e = b.get_extents ();
    b.set_extents (x, y, x + e.width, y + e.height);
    b.set_pos (x, y);
    x = e.right + dx;
  }

  lfs_attack_time.set_val (ATTACK_TIME);
  lfs_decay_time.set_val (DECAY_TIME);
  lfs_pitch_bend.set_val (PITCH_BEND);
  lfs_note_volume.set_val (NOTE_VOLUME);

  d_parameters.set_text ("parameters");
  lfs_attack_time.update ();
  lfs_decay_time.update ();
  lfs_pitch_bend.update ();
  lfs_note_volume.update ();

  anote.update_widgets ();

}

void ui_list::flash_gater () {
  static color clr [2] = {checkbutton::off_color, checkbutton::on_color};
  float dg = aout.gatr [aout.samples_per_channel - 1];
  if (fdr_gater.eval ()) {
    w_gater.enable (0);
    w_gater.blend_on_off_color (fdr_gater.alpha);
    color& c = const_cast<color&> (w_gater.get_color ());
    c *= dg;
  } else {
    w_gater.enable (1);
    int gi = din0.dinfo.gater;
    w_gater.set_color (dg * clr [gi].r, dg * clr [gi].g, dg * clr[gi].b);
  }
}

float ui_list::eval_fade (fader& fdr, checkbutton& cb) {
  if (fdr.eval ()) {
    cb.enable (0);
    cb.blend_on_off_color (fdr.alpha);
  } else cb.enable (1);

  return fdr.amount;
}

widget* ui_list::get_widget (const string& name) {
  for (size_vw i = 0, j = widgets.size (); i < j; ++i) {
    widget* wi = widgets[i];
    if (wi->get_name() == name) return wi;
  }
  return 0;
}

void setup_fade (fader& fdr, int& target, int what) {
  if (fdr.on == 0) {
    target = what;
    if (what) fdr.set (0, 1); else fdr.set (1, 0);
  }
}

void voice__listener::changed (checkbutton& cb) {
  int what = cb.is_on ();
  setup_fade (uis.fdr_voice, din0.dinfo.voice, what);
}

void gater__listener::changed (checkbutton& cb) {
  int what = cb.is_on ();
  setup_fade (uis.fdr_gater, din0.dinfo.gater, what);
}

void delay__listener::changed (checkbutton& cb) {
  int what = cb.is_on ();
  setup_fade (uis.fdr_delay, din0.dinfo.delay, what);
}

void compress__listener::changed (checkbutton& cb) {
  din0.dinfo.compress = cb.is_on ();
}

ui_list::~ui_list () {

  widget* pw1[5] = {&d_parameters, &lfs_attack_time, &lfs_decay_time, &lfs_pitch_bend, &lfs_note_volume};
  widget_save<widget> ("d_parameters", pw1, 5);

  widget* pw2[5] = {&d_min_max, &mml_min, &mml_max, &mmf_min, &mmf_max};
  widget_save<widget> ("d_min_max", pw2, 5);

  label_field_slider<float>* lim[3] = {&lfs_attack_time, &lfs_decay_time, &lfs_note_volume};
  ofstream flim ((dotdin + "d_parameter_limits").c_str(), ios::out);
  float lo, hi;
  for (int i = 0, j = 3; i < j; ++i) {
    label_field_slider<float>* lfs = lim[i];
    lfs->get_limits (lo, hi);
    flim << lfs->get_name () << ' ' << lo << ' ' << hi << endl;
  }

  int ilo, ihi;
  lfs_pitch_bend.get_limits (ilo, ihi);
  flim << lfs_pitch_bend.get_name () << ' ' << ilo << ' ' << ihi << endl;

}

void attack_val::operator() (const float& f) {

  ATTACK_TIME = f;

}

void decay_val::operator() (const float& f) {
  DECAY_TIME = f;
}

void pitch_bend_val::operator() (const int& i) {
  PITCH_BEND = i;
}

void note_volume_val::operator() (const float& f) {
  NOTE_VOLUME = f;
  keybd2.calc_visual_params ();
}

template <typename T> void widget_load (const string& fname, T** pw, int n) {
  ifstream f ((dotdin + fname).c_str(), ios::in);
  string ignore; int x, y;
  for (int i = 0; i < n; ++i) {
    f >> ignore >> x >> y;
    pw[i]->set_pos (x, y);
  }
}

template <typename T> void widget_save (const string& fname, T** pw, int n) {
  ofstream f ((dotdin + fname).c_str(), ios::out);
  for (int i = 0; i < n; ++i) pw[i]->save (f);
}

template <typename T> min_max_clicked<T>::min_max_clicked (label_field_slider<T>& _lfs) : lfs (_lfs) {
  lfs.set_button_listener (this);
}

template <typename T> void min_max_clicked<T>::clicked (button& b) {

  // toggle
  if (uis.d_min_max.visible ()) {
    if (&b == ui_list::mm_last) uis.d_min_max.hide ();
  } else {
    uis.d_min_max.show ();
  }

  // label min/max
  uis.d_min_max.set_text (lfs.get_text ());

  // setup color
  widget* w [5] = {&uis.d_min_max, &uis.mml_min, &uis.mml_max, &uis.mmf_min, &uis.mmf_max};
  const color& clr = lfs.get_color ();
  for (int i = 0; i < 5; ++i) w[i]->set_color (clr);

  // load min/max fields
  T low, high;
  lfs.get_limits (low, high);

  uis.mmf_min.set_text (low);
  uis.mmf_max.set_text (high);

  // setup listeners
  uis.mmf_min.set_listener (this);
  uis.mmf_max.set_listener (this);

  ui_list::mm_last = &b;


}

template <typename T> void min_max_clicked<T>::changed (field& f) {

  stringstream ss;
  ss << f.get_text ();
  T t; ss >> t;
  T l, h; lfs.get_limits (l, h);

  if (&f == &uis.mmf_min) {
    lfs.set_limits (t, h);
  } else {
    lfs.set_limits (l, t);
  }

  lfs.get_limits (l, h);
  uis.mmf_min.set_text (l);
  uis.mmf_max.set_text (h);

}
