/*
 *  linux/lib/vsprintf.c
 *
 *  Copyright (C) 2004-2009 FAUmachine Team.
 *  Copyright (C) 1991-1992  Linus Torvalds
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 */

/* vsprintf.c -- Lars Wirzenius & Linus Torvalds. */
/*
 * Wirzenius wrote this portably, Torvalds fucked it up :-)
 */

/*
 * Modified for FAUmachine by Volkmar Sieh.
 */

#include "compiler.h"
CODE16;

#include <stdarg.h>
#include "string.h"

#include "const.h"

/* we use this so that we can do without the ctype library */
#define is_digit(c)	((c) >= '0' && (c) <= '9')

static int skip_atoi(const char **s)
{
	int i=0;

	while (is_digit(const_get(**s))) {
		i = i*10 + const_get(**s) - '0';
		(*s)++;
	}
	return i;
}

#define ZEROPAD	1		/* pad with zero */
#define SIGN	2		/* unsigned/signed long */
#define PLUS	4		/* show plus */
#define SPACE	8		/* space if plus */
#define LEFT	16		/* left justified */
#define LARGE	32		/* use 'ABCDEF' instead of 'abcdef' */

#define INT_MAX         ((int)(~0U>>1))
#define size_t unsigned int

#define do_div(n,base) ({ \
int __res; \
__res = ((unsigned long) n) % (unsigned) base; \
n = ((unsigned long) n) / (unsigned) base; \
__res; })

static char *
number(char *str, long num, int base, int size, int precision, int type)
{
	const char *digits="0123456789abcdefghijklmnopqrstuvwxyz";
	char c;
	char sign;
	char tmp[66];
	int i;

	if (type & LARGE)
		digits = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
	if (type & LEFT)
		type &= ~ZEROPAD;
	if (base < 2 || base > 36)
		return 0;
	c = (type & ZEROPAD) ? '0' : ' ';
	sign = 0;
	if (type & SIGN) {
		if (num < 0) {
			sign = '-';
			num = -num;
			size--;
		} else if (type & PLUS) {
			sign = '+';
			size--;
		} else if (type & SPACE) {
			sign = ' ';
			size--;
		}
	}
	i = 0;
	if (num == 0)
		tmp[i++]='0';
	else while (num != 0) {
#ifndef CONFIG_80286_SUPPORT
		tmp[i++] = const_get(digits[do_div(num,base)]);
#else
		int __res;
		__res = ((unsigned long) num) % (unsigned) base;
		num = ((unsigned long) num) / (unsigned) base;
		tmp[i++] = const_get(digits[__res]);
#endif
	}
	if (i > precision)
		precision = i;
	size -= precision;
	if (!(type&(ZEROPAD+LEFT)))
		while(size-->0)
			*str++ = ' ';
	if (sign)
		*str++ = sign;
	if (!(type & LEFT))
		while (size-- > 0)
			*str++ = c;
	while (i < precision--)
		*str++ = '0';
	while (i-- > 0)
		*str++ = tmp[i];
	while (size-- > 0)
		*str++ = ' ';
	return str;
}

int
vsprintf(char *buf, const char *fmt, va_list args)
{
	int len;
	unsigned long num;
	int i, base;
	char * str;
	const char *s;

	int flags;		/* flags to number() */

	int field_width;	/* width of output field */
	int precision;		/* min. # of digits for integers; max
				   number of chars for from string */
	int qualifier;		/* 'h', 'l', or 'L' for integer fields */

	for (str=buf ; const_get(*fmt) ; ++fmt) {
		if (const_get(*fmt) != '%') {
			*str++ = const_get(*fmt);
			continue;
		}
			
		/* process flags */
		flags = 0;
		repeat:
			++fmt;		/* this also skips first '%' */
			if (const_get(*fmt) == '-') {
				flags |= LEFT;
				goto repeat;
			} else if (const_get(*fmt) == '+') {
				flags |= PLUS;
				goto repeat;
			} else if (const_get(*fmt) == ' ') {
				flags |= SPACE;
				goto repeat;
			} else if (const_get(*fmt) == '0') {
				flags |= ZEROPAD;
				goto repeat;
			}
		
		/* get field width */
		field_width = -1;
		if (is_digit(const_get(*fmt)))
			field_width = skip_atoi(&fmt);
		else if (const_get(*fmt) == '*') {
			++fmt;
			/* it's the next argument */
			field_width = va_arg(args, int);
			if (field_width < 0) {
				field_width = -field_width;
				flags |= LEFT;
			}
		}

		/* get the precision */
		precision = -1;
		if (const_get(*fmt) == '.') {
			++fmt;	
			if (is_digit(const_get(*fmt)))
				precision = skip_atoi(&fmt);
			else if (const_get(*fmt) == '*') {
				++fmt;
				/* it's the next argument */
				precision = va_arg(args, int);
			}
			if (precision < 0)
				precision = 0;
		}

		/* get the conversion qualifier */
		qualifier = -1;
		if (const_get(*fmt) == 'h' || const_get(*fmt) == 'l' || const_get(*fmt) == 'L') {
			qualifier = const_get(*fmt);
			++fmt;
		}

		/* default base */
		base = 10;

		if (const_get(*fmt) == 'c') {
			if (!(flags & LEFT))
				while (--field_width > 0)
					*str++ = ' ';
			*str++ = (unsigned char) va_arg(args, int);
			while (--field_width > 0)
				*str++ = ' ';
			continue;

		} else if (const_get(*fmt) == 's') {
			unsigned int seg;

			seg = va_arg(args, unsigned int);
			s = va_arg(args, char *);
			if (!s)
				s = "<NULL>";

			for (len = 0;
			    (precision == -1 || len < precision)
					&& get_byte(seg, s + len) != '\0';
			    len++) {
			}

			if (!(flags & LEFT))
				while (len < field_width--)
					*str++ = ' ';
			for (i = 0; i < len; ++i)
				*str++ = get_byte(seg, s++);
			while (len < field_width--)
				*str++ = ' ';
			continue;

		} else if (const_get(*fmt) == 'p') {
			if (field_width == -1) {
				field_width = 2*sizeof(void *);
				flags |= ZEROPAD;
			}
			str = number(str,
				(unsigned long) va_arg(args, void *), 16,
				field_width, precision, flags);
			continue;

		} else if (const_get(*fmt) == '%') {
			*str++ = '%';
			continue;

		/* integer number formats - set up the flags and "break" */
		} else if (const_get(*fmt) == 'o') {
			base = 8;
			goto print_num;

		} else if (const_get(*fmt) == 'X') {
			flags |= LARGE;
			base = 16;
			goto print_num;

		} else if (const_get(*fmt) == 'x') {
			base = 16;
			goto print_num;

		} else if (const_get(*fmt) == 'd' || const_get(*fmt) == 'i') {
			flags |= SIGN;
			goto print_num;

		} else if (const_get(*fmt) == 'u') {
			goto print_num;

		} else {
			*str++ = '%';
			if (const_get(*fmt))
				*str++ = const_get(*fmt);
			else
				--fmt;
			continue;
		}
	print_num:
		if (qualifier == 'l')
			num = va_arg(args, unsigned long);
		else if (qualifier == 'h') {
			num = (unsigned short) va_arg(args, int);
			if (flags & SIGN)
				num = (short) num;
		} else if (flags & SIGN)
			num = va_arg(args, int);
		else
			num = va_arg(args, unsigned int);
		str = number(str, num, base, field_width, precision, flags);
	}
	*str = '\0';
	return str-buf;
}

int
sprintf(char * buf, const char *fmt, ...)
{
	va_list args;
	int i;

	va_start(args, fmt);
	i=vsprintf(buf,fmt,args);
	va_end(args);
	return i;
}
