package latexDraw.ui.components;

import java.awt.Point;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import javax.swing.JEditorPane;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;

import latexDraw.lang.LaTeXDrawLang;
import latexDraw.ui.dialog.ExceptionFrameDialog;


/** 
 * This class define a popup menu allowing to cut, paste and copy text from an {@link JEditorPane}.
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 10/24/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 * @since 1.9<br>
 */
public class CopyPasteMenu extends JPopupMenu implements ActionListener, MouseListener
{
	private static final long serialVersionUID = 1L;

	/** The editor to manage. */
	protected JEditorPane editor;
	
	public static final String LABEL_COPY 	= LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.40"); //$NON-NLS-1$
	
	public static final String LABEL_CUT	= LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.44"); //$NON-NLS-1$
	
	public static final String LABEL_PASTE 	= LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.43"); //$NON-NLS-1$
	
	protected JMenuItem copyM;
	
	protected JMenuItem cutM;
	
	protected JMenuItem pasteM;
	
	
	/**
	 * The constructor.
	 * @param edit The editor, must not be null.
	 */
	public CopyPasteMenu(JEditorPane edit)
	{
		if(edit==null)
			throw new IllegalArgumentException("Invalid editor.");//$NON-NLS-1$
		
		editor = edit;
		
		copyM = new JMenuItem(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.40")); //$NON-NLS-1$
		copyM.setActionCommand(LABEL_COPY);
		copyM.addActionListener(this);
		add(copyM);
		cutM = new JMenuItem(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.44")); //$NON-NLS-1$
		cutM.setActionCommand(LABEL_CUT);
		cutM.addActionListener(this);
		add(cutM);
		pasteM = new JMenuItem(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.43")); //$NON-NLS-1$
		pasteM.setActionCommand(LABEL_PASTE);
		pasteM.addActionListener(this);
		add(pasteM);
	}




	/**
	 * Update the {@link JMenuItem} of the menu.
	 * @since 1.9
	 */
	public void updateMenus()
	{
		boolean ok = editor.getSelectedText()!=null;
		
		try 
		{
			Transferable transf = editor.getToolkit().getSystemClipboard().getContents(null);
			Object o = transf.getTransferData(DataFlavor.stringFlavor);
			pasteM.setEnabled(o!=null && o instanceof String);
		}
		catch(Exception e) { pasteM.setEnabled(false); }
		
		copyM.setEnabled(ok);
		cutM.setEnabled(ok);
	}




	public void actionPerformed(ActionEvent e)
	{
		Object o = e.getSource();
		
		if(o instanceof JMenuItem)
		{
			String actionCmd = ((JMenuItem)o).getActionCommand();
			
			if(actionCmd.equals(LABEL_COPY))
			{
				editor.copy();
				return ;
			}
			
			if(actionCmd.equals(LABEL_CUT))
			{
				editor.cut();
				return ;
			}
			
			if(actionCmd.equals(LABEL_PASTE))
			{
				editor.paste();
				return ;
			}
			
			return ;
		}
		
	}
	
	
	public void mouseClicked(MouseEvent e)
	{
		try
		{
	    	switch(e.getButton())
		    {
			    case MouseEvent.BUTTON3 : 
			    			Point point = e.getPoint();
			    			updateMenus();
			    			show(e.getComponent(), point.x, point.y);   
							break;
		    }
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
    	e.consume();
	}


	public void mouseEntered(MouseEvent e)
	{
		/* No action to do. */
	}

	public void mouseExited(MouseEvent e)
	{
		/* No action to do. */
	}

	public void mousePressed(MouseEvent e)
	{
		/* No action to do. */
	}

	public void mouseReleased(MouseEvent e)
	{
		/* No action to do. */
	}
}
