/*--------------------------------------------------------------------*/
/*--- ALBERTA:  an Adaptive multi Level finite element toolbox using -*/
/*---           Bisectioning refinement and Error control by Residual */
/*---           Techniques for scientific Applications             ---*/
/*---                                                              ---*/
/*--- file: lagrange_1_1d.c                                        ---*/
/*---                                                              ---*/
/*--- description: implementation of the basis functions           ---*/
/*---              lagrange1 in 1d                                 ---*/
/*---                                                              ---*/
/*--- created by: kgs on host enigma                               ---*/
/*---           at 17:19 on 28 of March 2003                       ---*/
/*--------------------------------------------------------------------*/
/*---                                                              ---*/
/*--- authors:   Alfred Schmidt                                    ---*/
/*---            Zentrum fuer Technomathematik                     ---*/
/*---            Fachbereich 3 Mathematik/Informatik               ---*/
/*---            Universitaet Bremen                               ---*/
/*---            Bibliothekstr. 2                                  ---*/
/*---            D-28359 Bremen, Germany                           ---*/
/*---                                                              ---*/
/*---            Kunibert G. Siebert                               ---*/
/*---            Institut fuer Mathematik                          ---*/
/*---            Universitaet Augsburg                             ---*/
/*---            Universitaetsstr. 14                              ---*/
/*---            D-86159 Augsburg, Germany                         ---*/
/*---                                                              ---*/
/*--- http://www.mathematik.uni-freiburg.de/IAM/ALBERTA            ---*/
/*---                                                              ---*/
/*--- (c) by A. Schmidt and K.G. Siebert (1996-2003)               ---*/
/*---                                                              ---*/
/*--------------------------------------------------------------------*/

static const REAL   bary1_1d[2][N_LAMBDA] = {{1.0, 0.0, 0.0, 0.0},
					     {0.0, 1.0, 0.0, 0.0}};

/*--------------------------------------------------------------------------*/
/*---  basisfunction 0 located at vertex 0                               ---*/
/*--------------------------------------------------------------------------*/

static REAL phi1_0_1d(const REAL lambda[N_LAMBDA])
{
  return(lambda[0]);
}

static const REAL *grd_phi1_0_1d(const REAL lambda[N_LAMBDA])
{
  static const REAL  grd[N_LAMBDA] = {1.0, 0.0, 0.0, 0.0};

  return(grd);
}

static const REAL (*D2_phi1_0_1d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0.0}};

  return(D2);
}

/*--------------------------------------------------------------------------*/
/*---  basisfunction 1 located at vertex 1                               ---*/
/*--------------------------------------------------------------------------*/

static REAL phi1_1_1d(const REAL lambda[N_LAMBDA])
{
  return(lambda[1]);
}

static const REAL *grd_phi1_1_1d(const REAL lambda[N_LAMBDA])
{
  static const REAL  grd[N_LAMBDA] = {0.0, 1.0, 0.0, 0.0};

  return(grd);
}

static const REAL (*D2_phi1_1_1d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0.0}};

  return(D2);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing local DOFs on an element              ---*/
/*--------------------------------------------------------------------*/

static const DOF *get_dof_indices1_1d(const EL *el, const DOF_ADMIN *admin,
				      DOF *idof)
{
  static DOF  dof_vec[2];
  DOF         *rvec = idof ? idof : dof_vec;
  int         ibas = 0, i, n0, node;
  DOF         **dof = el->dof;

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = admin->mesh->node[VERTEX];
  n0   = admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = dof[node+i][n0];


  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing boundary type of DOFs                 ---*/
/*--------------------------------------------------------------------*/

static const S_CHAR *get_bound1_1d(const EL_INFO *el_info, S_CHAR *vec)
{
  FUNCNAME("get_bound1_1d");
  static S_CHAR  my_vec[2];
  S_CHAR         *rvec = vec ? vec : my_vec;
  int            i;

  DEBUG_TEST_FLAG(FILL_BOUND, el_info);

/*--------------------------------------------------------------------*/
/*--- basis functions at vertices                                  ---*/
/*--------------------------------------------------------------------*/

  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[i] = el_info->vertex_bound[i];

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of scalar functions          ---*/
/*--------------------------------------------------------------------*/

static const REAL *interpol1_1d(const EL_INFO *el_info,
				int no, const int *b_no,
				REAL (*f)(const REAL_D),
				REAL (*f_loc)(const EL_INFO *,
					      const REAL [N_LAMBDA]), 
				REAL *vec)
{
  FUNCNAME("interpol1_1d");
  static REAL       my_vec[2];
  REAL             *rvec = vec ? vec : my_vec;
  int               i;
  const PARAMETRIC *parametric = el_info->mesh->parametric;

  DEBUG_TEST_EXIT(!b_no || (no > 0 && no <= 2), "not for %d points\n", no);

  if(f_loc) /* Evaluation in local coordinates */
    for (i = 0; i < 2; i++)
      rvec[i] = f_loc(el_info, bary1_1d[i]);
  else { /* Evaluation in world coordinates */
    if (parametric) {
      REAL_D     world[2];

      parametric->init_element(el_info, parametric);
      parametric->coord_to_world(el_info, nil, 2, bary1_1d, world);
      
      for (i = 0; i < 2; i++)
	rvec[i] = f(world[i]);
    }
    else { /* Vertex coordinates do not have to be calculated. */
      DEBUG_TEST_FLAG(FILL_COORDS, el_info);
      
      rvec[0] = f(el_info->coord[0]);
      rvec[1] = f(el_info->coord[1]);
    }
  }

  if(b_no) { /* Perform resorting if only certain indices are required. */
    REAL tmp[2];

    memcpy(tmp, rvec, 2 * sizeof(REAL));
    
    for(i = 0; i < no; i++)
      rvec[i] = tmp[b_no[i]];
  }

  
  return(rvec);
}


/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of vector functions          ---*/
/*--------------------------------------------------------------------*/

static const REAL_D *interpol_d1_1d(const EL_INFO *el_info,
				    int no, const int *b_no,
				    const REAL *(*f)(const REAL_D, REAL_D),
				    const REAL *(*f_loc)(const EL_INFO *,
							 const REAL [N_LAMBDA],
							 REAL_D),
				    REAL_D *vec)
{
  FUNCNAME("interpol_d1_1d");
  static REAL_D     my_vec[2];
  REAL_D           *rvec = vec ? vec : my_vec;
  int               i;
  const PARAMETRIC *parametric = el_info->mesh->parametric;

  DEBUG_TEST_EXIT(!b_no || (no > 0 && no <= 2), "not for %d points\n", no);

  if(f_loc) /* Evaluation in local coordinates */
    for (i = 0; i < 2; i++)
      f_loc(el_info, bary1_1d[i], rvec[i]);
  else { /* Evaluation in world coordinates */
    if (parametric) {
      REAL_D     world[2];

      parametric->init_element(el_info, parametric);
      parametric->coord_to_world(el_info, nil, 2, bary1_1d, world);
      
      for (i = 0; i < 2; i++)
	f(world[i], rvec[i]);
    }
    else { /* Vertex coordinates do not have to be calculated. */
      DEBUG_TEST_FLAG(FILL_COORDS, el_info);

      f(el_info->coord[0], rvec[0]);
      f(el_info->coord[1], rvec[1]);
    }
  }

  if(b_no) { /* Perform resorting if only certain indices are required. */
    REAL_D tmp[2];

    memcpy(tmp, rvec, 2 * sizeof(REAL_D));
    
    for(i = 0; i < no; i++)
      COPY_DOW(tmp[b_no[i]], rvec[i]);
  }

  return((const REAL_D *) rvec);
}
/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_INT_VEC                   ---*/
/*--------------------------------------------------------------------*/

static const int *get_int_vec1_1d(const EL *el, const DOF_INT_VEC *dv,
				  int *vec)
{
  FUNCNAME("get_int_vec1_1d");
  static int     my_vec[2];
  int            *rvec = vec ? vec : my_vec;
  int            *v = dv ? dv->vec : nil;
  int            ibas = 0, i, n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v, "no DOF_INT_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = v[dof[node+i][n0]];

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_REAL_VEC                  ---*/
/*--------------------------------------------------------------------*/

static const REAL *get_real_vec1_1d(const EL *el, const DOF_REAL_VEC *dv,
				    REAL *vec)
{
  FUNCNAME("get_real_vec1_1d");
  static REAL    my_vec[2];
  REAL           *rvec = vec ? vec : my_vec;
  REAL           *v = dv ? dv->vec : nil;
  int            ibas = 0, i, n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v, "no DOF_REAL_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = v[dof[node+i][n0]];

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_REAL_D_VEC                ---*/
/*--------------------------------------------------------------------*/

static const REAL_D *get_real_d_vec1_1d(const EL *el, const DOF_REAL_D_VEC *dv,
					REAL_D *vec)
{
  FUNCNAME("get_real_d_vec1_1d");
  static REAL_D  my_vec[2];
  REAL_D         *rvec = vec ? vec : my_vec;
  REAL_D         *v = dv ? dv->vec : nil;
  int            ibas = 0, i, n, n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v, "no DOF_REAL_D_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
  {
    for (n = 0; n < DIM_OF_WORLD; n++)
      rvec[ibas][n] = v[dof[node+i][n0]][n];
    ibas++;
  }

  return((const REAL_D *) rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_SCHAR_VEC                 ---*/
/*--------------------------------------------------------------------*/

static const S_CHAR *get_schar_vec1_1d(const EL *el, const DOF_SCHAR_VEC *dv,
				       S_CHAR *vec)
{
  FUNCNAME("get_schar_vec1_1d");
  static S_CHAR  my_vec[2];
  S_CHAR         *rvec = vec ? vec : my_vec;
  S_CHAR         *v = dv ? dv->vec : nil;
  int            ibas = 0, i, n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v, "no DOF_SCHAR_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = v[dof[node+i][n0]];

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_UCHAR_VEC                 ---*/
/*--------------------------------------------------------------------*/

static const U_CHAR *get_uchar_vec1_1d(const EL *el, const DOF_UCHAR_VEC *dv,
				       U_CHAR *vec)
{
  FUNCNAME("get_uchar_vec1_1d");
  static U_CHAR  my_vec[2];
  U_CHAR         *rvec = vec ? vec : my_vec;
  U_CHAR         *v = dv ? dv->vec : nil;
  int            ibas = 0, i, n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v, "no DOF_UCHAR_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = v[dof[node+i][n0]];

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for interpolaton DOF_REAL_VECs during refinement    ---*/
/*--------------------------------------------------------------------*/

static void refine_inter1_1d(DOF_REAL_VEC *dv, RC_LIST_EL *list, int n_el)
{
  EL              *el;
  const DOF       *cdof;
  const REAL      *pvec;
  const BAS_FCTS  *bas_fcts = dv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = dv->fe_space->admin;
  REAL            *v = dv->vec;

  el = list->el_info.el;
  pvec = (*bas_fcts->get_real_vec)(el, dv, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, first child                           ---*/
/*--------------------------------------------------------------------*/

  cdof = (*bas_fcts->get_dof_indices)(el->child[0], admin, nil);

  v[cdof[1]] = (0.5*pvec[0] + 0.5*pvec[1]);

  return;
}

/*--------------------------------------------------------------------*/
/*--- function for interpolaton DOF_REAL_D_VECs during refinement  ---*/
/*--------------------------------------------------------------------*/

static void refine_inter_d1_1d(DOF_REAL_D_VEC *dv, RC_LIST_EL *list, int n_el)
{
  EL              *el;
  const DOF       *cdof;
  const REAL_D    *pvec;
  int             n;
  const BAS_FCTS  *bas_fcts = dv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = dv->fe_space->admin;
  REAL_D          *v = dv->vec;

  el = list->el_info.el;
  pvec = (*bas_fcts->get_real_d_vec)(el, dv, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, first child                           ---*/
/*--------------------------------------------------------------------*/

  cdof = (*bas_fcts->get_dof_indices)(el->child[0], admin, nil);

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    v[cdof[1]][n] = (0.5*pvec[0][n] + 0.5*pvec[1][n]);
  }

  return;
}

/*--------------------------------------------------------------------*/
/*--- function for interpolaton DOF_REAL_VECs during coarsening    ---*/
/*--------------------------------------------------------------------*/

static void coarse_restr1_1d(DOF_REAL_VEC *dv, RC_LIST_EL *list, int n_el)
{
  EL              *el;
  const DOF       *pdof;
  const REAL      *cvec;
  const BAS_FCTS  *bas_fcts = dv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = dv->fe_space->admin;
  REAL            *v = dv->vec;

  el = list->el_info.el;
  pdof = (*bas_fcts->get_dof_indices)(el, admin, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, from first child                      ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_vec)(el->child[0], dv, nil);

  v[pdof[0]] += (0.5*cvec[1]);
  v[pdof[1]] += (0.5*cvec[1]);

  return;
}

/*--------------------------------------------------------------------*/
/*--- function for interpolaton DOF_REAL_D_VECs during coarsening  ---*/
/*--------------------------------------------------------------------*/

static void coarse_restr_d1_1d(DOF_REAL_D_VEC *dv, RC_LIST_EL *list, int n_el)
{
  EL              *el;
  const DOF       *pdof;
  const REAL_D    *cvec;
  int             n;
  const BAS_FCTS  *bas_fcts = dv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = dv->fe_space->admin;
  REAL_D          *v = dv->vec;

  el = list->el_info.el;
  pdof = (*bas_fcts->get_dof_indices)(el, admin, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, from first child                      ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_d_vec)(el->child[0], dv, nil);

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    v[pdof[0]][n] += (0.5*cvec[1][n]);
    v[pdof[1]][n] += (0.5*cvec[1][n]);
  }

  return;
}


/*--------------------------------------------------------------------*/
/*--- Collect all information about basis functions                ---*/
/*--------------------------------------------------------------------*/

static BAS_FCT *phi1_1d[2] =
{
  phi1_0_1d, phi1_1_1d
};

static GRD_BAS_FCT *grd_phi1_1d[2] =
{
  grd_phi1_0_1d, grd_phi1_1_1d
};
static D2_BAS_FCT *D2_phi1_1d[2] =
{
  D2_phi1_0_1d, D2_phi1_1_1d
};

static BAS_FCTS lagrange1_1d =
{
  "lagrange1_1d", 1, 2, 1, 
  {1, 0, 0, 0},    /* VERTEX, CENTER, EDGE, FACE   */
  nil,
  phi1_1d, grd_phi1_1d, D2_phi1_1d,
  get_dof_indices1_1d,
  get_bound1_1d,
  interpol1_1d,
  interpol_d1_1d,
  get_int_vec1_1d,
  get_real_vec1_1d,
  get_real_d_vec1_1d,
  get_uchar_vec1_1d,
  get_schar_vec1_1d,
  refine_inter1_1d,
  nil,
  coarse_restr1_1d,
  refine_inter_d1_1d,
  nil,
  coarse_restr_d1_1d,
  bary1_1d,
};
