/* Time-stamp: <2008-10-19 00:14:01 poser> */

/*
 * Copyright (C) 2007, 2008 William J. Poser.
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 3 of the GNU General Public License
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 * or go to the web page:  http://www.gnu.org/licenses/gpl.txt.
 */

/* These are functions for interpreting ISO8601 time/date stamps */

#include "config.h"
#include "compdefs.h"
#include <stdlib.h>
#ifdef HAVE_UNINUM_UNICODE_H
#include <uninum/unicode.h>
#else
#include "unicode.h"
#endif
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif
#include <stdio.h>
#include <string.h>
#ifdef LOCALE_GETTEXT
#include <libintl.h>
#define _(x) gettext(x)
#else
#define _(x) (x)
#endif
#include <time.h>
#include <wchar.h>
#include <errno.h>
#include "retcodes.h"

#ifdef HAVE_LONGDOUBLE
#define DOUBLE (long double)
#else
#define DOUBLE double
#endif


/*
 * Return the difference between UTC and the local time zone, in seconds.
 * Add this value to local time to obtain UTC.
 *
 * We call localtime() rather than tzset because localtime is guaranteed by
 * POSIX to set timezone while tzset is not.
 */

static long
TimezoneOffset(void) {
  time_t lt;
  /* timezone is defined in time.h */

  lt = time(NULL);
  (void) localtime(&lt);
  return((long) timezone);
}


/* 
 * Given a time as a string in one of the following formats, return the value in seconds:
 * 	hh
 * 	hhmm
 *	hh:mm
 * If the expression is ill-formed, set the error flag.
 */

static long BrokenTimeToSeconds (char *s, short *eflag) {
  int hours = 0;
  int minutes = 0;
  int FieldsFound;
  short status = 1;		/* No valid return value yet */


  FieldsFound = sscanf((char *)s,"%2u:%2u",&hours,&minutes);
  if(FieldsFound == 2) {
    status = 0;
    goto end;
  }

  FieldsFound = sscanf((char *)s,"%2u%2u",&hours,&minutes);
  if(FieldsFound == 2) {
    status = 0;
    goto end;
  }
  if(FieldsFound == 1) status = 0;

end:
  if (minutes > 59) status = 1;
  if (hours > 24) status = 1;
  *eflag = status;
  if (status == 0)  return (((hours * 60) + minutes) * 60);
  else return 0;
}

/* Extract the time zone info if any and return the number of seconds
 * that must be added to normalize the time to UTC. 
 * The time zone marker (Z,z,-,+) is ovewritten with a null,
 * modifying the original string.
 */

long ExtractTimeZone(UTF8 *s, short *eflag) {
  char *sep;
  long Offset;
  short err = 0;

  /* Z/z means the value is already UTC  */
  if ((sep = strchr((char *) s,'Z'))) {
    *sep = '\0';
    return 0;
  }

  if ((sep = strchr((char *) s,'z'))) {
    *sep = '\0';
    return 0;
  }

  /* + means subtract the following offset to get UTC  */
  if ((sep = strchr((char *) s,'+'))) {
    Offset = BrokenTimeToSeconds(sep+1,&err);
    *eflag = err;
    *sep = '\0';
    return (-Offset);
  }
  
  /* - means add the following offset to get UTC  */
  if ((sep = strchr((char *) s,'-'))) {
    Offset = BrokenTimeToSeconds(sep+1,&err);
    *eflag = err;
    *sep = '\0';
    return (Offset);
  }

  /*
   * If there is no time zone indicator, we default to the time zone of the
   * environment of this process.
   */

  return(TimezoneOffset());
}

/*
 * Remove hyphen-minus and colon from time-date strings
 * Return the adjusted location of the T.
 */

char *
StripTimeDateSeparators(char *s, char *tloc) {
  char *src;
  char *tgt;
  char c;
  int cnt = 0;			/* Number of hyphens stripped */

  src = tgt = s;
  while (src < tloc) {
    c = *src++;
    if (c == '-') {cnt++;continue;}
    *tgt++ = c;
  }
  while ( (c = *src++) != '\0') {
    if (c == ':') continue;
    *tgt++ = c;
  }
  *tgt = '\0';
  return(tloc-cnt);
}

/* 
 * This produces a value that is suitable for ordering dates and times.
 * It is NOT accurate for computing differences between dates and times
 * because the conversion of dates to days is not exact. For simplicity's
 * sake it treats years as containing 366 days and months as all containing
 * 31 days.
 *
 * This handles both basic and extended formats. It also permits an optional leading sign,
 * restricts the year to four digits.
 */
#define DTITYPE unsigned long
#define SECONDSINDAY (60*60*24)
int
GetISO8601Key(wchar_t *field, DOUBLE *key)
{
  unsigned int year = 0;
  unsigned int month =0;
  unsigned int day = 0;
  unsigned int hour = 0;
  unsigned int minute = 0;
  unsigned int second = 0;

  long TZOffset;		/* Number of seconds to add to normalize to UTC */
  DTITYPE Days;
  DTITYPE DaySeconds;
  int f1, f2;
  char *cfield;
  char *nscfield;		/* cfield without optional leading sign */
  char * Tloc;
  short err;
  int YearSign = 1;

  extern UTF8 * ws2u8(wchar_t *);

  cfield = (char *) ws2u8(field);
  Tloc = strchr(cfield,'T');
  if(!Tloc) {free(cfield);return(ERROR);}
  Tloc = StripTimeDateSeparators(cfield,Tloc);
  err=0;
  TZOffset = ExtractTimeZone(((UTF8 *)Tloc)+1,&err);
  if(err) return(ERROR);

  if(cfield[0] == '-') {
    YearSign = -1;
    nscfield = cfield + 1;
  }
  else if(cfield[0] == '+') nscfield = cfield + 1;
  else nscfield = cfield;

  f1=sscanf(nscfield,"%4u%2u%2uT%2u%2u.%2u", &year,&month,&day,&hour,&minute,&second);
  if(f1 != 6) {
    f2=sscanf(nscfield,"%4u%2u%2uT%2u%2u", &year,&month,&day,&hour,&minute);
    if(f2 != 5) {free(cfield);return(ERROR);}
  }
  free(cfield);
  if(month < 0 || day < 0) return (ERROR);
  if(month > 12) return (ERROR);
  if(day > 31) return (ERROR);
  if(hour < 0 || minute < 0 || second < 0) return (ERROR);
  if(hour > 24) return(ERROR);
  if(minute > 60 || second > 60) return (ERROR);

#ifdef UNNECESSARY
  /* 
   * If making actual date calculations, the following is necessary as per the spec.
   * That is, 2007-04-05T24:00 is the same instant as 2007-04-06T00:00
   * For our purposes it is unnecessary as it has no effect on ordering,
   * so we'd might as well be slightly more efficient and omit it.
   */
  if ((hour == 24) && (minute == 0)) {
    hour = 0;
    day += 1;
  }
#endif

  Days = (DTITYPE) ( (366 * (DTITYPE) year) + (31 * month) + day);
  DaySeconds = (DTITYPE) (((((DTITYPE) hour * 60) + (DTITYPE) minute) * 60) + (DTITYPE) second);
  *key = (DOUBLE) ((DOUBLE) YearSign * ( ( (DOUBLE)Days * (DOUBLE)SECONDSINDAY) + (DOUBLE)DaySeconds + (DOUBLE)TZOffset));
  return(SUCCESS);
}
