/*
 *@BEGIN LICENSE
 *
 * PSI4: an ab initio quantum chemistry software package
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 *@END LICENSE
 */

#include <libmints/vector.h>
#include "VWN3_Cfunctional.h"
#include "utility.h"
#include <cmath>

using namespace psi;

namespace psi {

VWN3_CFunctional::VWN3_CFunctional()
{
    name_ = "VWN3_C";
    description_ = "    VWN3 Correlation Functional\n";
    citation_ = "    S.H. Vosko, L. Wilk, and M. Nusair, Can. J. Phys., 58, 1200-1211, 1980\n";
    alpha_ = 1.0;
    omega_ = 0.0;
    lrc_ = false;
    gga_ = false;
    meta_ = false;
    parameters_["c"] =   6.2035049089939986E-01;
    parameters_["EcP_1"] =   3.1090699999999999E-02;
    parameters_["EcP_2"] =  -1.0498000000000000E-01;
    parameters_["EcP_3"] =   3.7274400000000001E+00;
    parameters_["EcP_4"] =   1.2935200000000000E+01;
    parameters_["EcF_1"] =   1.5545349999999999E-02;
    parameters_["EcF_2"] =  -3.2500000000000001E-01;
    parameters_["EcF_3"] =   7.0604199999999997E+00;
    parameters_["EcF_4"] =   1.8057800000000000E+01;
    parameters_["two_13"] =   1.2599210498948732E+00;
}
VWN3_CFunctional::~VWN3_CFunctional()
{
}
void VWN3_CFunctional::compute_functional(const std::map<std::string,SharedVector>& in, const std::map<std::string,SharedVector>& out, int npoints, int deriv, double alpha)
{
    double c = parameters_["c"];
    double EcP_1 = parameters_["EcP_1"];
    double EcP_2 = parameters_["EcP_2"];
    double EcP_3 = parameters_["EcP_3"];
    double EcP_4 = parameters_["EcP_4"];
    double EcF_1 = parameters_["EcF_1"];
    double EcF_2 = parameters_["EcF_2"];
    double EcF_3 = parameters_["EcF_3"];
    double EcF_4 = parameters_["EcF_4"];
    double two_13 = parameters_["two_13"];

    // Overall scale factor
    double scale = alpha_ * alpha;

    // => Input variables <= //

    double* rho_ap = NULL;
    double* rho_bp = NULL;
    double* gamma_aap = NULL;
    double* gamma_abp = NULL;
    double* gamma_bbp = NULL;
    double* tau_ap = NULL;
    double* tau_bp = NULL;

    if (true) {
        rho_ap = in.find("RHO_A")->second->pointer();
        rho_bp = in.find("RHO_B")->second->pointer();
    }
    if (gga_) {  
        gamma_aap = in.find("GAMMA_AA")->second->pointer();
        gamma_abp = in.find("GAMMA_AB")->second->pointer();
        gamma_bbp = in.find("GAMMA_BB")->second->pointer();
    } 
    if (meta_)  {
        tau_ap = in.find("TAU_A")->second->pointer();
        tau_bp = in.find("TAU_B")->second->pointer();
    }

    // => Outut variables <= //

    double* v = NULL;

    double* v_rho_a = NULL;
    double* v_rho_b = NULL;
    double* v_gamma_aa = NULL;
    double* v_gamma_ab = NULL;
    double* v_gamma_bb = NULL;
    double* v_tau_a = NULL;
    double* v_tau_b = NULL;
     
    double* v_rho_a_rho_a = NULL;
    double* v_rho_a_rho_b = NULL;
    double* v_rho_b_rho_b = NULL;
    double* v_gamma_aa_gamma_aa = NULL;
    double* v_gamma_aa_gamma_ab = NULL;
    double* v_gamma_aa_gamma_bb = NULL;
    double* v_gamma_ab_gamma_ab = NULL;
    double* v_gamma_ab_gamma_bb = NULL;
    double* v_gamma_bb_gamma_bb = NULL;
    double* v_tau_a_tau_a = NULL;
    double* v_tau_a_tau_b = NULL;
    double* v_tau_b_tau_b = NULL;
    double* v_rho_a_gamma_aa = NULL;
    double* v_rho_a_gamma_ab = NULL;
    double* v_rho_a_gamma_bb = NULL;
    double* v_rho_b_gamma_aa = NULL;
    double* v_rho_b_gamma_ab = NULL;
    double* v_rho_b_gamma_bb = NULL;
    double* v_rho_a_tau_a = NULL;
    double* v_rho_a_tau_b = NULL;
    double* v_rho_b_tau_a = NULL;
    double* v_rho_b_tau_b = NULL;
    double* v_gamma_aa_tau_a = NULL;
    double* v_gamma_aa_tau_b = NULL;
    double* v_gamma_ab_tau_a = NULL;
    double* v_gamma_ab_tau_b = NULL;
    double* v_gamma_bb_tau_a = NULL;
    double* v_gamma_bb_tau_b = NULL;

    if (deriv >= 0) {
        v = out.find("V")->second->pointer();
    } 
    if (deriv >= 1) {
        if (true) {
            v_rho_a = out.find("V_RHO_A")->second->pointer();
            v_rho_b = out.find("V_RHO_B")->second->pointer();
        }
        if (gga_) {
            v_gamma_aa = out.find("V_GAMMA_AA")->second->pointer();
            v_gamma_ab = out.find("V_GAMMA_AB")->second->pointer();
            v_gamma_bb = out.find("V_GAMMA_BB")->second->pointer();
        }
        if (meta_) {    
            v_tau_a = out.find("V_TAU_A")->second->pointer();
            v_tau_b = out.find("V_TAU_B")->second->pointer();
        }
    }
    if (deriv >= 2) {
        if (true) {
            v_rho_a_rho_a = out.find("V_RHO_A_RHO_A")->second->pointer();
            v_rho_a_rho_b = out.find("V_RHO_A_RHO_B")->second->pointer();
            v_rho_b_rho_b = out.find("V_RHO_B_RHO_B")->second->pointer();
        }
        if (gga_) {
            v_gamma_aa_gamma_aa = out.find("V_GAMMA_AA_GAMMA_AA")->second->pointer();
            v_gamma_aa_gamma_ab = out.find("V_GAMMA_AA_GAMMA_AB")->second->pointer();
            v_gamma_aa_gamma_bb = out.find("V_GAMMA_AA_GAMMA_BB")->second->pointer();
            v_gamma_ab_gamma_ab = out.find("V_GAMMA_AB_GAMMA_AB")->second->pointer();
            v_gamma_ab_gamma_bb = out.find("V_GAMMA_AB_GAMMA_BB")->second->pointer();
            v_gamma_bb_gamma_bb = out.find("V_GAMMA_BB_GAMMA_BB")->second->pointer();
        }
        if (meta_) {
            v_tau_a_tau_a = out.find("V_TAU_A_TAU_A")->second->pointer();
            v_tau_a_tau_b = out.find("V_TAU_A_TAU_B")->second->pointer();
            v_tau_b_tau_b = out.find("V_TAU_B_TAU_B")->second->pointer();
        }
        if (gga_) {
            v_rho_a_gamma_aa = out.find("V_RHO_A_GAMMA_AA")->second->pointer();
            v_rho_a_gamma_ab = out.find("V_RHO_A_GAMMA_AB")->second->pointer();
            v_rho_a_gamma_bb = out.find("V_RHO_A_GAMMA_BB")->second->pointer();
            v_rho_b_gamma_aa = out.find("V_RHO_B_GAMMA_AA")->second->pointer();
            v_rho_b_gamma_ab = out.find("V_RHO_B_GAMMA_AB")->second->pointer();
            v_rho_b_gamma_bb = out.find("V_RHO_B_GAMMA_BB")->second->pointer();
        }
        if (meta_) {
            v_rho_a_tau_a = out.find("V_RHO_A_TAU_A")->second->pointer();
            v_rho_a_tau_b = out.find("V_RHO_A_TAU_B")->second->pointer();
            v_rho_b_tau_a = out.find("V_RHO_B_TAU_A")->second->pointer();
            v_rho_b_tau_b = out.find("V_RHO_B_TAU_B")->second->pointer();
        }
        if (gga_ && meta_) {
            v_gamma_aa_tau_a = out.find("V_GAMMA_AA_TAU_A")->second->pointer();
            v_gamma_aa_tau_b = out.find("V_GAMMA_AA_TAU_B")->second->pointer();
            v_gamma_ab_tau_a = out.find("V_GAMMA_AB_TAU_A")->second->pointer();
            v_gamma_ab_tau_b = out.find("V_GAMMA_AB_TAU_B")->second->pointer();
            v_gamma_bb_tau_a = out.find("V_GAMMA_BB_TAU_A")->second->pointer();
            v_gamma_bb_tau_b = out.find("V_GAMMA_BB_TAU_B")->second->pointer();
        }
    }

    // => Loop over points <= //

    for (int Q = 0; Q < npoints; Q++) {

        // Input variables 
        double rho_a;
        double rho_b;
        double gamma_aa;
        double gamma_ab;
        double gamma_bb;
        double tau_a;
        double tau_b;

        if (true) {
            rho_a = rho_ap[Q];
            rho_b = rho_bp[Q];
        }        
        if (gga_) {
            gamma_aa = gamma_aap[Q];
            gamma_ab = gamma_abp[Q];
            gamma_bb = gamma_bbp[Q];
        }        
        if (meta_) {
            tau_a = tau_ap[Q];
            tau_b = tau_bp[Q];
        }        

        // Definitions (asymptotics to prevent numerical problems)
        if (rho_a < lsda_cutoff_ && rho_b < lsda_cutoff_) {
            continue;
        } else if (rho_a < lsda_cutoff_) {
            // v
            if (deriv >= 0) {
                double t7824 = rho_a+rho_b;
                double t7825 = 1.0/pow(t7824,1.0/3.0);
                double t7826 = c*t7825;
                double t7827 = sqrt(t7826);
                double t7828 = EcP_4*4.0;
                double t7829 = EcP_3*EcP_3;
                double t7830 = t7828-t7829;
                double t7831 = EcP_2-t7827;
                double t7832 = EcP_3*t7827;
                double t7833 = EcP_4+t7832+t7826;
                double t7834 = 1.0/t7833;
                double t7835 = sqrt(t7830);
                double t7836 = t7827*2.0;
                double t7837 = EcP_3+t7836;
                double t7838 = 1.0/t7837;
                double t7839 = t7835*t7838;
                double t7840 = atan(t7839);
                double t7841 = 1.0/sqrt(t7830);
                double t7842 = EcF_4*4.0;
                double t7843 = EcF_3*EcF_3;
                double t7844 = t7842-t7843;
                double t7845 = EcF_2-t7827;
                double t7846 = EcF_3*t7827;
                double t7847 = EcF_4+t7826+t7846;
                double t7848 = 1.0/t7847;
                double t7849 = sqrt(t7844);
                double t7850 = EcF_3+t7836;
                double t7851 = 1.0/t7850;
                double t7852 = t7851*t7849;
                double t7853 = atan(t7852);
                double t7854 = 1.0/sqrt(t7844);
                double t7855 = c*t7825*t7834;
                double t7856 = log(t7855);
                double t7857 = EcP_3*t7840*t7841*2.0;
                double t7858 = t7831*t7831;
                double t7859 = t7834*t7858;
                double t7860 = log(t7859);
                double t7861 = EcP_2*4.0;
                double t7862 = EcP_3*2.0;
                double t7863 = t7861+t7862;
                double t7864 = t7840*t7841*t7863;
                double t7865 = t7860+t7864;
                double t7866 = EcP_2*EcP_2;
                double t7867 = EcP_2*EcP_3;
                double t7868 = EcP_4+t7866+t7867;
                double t7869 = 1.0/t7868;
                double t7870 = t7856+t7857-EcP_2*EcP_3*t7865*t7869;
                double t7871 = EcP_1*t7870;
                v[Q] += scale * (t7824*(t7871-((pow(2.0,1.0/3.0)*2.0-2.0)*(t7871-EcF_1*(log(c*t7825*t7848)+EcF_3*t7853*t7854*2.0-(EcF_2*EcF_3*(log((t7845*t7845)*t7848)+t7853*t7854*(EcF_2*4.0+EcF_3*2.0)))/(EcF_4+EcF_2*EcF_2+EcF_2*EcF_3))))/(two_13*2.0-2.0)));
            }
            
            // v_rho_a
            if (deriv >= 1) {
                double t7873 = rho_a+rho_b;
                double t7874 = 1.0/pow(t7873,1.0/3.0);
                double t7875 = c*t7874;
                double t7876 = sqrt(t7875);
                double t7877 = EcP_4*4.0;
                double t7878 = EcP_3*EcP_3;
                double t7879 = t7877-t7878;
                double t7880 = EcP_2-t7876;
                double t7881 = EcP_3*t7876;
                double t7882 = EcP_4+t7881+t7875;
                double t7883 = 1.0/t7882;
                double t7884 = sqrt(t7879);
                double t7885 = t7876*2.0;
                double t7886 = EcP_3+t7885;
                double t7887 = 1.0/t7886;
                double t7888 = t7884*t7887;
                double t7889 = atan(t7888);
                double t7890 = 1.0/sqrt(t7879);
                double t7891 = 1.0/pow(t7873,4.0/3.0);
                double t7892 = c*t7891*(1.0/3.0);
                double t7893 = 1.0/sqrt(t7875);
                double t7894 = EcP_3*c*t7891*t7893*(1.0/6.0);
                double t7895 = t7892+t7894;
                double t7896 = t7880*t7880;
                double t7897 = 1.0/(t7882*t7882);
                double t7898 = EcP_2*4.0;
                double t7899 = EcP_3*2.0;
                double t7900 = t7898+t7899;
                double t7901 = 1.0/(t7886*t7886);
                double t7902 = EcP_2*EcP_2;
                double t7903 = EcP_2*EcP_3;
                double t7904 = EcP_4+t7902+t7903;
                double t7905 = 1.0/t7904;
                double t7906 = t7879*t7901;
                double t7907 = t7906+1.0;
                double t7908 = 1.0/t7907;
                double t7909 = 1.0/c;
                double t7910 = EcF_3*t7876;
                double t7911 = EcF_4+t7875+t7910;
                double t7912 = pow(t7873,1.0/3.0);
                double t7913 = EcF_3*c*t7891*t7893*(1.0/6.0);
                double t7914 = t7892+t7913;
                double t7915 = EcF_2-t7876;
                double t7916 = 1.0/(t7911*t7911);
                double t7917 = 1.0/t7911;
                double t7918 = EcF_3+t7885;
                double t7919 = 1.0/(t7918*t7918);
                double t7920 = EcF_4*4.0;
                double t7921 = EcF_3*EcF_3;
                double t7922 = t7920-t7921;
                double t7923 = t7922*t7919;
                double t7924 = t7923+1.0;
                double t7925 = 1.0/t7924;
                double t7926 = c*t7891*t7883*(1.0/3.0);
                double t7927 = t7926-c*t7874*t7895*t7897;
                double t7928 = t7882*t7912*t7909*t7927;
                double t7929 = t7895*t7896*t7897;
                double t7930 = c*t7880*t7891*t7883*t7893*(1.0/3.0);
                double t7931 = t7930+t7929;
                double t7932 = 1.0/(t7880*t7880);
                double t7933 = t7882*t7931*t7932;
                double t7934 = c*t7891*t7893*t7900*t7901*t7908*(1.0/3.0);
                double t7935 = t7933+t7934;
                double t7936 = EcP_2*EcP_3*t7905*t7935;
                double t7937 = t7936+t7928-EcP_3*c*t7891*t7893*t7901*t7908*(2.0/3.0);
                double t7938 = EcP_1*t7937;
                double t7939 = pow(2.0,1.0/3.0);
                double t7940 = t7939*2.0;
                double t7941 = t7940-2.0;
                double t7942 = two_13*2.0;
                double t7943 = t7942-2.0;
                double t7944 = 1.0/t7943;
                double t7945 = t7915*t7915;
                double t7946 = sqrt(t7922);
                double t7947 = 1.0/t7918;
                double t7948 = t7946*t7947;
                double t7949 = atan(t7948);
                double t7950 = 1.0/sqrt(t7922);
                double t7951 = EcF_2*4.0;
                double t7952 = EcF_3*2.0;
                double t7953 = t7951+t7952;
                double t7954 = EcF_2*EcF_2;
                double t7955 = EcF_2*EcF_3;
                double t7956 = EcF_4+t7954+t7955;
                double t7957 = 1.0/t7956;
                double t7958 = c*t7874*t7883;
                double t7959 = log(t7958);
                double t7960 = EcP_3*t7890*t7889*2.0;
                double t7961 = t7883*t7896;
                double t7962 = log(t7961);
                double t7963 = t7890*t7889*t7900;
                double t7964 = t7962+t7963;
                double t7965 = t7960+t7959-EcP_2*EcP_3*t7905*t7964;
                double t7966 = EcP_1*t7965;
                v_rho_a[Q] += scale * (t7966-t7873*(t7938-t7941*t7944*(t7938-EcF_1*(EcF_2*EcF_3*t7957*(t7911*1.0/(t7915*t7915)*(t7914*t7916*t7945+c*t7891*t7893*t7915*t7917*(1.0/3.0))+c*t7891*t7893*t7925*t7953*t7919*(1.0/3.0))+t7911*t7912*t7909*(c*t7891*t7917*(1.0/3.0)-c*t7874*t7914*t7916)-EcF_3*c*t7891*t7893*t7925*t7919*(2.0/3.0))))-t7941*t7944*(t7966-EcF_1*(log(c*t7874*t7917)+EcF_3*t7950*t7949*2.0-EcF_2*EcF_3*t7957*(log(t7917*t7945)+t7950*t7953*t7949))));
            }
            
            // v_rho_b
            if (deriv >= 1) {
                double t7968 = rho_a+rho_b;
                double t7969 = 1.0/pow(t7968,1.0/3.0);
                double t7970 = c*t7969;
                double t7971 = sqrt(t7970);
                double t7972 = EcP_4*4.0;
                double t7973 = EcP_3*EcP_3;
                double t7974 = t7972-t7973;
                double t7975 = EcP_2-t7971;
                double t7976 = EcP_3*t7971;
                double t7977 = EcP_4+t7970+t7976;
                double t7978 = 1.0/t7977;
                double t7979 = sqrt(t7974);
                double t7980 = t7971*2.0;
                double t7981 = EcP_3+t7980;
                double t7982 = 1.0/t7981;
                double t7983 = t7982*t7979;
                double t7984 = atan(t7983);
                double t7985 = 1.0/sqrt(t7974);
                double t7986 = 1.0/pow(t7968,4.0/3.0);
                double t7987 = c*t7986*(1.0/3.0);
                double t7988 = 1.0/sqrt(t7970);
                double t7989 = EcP_3*c*t7986*t7988*(1.0/6.0);
                double t7990 = t7987+t7989;
                double t7991 = t7975*t7975;
                double t7992 = 1.0/(t7977*t7977);
                double t7993 = EcP_2*4.0;
                double t7994 = EcP_3*2.0;
                double t7995 = t7993+t7994;
                double t7996 = 1.0/(t7981*t7981);
                double t7997 = EcP_2*EcP_2;
                double t7998 = EcP_2*EcP_3;
                double t7999 = EcP_4+t7997+t7998;
                double t8000 = 1.0/t7999;
                double t8001 = t7974*t7996;
                double t8002 = t8001+1.0;
                double t8003 = 1.0/t8002;
                double t8004 = 1.0/c;
                double t8005 = EcF_3*t7971;
                double t8006 = EcF_4+t7970+t8005;
                double t8007 = pow(t7968,1.0/3.0);
                double t8008 = EcF_3*c*t7986*t7988*(1.0/6.0);
                double t8009 = t7987+t8008;
                double t8010 = EcF_2-t7971;
                double t8011 = 1.0/(t8006*t8006);
                double t8012 = 1.0/t8006;
                double t8013 = EcF_3+t7980;
                double t8014 = 1.0/(t8013*t8013);
                double t8015 = EcF_4*4.0;
                double t8016 = EcF_3*EcF_3;
                double t8017 = t8015-t8016;
                double t8018 = t8014*t8017;
                double t8019 = t8018+1.0;
                double t8020 = 1.0/t8019;
                double t8021 = c*t7986*t7978*(1.0/3.0);
                double t8022 = t8021-c*t7990*t7992*t7969;
                double t8023 = t7977*t8004*t8022*t8007;
                double t8024 = t7990*t7991*t7992;
                double t8025 = c*t7975*t7986*t7978*t7988*(1.0/3.0);
                double t8026 = t8024+t8025;
                double t8027 = 1.0/(t7975*t7975);
                double t8028 = t7977*t8026*t8027;
                double t8029 = c*t7986*t7995*t7996*t7988*t8003*(1.0/3.0);
                double t8030 = t8028+t8029;
                double t8031 = EcP_2*EcP_3*t8000*t8030;
                double t8032 = t8031+t8023-EcP_3*c*t7986*t7996*t7988*t8003*(2.0/3.0);
                double t8033 = EcP_1*t8032;
                double t8034 = pow(2.0,1.0/3.0);
                double t8035 = t8034*2.0;
                double t8036 = t8035-2.0;
                double t8037 = two_13*2.0;
                double t8038 = t8037-2.0;
                double t8039 = 1.0/t8038;
                double t8040 = t8010*t8010;
                double t8041 = sqrt(t8017);
                double t8042 = 1.0/t8013;
                double t8043 = t8041*t8042;
                double t8044 = atan(t8043);
                double t8045 = 1.0/sqrt(t8017);
                double t8046 = EcF_2*4.0;
                double t8047 = EcF_3*2.0;
                double t8048 = t8046+t8047;
                double t8049 = EcF_2*EcF_2;
                double t8050 = EcF_2*EcF_3;
                double t8051 = EcF_4+t8050+t8049;
                double t8052 = 1.0/t8051;
                double t8053 = c*t7969*t7978;
                double t8054 = log(t8053);
                double t8055 = EcP_3*t7984*t7985*2.0;
                double t8056 = t7991*t7978;
                double t8057 = log(t8056);
                double t8058 = t7984*t7985*t7995;
                double t8059 = t8057+t8058;
                double t8060 = t8054+t8055-EcP_2*EcP_3*t8000*t8059;
                double t8061 = EcP_1*t8060;
                v_rho_b[Q] += scale * (t8061-t7968*(t8033-t8036*t8039*(t8033-EcF_1*(EcF_2*EcF_3*t8052*(1.0/(t8010*t8010)*t8006*(t8011*t8040*t8009+c*t7986*t7988*t8010*t8012*(1.0/3.0))+c*t7986*t7988*t8020*t8014*t8048*(1.0/3.0))+t8004*t8006*t8007*(c*t7986*t8012*(1.0/3.0)-c*t7969*t8011*t8009)-EcF_3*c*t7986*t7988*t8020*t8014*(2.0/3.0))))-t8036*t8039*(t8061-EcF_1*(log(c*t7969*t8012)+EcF_3*t8044*t8045*2.0-EcF_2*EcF_3*t8052*(log(t8012*t8040)+t8044*t8045*t8048))));
            }
            
            // v_rho_a_rho_a
            if (deriv >= 2) {
                double t8068 = rho_a+rho_b;
                double t8069 = 1.0/pow(t8068,1.0/3.0);
                double t8070 = c*t8069;
                double t8071 = 1.0/pow(t8068,4.0/3.0);
                double t8072 = sqrt(t8070);
                double t8073 = EcP_3*t8072;
                double t8074 = EcP_4+t8070+t8073;
                double t8075 = c*t8071*(1.0/3.0);
                double t8076 = 1.0/sqrt(t8070);
                double t8077 = EcP_3*c*t8071*t8076*(1.0/6.0);
                double t8078 = t8075+t8077;
                double t8079 = EcP_2-t8072;
                double t8080 = 1.0/(t8074*t8074);
                double t8081 = 1.0/t8074;
                double t8082 = t8072*2.0;
                double t8083 = EcP_3+t8082;
                double t8084 = 1.0/(t8083*t8083);
                double t8085 = EcP_4*4.0;
                double t8086 = EcP_3*EcP_3;
                double t8087 = t8085-t8086;
                double t8088 = t8084*t8087;
                double t8089 = t8088+1.0;
                double t8090 = 1.0/t8089;
                double t8091 = 1.0/c;
                double t8092 = c*t8071*t8081*(1.0/3.0);
                double t8095 = c*t8080*t8069*t8078;
                double t8093 = t8092-t8095;
                double t8094 = pow(t8068,1.0/3.0);
                double t8096 = 1.0/pow(t8068,7.0/3.0);
                double t8097 = EcP_2*EcP_2;
                double t8098 = EcP_2*EcP_3;
                double t8099 = EcP_4+t8097+t8098;
                double t8100 = 1.0/t8099;
                double t8101 = t8079*t8079;
                double t8102 = t8080*t8078*t8101;
                double t8103 = c*t8071*t8081*t8076*t8079*(1.0/3.0);
                double t8104 = t8102+t8103;
                double t8105 = 1.0/(t8079*t8079);
                double t8106 = c*t8096*(4.0/9.0);
                double t8107 = EcP_3*c*t8076*t8096*(2.0/9.0);
                double t8108 = c*c;
                double t8109 = 1.0/pow(t8068,8.0/3.0);
                double t8110 = 1.0/pow(t8070,3.0/2.0);
                double t8155 = EcP_3*t8110*t8108*t8109*(1.0/3.6E1);
                double t8111 = t8106+t8107-t8155;
                double t8112 = t8078*t8078;
                double t8113 = 1.0/(t8074*t8074*t8074);
                double t8114 = EcP_2*4.0;
                double t8115 = EcP_3*2.0;
                double t8116 = t8114+t8115;
                double t8117 = 1.0/(t8083*t8083*t8083);
                double t8118 = 1.0/(t8083*t8083*t8083*t8083*t8083);
                double t8119 = 1.0/(t8089*t8089);
                double t8120 = EcF_3*t8072;
                double t8121 = EcF_4+t8070+t8120;
                double t8122 = EcF_3*c*t8071*t8076*(1.0/6.0);
                double t8123 = t8075+t8122;
                double t8124 = 1.0/t8121;
                double t8125 = c*t8071*t8124*(1.0/3.0);
                double t8126 = 1.0/(t8121*t8121);
                double t8181 = c*t8069*t8123*t8126;
                double t8127 = t8125-t8181;
                double t8128 = 1.0/pow(t8068,2.0/3.0);
                double t8129 = EcF_2-t8072;
                double t8130 = 1.0/(t8129*t8129);
                double t8131 = t8129*t8129;
                double t8132 = EcF_3*c*t8076*t8096*(2.0/9.0);
                double t8133 = t8132+t8106-EcF_3*t8110*t8108*t8109*(1.0/3.6E1);
                double t8134 = t8123*t8123;
                double t8135 = 1.0/(t8121*t8121*t8121);
                double t8136 = EcF_3+t8082;
                double t8137 = EcF_2*4.0;
                double t8138 = EcF_3*2.0;
                double t8139 = t8137+t8138;
                double t8140 = 1.0/(t8136*t8136);
                double t8141 = EcF_4*4.0;
                double t8142 = EcF_3*EcF_3;
                double t8143 = t8141-t8142;
                double t8144 = t8140*t8143;
                double t8145 = t8144+1.0;
                double t8146 = 1.0/t8145;
                double t8147 = t8131*t8123*t8126;
                double t8148 = c*t8071*t8076*t8124*t8129*(1.0/3.0);
                double t8149 = t8147+t8148;
                double t8150 = 1.0/(t8136*t8136*t8136);
                double t8151 = 1.0/(t8136*t8136*t8136*t8136*t8136);
                double t8152 = 1.0/(t8145*t8145);
                double t8153 = t8091*t8093*t8094*t8078;
                double t8154 = c*t8081*t8096*(4.0/9.0);
                double t8156 = c*t8069*t8112*t8113*2.0;
                double t8157 = t8154+t8156-c*t8071*t8080*t8078*(2.0/3.0)-c*t8080*t8069*t8111;
                double t8158 = t8091*t8074*t8094*t8157;
                double t8159 = t8078*t8104*t8105;
                double t8160 = c*t8081*t8096*(1.0/1.8E1);
                double t8161 = t8101*t8112*t8113*2.0;
                double t8162 = t8081*t8079*t8110*t8108*t8109*(1.0/1.8E1);
                double t8163 = c*t8071*t8080*t8076*t8078*t8079*(2.0/3.0);
                double t8164 = t8160+t8161+t8162+t8163-t8080*t8101*t8111-c*t8081*t8076*t8096*t8079*(4.0/9.0);
                double t8165 = c*t8090*t8084*t8076*t8096*t8116*(4.0/9.0);
                double t8166 = 1.0/(t8079*t8079*t8079);
                double t8167 = c*t8071*t8074*t8076*t8104*t8166*(1.0/3.0);
                double t8168 = c*t8087*t8096*t8116*t8118*t8119*(2.0/9.0);
                double t8169 = t8165+t8167+t8159+t8168-t8074*t8105*t8164-c*t8090*t8096*t8116*t8117*(2.0/9.0)-t8090*t8084*t8110*t8116*t8108*t8109*(1.0/1.8E1);
                double t8170 = EcP_2*EcP_3*t8100*t8169;
                double t8171 = EcP_3*c*t8090*t8096*t8117*(4.0/9.0);
                double t8172 = EcP_3*t8090*t8084*t8110*t8108*t8109*(1.0/9.0);
                double t8173 = t8170+t8153+t8171+t8172+t8158-t8091*t8074*t8093*t8128*(1.0/3.0)-EcP_3*c*t8090*t8084*t8076*t8096*(8.0/9.0)-EcP_3*c*t8087*t8096*t8118*t8119*(4.0/9.0);
                double t8174 = EcP_1*t8173;
                double t8175 = pow(2.0,1.0/3.0);
                double t8176 = t8175*2.0;
                double t8177 = t8176-2.0;
                double t8178 = two_13*2.0;
                double t8179 = t8178-2.0;
                double t8180 = 1.0/t8179;
                double t8182 = EcF_2*EcF_2;
                double t8183 = EcF_2*EcF_3;
                double t8184 = EcF_4+t8182+t8183;
                double t8185 = 1.0/t8184;
                double t8186 = t8091*t8074*t8093*t8094;
                double t8187 = t8074*t8104*t8105;
                double t8188 = c*t8071*t8090*t8084*t8076*t8116*(1.0/3.0);
                double t8189 = t8187+t8188;
                double t8190 = EcP_2*EcP_3*t8100*t8189;
                double t8191 = t8190+t8186-EcP_3*c*t8071*t8090*t8084*t8076*(2.0/3.0);
                v_rho_a_rho_a[Q] += scale * (EcP_1*t8191*-2.0+t8068*(t8174-t8180*t8177*(t8174-EcF_1*(t8091*t8094*t8121*(c*t8096*t8124*(4.0/9.0)-c*t8071*t8123*t8126*(2.0/3.0)-c*t8069*t8133*t8126+c*t8069*t8134*t8135*2.0)+t8091*t8094*t8123*t8127-t8091*t8121*t8127*t8128*(1.0/3.0)+EcF_2*EcF_3*t8185*(-t8121*t8130*(c*t8096*t8124*(1.0/1.8E1)-t8131*t8133*t8126+t8131*t8134*t8135*2.0-c*t8076*t8096*t8124*t8129*(4.0/9.0)+t8110*t8124*t8108*t8109*t8129*(1.0/1.8E1)+c*t8071*t8076*t8123*t8126*t8129*(2.0/3.0))+t8130*t8123*t8149-c*t8096*t8150*t8146*t8139*(2.0/9.0)+c*t8076*t8096*t8140*t8146*t8139*(4.0/9.0)+c*t8096*t8151*t8143*t8152*t8139*(2.0/9.0)-t8110*t8140*t8108*t8109*t8146*t8139*(1.0/1.8E1)+c*t8071*t8076*t8121*1.0/(t8129*t8129*t8129)*t8149*(1.0/3.0))+EcF_3*c*t8096*t8150*t8146*(4.0/9.0)-EcF_3*c*t8076*t8096*t8140*t8146*(8.0/9.0)-EcF_3*c*t8096*t8151*t8143*t8152*(4.0/9.0)+EcF_3*t8110*t8140*t8108*t8109*t8146*(1.0/9.0))))+t8180*t8177*(EcP_1*t8191-EcF_1*(t8091*t8094*t8121*t8127+EcF_2*EcF_3*t8185*(t8121*t8130*t8149+c*t8071*t8076*t8140*t8146*t8139*(1.0/3.0))-EcF_3*c*t8071*t8076*t8140*t8146*(2.0/3.0)))*2.0);
            }
            
            // v_rho_a_rho_b
            if (deriv >= 2) {
                double t8193 = rho_a+rho_b;
                double t8194 = 1.0/pow(t8193,1.0/3.0);
                double t8195 = c*t8194;
                double t8196 = 1.0/pow(t8193,4.0/3.0);
                double t8197 = sqrt(t8195);
                double t8198 = EcP_3*t8197;
                double t8199 = EcP_4+t8195+t8198;
                double t8200 = c*t8196*(1.0/3.0);
                double t8201 = 1.0/sqrt(t8195);
                double t8202 = EcP_3*c*t8196*t8201*(1.0/6.0);
                double t8203 = t8200+t8202;
                double t8204 = EcP_2-t8197;
                double t8205 = 1.0/(t8199*t8199);
                double t8206 = 1.0/t8199;
                double t8207 = t8197*2.0;
                double t8208 = EcP_3+t8207;
                double t8209 = 1.0/(t8208*t8208);
                double t8210 = EcP_4*4.0;
                double t8211 = EcP_3*EcP_3;
                double t8212 = t8210-t8211;
                double t8213 = t8212*t8209;
                double t8214 = t8213+1.0;
                double t8215 = 1.0/t8214;
                double t8216 = 1.0/c;
                double t8217 = c*t8196*t8206*(1.0/3.0);
                double t8220 = c*t8194*t8203*t8205;
                double t8218 = -t8220+t8217;
                double t8219 = pow(t8193,1.0/3.0);
                double t8221 = 1.0/pow(t8193,7.0/3.0);
                double t8222 = EcP_2*EcP_2;
                double t8223 = EcP_2*EcP_3;
                double t8224 = EcP_4+t8222+t8223;
                double t8225 = 1.0/t8224;
                double t8226 = t8204*t8204;
                double t8227 = t8203*t8205*t8226;
                double t8228 = c*t8196*t8201*t8204*t8206*(1.0/3.0);
                double t8229 = t8227+t8228;
                double t8230 = 1.0/(t8204*t8204);
                double t8231 = c*t8221*(4.0/9.0);
                double t8232 = EcP_3*c*t8201*t8221*(2.0/9.0);
                double t8233 = c*c;
                double t8234 = 1.0/pow(t8193,8.0/3.0);
                double t8235 = 1.0/pow(t8195,3.0/2.0);
                double t8280 = EcP_3*t8233*t8234*t8235*(1.0/3.6E1);
                double t8236 = t8231+t8232-t8280;
                double t8237 = t8203*t8203;
                double t8238 = 1.0/(t8199*t8199*t8199);
                double t8239 = EcP_2*4.0;
                double t8240 = EcP_3*2.0;
                double t8241 = t8240+t8239;
                double t8242 = 1.0/(t8208*t8208*t8208);
                double t8243 = 1.0/(t8208*t8208*t8208*t8208*t8208);
                double t8244 = 1.0/(t8214*t8214);
                double t8245 = EcF_3*t8197;
                double t8246 = EcF_4+t8195+t8245;
                double t8247 = EcF_3*c*t8196*t8201*(1.0/6.0);
                double t8248 = t8200+t8247;
                double t8249 = 1.0/t8246;
                double t8250 = c*t8196*t8249*(1.0/3.0);
                double t8251 = 1.0/(t8246*t8246);
                double t8306 = c*t8194*t8251*t8248;
                double t8252 = t8250-t8306;
                double t8253 = 1.0/pow(t8193,2.0/3.0);
                double t8254 = EcF_2-t8197;
                double t8255 = 1.0/(t8254*t8254);
                double t8256 = t8254*t8254;
                double t8257 = EcF_3*c*t8201*t8221*(2.0/9.0);
                double t8258 = t8231+t8257-EcF_3*t8233*t8234*t8235*(1.0/3.6E1);
                double t8259 = t8248*t8248;
                double t8260 = 1.0/(t8246*t8246*t8246);
                double t8261 = EcF_3+t8207;
                double t8262 = EcF_2*4.0;
                double t8263 = EcF_3*2.0;
                double t8264 = t8262+t8263;
                double t8265 = 1.0/(t8261*t8261);
                double t8266 = EcF_4*4.0;
                double t8267 = EcF_3*EcF_3;
                double t8268 = t8266-t8267;
                double t8269 = t8265*t8268;
                double t8270 = t8269+1.0;
                double t8271 = 1.0/t8270;
                double t8272 = t8251*t8256*t8248;
                double t8273 = c*t8196*t8201*t8254*t8249*(1.0/3.0);
                double t8274 = t8272+t8273;
                double t8275 = 1.0/(t8261*t8261*t8261);
                double t8276 = 1.0/(t8261*t8261*t8261*t8261*t8261);
                double t8277 = 1.0/(t8270*t8270);
                double t8278 = t8203*t8216*t8218*t8219;
                double t8279 = c*t8221*t8206*(4.0/9.0);
                double t8281 = c*t8194*t8237*t8238*2.0;
                double t8282 = t8281+t8279-c*t8196*t8203*t8205*(2.0/3.0)-c*t8194*t8205*t8236;
                double t8283 = t8199*t8216*t8219*t8282;
                double t8284 = t8203*t8230*t8229;
                double t8285 = c*t8221*t8206*(1.0/1.8E1);
                double t8286 = t8226*t8237*t8238*2.0;
                double t8287 = t8204*t8206*t8233*t8234*t8235*(1.0/1.8E1);
                double t8288 = c*t8196*t8201*t8203*t8204*t8205*(2.0/3.0);
                double t8289 = t8285+t8286+t8287+t8288-t8205*t8226*t8236-c*t8201*t8221*t8204*t8206*(4.0/9.0);
                double t8290 = c*t8201*t8221*t8241*t8215*t8209*(4.0/9.0);
                double t8291 = 1.0/(t8204*t8204*t8204);
                double t8292 = c*t8196*t8199*t8201*t8291*t8229*(1.0/3.0);
                double t8293 = c*t8212*t8221*t8241*t8243*t8244*(2.0/9.0);
                double t8294 = t8290+t8292+t8284+t8293-t8199*t8230*t8289-c*t8221*t8241*t8215*t8242*(2.0/9.0)-t8241*t8215*t8233*t8234*t8235*t8209*(1.0/1.8E1);
                double t8295 = EcP_2*EcP_3*t8225*t8294;
                double t8296 = EcP_3*c*t8221*t8215*t8242*(4.0/9.0);
                double t8297 = EcP_3*t8215*t8233*t8234*t8235*t8209*(1.0/9.0);
                double t8298 = t8283+t8295+t8278+t8296+t8297-t8199*t8216*t8253*t8218*(1.0/3.0)-EcP_3*c*t8201*t8221*t8215*t8209*(8.0/9.0)-EcP_3*c*t8212*t8221*t8243*t8244*(4.0/9.0);
                double t8299 = EcP_1*t8298;
                double t8300 = pow(2.0,1.0/3.0);
                double t8301 = t8300*2.0;
                double t8302 = t8301-2.0;
                double t8303 = two_13*2.0;
                double t8304 = t8303-2.0;
                double t8305 = 1.0/t8304;
                double t8307 = EcF_2*EcF_2;
                double t8308 = EcF_2*EcF_3;
                double t8309 = EcF_4+t8307+t8308;
                double t8310 = 1.0/t8309;
                double t8311 = t8199*t8216*t8218*t8219;
                double t8312 = t8199*t8230*t8229;
                double t8313 = c*t8196*t8201*t8241*t8215*t8209*(1.0/3.0);
                double t8314 = t8312+t8313;
                double t8315 = EcP_2*EcP_3*t8225*t8314;
                double t8316 = t8311+t8315-EcP_3*c*t8196*t8201*t8215*t8209*(2.0/3.0);
                v_rho_a_rho_b[Q] += scale * (EcP_1*t8316*-2.0+t8193*(t8299-t8302*t8305*(t8299-EcF_1*(t8216*t8219*t8246*(c*t8221*t8249*(4.0/9.0)-c*t8194*t8251*t8258+c*t8194*t8260*t8259*2.0-c*t8196*t8251*t8248*(2.0/3.0))-t8216*t8252*t8253*t8246*(1.0/3.0)+t8216*t8252*t8219*t8248+EcF_2*EcF_3*t8310*(-t8246*t8255*(c*t8221*t8249*(1.0/1.8E1)-t8251*t8256*t8258+t8260*t8256*t8259*2.0-c*t8201*t8221*t8254*t8249*(4.0/9.0)+t8233*t8234*t8235*t8254*t8249*(1.0/1.8E1)+c*t8196*t8201*t8251*t8254*t8248*(2.0/3.0))+t8255*t8274*t8248-c*t8221*t8271*t8264*t8275*(2.0/9.0)+c*t8201*t8221*t8271*t8264*t8265*(4.0/9.0)+c*t8221*t8264*t8276*t8268*t8277*(2.0/9.0)-t8233*t8234*t8235*t8271*t8264*t8265*(1.0/1.8E1)+c*t8196*t8201*1.0/(t8254*t8254*t8254)*t8246*t8274*(1.0/3.0))+EcF_3*c*t8221*t8271*t8275*(4.0/9.0)-EcF_3*c*t8201*t8221*t8271*t8265*(8.0/9.0)-EcF_3*c*t8221*t8276*t8268*t8277*(4.0/9.0)+EcF_3*t8233*t8234*t8235*t8271*t8265*(1.0/9.0))))+t8302*t8305*(EcP_1*t8316-EcF_1*(t8216*t8252*t8219*t8246+EcF_2*EcF_3*t8310*(t8246*t8255*t8274+c*t8196*t8201*t8271*t8264*t8265*(1.0/3.0))-EcF_3*c*t8196*t8201*t8271*t8265*(2.0/3.0)))*2.0);
            }
            
            // v_rho_b_rho_b
            if (deriv >= 2) {
                double t8318 = rho_a+rho_b;
                double t8319 = 1.0/pow(t8318,1.0/3.0);
                double t8320 = c*t8319;
                double t8321 = 1.0/pow(t8318,4.0/3.0);
                double t8322 = sqrt(t8320);
                double t8323 = EcP_3*t8322;
                double t8324 = EcP_4+t8320+t8323;
                double t8325 = c*t8321*(1.0/3.0);
                double t8326 = 1.0/sqrt(t8320);
                double t8327 = EcP_3*c*t8321*t8326*(1.0/6.0);
                double t8328 = t8325+t8327;
                double t8329 = EcP_2-t8322;
                double t8330 = 1.0/(t8324*t8324);
                double t8331 = 1.0/t8324;
                double t8332 = t8322*2.0;
                double t8333 = EcP_3+t8332;
                double t8334 = 1.0/(t8333*t8333);
                double t8335 = EcP_4*4.0;
                double t8336 = EcP_3*EcP_3;
                double t8337 = t8335-t8336;
                double t8338 = t8334*t8337;
                double t8339 = t8338+1.0;
                double t8340 = 1.0/t8339;
                double t8341 = 1.0/c;
                double t8342 = c*t8321*t8331*(1.0/3.0);
                double t8345 = c*t8330*t8319*t8328;
                double t8343 = t8342-t8345;
                double t8344 = pow(t8318,1.0/3.0);
                double t8346 = 1.0/pow(t8318,7.0/3.0);
                double t8347 = EcP_2*EcP_2;
                double t8348 = EcP_2*EcP_3;
                double t8349 = EcP_4+t8347+t8348;
                double t8350 = 1.0/t8349;
                double t8351 = t8329*t8329;
                double t8352 = t8330*t8351*t8328;
                double t8353 = c*t8321*t8331*t8326*t8329*(1.0/3.0);
                double t8354 = t8352+t8353;
                double t8355 = 1.0/(t8329*t8329);
                double t8356 = c*t8346*(4.0/9.0);
                double t8357 = EcP_3*c*t8326*t8346*(2.0/9.0);
                double t8358 = c*c;
                double t8359 = 1.0/pow(t8318,8.0/3.0);
                double t8360 = 1.0/pow(t8320,3.0/2.0);
                double t8405 = EcP_3*t8360*t8358*t8359*(1.0/3.6E1);
                double t8361 = t8356+t8357-t8405;
                double t8362 = t8328*t8328;
                double t8363 = 1.0/(t8324*t8324*t8324);
                double t8364 = EcP_2*4.0;
                double t8365 = EcP_3*2.0;
                double t8366 = t8364+t8365;
                double t8367 = 1.0/(t8333*t8333*t8333);
                double t8368 = 1.0/(t8333*t8333*t8333*t8333*t8333);
                double t8369 = 1.0/(t8339*t8339);
                double t8370 = EcF_3*t8322;
                double t8371 = EcF_4+t8320+t8370;
                double t8372 = EcF_3*c*t8321*t8326*(1.0/6.0);
                double t8373 = t8325+t8372;
                double t8374 = 1.0/t8371;
                double t8375 = c*t8321*t8374*(1.0/3.0);
                double t8376 = 1.0/(t8371*t8371);
                double t8431 = c*t8319*t8373*t8376;
                double t8377 = t8375-t8431;
                double t8378 = 1.0/pow(t8318,2.0/3.0);
                double t8379 = EcF_2-t8322;
                double t8380 = 1.0/(t8379*t8379);
                double t8381 = t8379*t8379;
                double t8382 = EcF_3*c*t8326*t8346*(2.0/9.0);
                double t8383 = t8382+t8356-EcF_3*t8360*t8358*t8359*(1.0/3.6E1);
                double t8384 = t8373*t8373;
                double t8385 = 1.0/(t8371*t8371*t8371);
                double t8386 = EcF_3+t8332;
                double t8387 = EcF_2*4.0;
                double t8388 = EcF_3*2.0;
                double t8389 = t8387+t8388;
                double t8390 = 1.0/(t8386*t8386);
                double t8391 = EcF_4*4.0;
                double t8392 = EcF_3*EcF_3;
                double t8393 = t8391-t8392;
                double t8394 = t8390*t8393;
                double t8395 = t8394+1.0;
                double t8396 = 1.0/t8395;
                double t8397 = t8381*t8373*t8376;
                double t8398 = c*t8321*t8326*t8374*t8379*(1.0/3.0);
                double t8399 = t8397+t8398;
                double t8400 = 1.0/(t8386*t8386*t8386);
                double t8401 = 1.0/(t8386*t8386*t8386*t8386*t8386);
                double t8402 = 1.0/(t8395*t8395);
                double t8403 = t8341*t8343*t8344*t8328;
                double t8404 = c*t8331*t8346*(4.0/9.0);
                double t8406 = c*t8362*t8363*t8319*2.0;
                double t8407 = t8404+t8406-c*t8321*t8330*t8328*(2.0/3.0)-c*t8330*t8361*t8319;
                double t8408 = t8341*t8324*t8344*t8407;
                double t8409 = t8354*t8328*t8355;
                double t8410 = c*t8331*t8346*(1.0/1.8E1);
                double t8411 = t8351*t8362*t8363*2.0;
                double t8412 = t8331*t8360*t8329*t8358*t8359*(1.0/1.8E1);
                double t8413 = c*t8321*t8330*t8326*t8328*t8329*(2.0/3.0);
                double t8414 = t8410+t8411+t8412+t8413-t8330*t8351*t8361-c*t8331*t8326*t8346*t8329*(4.0/9.0);
                double t8415 = c*t8340*t8334*t8326*t8346*t8366*(4.0/9.0);
                double t8416 = 1.0/(t8329*t8329*t8329);
                double t8417 = c*t8321*t8324*t8326*t8354*t8416*(1.0/3.0);
                double t8418 = c*t8337*t8346*t8366*t8368*t8369*(2.0/9.0);
                double t8419 = t8415+t8417+t8409+t8418-t8324*t8355*t8414-c*t8340*t8346*t8366*t8367*(2.0/9.0)-t8340*t8360*t8334*t8366*t8358*t8359*(1.0/1.8E1);
                double t8420 = EcP_2*EcP_3*t8350*t8419;
                double t8421 = EcP_3*c*t8340*t8346*t8367*(4.0/9.0);
                double t8422 = EcP_3*t8340*t8360*t8334*t8358*t8359*(1.0/9.0);
                double t8423 = t8420+t8403+t8421+t8422+t8408-t8341*t8324*t8343*t8378*(1.0/3.0)-EcP_3*c*t8340*t8334*t8326*t8346*(8.0/9.0)-EcP_3*c*t8337*t8346*t8368*t8369*(4.0/9.0);
                double t8424 = EcP_1*t8423;
                double t8425 = pow(2.0,1.0/3.0);
                double t8426 = t8425*2.0;
                double t8427 = t8426-2.0;
                double t8428 = two_13*2.0;
                double t8429 = t8428-2.0;
                double t8430 = 1.0/t8429;
                double t8432 = EcF_2*EcF_2;
                double t8433 = EcF_2*EcF_3;
                double t8434 = EcF_4+t8432+t8433;
                double t8435 = 1.0/t8434;
                double t8436 = t8341*t8324*t8343*t8344;
                double t8437 = t8324*t8354*t8355;
                double t8438 = c*t8321*t8340*t8334*t8326*t8366*(1.0/3.0);
                double t8439 = t8437+t8438;
                double t8440 = EcP_2*EcP_3*t8350*t8439;
                double t8441 = t8440+t8436-EcP_3*c*t8321*t8340*t8334*t8326*(2.0/3.0);
                v_rho_b_rho_b[Q] += scale * (EcP_1*t8441*-2.0+t8318*(t8424-t8430*t8427*(t8424-EcF_1*(t8341*t8344*t8371*(c*t8346*t8374*(4.0/9.0)-c*t8321*t8373*t8376*(2.0/3.0)-c*t8319*t8383*t8376+c*t8319*t8384*t8385*2.0)+t8341*t8344*t8373*t8377-t8341*t8371*t8377*t8378*(1.0/3.0)+EcF_2*EcF_3*t8435*(-t8371*t8380*(c*t8346*t8374*(1.0/1.8E1)-t8381*t8383*t8376+t8381*t8384*t8385*2.0-c*t8326*t8346*t8374*t8379*(4.0/9.0)+t8360*t8374*t8358*t8359*t8379*(1.0/1.8E1)+c*t8321*t8326*t8373*t8376*t8379*(2.0/3.0))+t8380*t8373*t8399-c*t8346*t8396*t8389*t8400*(2.0/9.0)+c*t8326*t8390*t8346*t8396*t8389*(4.0/9.0)+c*t8346*t8393*t8389*t8401*t8402*(2.0/9.0)-t8360*t8390*t8358*t8359*t8396*t8389*(1.0/1.8E1)+c*t8321*t8326*t8371*1.0/(t8379*t8379*t8379)*t8399*(1.0/3.0))+EcF_3*c*t8346*t8396*t8400*(4.0/9.0)-EcF_3*c*t8326*t8390*t8346*t8396*(8.0/9.0)-EcF_3*c*t8346*t8393*t8401*t8402*(4.0/9.0)+EcF_3*t8360*t8390*t8358*t8359*t8396*(1.0/9.0))))+t8430*t8427*(EcP_1*t8441-EcF_1*(t8341*t8344*t8371*t8377+EcF_2*EcF_3*t8435*(t8371*t8380*t8399+c*t8321*t8326*t8390*t8396*t8389*(1.0/3.0))-EcF_3*c*t8321*t8326*t8390*t8396*(2.0/3.0)))*2.0);
            }
            
        } else if (rho_b < lsda_cutoff_) {
            // v
            if (deriv >= 0) {
                double t8468 = rho_a+rho_b;
                double t8469 = 1.0/pow(t8468,1.0/3.0);
                double t8470 = c*t8469;
                double t8471 = sqrt(t8470);
                double t8472 = EcP_4*4.0;
                double t8473 = EcP_3*EcP_3;
                double t8474 = t8472-t8473;
                double t8475 = EcP_2-t8471;
                double t8476 = EcP_3*t8471;
                double t8477 = EcP_4+t8470+t8476;
                double t8478 = 1.0/t8477;
                double t8479 = sqrt(t8474);
                double t8480 = t8471*2.0;
                double t8481 = EcP_3+t8480;
                double t8482 = 1.0/t8481;
                double t8483 = t8482*t8479;
                double t8484 = atan(t8483);
                double t8485 = 1.0/sqrt(t8474);
                double t8486 = EcF_4*4.0;
                double t8487 = EcF_3*EcF_3;
                double t8488 = t8486-t8487;
                double t8489 = EcF_2-t8471;
                double t8490 = EcF_3*t8471;
                double t8491 = EcF_4+t8470+t8490;
                double t8492 = 1.0/t8491;
                double t8493 = sqrt(t8488);
                double t8494 = EcF_3+t8480;
                double t8495 = 1.0/t8494;
                double t8496 = t8493*t8495;
                double t8497 = atan(t8496);
                double t8498 = 1.0/sqrt(t8488);
                double t8499 = c*t8469*t8478;
                double t8500 = log(t8499);
                double t8501 = EcP_3*t8484*t8485*2.0;
                double t8502 = t8475*t8475;
                double t8503 = t8478*t8502;
                double t8504 = log(t8503);
                double t8505 = EcP_2*4.0;
                double t8506 = EcP_3*2.0;
                double t8507 = t8505+t8506;
                double t8508 = t8484*t8485*t8507;
                double t8509 = t8504+t8508;
                double t8510 = EcP_2*EcP_2;
                double t8511 = EcP_2*EcP_3;
                double t8512 = EcP_4+t8510+t8511;
                double t8513 = 1.0/t8512;
                double t8514 = t8500+t8501-EcP_2*EcP_3*t8513*t8509;
                double t8515 = EcP_1*t8514;
                v[Q] += scale * (t8468*(t8515-((pow(2.0,1.0/3.0)*2.0-2.0)*(t8515-EcF_1*(log(c*t8492*t8469)+EcF_3*t8497*t8498*2.0-(EcF_2*EcF_3*(log(t8492*(t8489*t8489))+t8497*t8498*(EcF_2*4.0+EcF_3*2.0)))/(EcF_4+EcF_2*EcF_2+EcF_2*EcF_3))))/(two_13*2.0-2.0)));
            }
            
            // v_rho_a
            if (deriv >= 1) {
                double t8517 = rho_a+rho_b;
                double t8518 = 1.0/pow(t8517,1.0/3.0);
                double t8519 = c*t8518;
                double t8520 = sqrt(t8519);
                double t8521 = EcP_4*4.0;
                double t8522 = EcP_3*EcP_3;
                double t8523 = t8521-t8522;
                double t8524 = EcP_2-t8520;
                double t8525 = EcP_3*t8520;
                double t8526 = EcP_4+t8525+t8519;
                double t8527 = 1.0/t8526;
                double t8528 = sqrt(t8523);
                double t8529 = t8520*2.0;
                double t8530 = EcP_3+t8529;
                double t8531 = 1.0/t8530;
                double t8532 = t8531*t8528;
                double t8533 = atan(t8532);
                double t8534 = 1.0/sqrt(t8523);
                double t8535 = 1.0/pow(t8517,4.0/3.0);
                double t8536 = c*t8535*(1.0/3.0);
                double t8537 = 1.0/sqrt(t8519);
                double t8538 = EcP_3*c*t8535*t8537*(1.0/6.0);
                double t8539 = t8536+t8538;
                double t8540 = t8524*t8524;
                double t8541 = 1.0/(t8526*t8526);
                double t8542 = EcP_2*4.0;
                double t8543 = EcP_3*2.0;
                double t8544 = t8542+t8543;
                double t8545 = 1.0/(t8530*t8530);
                double t8546 = EcP_2*EcP_2;
                double t8547 = EcP_2*EcP_3;
                double t8548 = EcP_4+t8546+t8547;
                double t8549 = 1.0/t8548;
                double t8550 = t8523*t8545;
                double t8551 = t8550+1.0;
                double t8552 = 1.0/t8551;
                double t8553 = 1.0/c;
                double t8554 = EcF_3*t8520;
                double t8555 = EcF_4+t8554+t8519;
                double t8556 = pow(t8517,1.0/3.0);
                double t8557 = EcF_3*c*t8535*t8537*(1.0/6.0);
                double t8558 = t8536+t8557;
                double t8559 = EcF_2-t8520;
                double t8560 = 1.0/(t8555*t8555);
                double t8561 = 1.0/t8555;
                double t8562 = EcF_3+t8529;
                double t8563 = 1.0/(t8562*t8562);
                double t8564 = EcF_4*4.0;
                double t8565 = EcF_3*EcF_3;
                double t8566 = t8564-t8565;
                double t8567 = t8563*t8566;
                double t8568 = t8567+1.0;
                double t8569 = 1.0/t8568;
                double t8570 = c*t8535*t8527*(1.0/3.0);
                double t8571 = t8570-c*t8541*t8518*t8539;
                double t8572 = t8526*t8553*t8571*t8556;
                double t8573 = t8540*t8541*t8539;
                double t8574 = c*t8524*t8535*t8527*t8537*(1.0/3.0);
                double t8575 = t8573+t8574;
                double t8576 = 1.0/(t8524*t8524);
                double t8577 = t8526*t8575*t8576;
                double t8578 = c*t8552*t8535*t8544*t8545*t8537*(1.0/3.0);
                double t8579 = t8577+t8578;
                double t8580 = EcP_2*EcP_3*t8549*t8579;
                double t8581 = t8580+t8572-EcP_3*c*t8552*t8535*t8545*t8537*(2.0/3.0);
                double t8582 = EcP_1*t8581;
                double t8583 = pow(2.0,1.0/3.0);
                double t8584 = t8583*2.0;
                double t8585 = t8584-2.0;
                double t8586 = two_13*2.0;
                double t8587 = t8586-2.0;
                double t8588 = 1.0/t8587;
                double t8589 = t8559*t8559;
                double t8590 = sqrt(t8566);
                double t8591 = 1.0/t8562;
                double t8592 = t8590*t8591;
                double t8593 = atan(t8592);
                double t8594 = 1.0/sqrt(t8566);
                double t8595 = EcF_2*4.0;
                double t8596 = EcF_3*2.0;
                double t8597 = t8595+t8596;
                double t8598 = EcF_2*EcF_2;
                double t8599 = EcF_2*EcF_3;
                double t8600 = EcF_4+t8598+t8599;
                double t8601 = 1.0/t8600;
                double t8602 = c*t8518*t8527;
                double t8603 = log(t8602);
                double t8604 = EcP_3*t8533*t8534*2.0;
                double t8605 = t8540*t8527;
                double t8606 = log(t8605);
                double t8607 = t8533*t8534*t8544;
                double t8608 = t8606+t8607;
                double t8609 = t8603+t8604-EcP_2*EcP_3*t8549*t8608;
                double t8610 = EcP_1*t8609;
                v_rho_a[Q] += scale * (t8610-t8517*(t8582-t8585*t8588*(t8582-EcF_1*(EcF_2*EcF_3*t8601*(t8555*1.0/(t8559*t8559)*(t8560*t8558*t8589+c*t8561*t8535*t8537*t8559*(1.0/3.0))+c*t8535*t8563*t8537*t8569*t8597*(1.0/3.0))+t8553*t8555*t8556*(c*t8561*t8535*(1.0/3.0)-c*t8560*t8518*t8558)-EcF_3*c*t8535*t8563*t8537*t8569*(2.0/3.0))))-t8585*t8588*(t8610-EcF_1*(log(c*t8561*t8518)+EcF_3*t8593*t8594*2.0-EcF_2*EcF_3*t8601*(log(t8561*t8589)+t8593*t8594*t8597))));
            }
            
            // v_rho_b
            if (deriv >= 1) {
                double t8612 = rho_a+rho_b;
                double t8613 = 1.0/pow(t8612,1.0/3.0);
                double t8614 = c*t8613;
                double t8615 = sqrt(t8614);
                double t8616 = EcP_4*4.0;
                double t8617 = EcP_3*EcP_3;
                double t8618 = t8616-t8617;
                double t8619 = EcP_2-t8615;
                double t8620 = EcP_3*t8615;
                double t8621 = EcP_4+t8620+t8614;
                double t8622 = 1.0/t8621;
                double t8623 = sqrt(t8618);
                double t8624 = t8615*2.0;
                double t8625 = EcP_3+t8624;
                double t8626 = 1.0/t8625;
                double t8627 = t8623*t8626;
                double t8628 = atan(t8627);
                double t8629 = 1.0/sqrt(t8618);
                double t8630 = 1.0/pow(t8612,4.0/3.0);
                double t8631 = c*t8630*(1.0/3.0);
                double t8632 = 1.0/sqrt(t8614);
                double t8633 = EcP_3*c*t8630*t8632*(1.0/6.0);
                double t8634 = t8631+t8633;
                double t8635 = t8619*t8619;
                double t8636 = 1.0/(t8621*t8621);
                double t8637 = EcP_2*4.0;
                double t8638 = EcP_3*2.0;
                double t8639 = t8637+t8638;
                double t8640 = 1.0/(t8625*t8625);
                double t8641 = EcP_2*EcP_2;
                double t8642 = EcP_2*EcP_3;
                double t8643 = EcP_4+t8641+t8642;
                double t8644 = 1.0/t8643;
                double t8645 = t8640*t8618;
                double t8646 = t8645+1.0;
                double t8647 = 1.0/t8646;
                double t8648 = 1.0/c;
                double t8649 = EcF_3*t8615;
                double t8650 = EcF_4+t8614+t8649;
                double t8651 = pow(t8612,1.0/3.0);
                double t8652 = EcF_3*c*t8630*t8632*(1.0/6.0);
                double t8653 = t8631+t8652;
                double t8654 = EcF_2-t8615;
                double t8655 = 1.0/(t8650*t8650);
                double t8656 = 1.0/t8650;
                double t8657 = EcF_3+t8624;
                double t8658 = 1.0/(t8657*t8657);
                double t8659 = EcF_4*4.0;
                double t8660 = EcF_3*EcF_3;
                double t8661 = t8660-t8659;
                double t8662 = t8661*t8658;
                double t8663 = t8662-1.0;
                double t8664 = 1.0/t8663;
                double t8665 = c*t8630*t8622*(1.0/3.0);
                double t8666 = t8665-c*t8613*t8634*t8636;
                double t8667 = t8621*t8651*t8648*t8666;
                double t8668 = t8634*t8635*t8636;
                double t8669 = c*t8630*t8622*t8632*t8619*(1.0/3.0);
                double t8670 = t8668+t8669;
                double t8671 = 1.0/(t8619*t8619);
                double t8672 = t8621*t8670*t8671;
                double t8673 = c*t8630*t8640*t8632*t8647*t8639*(1.0/3.0);
                double t8674 = t8672+t8673;
                double t8675 = EcP_2*EcP_3*t8644*t8674;
                double t8676 = t8675+t8667-EcP_3*c*t8630*t8640*t8632*t8647*(2.0/3.0);
                double t8677 = EcP_1*t8676;
                double t8678 = pow(2.0,1.0/3.0);
                double t8679 = t8678*2.0;
                double t8680 = t8679-2.0;
                double t8681 = two_13*2.0;
                double t8682 = t8681-2.0;
                double t8683 = 1.0/t8682;
                double t8684 = -t8660+t8659;
                double t8685 = t8654*t8654;
                double t8686 = sqrt(t8684);
                double t8687 = 1.0/t8657;
                double t8688 = t8686*t8687;
                double t8689 = atan(t8688);
                double t8690 = 1.0/sqrt(t8684);
                double t8691 = EcF_2*4.0;
                double t8692 = EcF_3*2.0;
                double t8693 = t8691+t8692;
                double t8694 = EcF_2*EcF_2;
                double t8695 = EcF_2*EcF_3;
                double t8696 = EcF_4+t8694+t8695;
                double t8697 = 1.0/t8696;
                double t8698 = c*t8613*t8622;
                double t8699 = log(t8698);
                double t8700 = EcP_3*t8628*t8629*2.0;
                double t8701 = t8622*t8635;
                double t8702 = log(t8701);
                double t8703 = t8628*t8629*t8639;
                double t8704 = t8702+t8703;
                double t8705 = t8699+t8700-EcP_2*EcP_3*t8644*t8704;
                double t8706 = EcP_1*t8705;
                v_rho_b[Q] += scale * (t8706-t8612*(t8677-t8680*t8683*(t8677-EcF_1*(EcF_2*EcF_3*t8697*(t8650*1.0/(t8654*t8654)*(t8653*t8655*t8685+c*t8630*t8632*t8654*t8656*(1.0/3.0))-c*t8630*t8632*t8664*t8693*t8658*(1.0/3.0))+t8650*t8651*t8648*(c*t8630*t8656*(1.0/3.0)-c*t8613*t8653*t8655)+EcF_3*c*t8630*t8632*t8664*t8658*(2.0/3.0))))-t8680*t8683*(t8706-EcF_1*(log(c*t8613*t8656)+EcF_3*t8690*t8689*2.0-EcF_2*EcF_3*t8697*(log(t8656*t8685)+t8690*t8693*t8689))));
            }
            
            // v_rho_a_rho_a
            if (deriv >= 2) {
                double t8713 = rho_a+rho_b;
                double t8714 = 1.0/pow(t8713,1.0/3.0);
                double t8715 = c*t8714;
                double t8716 = 1.0/pow(t8713,4.0/3.0);
                double t8717 = sqrt(t8715);
                double t8718 = EcP_3*t8717;
                double t8719 = EcP_4+t8715+t8718;
                double t8720 = c*t8716*(1.0/3.0);
                double t8721 = 1.0/sqrt(t8715);
                double t8722 = EcP_3*c*t8721*t8716*(1.0/6.0);
                double t8723 = t8720+t8722;
                double t8724 = EcP_2-t8717;
                double t8725 = 1.0/(t8719*t8719);
                double t8726 = 1.0/t8719;
                double t8727 = t8717*2.0;
                double t8728 = EcP_3+t8727;
                double t8729 = 1.0/(t8728*t8728);
                double t8730 = EcP_4*4.0;
                double t8731 = EcP_3*EcP_3;
                double t8732 = t8730-t8731;
                double t8733 = t8732*t8729;
                double t8734 = t8733+1.0;
                double t8735 = 1.0/t8734;
                double t8736 = 1.0/c;
                double t8737 = c*t8716*t8726*(1.0/3.0);
                double t8740 = c*t8714*t8723*t8725;
                double t8738 = -t8740+t8737;
                double t8739 = pow(t8713,1.0/3.0);
                double t8741 = 1.0/pow(t8713,7.0/3.0);
                double t8742 = EcP_2*EcP_2;
                double t8743 = EcP_2*EcP_3;
                double t8744 = EcP_4+t8742+t8743;
                double t8745 = 1.0/t8744;
                double t8746 = t8724*t8724;
                double t8747 = t8723*t8725*t8746;
                double t8748 = c*t8721*t8724*t8716*t8726*(1.0/3.0);
                double t8749 = t8747+t8748;
                double t8750 = 1.0/(t8724*t8724);
                double t8751 = c*t8741*(4.0/9.0);
                double t8752 = EcP_3*c*t8721*t8741*(2.0/9.0);
                double t8753 = c*c;
                double t8754 = 1.0/pow(t8713,8.0/3.0);
                double t8755 = 1.0/pow(t8715,3.0/2.0);
                double t8800 = EcP_3*t8753*t8754*t8755*(1.0/3.6E1);
                double t8756 = t8751+t8752-t8800;
                double t8757 = t8723*t8723;
                double t8758 = 1.0/(t8719*t8719*t8719);
                double t8759 = EcP_2*4.0;
                double t8760 = EcP_3*2.0;
                double t8761 = t8760+t8759;
                double t8762 = 1.0/(t8728*t8728*t8728);
                double t8763 = 1.0/(t8728*t8728*t8728*t8728*t8728);
                double t8764 = 1.0/(t8734*t8734);
                double t8765 = EcF_3*t8717;
                double t8766 = EcF_4+t8715+t8765;
                double t8767 = EcF_3*c*t8721*t8716*(1.0/6.0);
                double t8768 = t8720+t8767;
                double t8769 = 1.0/t8766;
                double t8770 = c*t8716*t8769*(1.0/3.0);
                double t8771 = 1.0/(t8766*t8766);
                double t8826 = c*t8714*t8771*t8768;
                double t8772 = t8770-t8826;
                double t8773 = 1.0/pow(t8713,2.0/3.0);
                double t8774 = EcF_2-t8717;
                double t8775 = 1.0/(t8774*t8774);
                double t8776 = t8774*t8774;
                double t8777 = EcF_3*c*t8721*t8741*(2.0/9.0);
                double t8778 = t8751+t8777-EcF_3*t8753*t8754*t8755*(1.0/3.6E1);
                double t8779 = t8768*t8768;
                double t8780 = 1.0/(t8766*t8766*t8766);
                double t8781 = EcF_3+t8727;
                double t8782 = EcF_2*4.0;
                double t8783 = EcF_3*2.0;
                double t8784 = t8782+t8783;
                double t8785 = 1.0/(t8781*t8781);
                double t8786 = EcF_4*4.0;
                double t8787 = EcF_3*EcF_3;
                double t8788 = t8786-t8787;
                double t8789 = t8785*t8788;
                double t8790 = t8789+1.0;
                double t8791 = 1.0/t8790;
                double t8792 = t8771*t8776*t8768;
                double t8793 = c*t8721*t8716*t8774*t8769*(1.0/3.0);
                double t8794 = t8792+t8793;
                double t8795 = 1.0/(t8781*t8781*t8781);
                double t8796 = 1.0/(t8781*t8781*t8781*t8781*t8781);
                double t8797 = 1.0/(t8790*t8790);
                double t8798 = t8723*t8736*t8738*t8739;
                double t8799 = c*t8741*t8726*(4.0/9.0);
                double t8801 = c*t8714*t8757*t8758*2.0;
                double t8802 = t8799+t8801-c*t8723*t8716*t8725*(2.0/3.0)-c*t8714*t8725*t8756;
                double t8803 = t8736*t8719*t8739*t8802;
                double t8804 = t8723*t8750*t8749;
                double t8805 = c*t8741*t8726*(1.0/1.8E1);
                double t8806 = t8746*t8757*t8758*2.0;
                double t8807 = t8724*t8726*t8753*t8754*t8755*(1.0/1.8E1);
                double t8808 = c*t8721*t8723*t8724*t8716*t8725*(2.0/3.0);
                double t8809 = t8805+t8806+t8807+t8808-t8725*t8746*t8756-c*t8721*t8741*t8724*t8726*(4.0/9.0);
                double t8810 = c*t8721*t8741*t8761*t8735*t8729*(4.0/9.0);
                double t8811 = 1.0/(t8724*t8724*t8724);
                double t8812 = c*t8721*t8716*t8719*t8749*t8811*(1.0/3.0);
                double t8813 = c*t8732*t8741*t8761*t8763*t8764*(2.0/9.0);
                double t8814 = t8810+t8812+t8804+t8813-t8750*t8719*t8809-c*t8741*t8761*t8735*t8762*(2.0/9.0)-t8761*t8735*t8753*t8754*t8755*t8729*(1.0/1.8E1);
                double t8815 = EcP_2*EcP_3*t8745*t8814;
                double t8816 = EcP_3*c*t8741*t8735*t8762*(4.0/9.0);
                double t8817 = EcP_3*t8735*t8753*t8754*t8755*t8729*(1.0/9.0);
                double t8818 = t8798+t8803+t8815+t8816+t8817-t8736*t8719*t8773*t8738*(1.0/3.0)-EcP_3*c*t8721*t8741*t8735*t8729*(8.0/9.0)-EcP_3*c*t8732*t8741*t8763*t8764*(4.0/9.0);
                double t8819 = EcP_1*t8818;
                double t8820 = pow(2.0,1.0/3.0);
                double t8821 = t8820*2.0;
                double t8822 = t8821-2.0;
                double t8823 = two_13*2.0;
                double t8824 = t8823-2.0;
                double t8825 = 1.0/t8824;
                double t8827 = EcF_2*EcF_2;
                double t8828 = EcF_2*EcF_3;
                double t8829 = EcF_4+t8827+t8828;
                double t8830 = 1.0/t8829;
                double t8831 = t8736*t8719*t8738*t8739;
                double t8832 = t8750*t8719*t8749;
                double t8833 = c*t8721*t8716*t8761*t8735*t8729*(1.0/3.0);
                double t8834 = t8832+t8833;
                double t8835 = EcP_2*EcP_3*t8745*t8834;
                double t8836 = t8831+t8835-EcP_3*c*t8721*t8716*t8735*t8729*(2.0/3.0);
                v_rho_a_rho_a[Q] += scale * (EcP_1*t8836*-2.0+t8713*(t8819-t8822*t8825*(t8819-EcF_1*(t8736*t8739*t8766*(c*t8741*t8769*(4.0/9.0)-c*t8714*t8771*t8778+c*t8714*t8780*t8779*2.0-c*t8716*t8771*t8768*(2.0/3.0))-t8736*t8772*t8773*t8766*(1.0/3.0)+t8736*t8772*t8739*t8768+EcF_2*EcF_3*t8830*(-t8766*t8775*(c*t8741*t8769*(1.0/1.8E1)-t8771*t8776*t8778+t8780*t8776*t8779*2.0-c*t8721*t8741*t8774*t8769*(4.0/9.0)+t8753*t8754*t8755*t8774*t8769*(1.0/1.8E1)+c*t8721*t8716*t8771*t8774*t8768*(2.0/3.0))+t8775*t8794*t8768-c*t8741*t8791*t8784*t8795*(2.0/9.0)+c*t8721*t8741*t8791*t8784*t8785*(4.0/9.0)+c*t8741*t8784*t8796*t8788*t8797*(2.0/9.0)-t8753*t8754*t8755*t8791*t8784*t8785*(1.0/1.8E1)+c*t8721*t8716*1.0/(t8774*t8774*t8774)*t8766*t8794*(1.0/3.0))+EcF_3*c*t8741*t8791*t8795*(4.0/9.0)-EcF_3*c*t8721*t8741*t8791*t8785*(8.0/9.0)-EcF_3*c*t8741*t8796*t8788*t8797*(4.0/9.0)+EcF_3*t8753*t8754*t8755*t8791*t8785*(1.0/9.0))))+t8822*t8825*(EcP_1*t8836-EcF_1*(t8736*t8772*t8739*t8766+EcF_2*EcF_3*t8830*(t8766*t8775*t8794+c*t8721*t8716*t8791*t8784*t8785*(1.0/3.0))-EcF_3*c*t8721*t8716*t8791*t8785*(2.0/3.0)))*2.0);
            }
            
            // v_rho_a_rho_b
            if (deriv >= 2) {
                double t8838 = rho_a+rho_b;
                double t8839 = 1.0/pow(t8838,1.0/3.0);
                double t8840 = c*t8839;
                double t8841 = 1.0/pow(t8838,4.0/3.0);
                double t8842 = sqrt(t8840);
                double t8843 = EcP_3*t8842;
                double t8844 = EcP_4+t8840+t8843;
                double t8845 = c*t8841*(1.0/3.0);
                double t8846 = 1.0/sqrt(t8840);
                double t8847 = EcP_3*c*t8841*t8846*(1.0/6.0);
                double t8848 = t8845+t8847;
                double t8849 = EcP_2-t8842;
                double t8850 = 1.0/(t8844*t8844);
                double t8851 = 1.0/t8844;
                double t8852 = t8842*2.0;
                double t8853 = EcP_3+t8852;
                double t8854 = 1.0/(t8853*t8853);
                double t8855 = EcP_4*4.0;
                double t8856 = EcP_3*EcP_3;
                double t8857 = t8855-t8856;
                double t8858 = t8854*t8857;
                double t8859 = t8858+1.0;
                double t8860 = 1.0/t8859;
                double t8861 = 1.0/c;
                double t8862 = c*t8841*t8851*(1.0/3.0);
                double t8865 = c*t8850*t8839*t8848;
                double t8863 = t8862-t8865;
                double t8864 = pow(t8838,1.0/3.0);
                double t8866 = 1.0/pow(t8838,7.0/3.0);
                double t8867 = EcP_2*EcP_2;
                double t8868 = EcP_2*EcP_3;
                double t8869 = EcP_4+t8867+t8868;
                double t8870 = 1.0/t8869;
                double t8871 = t8849*t8849;
                double t8872 = t8850*t8871*t8848;
                double t8873 = c*t8841*t8851*t8846*t8849*(1.0/3.0);
                double t8874 = t8872+t8873;
                double t8875 = 1.0/(t8849*t8849);
                double t8876 = c*t8866*(4.0/9.0);
                double t8877 = EcP_3*c*t8846*t8866*(2.0/9.0);
                double t8878 = c*c;
                double t8879 = 1.0/pow(t8838,8.0/3.0);
                double t8880 = 1.0/pow(t8840,3.0/2.0);
                double t8925 = EcP_3*t8880*t8878*t8879*(1.0/3.6E1);
                double t8881 = t8876+t8877-t8925;
                double t8882 = t8848*t8848;
                double t8883 = 1.0/(t8844*t8844*t8844);
                double t8884 = EcP_2*4.0;
                double t8885 = EcP_3*2.0;
                double t8886 = t8884+t8885;
                double t8887 = 1.0/(t8853*t8853*t8853);
                double t8888 = 1.0/(t8853*t8853*t8853*t8853*t8853);
                double t8889 = 1.0/(t8859*t8859);
                double t8890 = EcF_3*t8842;
                double t8891 = EcF_4+t8840+t8890;
                double t8892 = EcF_3*c*t8841*t8846*(1.0/6.0);
                double t8893 = t8845+t8892;
                double t8894 = 1.0/t8891;
                double t8895 = c*t8841*t8894*(1.0/3.0);
                double t8896 = 1.0/(t8891*t8891);
                double t8951 = c*t8839*t8893*t8896;
                double t8897 = t8895-t8951;
                double t8898 = 1.0/pow(t8838,2.0/3.0);
                double t8899 = EcF_2-t8842;
                double t8900 = 1.0/(t8899*t8899);
                double t8901 = t8899*t8899;
                double t8902 = EcF_3*c*t8846*t8866*(2.0/9.0);
                double t8903 = t8876+t8902-EcF_3*t8880*t8878*t8879*(1.0/3.6E1);
                double t8904 = t8893*t8893;
                double t8905 = 1.0/(t8891*t8891*t8891);
                double t8906 = EcF_3+t8852;
                double t8907 = EcF_2*4.0;
                double t8908 = EcF_3*2.0;
                double t8909 = t8907+t8908;
                double t8910 = 1.0/(t8906*t8906);
                double t8911 = EcF_4*4.0;
                double t8912 = EcF_3*EcF_3;
                double t8913 = t8911-t8912;
                double t8914 = t8910*t8913;
                double t8915 = t8914+1.0;
                double t8916 = 1.0/t8915;
                double t8917 = t8893*t8896*t8901;
                double t8918 = c*t8841*t8846*t8894*t8899*(1.0/3.0);
                double t8919 = t8917+t8918;
                double t8920 = 1.0/(t8906*t8906*t8906);
                double t8921 = 1.0/(t8906*t8906*t8906*t8906*t8906);
                double t8922 = 1.0/(t8915*t8915);
                double t8923 = t8861*t8863*t8864*t8848;
                double t8924 = c*t8851*t8866*(4.0/9.0);
                double t8926 = c*t8882*t8883*t8839*2.0;
                double t8927 = t8924+t8926-c*t8841*t8850*t8848*(2.0/3.0)-c*t8850*t8881*t8839;
                double t8928 = t8861*t8844*t8864*t8927;
                double t8929 = t8874*t8848*t8875;
                double t8930 = c*t8851*t8866*(1.0/1.8E1);
                double t8931 = t8871*t8882*t8883*2.0;
                double t8932 = t8851*t8880*t8849*t8878*t8879*(1.0/1.8E1);
                double t8933 = c*t8841*t8850*t8846*t8848*t8849*(2.0/3.0);
                double t8934 = t8930+t8931+t8932+t8933-t8850*t8871*t8881-c*t8851*t8846*t8866*t8849*(4.0/9.0);
                double t8935 = c*t8860*t8854*t8846*t8866*t8886*(4.0/9.0);
                double t8936 = 1.0/(t8849*t8849*t8849);
                double t8937 = c*t8841*t8844*t8846*t8874*t8936*(1.0/3.0);
                double t8938 = c*t8857*t8866*t8886*t8888*t8889*(2.0/9.0);
                double t8939 = t8935+t8937+t8929+t8938-t8844*t8875*t8934-c*t8860*t8866*t8886*t8887*(2.0/9.0)-t8860*t8880*t8854*t8886*t8878*t8879*(1.0/1.8E1);
                double t8940 = EcP_2*EcP_3*t8870*t8939;
                double t8941 = EcP_3*c*t8860*t8866*t8887*(4.0/9.0);
                double t8942 = EcP_3*t8860*t8880*t8854*t8878*t8879*(1.0/9.0);
                double t8943 = t8940+t8923+t8941+t8942+t8928-t8861*t8844*t8863*t8898*(1.0/3.0)-EcP_3*c*t8860*t8854*t8846*t8866*(8.0/9.0)-EcP_3*c*t8857*t8866*t8888*t8889*(4.0/9.0);
                double t8944 = EcP_1*t8943;
                double t8945 = pow(2.0,1.0/3.0);
                double t8946 = t8945*2.0;
                double t8947 = t8946-2.0;
                double t8948 = two_13*2.0;
                double t8949 = t8948-2.0;
                double t8950 = 1.0/t8949;
                double t8952 = EcF_2*EcF_2;
                double t8953 = EcF_2*EcF_3;
                double t8954 = EcF_4+t8952+t8953;
                double t8955 = 1.0/t8954;
                double t8956 = t8861*t8844*t8863*t8864;
                double t8957 = t8844*t8874*t8875;
                double t8958 = c*t8841*t8860*t8854*t8846*t8886*(1.0/3.0);
                double t8959 = t8957+t8958;
                double t8960 = EcP_2*EcP_3*t8870*t8959;
                double t8961 = t8960+t8956-EcP_3*c*t8841*t8860*t8854*t8846*(2.0/3.0);
                v_rho_a_rho_b[Q] += scale * (EcP_1*t8961*-2.0+t8838*(t8944-t8950*t8947*(t8944-EcF_1*(t8861*t8864*t8891*(c*t8866*t8894*(4.0/9.0)-c*t8841*t8893*t8896*(2.0/3.0)-c*t8839*t8896*t8903+c*t8839*t8904*t8905*2.0)+t8861*t8864*t8893*t8897-t8861*t8891*t8897*t8898*(1.0/3.0)+EcF_2*EcF_3*t8955*(-t8891*t8900*(c*t8866*t8894*(1.0/1.8E1)-t8896*t8901*t8903+t8901*t8904*t8905*2.0-c*t8846*t8866*t8894*t8899*(4.0/9.0)+t8880*t8894*t8878*t8879*t8899*(1.0/1.8E1)+c*t8841*t8846*t8893*t8896*t8899*(2.0/3.0))+t8893*t8900*t8919-c*t8866*t8920*t8916*t8909*(2.0/9.0)+c*t8846*t8866*t8910*t8916*t8909*(4.0/9.0)+c*t8866*t8921*t8913*t8922*t8909*(2.0/9.0)-t8880*t8878*t8879*t8910*t8916*t8909*(1.0/1.8E1)+c*t8841*t8846*t8891*1.0/(t8899*t8899*t8899)*t8919*(1.0/3.0))+EcF_3*c*t8866*t8920*t8916*(4.0/9.0)-EcF_3*c*t8846*t8866*t8910*t8916*(8.0/9.0)-EcF_3*c*t8866*t8921*t8913*t8922*(4.0/9.0)+EcF_3*t8880*t8878*t8879*t8910*t8916*(1.0/9.0))))+t8950*t8947*(EcP_1*t8961-EcF_1*(t8861*t8864*t8891*t8897+EcF_2*EcF_3*t8955*(t8891*t8900*t8919+c*t8841*t8846*t8910*t8916*t8909*(1.0/3.0))-EcF_3*c*t8841*t8846*t8910*t8916*(2.0/3.0)))*2.0);
            }
            
            // v_rho_b_rho_b
            if (deriv >= 2) {
                double t8963 = rho_a+rho_b;
                double t8964 = 1.0/pow(t8963,1.0/3.0);
                double t8965 = c*t8964;
                double t8966 = 1.0/pow(t8963,4.0/3.0);
                double t8967 = sqrt(t8965);
                double t8968 = EcP_3*t8967;
                double t8969 = EcP_4+t8965+t8968;
                double t8970 = c*t8966*(1.0/3.0);
                double t8971 = 1.0/sqrt(t8965);
                double t8972 = EcP_3*c*t8971*t8966*(1.0/6.0);
                double t8973 = t8970+t8972;
                double t8974 = EcP_2-t8967;
                double t8975 = 1.0/(t8969*t8969);
                double t8976 = 1.0/t8969;
                double t8977 = t8967*2.0;
                double t8978 = EcP_3+t8977;
                double t8979 = 1.0/(t8978*t8978);
                double t8980 = EcP_4*4.0;
                double t8981 = EcP_3*EcP_3;
                double t8982 = t8980-t8981;
                double t8983 = t8982*t8979;
                double t8984 = t8983+1.0;
                double t8985 = 1.0/t8984;
                double t8986 = 1.0/c;
                double t8987 = c*t8966*t8976*(1.0/3.0);
                double t8990 = c*t8964*t8973*t8975;
                double t8988 = -t8990+t8987;
                double t8989 = pow(t8963,1.0/3.0);
                double t8991 = 1.0/pow(t8963,7.0/3.0);
                double t8992 = EcP_2*EcP_2;
                double t8993 = EcP_2*EcP_3;
                double t8994 = EcP_4+t8992+t8993;
                double t8995 = 1.0/t8994;
                double t8996 = t8974*t8974;
                double t8997 = t8973*t8975*t8996;
                double t8998 = c*t8971*t8974*t8966*t8976*(1.0/3.0);
                double t8999 = t8997+t8998;
                double t9000 = 1.0/(t8974*t8974);
                double t9001 = c*t8991*(4.0/9.0);
                double t9002 = EcP_3*c*t8971*t8991*(2.0/9.0);
                double t9003 = c*c;
                double t9004 = 1.0/pow(t8963,8.0/3.0);
                double t9005 = 1.0/pow(t8965,3.0/2.0);
                double t9050 = EcP_3*t9003*t9004*t9005*(1.0/3.6E1);
                double t9006 = t9001+t9002-t9050;
                double t9007 = t8973*t8973;
                double t9008 = 1.0/(t8969*t8969*t8969);
                double t9009 = EcP_2*4.0;
                double t9010 = EcP_3*2.0;
                double t9011 = t9010+t9009;
                double t9012 = 1.0/(t8978*t8978*t8978);
                double t9013 = 1.0/(t8978*t8978*t8978*t8978*t8978);
                double t9014 = 1.0/(t8984*t8984);
                double t9015 = EcF_3*t8967;
                double t9016 = EcF_4+t8965+t9015;
                double t9017 = EcF_3*c*t8971*t8966*(1.0/6.0);
                double t9018 = t8970+t9017;
                double t9019 = 1.0/t9016;
                double t9020 = c*t8966*t9019*(1.0/3.0);
                double t9021 = 1.0/(t9016*t9016);
                double t9076 = c*t8964*t9021*t9018;
                double t9022 = t9020-t9076;
                double t9023 = 1.0/pow(t8963,2.0/3.0);
                double t9024 = EcF_2-t8967;
                double t9025 = 1.0/(t9024*t9024);
                double t9026 = t9024*t9024;
                double t9027 = EcF_3*c*t8971*t8991*(2.0/9.0);
                double t9028 = t9001+t9027-EcF_3*t9003*t9004*t9005*(1.0/3.6E1);
                double t9029 = t9018*t9018;
                double t9030 = 1.0/(t9016*t9016*t9016);
                double t9031 = EcF_3+t8977;
                double t9032 = EcF_2*4.0;
                double t9033 = EcF_3*2.0;
                double t9034 = t9032+t9033;
                double t9035 = 1.0/(t9031*t9031);
                double t9036 = EcF_4*4.0;
                double t9037 = EcF_3*EcF_3;
                double t9038 = t9036-t9037;
                double t9039 = t9035*t9038;
                double t9040 = t9039+1.0;
                double t9041 = 1.0/t9040;
                double t9042 = t9021*t9026*t9018;
                double t9043 = c*t8971*t8966*t9024*t9019*(1.0/3.0);
                double t9044 = t9042+t9043;
                double t9045 = 1.0/(t9031*t9031*t9031);
                double t9046 = 1.0/(t9031*t9031*t9031*t9031*t9031);
                double t9047 = 1.0/(t9040*t9040);
                double t9048 = t8973*t8986*t8988*t8989;
                double t9049 = c*t8991*t8976*(4.0/9.0);
                double t9051 = c*t8964*t9007*t9008*2.0;
                double t9052 = t9051+t9049-c*t8973*t8966*t8975*(2.0/3.0)-c*t8964*t8975*t9006;
                double t9053 = t8986*t8969*t8989*t9052;
                double t9054 = t8973*t8999*t9000;
                double t9055 = c*t8991*t8976*(1.0/1.8E1);
                double t9056 = t8996*t9007*t9008*2.0;
                double t9057 = t8974*t8976*t9003*t9004*t9005*(1.0/1.8E1);
                double t9058 = c*t8971*t8973*t8974*t8966*t8975*(2.0/3.0);
                double t9059 = t9055+t9056+t9057+t9058-t8975*t8996*t9006-c*t8971*t8991*t8974*t8976*(4.0/9.0);
                double t9060 = c*t8971*t8991*t8985*t8979*t9011*(4.0/9.0);
                double t9061 = 1.0/(t8974*t8974*t8974);
                double t9062 = c*t8971*t8966*t8969*t8999*t9061*(1.0/3.0);
                double t9063 = c*t8982*t8991*t9011*t9013*t9014*(2.0/9.0);
                double t9064 = t9060+t9062+t9054+t9063-t8969*t9000*t9059-c*t8991*t8985*t9011*t9012*(2.0/9.0)-t8985*t8979*t9011*t9003*t9004*t9005*(1.0/1.8E1);
                double t9065 = EcP_2*EcP_3*t8995*t9064;
                double t9066 = EcP_3*c*t8991*t8985*t9012*(4.0/9.0);
                double t9067 = EcP_3*t8985*t8979*t9003*t9004*t9005*(1.0/9.0);
                double t9068 = t9053+t9065+t9048+t9066+t9067-t8986*t8969*t8988*t9023*(1.0/3.0)-EcP_3*c*t8971*t8991*t8985*t8979*(8.0/9.0)-EcP_3*c*t8982*t8991*t9013*t9014*(4.0/9.0);
                double t9069 = EcP_1*t9068;
                double t9070 = pow(2.0,1.0/3.0);
                double t9071 = t9070*2.0;
                double t9072 = t9071-2.0;
                double t9073 = two_13*2.0;
                double t9074 = t9073-2.0;
                double t9075 = 1.0/t9074;
                double t9077 = EcF_2*EcF_2;
                double t9078 = EcF_2*EcF_3;
                double t9079 = EcF_4+t9077+t9078;
                double t9080 = 1.0/t9079;
                double t9081 = t8986*t8969*t8988*t8989;
                double t9082 = t8969*t8999*t9000;
                double t9083 = c*t8971*t8966*t8985*t8979*t9011*(1.0/3.0);
                double t9084 = t9082+t9083;
                double t9085 = EcP_2*EcP_3*t8995*t9084;
                double t9086 = t9081+t9085-EcP_3*c*t8971*t8966*t8985*t8979*(2.0/3.0);
                v_rho_b_rho_b[Q] += scale * (EcP_1*t9086*-2.0+t8963*(t9069-t9072*t9075*(t9069-EcF_1*(t8986*t8989*t9016*(c*t8991*t9019*(4.0/9.0)-c*t8964*t9021*t9028+c*t8964*t9030*t9029*2.0-c*t8966*t9021*t9018*(2.0/3.0))+t8986*t8989*t9022*t9018-t8986*t9022*t9023*t9016*(1.0/3.0)+EcF_2*EcF_3*t9080*(-t9016*t9025*(c*t8991*t9019*(1.0/1.8E1)-t9021*t9026*t9028+t9030*t9026*t9029*2.0-c*t8971*t8991*t9024*t9019*(4.0/9.0)+t9003*t9004*t9005*t9024*t9019*(1.0/1.8E1)+c*t8971*t8966*t9021*t9024*t9018*(2.0/3.0))+t9025*t9044*t9018-c*t8991*t9041*t9034*t9045*(2.0/9.0)+c*t8971*t8991*t9041*t9034*t9035*(4.0/9.0)+c*t8991*t9034*t9046*t9038*t9047*(2.0/9.0)-t9003*t9004*t9005*t9041*t9034*t9035*(1.0/1.8E1)+c*t8971*t8966*1.0/(t9024*t9024*t9024)*t9016*t9044*(1.0/3.0))+EcF_3*c*t8991*t9041*t9045*(4.0/9.0)-EcF_3*c*t8971*t8991*t9041*t9035*(8.0/9.0)-EcF_3*c*t8991*t9046*t9038*t9047*(4.0/9.0)+EcF_3*t9003*t9004*t9005*t9041*t9035*(1.0/9.0))))+t9072*t9075*(EcP_1*t9086-EcF_1*(t8986*t8989*t9022*t9016+EcF_2*EcF_3*t9080*(t9016*t9025*t9044+c*t8971*t8966*t9041*t9034*t9035*(1.0/3.0))-EcF_3*c*t8971*t8966*t9041*t9035*(2.0/3.0)))*2.0);
            }
            
        } else {
            // v
            if (deriv >= 0) {
                double t6979 = rho_a+rho_b;
                double t6980 = 1.0/pow(t6979,1.0/3.0);
                double t6981 = c*t6980;
                double t6982 = sqrt(t6981);
                double t6983 = EcP_4*4.0;
                double t6984 = EcP_3*EcP_3;
                double t6985 = t6983-t6984;
                double t6986 = EcP_2-t6982;
                double t6987 = EcP_3*t6982;
                double t6988 = EcP_4+t6981+t6987;
                double t6989 = 1.0/t6988;
                double t6990 = sqrt(t6985);
                double t6991 = t6982*2.0;
                double t6992 = EcP_3+t6991;
                double t6993 = 1.0/t6992;
                double t6994 = t6990*t6993;
                double t6995 = atan(t6994);
                double t6996 = 1.0/sqrt(t6985);
                double t6997 = EcF_4*4.0;
                double t6998 = EcF_3*EcF_3;
                double t6999 = t6997-t6998;
                double t7000 = EcF_2-t6982;
                double t7001 = EcF_3*t6982;
                double t7002 = EcF_4+t6981+t7001;
                double t7003 = 1.0/t7002;
                double t7004 = sqrt(t6999);
                double t7005 = EcF_3+t6991;
                double t7006 = 1.0/t7005;
                double t7007 = t7004*t7006;
                double t7008 = atan(t7007);
                double t7009 = 1.0/sqrt(t6999);
                double t7010 = c*t6980*t6989;
                double t7011 = log(t7010);
                double t7012 = EcP_3*t6995*t6996*2.0;
                double t7013 = t6986*t6986;
                double t7014 = t6989*t7013;
                double t7015 = log(t7014);
                double t7016 = EcP_2*4.0;
                double t7017 = EcP_3*2.0;
                double t7018 = t7016+t7017;
                double t7019 = t6995*t6996*t7018;
                double t7020 = t7015+t7019;
                double t7021 = EcP_2*EcP_2;
                double t7022 = EcP_2*EcP_3;
                double t7023 = EcP_4+t7021+t7022;
                double t7024 = 1.0/t7023;
                double t7025 = t7011+t7012-EcP_2*EcP_3*t7020*t7024;
                double t7026 = EcP_1*t7025;
                double t7027 = 1.0/t6979;
                double t7028 = rho_a-rho_b;
                double t7029 = t7027*t7028;
                v[Q] += scale * (t6979*(t7026-((t7026-EcF_1*(log(c*t6980*t7003)+EcF_3*t7008*t7009*2.0-(EcF_2*EcF_3*(log((t7000*t7000)*t7003)+t7008*t7009*(EcF_2*4.0+EcF_3*2.0)))/(EcF_4+EcF_2*EcF_2+EcF_2*EcF_3)))*(pow(t7029+1.0,4.0/3.0)+pow(-t7029+1.0,4.0/3.0)-2.0))/(two_13*2.0-2.0)));
            }
            
            // v_rho_a
            if (deriv >= 1) {
                double t7031 = rho_a+rho_b;
                double t7032 = 1.0/pow(t7031,1.0/3.0);
                double t7033 = c*t7032;
                double t7034 = 1.0/pow(t7031,4.0/3.0);
                double t7035 = sqrt(t7033);
                double t7036 = EcP_3*t7035;
                double t7037 = EcP_4+t7033+t7036;
                double t7038 = c*t7034*(1.0/3.0);
                double t7039 = 1.0/sqrt(t7033);
                double t7040 = EcP_3*c*t7034*t7039*(1.0/6.0);
                double t7041 = t7040+t7038;
                double t7042 = EcP_2-t7035;
                double t7043 = 1.0/(t7037*t7037);
                double t7044 = 1.0/t7037;
                double t7045 = t7035*2.0;
                double t7046 = EcP_3+t7045;
                double t7047 = 1.0/(t7046*t7046);
                double t7048 = EcP_4*4.0;
                double t7049 = EcP_3*EcP_3;
                double t7050 = t7048-t7049;
                double t7051 = t7050*t7047;
                double t7052 = t7051+1.0;
                double t7053 = 1.0/t7052;
                double t7054 = 1.0/t7031;
                double t7055 = rho_a-rho_b;
                double t7056 = t7054*t7055;
                double t7057 = 1.0/(t7031*t7031);
                double t7058 = t7054-t7055*t7057;
                double t7059 = EcF_4*4.0;
                double t7060 = EcF_3*EcF_3;
                double t7061 = -t7060+t7059;
                double t7062 = EcF_2-t7035;
                double t7063 = EcF_3*t7035;
                double t7064 = EcF_4+t7033+t7063;
                double t7065 = 1.0/t7064;
                double t7066 = sqrt(t7061);
                double t7067 = EcF_3+t7045;
                double t7068 = 1.0/t7067;
                double t7069 = t7066*t7068;
                double t7070 = atan(t7069);
                double t7071 = 1.0/sqrt(t7061);
                double t7072 = t7042*t7042;
                double t7073 = sqrt(t7050);
                double t7074 = 1.0/t7046;
                double t7075 = t7073*t7074;
                double t7076 = atan(t7075);
                double t7077 = 1.0/sqrt(t7050);
                double t7078 = EcP_2*4.0;
                double t7079 = EcP_3*2.0;
                double t7080 = t7078+t7079;
                double t7081 = EcP_2*EcP_2;
                double t7082 = EcP_2*EcP_3;
                double t7083 = EcP_4+t7081+t7082;
                double t7084 = 1.0/t7083;
                double t7085 = two_13*2.0;
                double t7086 = t7085-2.0;
                double t7087 = 1.0/t7086;
                double t7088 = 1.0/c;
                double t7089 = pow(t7031,1.0/3.0);
                double t7090 = EcF_3*c*t7034*t7039*(1.0/6.0);
                double t7091 = t7090+t7038;
                double t7092 = t7062*t7062;
                double t7093 = 1.0/(t7064*t7064);
                double t7094 = EcF_2*4.0;
                double t7095 = EcF_3*2.0;
                double t7096 = t7094+t7095;
                double t7097 = 1.0/(t7067*t7067);
                double t7098 = EcF_2*EcF_2;
                double t7099 = EcF_2*EcF_3;
                double t7100 = EcF_4+t7098+t7099;
                double t7101 = 1.0/t7100;
                double t7102 = t7061*t7097;
                double t7103 = t7102+1.0;
                double t7104 = 1.0/t7103;
                double t7105 = c*t7034*t7044*(1.0/3.0);
                double t7106 = t7105-c*t7032*t7041*t7043;
                double t7107 = t7037*t7088*t7089*t7106;
                double t7108 = t7041*t7043*t7072;
                double t7109 = c*t7042*t7034*t7044*t7039*(1.0/3.0);
                double t7110 = t7108+t7109;
                double t7111 = 1.0/(t7042*t7042);
                double t7112 = t7037*t7110*t7111;
                double t7113 = c*t7034*t7053*t7080*t7047*t7039*(1.0/3.0);
                double t7114 = t7112+t7113;
                double t7115 = EcP_2*EcP_3*t7084*t7114;
                double t7116 = t7115+t7107-EcP_3*c*t7034*t7053*t7047*t7039*(2.0/3.0);
                double t7117 = EcP_1*t7116;
                double t7118 = t7056+1.0;
                double t7119 = -t7056+1.0;
                double t7120 = c*t7032*t7044;
                double t7121 = log(t7120);
                double t7122 = EcP_3*t7076*t7077*2.0;
                double t7123 = t7044*t7072;
                double t7124 = log(t7123);
                double t7125 = t7080*t7076*t7077;
                double t7126 = t7124+t7125;
                double t7137 = EcP_2*EcP_3*t7084*t7126;
                double t7127 = t7121+t7122-t7137;
                double t7128 = EcP_1*t7127;
                double t7129 = c*t7032*t7065;
                double t7130 = log(t7129);
                double t7131 = EcF_3*t7070*t7071*2.0;
                double t7132 = t7065*t7092;
                double t7133 = log(t7132);
                double t7134 = t7070*t7071*t7096;
                double t7135 = t7133+t7134;
                double t7136 = t7130+t7131-EcF_2*EcF_3*t7101*t7135;
                double t7138 = pow(t7118,4.0/3.0);
                double t7139 = pow(t7119,4.0/3.0);
                double t7140 = t7138+t7139-2.0;
                double t7141 = t7128-EcF_1*t7136;
                v_rho_a[Q] += scale * (t7128-t7031*(t7117+t7087*t7141*(t7058*pow(t7118,1.0/3.0)*(4.0/3.0)-t7058*pow(t7119,1.0/3.0)*(4.0/3.0))-t7087*t7140*(t7117-EcF_1*(EcF_2*EcF_3*t7101*(1.0/(t7062*t7062)*t7064*(t7091*t7092*t7093+c*t7034*t7062*t7065*t7039*(1.0/3.0))+c*t7034*t7039*t7096*t7097*t7104*(1.0/3.0))+t7064*t7088*t7089*(c*t7034*t7065*(1.0/3.0)-c*t7032*t7091*t7093)-EcF_3*c*t7034*t7039*t7097*t7104*(2.0/3.0))))-t7087*t7140*t7141);
            }
            
            // v_rho_b
            if (deriv >= 1) {
                double t7143 = rho_a+rho_b;
                double t7144 = 1.0/pow(t7143,1.0/3.0);
                double t7145 = c*t7144;
                double t7146 = 1.0/pow(t7143,4.0/3.0);
                double t7147 = sqrt(t7145);
                double t7148 = EcP_3*t7147;
                double t7149 = EcP_4+t7145+t7148;
                double t7150 = c*t7146*(1.0/3.0);
                double t7151 = 1.0/sqrt(t7145);
                double t7152 = EcP_3*c*t7151*t7146*(1.0/6.0);
                double t7153 = t7150+t7152;
                double t7154 = EcP_2-t7147;
                double t7155 = 1.0/(t7149*t7149);
                double t7156 = 1.0/t7149;
                double t7157 = t7147*2.0;
                double t7158 = EcP_3+t7157;
                double t7159 = 1.0/(t7158*t7158);
                double t7160 = EcP_4*4.0;
                double t7161 = EcP_3*EcP_3;
                double t7162 = t7160-t7161;
                double t7163 = t7162*t7159;
                double t7164 = t7163+1.0;
                double t7165 = 1.0/t7164;
                double t7166 = 1.0/t7143;
                double t7167 = rho_a-rho_b;
                double t7168 = t7166*t7167;
                double t7169 = 1.0/(t7143*t7143);
                double t7170 = t7167*t7169;
                double t7171 = t7170+t7166;
                double t7172 = EcF_4*4.0;
                double t7173 = EcF_3*EcF_3;
                double t7174 = t7172-t7173;
                double t7175 = EcF_2-t7147;
                double t7176 = EcF_3*t7147;
                double t7177 = EcF_4+t7145+t7176;
                double t7178 = 1.0/t7177;
                double t7179 = sqrt(t7174);
                double t7180 = EcF_3+t7157;
                double t7181 = 1.0/t7180;
                double t7182 = t7181*t7179;
                double t7183 = atan(t7182);
                double t7184 = 1.0/sqrt(t7174);
                double t7185 = t7154*t7154;
                double t7186 = sqrt(t7162);
                double t7187 = 1.0/t7158;
                double t7188 = t7186*t7187;
                double t7189 = atan(t7188);
                double t7190 = 1.0/sqrt(t7162);
                double t7191 = EcP_2*4.0;
                double t7192 = EcP_3*2.0;
                double t7193 = t7191+t7192;
                double t7194 = EcP_2*EcP_2;
                double t7195 = EcP_2*EcP_3;
                double t7196 = EcP_4+t7194+t7195;
                double t7197 = 1.0/t7196;
                double t7198 = two_13*2.0;
                double t7199 = t7198-2.0;
                double t7200 = 1.0/t7199;
                double t7201 = 1.0/c;
                double t7202 = pow(t7143,1.0/3.0);
                double t7203 = EcF_3*c*t7151*t7146*(1.0/6.0);
                double t7204 = t7150+t7203;
                double t7205 = t7175*t7175;
                double t7206 = 1.0/(t7177*t7177);
                double t7207 = EcF_2*4.0;
                double t7208 = EcF_3*2.0;
                double t7209 = t7207+t7208;
                double t7210 = 1.0/(t7180*t7180);
                double t7211 = EcF_2*EcF_2;
                double t7212 = EcF_2*EcF_3;
                double t7213 = EcF_4+t7211+t7212;
                double t7214 = 1.0/t7213;
                double t7215 = t7174*t7210;
                double t7216 = t7215+1.0;
                double t7217 = 1.0/t7216;
                double t7218 = c*t7146*t7156*(1.0/3.0);
                double t7219 = t7218-c*t7144*t7153*t7155;
                double t7220 = t7149*t7201*t7202*t7219;
                double t7221 = t7153*t7155*t7185;
                double t7222 = c*t7151*t7154*t7146*t7156*(1.0/3.0);
                double t7223 = t7221+t7222;
                double t7224 = 1.0/(t7154*t7154);
                double t7225 = t7149*t7223*t7224;
                double t7226 = c*t7151*t7146*t7165*t7193*t7159*(1.0/3.0);
                double t7227 = t7225+t7226;
                double t7228 = EcP_2*EcP_3*t7197*t7227;
                double t7229 = t7220+t7228-EcP_3*c*t7151*t7146*t7165*t7159*(2.0/3.0);
                double t7230 = EcP_1*t7229;
                double t7231 = t7168+1.0;
                double t7232 = -t7168+1.0;
                double t7233 = c*t7144*t7156;
                double t7234 = log(t7233);
                double t7235 = EcP_3*t7190*t7189*2.0;
                double t7236 = t7156*t7185;
                double t7237 = log(t7236);
                double t7238 = t7190*t7193*t7189;
                double t7239 = t7237+t7238;
                double t7250 = EcP_2*EcP_3*t7197*t7239;
                double t7240 = -t7250+t7234+t7235;
                double t7241 = EcP_1*t7240;
                double t7242 = c*t7144*t7178;
                double t7243 = log(t7242);
                double t7244 = EcF_3*t7183*t7184*2.0;
                double t7245 = t7178*t7205;
                double t7246 = log(t7245);
                double t7247 = t7183*t7184*t7209;
                double t7248 = t7246+t7247;
                double t7249 = t7243+t7244-EcF_2*EcF_3*t7214*t7248;
                double t7251 = pow(t7231,4.0/3.0);
                double t7252 = pow(t7232,4.0/3.0);
                double t7253 = t7251+t7252-2.0;
                double t7254 = t7241-EcF_1*t7249;
                v_rho_b[Q] += scale * (t7241+t7143*(-t7230+t7200*t7254*(t7171*pow(t7231,1.0/3.0)*(4.0/3.0)-t7171*pow(t7232,1.0/3.0)*(4.0/3.0))+t7200*t7253*(t7230-EcF_1*(EcF_2*EcF_3*t7214*(1.0/(t7175*t7175)*t7177*(t7204*t7205*t7206+c*t7151*t7146*t7175*t7178*(1.0/3.0))+c*t7151*t7146*t7210*t7217*t7209*(1.0/3.0))+t7177*t7201*t7202*(c*t7146*t7178*(1.0/3.0)-c*t7144*t7204*t7206)-EcF_3*c*t7151*t7146*t7210*t7217*(2.0/3.0))))-t7200*t7253*t7254);
            }
            
            // v_rho_a_rho_a
            if (deriv >= 2) {
                double t7261 = rho_a+rho_b;
                double t7262 = 1.0/pow(t7261,1.0/3.0);
                double t7263 = c*t7262;
                double t7264 = 1.0/pow(t7261,4.0/3.0);
                double t7265 = sqrt(t7263);
                double t7266 = EcP_3*t7265;
                double t7267 = EcP_4+t7263+t7266;
                double t7268 = c*t7264*(1.0/3.0);
                double t7269 = 1.0/sqrt(t7263);
                double t7270 = EcP_3*c*t7264*t7269*(1.0/6.0);
                double t7271 = t7270+t7268;
                double t7272 = EcP_2-t7265;
                double t7273 = 1.0/(t7267*t7267);
                double t7274 = 1.0/t7267;
                double t7275 = t7265*2.0;
                double t7276 = EcP_3+t7275;
                double t7277 = 1.0/(t7276*t7276);
                double t7278 = EcP_4*4.0;
                double t7279 = EcP_3*EcP_3;
                double t7280 = t7278-t7279;
                double t7281 = t7280*t7277;
                double t7282 = t7281+1.0;
                double t7283 = 1.0/t7282;
                double t7284 = 1.0/c;
                double t7285 = c*t7264*t7274*(1.0/3.0);
                double t7288 = c*t7262*t7271*t7273;
                double t7286 = t7285-t7288;
                double t7287 = pow(t7261,1.0/3.0);
                double t7289 = 1.0/pow(t7261,7.0/3.0);
                double t7290 = EcP_2*EcP_2;
                double t7291 = EcP_2*EcP_3;
                double t7292 = EcP_4+t7290+t7291;
                double t7293 = 1.0/t7292;
                double t7294 = t7272*t7272;
                double t7295 = t7271*t7273*t7294;
                double t7296 = c*t7272*t7264*t7274*t7269*(1.0/3.0);
                double t7297 = t7295+t7296;
                double t7298 = 1.0/(t7272*t7272);
                double t7299 = c*t7289*(4.0/9.0);
                double t7300 = EcP_3*c*t7269*t7289*(2.0/9.0);
                double t7301 = c*c;
                double t7302 = 1.0/pow(t7261,8.0/3.0);
                double t7303 = 1.0/pow(t7263,3.0/2.0);
                double t7348 = EcP_3*t7301*t7302*t7303*(1.0/3.6E1);
                double t7304 = t7299+t7300-t7348;
                double t7305 = t7271*t7271;
                double t7306 = 1.0/(t7267*t7267*t7267);
                double t7307 = EcP_2*4.0;
                double t7308 = EcP_3*2.0;
                double t7309 = t7307+t7308;
                double t7310 = 1.0/(t7276*t7276*t7276);
                double t7311 = 1.0/(t7276*t7276*t7276*t7276*t7276);
                double t7312 = 1.0/(t7282*t7282);
                double t7313 = EcF_3*t7265;
                double t7314 = EcF_4+t7263+t7313;
                double t7315 = EcF_3*c*t7264*t7269*(1.0/6.0);
                double t7316 = t7268+t7315;
                double t7317 = 1.0/t7314;
                double t7318 = c*t7264*t7317*(1.0/3.0);
                double t7319 = 1.0/(t7314*t7314);
                double t7378 = c*t7262*t7316*t7319;
                double t7320 = t7318-t7378;
                double t7321 = 1.0/pow(t7261,2.0/3.0);
                double t7322 = EcF_2-t7265;
                double t7323 = 1.0/(t7322*t7322);
                double t7324 = t7322*t7322;
                double t7325 = EcF_3*c*t7269*t7289*(2.0/9.0);
                double t7326 = t7299+t7325-EcF_3*t7301*t7302*t7303*(1.0/3.6E1);
                double t7327 = t7316*t7316;
                double t7328 = 1.0/(t7314*t7314*t7314);
                double t7329 = EcF_3+t7275;
                double t7330 = EcF_2*4.0;
                double t7331 = EcF_3*2.0;
                double t7332 = t7330+t7331;
                double t7333 = 1.0/(t7329*t7329);
                double t7334 = EcF_4*4.0;
                double t7335 = EcF_3*EcF_3;
                double t7336 = t7334-t7335;
                double t7337 = t7333*t7336;
                double t7338 = t7337+1.0;
                double t7339 = 1.0/t7338;
                double t7340 = t7324*t7316*t7319;
                double t7341 = c*t7264*t7269*t7322*t7317*(1.0/3.0);
                double t7342 = t7340+t7341;
                double t7343 = 1.0/(t7329*t7329*t7329);
                double t7344 = 1.0/(t7329*t7329*t7329*t7329*t7329);
                double t7345 = 1.0/(t7338*t7338);
                double t7346 = t7271*t7284*t7286*t7287;
                double t7347 = c*t7274*t7289*(4.0/9.0);
                double t7349 = c*t7262*t7305*t7306*2.0;
                double t7350 = t7347+t7349-c*t7271*t7264*t7273*(2.0/3.0)-c*t7262*t7273*t7304;
                double t7351 = t7284*t7267*t7287*t7350;
                double t7352 = t7271*t7297*t7298;
                double t7353 = c*t7274*t7289*(1.0/1.8E1);
                double t7354 = t7294*t7305*t7306*2.0;
                double t7355 = t7272*t7274*t7301*t7302*t7303*(1.0/1.8E1);
                double t7356 = c*t7271*t7272*t7264*t7273*t7269*(2.0/3.0);
                double t7357 = t7353+t7354+t7355+t7356-t7273*t7294*t7304-c*t7272*t7274*t7269*t7289*(4.0/9.0);
                double t7358 = c*t7283*t7277*t7269*t7289*t7309*(4.0/9.0);
                double t7359 = 1.0/(t7272*t7272*t7272);
                double t7360 = c*t7264*t7267*t7269*t7297*t7359*(1.0/3.0);
                double t7361 = c*t7280*t7289*t7311*t7312*t7309*(2.0/9.0);
                double t7362 = t7360+t7352+t7361+t7358-t7267*t7298*t7357-c*t7283*t7289*t7310*t7309*(2.0/9.0)-t7283*t7277*t7301*t7302*t7303*t7309*(1.0/1.8E1);
                double t7363 = EcP_2*EcP_3*t7293*t7362;
                double t7364 = EcP_3*c*t7283*t7289*t7310*(4.0/9.0);
                double t7365 = EcP_3*t7283*t7277*t7301*t7302*t7303*(1.0/9.0);
                double t7366 = t7351+t7363+t7346+t7364+t7365-t7284*t7267*t7286*t7321*(1.0/3.0)-EcP_3*c*t7283*t7277*t7269*t7289*(8.0/9.0)-EcP_3*c*t7280*t7289*t7311*t7312*(4.0/9.0);
                double t7367 = EcP_1*t7366;
                double t7368 = 1.0/t7261;
                double t7369 = rho_a-rho_b;
                double t7370 = t7368*t7369;
                double t7371 = two_13*2.0;
                double t7372 = t7371-2.0;
                double t7373 = 1.0/t7372;
                double t7374 = t7370+1.0;
                double t7375 = -t7370+1.0;
                double t7376 = 1.0/(t7261*t7261);
                double t7404 = t7376*t7369;
                double t7377 = t7368-t7404;
                double t7379 = EcF_2*EcF_2;
                double t7380 = EcF_2*EcF_3;
                double t7381 = EcF_4+t7380+t7379;
                double t7382 = 1.0/t7381;
                double t7383 = t7284*t7267*t7286*t7287;
                double t7384 = t7267*t7297*t7298;
                double t7385 = c*t7264*t7283*t7277*t7269*t7309*(1.0/3.0);
                double t7386 = t7384+t7385;
                double t7387 = EcP_2*EcP_3*t7293*t7386;
                double t7433 = EcP_3*c*t7264*t7283*t7277*t7269*(2.0/3.0);
                double t7388 = t7383+t7387-t7433;
                double t7389 = sqrt(t7336);
                double t7390 = 1.0/t7329;
                double t7391 = t7390*t7389;
                double t7392 = atan(t7391);
                double t7393 = 1.0/sqrt(t7336);
                double t7394 = sqrt(t7280);
                double t7395 = 1.0/t7276;
                double t7396 = t7394*t7395;
                double t7397 = atan(t7396);
                double t7398 = 1.0/sqrt(t7280);
                double t7399 = pow(t7374,1.0/3.0);
                double t7400 = pow(t7375,1.0/3.0);
                double t7401 = t7376*2.0;
                double t7402 = 1.0/(t7261*t7261*t7261);
                double t7403 = t7401-t7369*t7402*2.0;
                double t7405 = t7377*t7377;
                double t7406 = t7377*t7399*(4.0/3.0);
                double t7407 = t7406-t7377*t7400*(4.0/3.0);
                double t7408 = c*t7262*t7317;
                double t7409 = log(t7408);
                double t7410 = EcF_3*t7392*t7393*2.0;
                double t7411 = t7324*t7317;
                double t7412 = log(t7411);
                double t7413 = t7332*t7392*t7393;
                double t7414 = t7412+t7413;
                double t7415 = t7410+t7409-EcF_2*EcF_3*t7382*t7414;
                double t7416 = EcF_1*t7415;
                double t7417 = c*t7262*t7274;
                double t7418 = log(t7417);
                double t7419 = EcP_3*t7397*t7398*2.0;
                double t7420 = t7274*t7294;
                double t7421 = log(t7420);
                double t7422 = t7309*t7397*t7398;
                double t7423 = t7421+t7422;
                double t7424 = t7418+t7419-EcP_2*EcP_3*t7293*t7423;
                double t7425 = t7416-EcP_1*t7424;
                double t7426 = t7284*t7287*t7320*t7314;
                double t7427 = t7314*t7323*t7342;
                double t7428 = c*t7264*t7269*t7332*t7333*t7339*(1.0/3.0);
                double t7429 = t7427+t7428;
                double t7430 = EcF_2*EcF_3*t7382*t7429;
                double t7431 = t7430+t7426-EcF_3*c*t7264*t7269*t7333*t7339*(2.0/3.0);
                double t7432 = EcF_1*t7431;
                double t7434 = t7432-EcP_1*t7388;
                double t7435 = pow(t7374,4.0/3.0);
                double t7436 = pow(t7375,4.0/3.0);
                double t7437 = t7435+t7436-2.0;
                v_rho_a_rho_a[Q] += scale * (EcP_1*t7388*-2.0+t7261*(t7367-t7373*t7437*(t7367-EcF_1*(t7284*t7287*t7314*(c*t7289*t7317*(4.0/9.0)-c*t7262*t7326*t7319+c*t7262*t7327*t7328*2.0-c*t7264*t7316*t7319*(2.0/3.0))+t7284*t7287*t7320*t7316-t7284*t7320*t7321*t7314*(1.0/3.0)+EcF_2*EcF_3*t7382*(-t7314*t7323*(c*t7289*t7317*(1.0/1.8E1)-t7324*t7326*t7319+t7324*t7327*t7328*2.0-c*t7269*t7289*t7322*t7317*(4.0/9.0)+t7301*t7302*t7303*t7322*t7317*(1.0/1.8E1)+c*t7264*t7269*t7322*t7316*t7319*(2.0/3.0))+t7323*t7342*t7316-c*t7289*t7332*t7343*t7339*(2.0/9.0)+c*t7269*t7289*t7332*t7333*t7339*(4.0/9.0)+c*t7289*t7332*t7344*t7336*t7345*(2.0/9.0)-t7301*t7302*t7303*t7332*t7333*t7339*(1.0/1.8E1)+c*t7264*t7269*1.0/(t7322*t7322*t7322)*t7314*t7342*(1.0/3.0))+EcF_3*c*t7289*t7343*t7339*(4.0/9.0)-EcF_3*c*t7269*t7289*t7333*t7339*(8.0/9.0)-EcF_3*c*t7289*t7344*t7336*t7345*(4.0/9.0)+EcF_3*t7301*t7302*t7303*t7333*t7339*(1.0/9.0)))+t7373*t7425*(t7399*t7403*(-4.0/3.0)+t7400*t7403*(4.0/3.0)+1.0/pow(t7374,2.0/3.0)*t7405*(4.0/9.0)+1.0/pow(t7375,2.0/3.0)*t7405*(4.0/9.0))-t7373*t7407*t7434*2.0)+t7373*t7407*t7425*2.0-t7373*t7434*t7437*2.0);
            }
            
            // v_rho_a_rho_b
            if (deriv >= 2) {
                double t7439 = rho_a+rho_b;
                double t7440 = 1.0/pow(t7439,1.0/3.0);
                double t7441 = c*t7440;
                double t7442 = 1.0/pow(t7439,4.0/3.0);
                double t7443 = sqrt(t7441);
                double t7444 = EcP_3*t7443;
                double t7445 = EcP_4+t7441+t7444;
                double t7446 = c*t7442*(1.0/3.0);
                double t7447 = 1.0/sqrt(t7441);
                double t7448 = EcP_3*c*t7442*t7447*(1.0/6.0);
                double t7449 = t7446+t7448;
                double t7450 = EcP_2-t7443;
                double t7451 = 1.0/(t7445*t7445);
                double t7452 = 1.0/t7445;
                double t7453 = t7443*2.0;
                double t7454 = EcP_3+t7453;
                double t7455 = 1.0/(t7454*t7454);
                double t7456 = EcP_4*4.0;
                double t7457 = EcP_3*EcP_3;
                double t7458 = t7456-t7457;
                double t7459 = t7455*t7458;
                double t7460 = t7459+1.0;
                double t7461 = 1.0/t7460;
                double t7462 = 1.0/c;
                double t7463 = c*t7442*t7452*(1.0/3.0);
                double t7466 = c*t7440*t7451*t7449;
                double t7464 = t7463-t7466;
                double t7465 = pow(t7439,1.0/3.0);
                double t7467 = 1.0/pow(t7439,7.0/3.0);
                double t7468 = EcP_2*EcP_2;
                double t7469 = EcP_2*EcP_3;
                double t7470 = EcP_4+t7468+t7469;
                double t7471 = 1.0/t7470;
                double t7472 = t7450*t7450;
                double t7473 = t7451*t7472*t7449;
                double t7474 = c*t7450*t7442*t7452*t7447*(1.0/3.0);
                double t7475 = t7473+t7474;
                double t7476 = 1.0/(t7450*t7450);
                double t7477 = c*t7467*(4.0/9.0);
                double t7478 = EcP_3*c*t7447*t7467*(2.0/9.0);
                double t7479 = c*c;
                double t7480 = 1.0/pow(t7439,8.0/3.0);
                double t7481 = 1.0/pow(t7441,3.0/2.0);
                double t7526 = EcP_3*t7480*t7481*t7479*(1.0/3.6E1);
                double t7482 = t7477+t7478-t7526;
                double t7483 = t7449*t7449;
                double t7484 = 1.0/(t7445*t7445*t7445);
                double t7485 = EcP_2*4.0;
                double t7486 = EcP_3*2.0;
                double t7487 = t7485+t7486;
                double t7488 = 1.0/(t7454*t7454*t7454);
                double t7489 = 1.0/(t7454*t7454*t7454*t7454*t7454);
                double t7490 = 1.0/(t7460*t7460);
                double t7491 = EcF_3*t7443;
                double t7492 = EcF_4+t7441+t7491;
                double t7493 = EcF_3*c*t7442*t7447*(1.0/6.0);
                double t7494 = t7446+t7493;
                double t7495 = 1.0/t7492;
                double t7496 = c*t7442*t7495*(1.0/3.0);
                double t7497 = 1.0/(t7492*t7492);
                double t7557 = c*t7440*t7494*t7497;
                double t7498 = t7496-t7557;
                double t7499 = 1.0/pow(t7439,2.0/3.0);
                double t7500 = EcF_2-t7443;
                double t7501 = 1.0/(t7500*t7500);
                double t7502 = t7500*t7500;
                double t7503 = EcF_3*c*t7447*t7467*(2.0/9.0);
                double t7504 = t7477+t7503-EcF_3*t7480*t7481*t7479*(1.0/3.6E1);
                double t7505 = t7494*t7494;
                double t7506 = 1.0/(t7492*t7492*t7492);
                double t7507 = EcF_3+t7453;
                double t7508 = EcF_2*4.0;
                double t7509 = EcF_3*2.0;
                double t7510 = t7508+t7509;
                double t7511 = 1.0/(t7507*t7507);
                double t7512 = EcF_4*4.0;
                double t7513 = EcF_3*EcF_3;
                double t7514 = t7512-t7513;
                double t7515 = t7511*t7514;
                double t7516 = t7515+1.0;
                double t7517 = 1.0/t7516;
                double t7518 = t7494*t7497*t7502;
                double t7519 = c*t7442*t7447*t7495*t7500*(1.0/3.0);
                double t7520 = t7518+t7519;
                double t7521 = 1.0/(t7507*t7507*t7507);
                double t7522 = 1.0/(t7507*t7507*t7507*t7507*t7507);
                double t7523 = 1.0/(t7516*t7516);
                double t7524 = t7462*t7464*t7465*t7449;
                double t7525 = c*t7452*t7467*(4.0/9.0);
                double t7527 = c*t7440*t7483*t7484*2.0;
                double t7528 = t7525+t7527-c*t7440*t7451*t7482-c*t7442*t7451*t7449*(2.0/3.0);
                double t7529 = t7462*t7445*t7465*t7528;
                double t7530 = t7475*t7449*t7476;
                double t7531 = c*t7452*t7467*(1.0/1.8E1);
                double t7532 = t7472*t7483*t7484*2.0;
                double t7533 = t7450*t7452*t7480*t7481*t7479*(1.0/1.8E1);
                double t7534 = c*t7450*t7442*t7451*t7447*t7449*(2.0/3.0);
                double t7535 = t7531+t7532+t7533+t7534-t7451*t7472*t7482-c*t7450*t7452*t7447*t7467*(4.0/9.0);
                double t7536 = c*t7461*t7455*t7447*t7467*t7487*(4.0/9.0);
                double t7537 = 1.0/(t7450*t7450*t7450);
                double t7538 = c*t7442*t7445*t7447*t7475*t7537*(1.0/3.0);
                double t7539 = c*t7490*t7458*t7467*t7487*t7489*(2.0/9.0);
                double t7540 = t7530+t7536+t7538+t7539-t7445*t7476*t7535-c*t7461*t7467*t7487*t7488*(2.0/9.0)-t7461*t7480*t7481*t7455*t7487*t7479*(1.0/1.8E1);
                double t7541 = EcP_2*EcP_3*t7471*t7540;
                double t7542 = EcP_3*c*t7461*t7467*t7488*(4.0/9.0);
                double t7543 = EcP_3*t7461*t7480*t7481*t7455*t7479*(1.0/9.0);
                double t7544 = t7541+t7524+t7542+t7543+t7529-t7462*t7445*t7464*t7499*(1.0/3.0)-EcP_3*c*t7461*t7455*t7447*t7467*(8.0/9.0)-EcP_3*c*t7490*t7458*t7467*t7489*(4.0/9.0);
                double t7545 = EcP_1*t7544;
                double t7546 = 1.0/t7439;
                double t7547 = rho_a-rho_b;
                double t7548 = t7546*t7547;
                double t7549 = two_13*2.0;
                double t7550 = t7549-2.0;
                double t7551 = 1.0/t7550;
                double t7552 = t7548+1.0;
                double t7553 = -t7548+1.0;
                double t7554 = 1.0/(t7439*t7439);
                double t7555 = t7554*t7547;
                double t7556 = t7546+t7555;
                double t7558 = EcF_2*EcF_2;
                double t7559 = EcF_2*EcF_3;
                double t7560 = EcF_4+t7558+t7559;
                double t7561 = 1.0/t7560;
                double t7562 = t7462*t7445*t7464*t7465;
                double t7563 = t7445*t7475*t7476;
                double t7564 = c*t7442*t7461*t7455*t7447*t7487*(1.0/3.0);
                double t7565 = t7563+t7564;
                double t7566 = EcP_2*EcP_3*t7471*t7565;
                double t7578 = EcP_3*c*t7442*t7461*t7455*t7447*(2.0/3.0);
                double t7567 = t7562+t7566-t7578;
                double t7568 = pow(t7552,1.0/3.0);
                double t7569 = pow(t7553,1.0/3.0);
                double t7570 = t7546-t7555;
                double t7571 = t7462*t7465*t7492*t7498;
                double t7572 = t7492*t7501*t7520;
                double t7573 = c*t7442*t7447*t7510*t7511*t7517*(1.0/3.0);
                double t7574 = t7572+t7573;
                double t7575 = EcF_2*EcF_3*t7561*t7574;
                double t7616 = EcF_3*c*t7442*t7447*t7511*t7517*(2.0/3.0);
                double t7576 = t7571+t7575-t7616;
                double t7577 = EcF_1*t7576;
                double t7617 = EcP_1*t7567;
                double t7579 = t7577-t7617;
                double t7580 = sqrt(t7514);
                double t7581 = 1.0/t7507;
                double t7582 = t7580*t7581;
                double t7583 = atan(t7582);
                double t7584 = 1.0/sqrt(t7514);
                double t7585 = sqrt(t7458);
                double t7586 = 1.0/t7454;
                double t7587 = t7585*t7586;
                double t7588 = atan(t7587);
                double t7589 = 1.0/sqrt(t7458);
                double t7590 = 1.0/(t7439*t7439*t7439);
                double t7591 = t7556*t7568*(4.0/3.0);
                double t7592 = t7591-t7556*t7569*(4.0/3.0);
                double t7593 = c*t7440*t7495;
                double t7594 = log(t7593);
                double t7595 = EcF_3*t7583*t7584*2.0;
                double t7596 = t7495*t7502;
                double t7597 = log(t7596);
                double t7598 = t7510*t7583*t7584;
                double t7599 = t7597+t7598;
                double t7613 = EcF_2*EcF_3*t7561*t7599;
                double t7600 = t7594+t7595-t7613;
                double t7601 = EcF_1*t7600;
                double t7602 = c*t7440*t7452;
                double t7603 = log(t7602);
                double t7604 = EcP_3*t7588*t7589*2.0;
                double t7605 = t7452*t7472;
                double t7606 = log(t7605);
                double t7607 = t7487*t7588*t7589;
                double t7608 = t7606+t7607;
                double t7614 = EcP_2*EcP_3*t7471*t7608;
                double t7609 = t7603+t7604-t7614;
                double t7615 = EcP_1*t7609;
                double t7610 = t7601-t7615;
                double t7611 = t7570*t7568*(4.0/3.0);
                double t7612 = t7611-t7570*t7569*(4.0/3.0);
                double t7618 = pow(t7552,4.0/3.0);
                double t7619 = pow(t7553,4.0/3.0);
                double t7620 = t7618+t7619-2.0;
                v_rho_a_rho_b[Q] += scale * (EcP_1*t7567*-2.0-t7439*(-t7545+t7551*t7620*(t7545-EcF_1*(t7462*t7465*t7492*(c*t7467*t7495*(4.0/9.0)-c*t7442*t7494*t7497*(2.0/3.0)-c*t7440*t7497*t7504+c*t7440*t7505*t7506*2.0)+t7462*t7465*t7494*t7498-t7462*t7492*t7498*t7499*(1.0/3.0)+EcF_2*EcF_3*t7561*(-t7492*t7501*(c*t7467*t7495*(1.0/1.8E1)-t7497*t7502*t7504+t7502*t7505*t7506*2.0-c*t7447*t7467*t7495*t7500*(4.0/9.0)+t7480*t7481*t7495*t7479*t7500*(1.0/1.8E1)+c*t7442*t7447*t7494*t7497*t7500*(2.0/3.0))+t7494*t7501*t7520-c*t7467*t7510*t7521*t7517*(2.0/9.0)+c*t7447*t7467*t7510*t7511*t7517*(4.0/9.0)+c*t7467*t7510*t7522*t7514*t7523*(2.0/9.0)-t7480*t7481*t7479*t7510*t7511*t7517*(1.0/1.8E1)+c*t7442*t7447*t7492*1.0/(t7500*t7500*t7500)*t7520*(1.0/3.0))+EcF_3*c*t7467*t7521*t7517*(4.0/9.0)-EcF_3*c*t7447*t7467*t7511*t7517*(8.0/9.0)-EcF_3*c*t7467*t7522*t7514*t7523*(4.0/9.0)+EcF_3*t7480*t7481*t7479*t7511*t7517*(1.0/9.0)))+t7551*t7610*(1.0/pow(t7552,2.0/3.0)*t7570*t7556*(4.0/9.0)+t7570*1.0/pow(t7553,2.0/3.0)*t7556*(4.0/9.0)-t7590*t7547*t7568*(8.0/3.0)+t7590*t7547*t7569*(8.0/3.0))-t7551*t7592*t7579+t7551*t7579*t7612)-t7551*t7592*t7610-t7551*t7579*t7620*2.0+t7551*t7610*t7612);
            }
            
            // v_rho_b_rho_b
            if (deriv >= 2) {
                double t7622 = rho_a+rho_b;
                double t7623 = 1.0/pow(t7622,1.0/3.0);
                double t7624 = c*t7623;
                double t7625 = 1.0/pow(t7622,4.0/3.0);
                double t7626 = sqrt(t7624);
                double t7627 = EcP_3*t7626;
                double t7628 = EcP_4+t7624+t7627;
                double t7629 = c*t7625*(1.0/3.0);
                double t7630 = 1.0/sqrt(t7624);
                double t7631 = EcP_3*c*t7630*t7625*(1.0/6.0);
                double t7632 = t7631+t7629;
                double t7633 = EcP_2-t7626;
                double t7634 = 1.0/(t7628*t7628);
                double t7635 = 1.0/t7628;
                double t7636 = t7626*2.0;
                double t7637 = EcP_3+t7636;
                double t7638 = 1.0/(t7637*t7637);
                double t7639 = EcP_4*4.0;
                double t7640 = EcP_3*EcP_3;
                double t7641 = t7640-t7639;
                double t7642 = t7641*t7638;
                double t7643 = t7642-1.0;
                double t7644 = 1.0/t7643;
                double t7645 = 1.0/c;
                double t7646 = c*t7625*t7635*(1.0/3.0);
                double t7649 = c*t7623*t7632*t7634;
                double t7647 = t7646-t7649;
                double t7648 = pow(t7622,1.0/3.0);
                double t7650 = 1.0/pow(t7622,7.0/3.0);
                double t7651 = EcP_2*EcP_2;
                double t7652 = EcP_2*EcP_3;
                double t7653 = EcP_4+t7651+t7652;
                double t7654 = 1.0/t7653;
                double t7655 = t7633*t7633;
                double t7656 = t7632*t7634*t7655;
                double t7657 = c*t7630*t7633*t7625*t7635*(1.0/3.0);
                double t7658 = t7656+t7657;
                double t7659 = 1.0/(t7633*t7633);
                double t7660 = c*t7650*(4.0/9.0);
                double t7661 = EcP_3*c*t7630*t7650*(2.0/9.0);
                double t7662 = c*c;
                double t7663 = 1.0/pow(t7622,8.0/3.0);
                double t7664 = 1.0/pow(t7624,3.0/2.0);
                double t7713 = EcP_3*t7662*t7663*t7664*(1.0/3.6E1);
                double t7665 = t7660+t7661-t7713;
                double t7666 = t7632*t7632;
                double t7667 = 1.0/(t7628*t7628*t7628);
                double t7668 = EcP_2*4.0;
                double t7669 = EcP_3*2.0;
                double t7670 = t7668+t7669;
                double t7671 = -t7640+t7639;
                double t7672 = t7671*t7638;
                double t7673 = t7672+1.0;
                double t7674 = 1.0/(t7637*t7637*t7637);
                double t7675 = 1.0/t7673;
                double t7676 = 1.0/(t7637*t7637*t7637*t7637*t7637);
                double t7677 = 1.0/(t7673*t7673);
                double t7678 = EcF_3*t7626;
                double t7679 = EcF_4+t7624+t7678;
                double t7680 = EcF_3*c*t7630*t7625*(1.0/6.0);
                double t7681 = t7680+t7629;
                double t7682 = 1.0/t7679;
                double t7683 = c*t7625*t7682*(1.0/3.0);
                double t7684 = 1.0/(t7679*t7679);
                double t7739 = c*t7623*t7681*t7684;
                double t7685 = t7683-t7739;
                double t7686 = 1.0/pow(t7622,2.0/3.0);
                double t7687 = EcF_2-t7626;
                double t7688 = 1.0/(t7687*t7687);
                double t7689 = t7687*t7687;
                double t7690 = EcF_3*c*t7630*t7650*(2.0/9.0);
                double t7691 = t7660+t7690-EcF_3*t7662*t7663*t7664*(1.0/3.6E1);
                double t7692 = t7681*t7681;
                double t7693 = 1.0/(t7679*t7679*t7679);
                double t7694 = EcF_3+t7636;
                double t7695 = EcF_2*4.0;
                double t7696 = EcF_3*2.0;
                double t7697 = t7695+t7696;
                double t7698 = 1.0/(t7694*t7694);
                double t7699 = EcF_4*4.0;
                double t7700 = EcF_3*EcF_3;
                double t7701 = t7699-t7700;
                double t7702 = t7698*t7701;
                double t7703 = t7702+1.0;
                double t7704 = 1.0/t7703;
                double t7705 = t7681*t7684*t7689;
                double t7706 = c*t7630*t7625*t7682*t7687*(1.0/3.0);
                double t7707 = t7705+t7706;
                double t7708 = 1.0/(t7694*t7694*t7694);
                double t7709 = 1.0/(t7694*t7694*t7694*t7694*t7694);
                double t7710 = 1.0/(t7703*t7703);
                double t7711 = t7632*t7645*t7647*t7648;
                double t7712 = c*t7650*t7635*(4.0/9.0);
                double t7714 = c*t7623*t7666*t7667*2.0;
                double t7715 = t7712+t7714-c*t7632*t7625*t7634*(2.0/3.0)-c*t7623*t7634*t7665;
                double t7716 = t7645*t7628*t7648*t7715;
                double t7717 = t7632*t7658*t7659;
                double t7718 = c*t7650*t7635*(1.0/1.8E1);
                double t7719 = t7655*t7666*t7667*2.0;
                double t7720 = t7633*t7635*t7662*t7663*t7664*(1.0/1.8E1);
                double t7721 = c*t7630*t7632*t7633*t7625*t7634*(2.0/3.0);
                double t7722 = t7720+t7721+t7718+t7719-t7634*t7655*t7665-c*t7630*t7650*t7633*t7635*(4.0/9.0);
                double t7723 = 1.0/(t7633*t7633*t7633);
                double t7724 = c*t7630*t7625*t7628*t7658*t7723*(1.0/3.0);
                double t7725 = c*t7650*t7670*t7671*t7676*t7677*(2.0/9.0);
                double t7726 = EcP_3*c*t7650*t7674*t7675*(4.0/9.0);
                double t7727 = EcP_3*t7662*t7663*t7664*t7638*t7675*(1.0/9.0);
                double t7728 = 1.0/t7622;
                double t7729 = rho_a-rho_b;
                double t7730 = t7728*t7729;
                double t7731 = two_13*2.0;
                double t7732 = t7731-2.0;
                double t7733 = 1.0/t7732;
                double t7734 = t7730+1.0;
                double t7735 = -t7730+1.0;
                double t7736 = 1.0/(t7622*t7622);
                double t7737 = t7736*t7729;
                double t7738 = t7728+t7737;
                double t7740 = EcF_2*EcF_2;
                double t7741 = EcF_2*EcF_3;
                double t7742 = EcF_4+t7740+t7741;
                double t7743 = 1.0/t7742;
                double t7744 = t7645*t7628*t7647*t7648;
                double t7745 = t7628*t7658*t7659;
                double t7746 = sqrt(t7701);
                double t7747 = 1.0/t7694;
                double t7748 = t7746*t7747;
                double t7749 = atan(t7748);
                double t7750 = 1.0/sqrt(t7701);
                double t7751 = sqrt(t7671);
                double t7752 = 1.0/t7637;
                double t7753 = t7751*t7752;
                double t7754 = atan(t7753);
                double t7755 = 1.0/sqrt(t7671);
                double t7756 = pow(t7734,1.0/3.0);
                double t7757 = pow(t7735,1.0/3.0);
                double t7758 = t7736*2.0;
                double t7759 = 1.0/(t7622*t7622*t7622);
                double t7760 = t7729*t7759*2.0;
                double t7761 = t7760+t7758;
                double t7762 = t7738*t7738;
                double t7763 = t7738*t7756*(4.0/3.0);
                double t7764 = t7763-t7738*t7757*(4.0/3.0);
                double t7765 = c*t7623*t7682;
                double t7766 = log(t7765);
                double t7767 = EcF_3*t7750*t7749*2.0;
                double t7768 = t7682*t7689;
                double t7769 = log(t7768);
                double t7770 = t7697*t7750*t7749;
                double t7771 = t7770+t7769;
                double t7772 = t7766+t7767-EcF_2*EcF_3*t7743*t7771;
                double t7773 = EcF_1*t7772;
                double t7774 = c*t7623*t7635;
                double t7775 = log(t7774);
                double t7776 = EcP_3*t7754*t7755*2.0;
                double t7777 = t7635*t7655;
                double t7778 = log(t7777);
                double t7779 = t7670*t7754*t7755;
                double t7780 = t7778+t7779;
                double t7781 = t7775+t7776-EcP_2*EcP_3*t7654*t7780;
                double t7782 = t7773-EcP_1*t7781;
                double t7783 = t7645*t7648*t7685*t7679;
                double t7784 = t7679*t7688*t7707;
                double t7785 = c*t7630*t7625*t7697*t7698*t7704*(1.0/3.0);
                double t7786 = t7784+t7785;
                double t7787 = EcF_2*EcF_3*t7743*t7786;
                double t7788 = t7783+t7787-EcF_3*c*t7630*t7625*t7698*t7704*(2.0/3.0);
                double t7789 = EcF_1*t7788;
                double t7790 = c*t7630*t7625*t7670*t7638*t7675*(1.0/3.0);
                double t7791 = t7745+t7790;
                double t7792 = EcP_2*EcP_3*t7654*t7791;
                double t7793 = t7744+t7792-EcP_3*c*t7630*t7625*t7638*t7675*(2.0/3.0);
                double t7794 = t7789-EcP_1*t7793;
                double t7795 = pow(t7734,4.0/3.0);
                double t7796 = pow(t7735,4.0/3.0);
                double t7797 = t7795+t7796-2.0;
                v_rho_b_rho_b[Q] += scale * (EcP_1*(t7744+EcP_2*EcP_3*t7654*(t7745-c*t7630*t7625*t7670*t7644*t7638*(1.0/3.0))+EcP_3*c*t7630*t7625*t7644*t7638*(2.0/3.0))*-2.0+t7622*(EcP_1*(t7711+t7716+t7726+t7727-t7645*t7628*t7647*t7686*(1.0/3.0)+EcP_2*EcP_3*t7654*(t7724+t7725+t7717-t7628*t7659*t7722+c*t7650*t7670*t7644*t7674*(2.0/9.0)-c*t7630*t7650*t7670*t7644*t7638*(4.0/9.0)-t7670*t7662*t7663*t7664*t7638*t7675*(1.0/1.8E1))-EcP_3*c*t7630*t7650*t7638*t7675*(8.0/9.0)-EcP_3*c*t7650*t7671*t7676*t7677*(4.0/9.0))+t7733*t7797*(EcF_1*(t7645*t7648*t7679*(c*t7650*t7682*(4.0/9.0)-c*t7623*t7691*t7684+c*t7623*t7692*t7693*2.0-c*t7625*t7681*t7684*(2.0/3.0))+t7645*t7681*t7648*t7685-t7645*t7685*t7686*t7679*(1.0/3.0)+EcF_2*EcF_3*t7743*(-t7679*t7688*(c*t7650*t7682*(1.0/1.8E1)-t7691*t7684*t7689+t7692*t7693*t7689*2.0-c*t7630*t7650*t7682*t7687*(4.0/9.0)+t7662*t7663*t7664*t7682*t7687*(1.0/1.8E1)+c*t7630*t7625*t7681*t7684*t7687*(2.0/3.0))+t7681*t7688*t7707-c*t7650*t7697*t7704*t7708*(2.0/9.0)+c*t7630*t7650*t7697*t7698*t7704*(4.0/9.0)+c*t7650*t7697*t7701*t7710*t7709*(2.0/9.0)-t7662*t7663*t7664*t7697*t7698*t7704*(1.0/1.8E1)+c*t7630*t7625*1.0/(t7687*t7687*t7687)*t7679*t7707*(1.0/3.0))+EcF_3*c*t7650*t7704*t7708*(4.0/9.0)-EcF_3*c*t7630*t7650*t7698*t7704*(8.0/9.0)-EcF_3*c*t7650*t7701*t7710*t7709*(4.0/9.0)+EcF_3*t7662*t7663*t7664*t7698*t7704*(1.0/9.0))-EcP_1*(t7711+t7716+t7726+t7727-t7645*t7628*t7647*t7686*(1.0/3.0)+EcP_2*EcP_3*t7654*(t7724+t7725+t7717-t7628*t7659*t7722-c*t7650*t7670*t7674*t7675*(2.0/9.0)+c*t7630*t7650*t7670*t7638*t7675*(4.0/9.0)-t7670*t7662*t7663*t7664*t7638*t7675*(1.0/1.8E1))-EcP_3*c*t7630*t7650*t7638*t7675*(8.0/9.0)-EcP_3*c*t7650*t7671*t7676*t7677*(4.0/9.0)))+t7733*t7782*(t7761*t7756*(4.0/3.0)-t7761*t7757*(4.0/3.0)+1.0/pow(t7734,2.0/3.0)*t7762*(4.0/9.0)+1.0/pow(t7735,2.0/3.0)*t7762*(4.0/9.0))+t7733*t7764*t7794*2.0)-t7733*t7764*t7782*2.0-t7733*t7794*t7797*2.0);
            }
            
        }
    }
}

}
