from __future__ import division
import unittest
import sys
import os

# Add the directory containing 'grendel' to the sys.path
sys.path.append(os.path.join(os.path.dirname(__file__), os.pardir, os.pardir, os.pardir))

# Add the directory containing the 'grendel_tests' package to sys.path
sys.path.append(os.path.join(os.path.dirname(__file__), os.pardir, os.pardir))

from numpy.testing import assert_array_almost_equal

from grendel_tests import long_test
from grendel import *
from grendel.gmath.einsum import EinsumContraction

import numpy as np

# Quick little helper function...
def range_tensor(shape, start=1, dtype=float, **kwargs):
    tot = 1
    for i in shape: tot *= i
    data = []
    for i in xrange(start, start + tot):
        data.append(dtype(i))
    return Tensor(data, dtype=dtype, **kwargs).reshape(shape)

# another helper function...
def index_range_tensor(idxs, start=1, dtype=float, **kwargs):
    in_set = kwargs.pop('in_set')
    tmp = Tensor(indices=idxs, in_set=in_set)
    tot = 1
    for i in tmp.shape: tot*=i
    data = []
    for i in xrange(start, start + tot):
        data.append(dtype(i))
    return Tensor(data, indices=idxs, in_set=in_set, **kwargs)

class EinsumTest(unittest.TestCase):

    def setUp(self):
        IndexRange.reset_ranges()

    def tearDown(self):
        IndexRange.reset_ranges()

    #--------------------------------------------------------------------------------#
    # Contractions

    def test_contract_1(self):
        # First example from numpy documentation
        a = range_tensor((3,4,5), 0)
        b = range_tensor((4,3,2), 0)
        res = Tensor(shape=(5,2))
        res['k,l'] = a['i,j,k'] * b['j,i,l']
        self.assertSequenceEqual(
            res,
            Tensor(
                [[ 4400.,  4730.],
                 [ 4532.,  4874.],
                 [ 4664.,  5018.],
                 [ 4796.,  5162.],
                 [ 4928.,  5306.]]
            )
        )

    def test_contract_2(self):
        # Now some tests of my own; checked with Mathematica
        a = range_tensor([2]*8)
        b = range_tensor([2]*7)
        res = Tensor(shape=(2,))
        res['i'] = a['i,j,k,l,m,n,o,p'] * b['j,k,l,m,n,o,p']
        self.assertSequenceEqual(res, Tensor(707264, 1764032))
        res['i'] = a['i,j,k,l,m,n,o,p'] * b['j,l,k,m,n,o,p']
        self.assertSequenceEqual(res, Tensor(699072, 1755840))
        res['i'] = a['i,j,k,l,m,n,o,p'] * b['j,p,l,m,k,n,o']
        self.assertSequenceEqual(res, Tensor(679264, 1736032))
        res['i'] = a['p,j,l,k,n,m,i,o'] * b['j,p,l,m,k,n,o']
        self.assertSequenceEqual(res, Tensor(1334784, 1351296))

    def test_contract_3(self):
        # Generated by mathematica
        a = range_tensor([3, 3, 3, 3, 3, 3, 3, 3])
        b = range_tensor([3, 3, 3, 3, 3, 3, 3])
        res = Tensor(shape=(3,))
        res['i'] = a['p,j,l,k,n,m,i,o'] * b['j,p,l,m,k,n,o']
        self.assertSequenceEqual(res, Tensor(9422787186, 9429964920, 9437142654))

    def test_contract_4(self):
        # Clearly generated by Mathematica...
        a = range_tensor([2, 3, 4, 5])
        b = range_tensor([5, 7, 3, 6])
        res = Tensor(shape=[2, 4, 7, 6])
        res['i,k,m,n'] = a['i,j,k,l'] * b['l,m,j,n']
        self.assertSequenceEqual(res, Tensor([[[[94335, 94680, 95025, 95370, 95715, 96060], [100545, 100890, 101235, 101580, 101925, 102270], [106755, 107100, 107445, 107790, 108135, 108480], [112965, 113310, 113655, 114000, 114345, 114690], [119175, 119520, 119865, 120210, 120555, 120900], [125385, 125730, 126075, 126420, 126765, 127110], [131595, 131940, 132285, 132630, 132975, 133320]], [[113760, 114180, 114600, 115020, 115440, 115860], [121320, 121740, 122160, 122580, 123000, 123420], [128880, 129300, 129720, 130140, 130560, 130980], [136440, 136860, 137280, 137700, 138120, 138540], [144000, 144420, 144840, 145260, 145680, 146100], [151560, 151980, 152400, 152820, 153240, 153660], [159120, 159540, 159960, 160380, 160800, 161220]], [[133185, 133680, 134175, 134670, 135165, 135660], [142095, 142590, 143085, 143580, 144075, 144570], [151005, 151500, 151995, 152490, 152985, 153480], [159915, 160410, 160905, 161400, 161895, 162390], [168825, 169320, 169815, 170310, 170805, 171300], [177735, 178230, 178725, 179220, 179715, 180210], [186645, 187140, 187635, 188130, 188625, 189120]], [[152610, 153180, 153750, 154320, 154890, 155460], [162870, 163440, 164010, 164580, 165150, 165720], [173130, 173700, 174270, 174840, 175410, 175980], [183390, 183960, 184530, 185100, 185670, 186240], [193650, 194220, 194790, 195360, 195930, 196500], [203910, 204480, 205050, 205620, 206190, 206760], [214170, 214740, 215310, 215880, 216450, 217020]]], [[[327435, 328680, 329925, 331170, 332415, 333660], [349845, 351090, 352335, 353580, 354825, 356070], [372255, 373500, 374745, 375990, 377235, 378480], [394665, 395910, 397155, 398400, 399645, 400890], [417075, 418320, 419565, 420810, 422055, 423300], [439485, 440730, 441975, 443220, 444465, 445710], [461895, 463140, 464385, 465630, 466875, 468120]], [[346860, 348180, 349500, 350820, 352140, 353460], [370620, 371940, 373260, 374580, 375900, 377220], [394380, 395700, 397020, 398340, 399660, 400980], [418140, 419460, 420780, 422100, 423420, 424740], [441900, 443220, 444540, 445860, 447180, 448500], [465660, 466980, 468300, 469620, 470940, 472260], [489420, 490740, 492060, 493380, 494700, 496020]], [[366285, 367680, 369075, 370470, 371865, 373260], [391395, 392790, 394185, 395580, 396975, 398370], [416505, 417900, 419295, 420690, 422085, 423480], [441615, 443010, 444405, 445800, 447195, 448590], [466725, 468120, 469515, 470910, 472305, 473700], [491835, 493230, 494625, 496020, 497415, 498810], [516945, 518340, 519735, 521130, 522525, 523920]], [[385710, 387180, 388650, 390120, 391590, 393060], [412170, 413640, 415110, 416580, 418050, 419520], [438630, 440100, 441570, 443040, 444510, 445980], [465090, 466560, 468030, 469500, 470970, 472440], [491550, 493020, 494490, 495960, 497430, 498900], [518010, 519480, 520950, 522420, 523890, 525360], [544470, 545940, 547410, 548880, 550350, 551820]]]]))

    def test_contract_5(self):
        # Also clearly generated by Mathematica, an outer product...
        a = range_tensor([2, 3, 4, 5])
        b = range_tensor([5, 7, 3, 6])
        res = Tensor(shape=[2, 3, 4, 7, 3, 6])
        res['i,j,k,m,n,o'] = a['i,j,k,l'] * b['l,m,n,o']
        self.assertSequenceEqual(res, Tensor([[[[[[5055, 5070, 5085, 5100, 5115, 5130], [5145, 5160, 5175, 5190, 5205, 5220], [5235, 5250, 5265, 5280, 5295, 5310]], [[5325, 5340, 5355, 5370, 5385, 5400], [5415, 5430, 5445, 5460, 5475, 5490], [5505, 5520, 5535, 5550, 5565, 5580]], [[5595, 5610, 5625, 5640, 5655, 5670], [5685, 5700, 5715, 5730, 5745, 5760], [5775, 5790, 5805, 5820, 5835, 5850]], [[5865, 5880, 5895, 5910, 5925, 5940], [5955, 5970, 5985, 6000, 6015, 6030], [6045, 6060, 6075, 6090, 6105, 6120]], [[6135, 6150, 6165, 6180, 6195, 6210], [6225, 6240, 6255, 6270, 6285, 6300], [6315, 6330, 6345, 6360, 6375, 6390]], [[6405, 6420, 6435, 6450, 6465, 6480], [6495, 6510, 6525, 6540, 6555, 6570], [6585, 6600, 6615, 6630, 6645, 6660]], [[6675, 6690, 6705, 6720, 6735, 6750], [6765, 6780, 6795, 6810, 6825, 6840], [6855, 6870, 6885, 6900, 6915, 6930]]], [[[11380, 11420, 11460, 11500, 11540, 11580], [11620, 11660, 11700, 11740, 11780, 11820], [11860, 11900, 11940, 11980, 12020, 12060]], [[12100, 12140, 12180, 12220, 12260, 12300], [12340, 12380, 12420, 12460, 12500, 12540], [12580, 12620, 12660, 12700, 12740, 12780]], [[12820, 12860, 12900, 12940, 12980, 13020], [13060, 13100, 13140, 13180, 13220, 13260], [13300, 13340, 13380, 13420, 13460, 13500]], [[13540, 13580, 13620, 13660, 13700, 13740], [13780, 13820, 13860, 13900, 13940, 13980], [14020, 14060, 14100, 14140, 14180, 14220]], [[14260, 14300, 14340, 14380, 14420, 14460], [14500, 14540, 14580, 14620, 14660, 14700], [14740, 14780, 14820, 14860, 14900, 14940]], [[14980, 15020, 15060, 15100, 15140, 15180], [15220, 15260, 15300, 15340, 15380, 15420], [15460, 15500, 15540, 15580, 15620, 15660]], [[15700, 15740, 15780, 15820, 15860, 15900], [15940, 15980, 16020, 16060, 16100, 16140], [16180, 16220, 16260, 16300, 16340, 16380]]], [[[17705, 17770, 17835, 17900, 17965, 18030], [18095, 18160, 18225, 18290, 18355, 18420], [18485, 18550, 18615, 18680, 18745, 18810]], [[18875, 18940, 19005, 19070, 19135, 19200], [19265, 19330, 19395, 19460, 19525, 19590], [19655, 19720, 19785, 19850, 19915, 19980]], [[20045, 20110, 20175, 20240, 20305, 20370], [20435, 20500, 20565, 20630, 20695, 20760], [20825, 20890, 20955, 21020, 21085, 21150]], [[21215, 21280, 21345, 21410, 21475, 21540], [21605, 21670, 21735, 21800, 21865, 21930], [21995, 22060, 22125, 22190, 22255, 22320]], [[22385, 22450, 22515, 22580, 22645, 22710], [22775, 22840, 22905, 22970, 23035, 23100], [23165, 23230, 23295, 23360, 23425, 23490]], [[23555, 23620, 23685, 23750, 23815, 23880], [23945, 24010, 24075, 24140, 24205, 24270], [24335, 24400, 24465, 24530, 24595, 24660]], [[24725, 24790, 24855, 24920, 24985, 25050], [25115, 25180, 25245, 25310, 25375, 25440], [25505, 25570, 25635, 25700, 25765, 25830]]], [[[24030, 24120, 24210, 24300, 24390, 24480], [24570, 24660, 24750, 24840, 24930, 25020], [25110, 25200, 25290, 25380, 25470, 25560]], [[25650, 25740, 25830, 25920, 26010, 26100], [26190, 26280, 26370, 26460, 26550, 26640], [26730, 26820, 26910, 27000, 27090, 27180]], [[27270, 27360, 27450, 27540, 27630, 27720], [27810, 27900, 27990, 28080, 28170, 28260], [28350, 28440, 28530, 28620, 28710, 28800]], [[28890, 28980, 29070, 29160, 29250, 29340], [29430, 29520, 29610, 29700, 29790, 29880], [29970, 30060, 30150, 30240, 30330, 30420]], [[30510, 30600, 30690, 30780, 30870, 30960], [31050, 31140, 31230, 31320, 31410, 31500], [31590, 31680, 31770, 31860, 31950, 32040]], [[32130, 32220, 32310, 32400, 32490, 32580], [32670, 32760, 32850, 32940, 33030, 33120], [33210, 33300, 33390, 33480, 33570, 33660]], [[33750, 33840, 33930, 34020, 34110, 34200], [34290, 34380, 34470, 34560, 34650, 34740], [34830, 34920, 35010, 35100, 35190, 35280]]]], [[[[30355, 30470, 30585, 30700, 30815, 30930], [31045, 31160, 31275, 31390, 31505, 31620], [31735, 31850, 31965, 32080, 32195, 32310]], [[32425, 32540, 32655, 32770, 32885, 33000], [33115, 33230, 33345, 33460, 33575, 33690], [33805, 33920, 34035, 34150, 34265, 34380]], [[34495, 34610, 34725, 34840, 34955, 35070], [35185, 35300, 35415, 35530, 35645, 35760], [35875, 35990, 36105, 36220, 36335, 36450]], [[36565, 36680, 36795, 36910, 37025, 37140], [37255, 37370, 37485, 37600, 37715, 37830], [37945, 38060, 38175, 38290, 38405, 38520]], [[38635, 38750, 38865, 38980, 39095, 39210], [39325, 39440, 39555, 39670, 39785, 39900], [40015, 40130, 40245, 40360, 40475, 40590]], [[40705, 40820, 40935, 41050, 41165, 41280], [41395, 41510, 41625, 41740, 41855, 41970], [42085, 42200, 42315, 42430, 42545, 42660]], [[42775, 42890, 43005, 43120, 43235, 43350], [43465, 43580, 43695, 43810, 43925, 44040], [44155, 44270, 44385, 44500, 44615, 44730]]], [[[36680, 36820, 36960, 37100, 37240, 37380], [37520, 37660, 37800, 37940, 38080, 38220], [38360, 38500, 38640, 38780, 38920, 39060]], [[39200, 39340, 39480, 39620, 39760, 39900], [40040, 40180, 40320, 40460, 40600, 40740], [40880, 41020, 41160, 41300, 41440, 41580]], [[41720, 41860, 42000, 42140, 42280, 42420], [42560, 42700, 42840, 42980, 43120, 43260], [43400, 43540, 43680, 43820, 43960, 44100]], [[44240, 44380, 44520, 44660, 44800, 44940], [45080, 45220, 45360, 45500, 45640, 45780], [45920, 46060, 46200, 46340, 46480, 46620]], [[46760, 46900, 47040, 47180, 47320, 47460], [47600, 47740, 47880, 48020, 48160, 48300], [48440, 48580, 48720, 48860, 49000, 49140]], [[49280, 49420, 49560, 49700, 49840, 49980], [50120, 50260, 50400, 50540, 50680, 50820], [50960, 51100, 51240, 51380, 51520, 51660]], [[51800, 51940, 52080, 52220, 52360, 52500], [52640, 52780, 52920, 53060, 53200, 53340], [53480, 53620, 53760, 53900, 54040, 54180]]], [[[43005, 43170, 43335, 43500, 43665, 43830], [43995, 44160, 44325, 44490, 44655, 44820], [44985, 45150, 45315, 45480, 45645, 45810]], [[45975, 46140, 46305, 46470, 46635, 46800], [46965, 47130, 47295, 47460, 47625, 47790], [47955, 48120, 48285, 48450, 48615, 48780]], [[48945, 49110, 49275, 49440, 49605, 49770], [49935, 50100, 50265, 50430, 50595, 50760], [50925, 51090, 51255, 51420, 51585, 51750]], [[51915, 52080, 52245, 52410, 52575, 52740], [52905, 53070, 53235, 53400, 53565, 53730], [53895, 54060, 54225, 54390, 54555, 54720]], [[54885, 55050, 55215, 55380, 55545, 55710], [55875, 56040, 56205, 56370, 56535, 56700], [56865, 57030, 57195, 57360, 57525, 57690]], [[57855, 58020, 58185, 58350, 58515, 58680], [58845, 59010, 59175, 59340, 59505, 59670], [59835, 60000, 60165, 60330, 60495, 60660]], [[60825, 60990, 61155, 61320, 61485, 61650], [61815, 61980, 62145, 62310, 62475, 62640], [62805, 62970, 63135, 63300, 63465, 63630]]], [[[49330, 49520, 49710, 49900, 50090, 50280], [50470, 50660, 50850, 51040, 51230, 51420], [51610, 51800, 51990, 52180, 52370, 52560]], [[52750, 52940, 53130, 53320, 53510, 53700], [53890, 54080, 54270, 54460, 54650, 54840], [55030, 55220, 55410, 55600, 55790, 55980]], [[56170, 56360, 56550, 56740, 56930, 57120], [57310, 57500, 57690, 57880, 58070, 58260], [58450, 58640, 58830, 59020, 59210, 59400]], [[59590, 59780, 59970, 60160, 60350, 60540], [60730, 60920, 61110, 61300, 61490, 61680], [61870, 62060, 62250, 62440, 62630, 62820]], [[63010, 63200, 63390, 63580, 63770, 63960], [64150, 64340, 64530, 64720, 64910, 65100], [65290, 65480, 65670, 65860, 66050, 66240]], [[66430, 66620, 66810, 67000, 67190, 67380], [67570, 67760, 67950, 68140, 68330, 68520], [68710, 68900, 69090, 69280, 69470, 69660]], [[69850, 70040, 70230, 70420, 70610, 70800], [70990, 71180, 71370, 71560, 71750, 71940], [72130, 72320, 72510, 72700, 72890, 73080]]]], [[[[55655, 55870, 56085, 56300, 56515, 56730], [56945, 57160, 57375, 57590, 57805, 58020], [58235, 58450, 58665, 58880, 59095, 59310]], [[59525, 59740, 59955, 60170, 60385, 60600], [60815, 61030, 61245, 61460, 61675, 61890], [62105, 62320, 62535, 62750, 62965, 63180]], [[63395, 63610, 63825, 64040, 64255, 64470], [64685, 64900, 65115, 65330, 65545, 65760], [65975, 66190, 66405, 66620, 66835, 67050]], [[67265, 67480, 67695, 67910, 68125, 68340], [68555, 68770, 68985, 69200, 69415, 69630], [69845, 70060, 70275, 70490, 70705, 70920]], [[71135, 71350, 71565, 71780, 71995, 72210], [72425, 72640, 72855, 73070, 73285, 73500], [73715, 73930, 74145, 74360, 74575, 74790]], [[75005, 75220, 75435, 75650, 75865, 76080], [76295, 76510, 76725, 76940, 77155, 77370], [77585, 77800, 78015, 78230, 78445, 78660]], [[78875, 79090, 79305, 79520, 79735, 79950], [80165, 80380, 80595, 80810, 81025, 81240], [81455, 81670, 81885, 82100, 82315, 82530]]], [[[61980, 62220, 62460, 62700, 62940, 63180], [63420, 63660, 63900, 64140, 64380, 64620], [64860, 65100, 65340, 65580, 65820, 66060]], [[66300, 66540, 66780, 67020, 67260, 67500], [67740, 67980, 68220, 68460, 68700, 68940], [69180, 69420, 69660, 69900, 70140, 70380]], [[70620, 70860, 71100, 71340, 71580, 71820], [72060, 72300, 72540, 72780, 73020, 73260], [73500, 73740, 73980, 74220, 74460, 74700]], [[74940, 75180, 75420, 75660, 75900, 76140], [76380, 76620, 76860, 77100, 77340, 77580], [77820, 78060, 78300, 78540, 78780, 79020]], [[79260, 79500, 79740, 79980, 80220, 80460], [80700, 80940, 81180, 81420, 81660, 81900], [82140, 82380, 82620, 82860, 83100, 83340]], [[83580, 83820, 84060, 84300, 84540, 84780], [85020, 85260, 85500, 85740, 85980, 86220], [86460, 86700, 86940, 87180, 87420, 87660]], [[87900, 88140, 88380, 88620, 88860, 89100], [89340, 89580, 89820, 90060, 90300, 90540], [90780, 91020, 91260, 91500, 91740, 91980]]], [[[68305, 68570, 68835, 69100, 69365, 69630], [69895, 70160, 70425, 70690, 70955, 71220], [71485, 71750, 72015, 72280, 72545, 72810]], [[73075, 73340, 73605, 73870, 74135, 74400], [74665, 74930, 75195, 75460, 75725, 75990], [76255, 76520, 76785, 77050, 77315, 77580]], [[77845, 78110, 78375, 78640, 78905, 79170], [79435, 79700, 79965, 80230, 80495, 80760], [81025, 81290, 81555, 81820, 82085, 82350]], [[82615, 82880, 83145, 83410, 83675, 83940], [84205, 84470, 84735, 85000, 85265, 85530], [85795, 86060, 86325, 86590, 86855, 87120]], [[87385, 87650, 87915, 88180, 88445, 88710], [88975, 89240, 89505, 89770, 90035, 90300], [90565, 90830, 91095, 91360, 91625, 91890]], [[92155, 92420, 92685, 92950, 93215, 93480], [93745, 94010, 94275, 94540, 94805, 95070], [95335, 95600, 95865, 96130, 96395, 96660]], [[96925, 97190, 97455, 97720, 97985, 98250], [98515, 98780, 99045, 99310, 99575, 99840], [100105, 100370, 100635, 100900, 101165, 101430]]], [[[74630, 74920, 75210, 75500, 75790, 76080], [76370, 76660, 76950, 77240, 77530, 77820], [78110, 78400, 78690, 78980, 79270, 79560]], [[79850, 80140, 80430, 80720, 81010, 81300], [81590, 81880, 82170, 82460, 82750, 83040], [83330, 83620, 83910, 84200, 84490, 84780]], [[85070, 85360, 85650, 85940, 86230, 86520], [86810, 87100, 87390, 87680, 87970, 88260], [88550, 88840, 89130, 89420, 89710, 90000]], [[90290, 90580, 90870, 91160, 91450, 91740], [92030, 92320, 92610, 92900, 93190, 93480], [93770, 94060, 94350, 94640, 94930, 95220]], [[95510, 95800, 96090, 96380, 96670, 96960], [97250, 97540, 97830, 98120, 98410, 98700], [98990, 99280, 99570, 99860, 100150, 100440]], [[100730, 101020, 101310, 101600, 101890, 102180], [102470, 102760, 103050, 103340, 103630, 103920], [104210, 104500, 104790, 105080, 105370, 105660]], [[105950, 106240, 106530, 106820, 107110, 107400], [107690, 107980, 108270, 108560, 108850, 109140], [109430, 109720, 110010, 110300, 110590, 110880]]]]], [[[[[80955, 81270, 81585, 81900, 82215, 82530], [82845, 83160, 83475, 83790, 84105, 84420], [84735, 85050, 85365, 85680, 85995, 86310]], [[86625, 86940, 87255, 87570, 87885, 88200], [88515, 88830, 89145, 89460, 89775, 90090], [90405, 90720, 91035, 91350, 91665, 91980]], [[92295, 92610, 92925, 93240, 93555, 93870], [94185, 94500, 94815, 95130, 95445, 95760], [96075, 96390, 96705, 97020, 97335, 97650]], [[97965, 98280, 98595, 98910, 99225, 99540], [99855, 100170, 100485, 100800, 101115, 101430], [101745, 102060, 102375, 102690, 103005, 103320]], [[103635, 103950, 104265, 104580, 104895, 105210], [105525, 105840, 106155, 106470, 106785, 107100], [107415, 107730, 108045, 108360, 108675, 108990]], [[109305, 109620, 109935, 110250, 110565, 110880], [111195, 111510, 111825, 112140, 112455, 112770], [113085, 113400, 113715, 114030, 114345, 114660]], [[114975, 115290, 115605, 115920, 116235, 116550], [116865, 117180, 117495, 117810, 118125, 118440], [118755, 119070, 119385, 119700, 120015, 120330]]], [[[87280, 87620, 87960, 88300, 88640, 88980], [89320, 89660, 90000, 90340, 90680, 91020], [91360, 91700, 92040, 92380, 92720, 93060]], [[93400, 93740, 94080, 94420, 94760, 95100], [95440, 95780, 96120, 96460, 96800, 97140], [97480, 97820, 98160, 98500, 98840, 99180]], [[99520, 99860, 100200, 100540, 100880, 101220], [101560, 101900, 102240, 102580, 102920, 103260], [103600, 103940, 104280, 104620, 104960, 105300]], [[105640, 105980, 106320, 106660, 107000, 107340], [107680, 108020, 108360, 108700, 109040, 109380], [109720, 110060, 110400, 110740, 111080, 111420]], [[111760, 112100, 112440, 112780, 113120, 113460], [113800, 114140, 114480, 114820, 115160, 115500], [115840, 116180, 116520, 116860, 117200, 117540]], [[117880, 118220, 118560, 118900, 119240, 119580], [119920, 120260, 120600, 120940, 121280, 121620], [121960, 122300, 122640, 122980, 123320, 123660]], [[124000, 124340, 124680, 125020, 125360, 125700], [126040, 126380, 126720, 127060, 127400, 127740], [128080, 128420, 128760, 129100, 129440, 129780]]], [[[93605, 93970, 94335, 94700, 95065, 95430], [95795, 96160, 96525, 96890, 97255, 97620], [97985, 98350, 98715, 99080, 99445, 99810]], [[100175, 100540, 100905, 101270, 101635, 102000], [102365, 102730, 103095, 103460, 103825, 104190], [104555, 104920, 105285, 105650, 106015, 106380]], [[106745, 107110, 107475, 107840, 108205, 108570], [108935, 109300, 109665, 110030, 110395, 110760], [111125, 111490, 111855, 112220, 112585, 112950]], [[113315, 113680, 114045, 114410, 114775, 115140], [115505, 115870, 116235, 116600, 116965, 117330], [117695, 118060, 118425, 118790, 119155, 119520]], [[119885, 120250, 120615, 120980, 121345, 121710], [122075, 122440, 122805, 123170, 123535, 123900], [124265, 124630, 124995, 125360, 125725, 126090]], [[126455, 126820, 127185, 127550, 127915, 128280], [128645, 129010, 129375, 129740, 130105, 130470], [130835, 131200, 131565, 131930, 132295, 132660]], [[133025, 133390, 133755, 134120, 134485, 134850], [135215, 135580, 135945, 136310, 136675, 137040], [137405, 137770, 138135, 138500, 138865, 139230]]], [[[99930, 100320, 100710, 101100, 101490, 101880], [102270, 102660, 103050, 103440, 103830, 104220], [104610, 105000, 105390, 105780, 106170, 106560]], [[106950, 107340, 107730, 108120, 108510, 108900], [109290, 109680, 110070, 110460, 110850, 111240], [111630, 112020, 112410, 112800, 113190, 113580]], [[113970, 114360, 114750, 115140, 115530, 115920], [116310, 116700, 117090, 117480, 117870, 118260], [118650, 119040, 119430, 119820, 120210, 120600]], [[120990, 121380, 121770, 122160, 122550, 122940], [123330, 123720, 124110, 124500, 124890, 125280], [125670, 126060, 126450, 126840, 127230, 127620]], [[128010, 128400, 128790, 129180, 129570, 129960], [130350, 130740, 131130, 131520, 131910, 132300], [132690, 133080, 133470, 133860, 134250, 134640]], [[135030, 135420, 135810, 136200, 136590, 136980], [137370, 137760, 138150, 138540, 138930, 139320], [139710, 140100, 140490, 140880, 141270, 141660]], [[142050, 142440, 142830, 143220, 143610, 144000], [144390, 144780, 145170, 145560, 145950, 146340], [146730, 147120, 147510, 147900, 148290, 148680]]]], [[[[106255, 106670, 107085, 107500, 107915, 108330], [108745, 109160, 109575, 109990, 110405, 110820], [111235, 111650, 112065, 112480, 112895, 113310]], [[113725, 114140, 114555, 114970, 115385, 115800], [116215, 116630, 117045, 117460, 117875, 118290], [118705, 119120, 119535, 119950, 120365, 120780]], [[121195, 121610, 122025, 122440, 122855, 123270], [123685, 124100, 124515, 124930, 125345, 125760], [126175, 126590, 127005, 127420, 127835, 128250]], [[128665, 129080, 129495, 129910, 130325, 130740], [131155, 131570, 131985, 132400, 132815, 133230], [133645, 134060, 134475, 134890, 135305, 135720]], [[136135, 136550, 136965, 137380, 137795, 138210], [138625, 139040, 139455, 139870, 140285, 140700], [141115, 141530, 141945, 142360, 142775, 143190]], [[143605, 144020, 144435, 144850, 145265, 145680], [146095, 146510, 146925, 147340, 147755, 148170], [148585, 149000, 149415, 149830, 150245, 150660]], [[151075, 151490, 151905, 152320, 152735, 153150], [153565, 153980, 154395, 154810, 155225, 155640], [156055, 156470, 156885, 157300, 157715, 158130]]], [[[112580, 113020, 113460, 113900, 114340, 114780], [115220, 115660, 116100, 116540, 116980, 117420], [117860, 118300, 118740, 119180, 119620, 120060]], [[120500, 120940, 121380, 121820, 122260, 122700], [123140, 123580, 124020, 124460, 124900, 125340], [125780, 126220, 126660, 127100, 127540, 127980]], [[128420, 128860, 129300, 129740, 130180, 130620], [131060, 131500, 131940, 132380, 132820, 133260], [133700, 134140, 134580, 135020, 135460, 135900]], [[136340, 136780, 137220, 137660, 138100, 138540], [138980, 139420, 139860, 140300, 140740, 141180], [141620, 142060, 142500, 142940, 143380, 143820]], [[144260, 144700, 145140, 145580, 146020, 146460], [146900, 147340, 147780, 148220, 148660, 149100], [149540, 149980, 150420, 150860, 151300, 151740]], [[152180, 152620, 153060, 153500, 153940, 154380], [154820, 155260, 155700, 156140, 156580, 157020], [157460, 157900, 158340, 158780, 159220, 159660]], [[160100, 160540, 160980, 161420, 161860, 162300], [162740, 163180, 163620, 164060, 164500, 164940], [165380, 165820, 166260, 166700, 167140, 167580]]], [[[118905, 119370, 119835, 120300, 120765, 121230], [121695, 122160, 122625, 123090, 123555, 124020], [124485, 124950, 125415, 125880, 126345, 126810]], [[127275, 127740, 128205, 128670, 129135, 129600], [130065, 130530, 130995, 131460, 131925, 132390], [132855, 133320, 133785, 134250, 134715, 135180]], [[135645, 136110, 136575, 137040, 137505, 137970], [138435, 138900, 139365, 139830, 140295, 140760], [141225, 141690, 142155, 142620, 143085, 143550]], [[144015, 144480, 144945, 145410, 145875, 146340], [146805, 147270, 147735, 148200, 148665, 149130], [149595, 150060, 150525, 150990, 151455, 151920]], [[152385, 152850, 153315, 153780, 154245, 154710], [155175, 155640, 156105, 156570, 157035, 157500], [157965, 158430, 158895, 159360, 159825, 160290]], [[160755, 161220, 161685, 162150, 162615, 163080], [163545, 164010, 164475, 164940, 165405, 165870], [166335, 166800, 167265, 167730, 168195, 168660]], [[169125, 169590, 170055, 170520, 170985, 171450], [171915, 172380, 172845, 173310, 173775, 174240], [174705, 175170, 175635, 176100, 176565, 177030]]], [[[125230, 125720, 126210, 126700, 127190, 127680], [128170, 128660, 129150, 129640, 130130, 130620], [131110, 131600, 132090, 132580, 133070, 133560]], [[134050, 134540, 135030, 135520, 136010, 136500], [136990, 137480, 137970, 138460, 138950, 139440], [139930, 140420, 140910, 141400, 141890, 142380]], [[142870, 143360, 143850, 144340, 144830, 145320], [145810, 146300, 146790, 147280, 147770, 148260], [148750, 149240, 149730, 150220, 150710, 151200]], [[151690, 152180, 152670, 153160, 153650, 154140], [154630, 155120, 155610, 156100, 156590, 157080], [157570, 158060, 158550, 159040, 159530, 160020]], [[160510, 161000, 161490, 161980, 162470, 162960], [163450, 163940, 164430, 164920, 165410, 165900], [166390, 166880, 167370, 167860, 168350, 168840]], [[169330, 169820, 170310, 170800, 171290, 171780], [172270, 172760, 173250, 173740, 174230, 174720], [175210, 175700, 176190, 176680, 177170, 177660]], [[178150, 178640, 179130, 179620, 180110, 180600], [181090, 181580, 182070, 182560, 183050, 183540], [184030, 184520, 185010, 185500, 185990, 186480]]]], [[[[131555, 132070, 132585, 133100, 133615, 134130], [134645, 135160, 135675, 136190, 136705, 137220], [137735, 138250, 138765, 139280, 139795, 140310]], [[140825, 141340, 141855, 142370, 142885, 143400], [143915, 144430, 144945, 145460, 145975, 146490], [147005, 147520, 148035, 148550, 149065, 149580]], [[150095, 150610, 151125, 151640, 152155, 152670], [153185, 153700, 154215, 154730, 155245, 155760], [156275, 156790, 157305, 157820, 158335, 158850]], [[159365, 159880, 160395, 160910, 161425, 161940], [162455, 162970, 163485, 164000, 164515, 165030], [165545, 166060, 166575, 167090, 167605, 168120]], [[168635, 169150, 169665, 170180, 170695, 171210], [171725, 172240, 172755, 173270, 173785, 174300], [174815, 175330, 175845, 176360, 176875, 177390]], [[177905, 178420, 178935, 179450, 179965, 180480], [180995, 181510, 182025, 182540, 183055, 183570], [184085, 184600, 185115, 185630, 186145, 186660]], [[187175, 187690, 188205, 188720, 189235, 189750], [190265, 190780, 191295, 191810, 192325, 192840], [193355, 193870, 194385, 194900, 195415, 195930]]], [[[137880, 138420, 138960, 139500, 140040, 140580], [141120, 141660, 142200, 142740, 143280, 143820], [144360, 144900, 145440, 145980, 146520, 147060]], [[147600, 148140, 148680, 149220, 149760, 150300], [150840, 151380, 151920, 152460, 153000, 153540], [154080, 154620, 155160, 155700, 156240, 156780]], [[157320, 157860, 158400, 158940, 159480, 160020], [160560, 161100, 161640, 162180, 162720, 163260], [163800, 164340, 164880, 165420, 165960, 166500]], [[167040, 167580, 168120, 168660, 169200, 169740], [170280, 170820, 171360, 171900, 172440, 172980], [173520, 174060, 174600, 175140, 175680, 176220]], [[176760, 177300, 177840, 178380, 178920, 179460], [180000, 180540, 181080, 181620, 182160, 182700], [183240, 183780, 184320, 184860, 185400, 185940]], [[186480, 187020, 187560, 188100, 188640, 189180], [189720, 190260, 190800, 191340, 191880, 192420], [192960, 193500, 194040, 194580, 195120, 195660]], [[196200, 196740, 197280, 197820, 198360, 198900], [199440, 199980, 200520, 201060, 201600, 202140], [202680, 203220, 203760, 204300, 204840, 205380]]], [[[144205, 144770, 145335, 145900, 146465, 147030], [147595, 148160, 148725, 149290, 149855, 150420], [150985, 151550, 152115, 152680, 153245, 153810]], [[154375, 154940, 155505, 156070, 156635, 157200], [157765, 158330, 158895, 159460, 160025, 160590], [161155, 161720, 162285, 162850, 163415, 163980]], [[164545, 165110, 165675, 166240, 166805, 167370], [167935, 168500, 169065, 169630, 170195, 170760], [171325, 171890, 172455, 173020, 173585, 174150]], [[174715, 175280, 175845, 176410, 176975, 177540], [178105, 178670, 179235, 179800, 180365, 180930], [181495, 182060, 182625, 183190, 183755, 184320]], [[184885, 185450, 186015, 186580, 187145, 187710], [188275, 188840, 189405, 189970, 190535, 191100], [191665, 192230, 192795, 193360, 193925, 194490]], [[195055, 195620, 196185, 196750, 197315, 197880], [198445, 199010, 199575, 200140, 200705, 201270], [201835, 202400, 202965, 203530, 204095, 204660]], [[205225, 205790, 206355, 206920, 207485, 208050], [208615, 209180, 209745, 210310, 210875, 211440], [212005, 212570, 213135, 213700, 214265, 214830]]], [[[150530, 151120, 151710, 152300, 152890, 153480], [154070, 154660, 155250, 155840, 156430, 157020], [157610, 158200, 158790, 159380, 159970, 160560]], [[161150, 161740, 162330, 162920, 163510, 164100], [164690, 165280, 165870, 166460, 167050, 167640], [168230, 168820, 169410, 170000, 170590, 171180]], [[171770, 172360, 172950, 173540, 174130, 174720], [175310, 175900, 176490, 177080, 177670, 178260], [178850, 179440, 180030, 180620, 181210, 181800]], [[182390, 182980, 183570, 184160, 184750, 185340], [185930, 186520, 187110, 187700, 188290, 188880], [189470, 190060, 190650, 191240, 191830, 192420]], [[193010, 193600, 194190, 194780, 195370, 195960], [196550, 197140, 197730, 198320, 198910, 199500], [200090, 200680, 201270, 201860, 202450, 203040]], [[203630, 204220, 204810, 205400, 205990, 206580], [207170, 207760, 208350, 208940, 209530, 210120], [210710, 211300, 211890, 212480, 213070, 213660]], [[214250, 214840, 215430, 216020, 216610, 217200], [217790, 218380, 218970, 219560, 220150, 220740], [221330, 221920, 222510, 223100, 223690, 224280]]]]]]))

    @long_test
    def test_big_contractions_1(self):
        # A huge n^10 contraction with n=4
        a = range_tensor([4, 4, 4, 4, 4, 4, 4, 4])
        b = range_tensor([4, 4, 4, 4, 4, 4, 4, 4])
        res = Tensor(shape=[4, 4, 4, 4])
        res['o,p,q,s'] = a['i,j,k,l,m,n,o,p'] * b['j,i,l,k,n,m,q,s']
        self.assertSequenceEqual(res, Tensor([[[[5086057271296, 5086191460352, 5086325649408, 5086459838464], [5086594027520, 5086728216576, 5086862405632, 5086996594688], [5087130783744, 5087264972800, 5087399161856, 5087533350912], [5087667539968, 5087801729024, 5087935918080, 5088070107136]], [[5086191460352, 5086325653504, 5086459846656, 5086594039808], [5086728232960, 5086862426112, 5086996619264, 5087130812416], [5087265005568, 5087399198720, 5087533391872, 5087667585024], [5087801778176, 5087935971328, 5088070164480, 5088204357632]], [[5086325649408, 5086459846656, 5086594043904, 5086728241152], [5086862438400, 5086996635648, 5087130832896, 5087265030144], [5087399227392, 5087533424640, 5087667621888, 5087801819136], [5087936016384, 5088070213632, 5088204410880, 5088338608128]], [[5086459838464, 5086594039808, 5086728241152, 5086862442496], [5086996643840, 5087130845184, 5087265046528, 5087399247872], [5087533449216, 5087667650560, 5087801851904, 5087936053248], [5088070254592, 5088204455936, 5088338657280, 5088472858624]]], [[[5086594027520, 5086728232960, 5086862438400, 5086996643840], [5087130849280, 5087265054720, 5087399260160, 5087533465600], [5087667671040, 5087801876480, 5087936081920, 5088070287360], [5088204492800, 5088338698240, 5088472903680, 5088607109120]], [[5086728216576, 5086862426112, 5086996635648, 5087130845184], [5087265054720, 5087399264256, 5087533473792, 5087667683328], [5087801892864, 5087936102400, 5088070311936, 5088204521472], [5088338731008, 5088472940544, 5088607150080, 5088741359616]], [[5086862405632, 5086996619264, 5087130832896, 5087265046528], [5087399260160, 5087533473792, 5087667687424, 5087801901056], [5087936114688, 5088070328320, 5088204541952, 5088338755584], [5088472969216, 5088607182848, 5088741396480, 5088875610112]], [[5086996594688, 5087130812416, 5087265030144, 5087399247872], [5087533465600, 5087667683328, 5087801901056, 5087936118784], [5088070336512, 5088204554240, 5088338771968, 5088472989696], [5088607207424, 5088741425152, 5088875642880, 5089009860608]]], [[[5087130783744, 5087265005568, 5087399227392, 5087533449216], [5087667671040, 5087801892864, 5087936114688, 5088070336512], [5088204558336, 5088338780160, 5088473001984, 5088607223808], [5088741445632, 5088875667456, 5089009889280, 5089144111104]], [[5087264972800, 5087399198720, 5087533424640, 5087667650560], [5087801876480, 5087936102400, 5088070328320, 5088204554240], [5088338780160, 5088473006080, 5088607232000, 5088741457920], [5088875683840, 5089009909760, 5089144135680, 5089278361600]], [[5087399161856, 5087533391872, 5087667621888, 5087801851904], [5087936081920, 5088070311936, 5088204541952, 5088338771968], [5088473001984, 5088607232000, 5088741462016, 5088875692032], [5089009922048, 5089144152064, 5089278382080, 5089412612096]], [[5087533350912, 5087667585024, 5087801819136, 5087936053248], [5088070287360, 5088204521472, 5088338755584, 5088472989696], [5088607223808, 5088741457920, 5088875692032, 5089009926144], [5089144160256, 5089278394368, 5089412628480, 5089546862592]]], [[[5087667539968, 5087801778176, 5087936016384, 5088070254592], [5088204492800, 5088338731008, 5088472969216, 5088607207424], [5088741445632, 5088875683840, 5089009922048, 5089144160256], [5089278398464, 5089412636672, 5089546874880, 5089681113088]], [[5087801729024, 5087935971328, 5088070213632, 5088204455936], [5088338698240, 5088472940544, 5088607182848, 5088741425152], [5088875667456, 5089009909760, 5089144152064, 5089278394368], [5089412636672, 5089546878976, 5089681121280, 5089815363584]], [[5087935918080, 5088070164480, 5088204410880, 5088338657280], [5088472903680, 5088607150080, 5088741396480, 5088875642880], [5089009889280, 5089144135680, 5089278382080, 5089412628480], [5089546874880, 5089681121280, 5089815367680, 5089949614080]], [[5088070107136, 5088204357632, 5088338608128, 5088472858624], [5088607109120, 5088741359616, 5088875610112, 5089009860608], [5089144111104, 5089278361600, 5089412612096, 5089546862592], [5089681113088, 5089815363584, 5089949614080, 5090083864576]]]]))

    @long_test
    def test_big_contractions_2(self):
        # An n^8 contraction with n=7; need to use long ints
        res = Tensor(shape=[7], dtype=long)
        a = range_tensor([7, 7, 7, 7, 7, 7, 7, 7], dtype=long)
        b = range_tensor([7, 7, 7, 7, 7, 7, 7], dtype=long)
        res['p'] = a['i,j,k,l,m,n,o,p'] * b['i,j,k,l,m,n,o']
        self.assertSequenceEqual(res, Tensor([1303274021971022612l, 1303274361082970808l, 1303274700194919004l, 1303275039306867200l, 1303275378418815396l, 1303275717530763592l, 1303276056642711788l], dtype=long))


    def test_contract_imul(self):
        # First example from numpy documentation
        a = range_tensor((3,4,5), 0)
        b = range_tensor((4,3,2), 0)
        res = Tensor(shape=(5,2))
        res2 = Tensor(shape=(5,2))
        res['k,l'] = a['i,j,k'] * b['j,i,l'] + a['i,j,k'] * b['j,i,l']
        res2['k,l'] = a['i,j,k'] * b['j,i,l']
        res2['k,l'] *= 2
        self.assertSequenceEqual(res, res2)

    def test_contract_mul_contraction(self):
        # First example from numpy documentation
        a = range_tensor((3,4,5), 0)
        b = range_tensor((4,3,2), 0)
        res = Tensor(shape=(5,2))
        res2 = Tensor(shape=(5,2))
        res['k,l'] = a['i,j,k'] * b['j,i,l'] + a['i,j,k'] * b['j,i,l']
        res2['k,l'] = 2 * (a['i,j,k'] * b['j,i,l'])
        self.assertSequenceEqual(res, res2)

    def test_contract_rmul_sum(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((4,3,2), 0)
        res = Tensor(shape=(5,2))
        res2 = Tensor(shape=(5,2))
        res['k,l'] = 3 * (a['i,j,k'] * b['j,i,l'] + a['i,j,k'] * b['j,i,l'])
        res2['k,l'] = 6 * (a['i,j,k'] * b['j,i,l'])
        self.assertSequenceEqual(res, res2)

    #--------------------------------------------------------------------------------#
    # EinsumTensor Arithmetic operators

    def test_neg(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((4,3,2), 0)
        res = Tensor(shape=(5,2))
        res2 = Tensor(shape=(5,2))
        res['k,l'] = -a['i,j,k'] * b['j,i,l']
        res2['k,l'] = a['i,j,k'] * b['j,i,l']
        self.assertSequenceEqual(res, -res2)

    def test_mul_contraction(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((4,3,2), 0)
        res1 = Tensor(shape=(4,5,3))
        res2 = Tensor(shape=(4,5,3))
        res1['m,k,n'] = b['m,n,l'] * (a['i,j,k'] * b['j,i,l'])
        res2['m,k,n'] = b['m,n,l'] * a['i,j,k'] * b['j,i,l']
        self.assertSequenceEqual(res1, res2)

    def test_mul_number(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((4,3,2), 0)
        res = Tensor(shape=(5,2))
        res2 = Tensor(shape=(5,2))
        res['k,l'] = (2 * a['i,j,k']) * b['j,i,l']
        res2['k,l'] = a['i,j,k'] * b['j,i,l']
        self.assertSequenceEqual(res, 2 * res2)

    def test_add_sum_1(self):
        a = range_tensor((5,5,5), 0)
        res1 = Tensor(shape=(5,5,5))
        res1['i,j,k'] = a['i,j,k'] + (a['i,k,j'] + a['k,j,i'])
        res2 = Tensor(shape=(5,5,5))
        for i, j, k in product(xrange(5), repeat=3):
            res2[i,j,k] = a[i,j,k] + (a[i,k,j] + a[k,j,i])
        self.assertSequenceEqual(res1, res2)

    def test_add_sum_2(self):
        a = range_tensor((5,5,5), 0)
        res1 = Tensor(shape=(5,5,5))
        res1['j,k,i'] = a['i,j,k'] + a['i,k,j'] + a['k,j,i']
        res2 = Tensor(shape=(5,5,5))
        for i, j, k in product(xrange(5), repeat=3):
            res2[j,k,i] = a[i,j,k] + (a[i,k,j] + a[k,j,i])
        self.assertSequenceEqual(res1, res2)

    def test_add_sum_3(self):
        a,b,c,d = np.arange(5), np.arange(3,8), np.arange(12,17), np.arange(7,12)
        a,b,c,d = map(Tensor, (a,b,c,d))
        res1 = Tensor(shape=(5,5,5,5))
        res2 = Tensor(shape=(5,5,5,5))
        res1['i,j,k,l'] = a['i'] + b['j'] + c['k'] + d['l']
        for i, j, k, l in product(xrange(5), repeat=4):
            res2[i,j,k,l] = a[i] + b[j] + c[k] + d[l]
        assert_array_almost_equal(res1, res2)

    def test_add_sum_4(self):
        a,b = np.arange(5), np.arange(3,8)
        a,b = map(Tensor, (a,b))
        res1 = Tensor(shape=(5,5))
        res2 = Tensor(shape=(5,5))
        res1['i,j'] = a['i'] + b['j']
        for i, j in product(xrange(5), xrange(5)):
            res2[i,j] = a[i] + b[j]
        assert_array_almost_equal(res1, res2)

    def test_add_sum_5(self):
        a,b = np.arange(5), np.arange(8)
        a,b = map(Tensor, (a,b))
        res1 = Tensor(shape=(5,8))
        res2 = Tensor(shape=(5,8))
        res1['i,j'] = a['i'] + b['j']
        for i, j in product(xrange(5), xrange(8)):
            res2[i,j] = a[i] + b[j]
        assert_array_almost_equal(res1, res2)

    def test_add_sum_6(self):
        a,b,c = np.arange(3,8), np.arange(12,17), range_tensor((5,5))
        a,b,c = map(Tensor, (a,b,c))
        res1 = Tensor(shape=(5,5,5,5))
        res2 = Tensor(shape=(5,5,5,5))
        res1['i,k,j,l'] = a['i'] + b['j'] + c['l','k']
        for i, j, k, l in product(xrange(5), repeat=4):
            res2[i,k,j,l] = a[i] + b[j] + c[l,k]
        assert_array_almost_equal(res1, res2)

    def test_iadd_1(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((3,4,5), 12)
        res = a + b
        a['i,j,k'] += b['i,j,k']
        assert_array_almost_equal(res, a)

    def test_iadd_2(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((3,4,5), 12)
        c = range_tensor((3,5,4), 3)
        res = a + b + c.transpose((0,2,1))
        a['i,j,k'] += b['i,j,k'] + c['i,k,j']
        assert_array_almost_equal(res, a)

    def test_sub_1(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((3,4,5), 12)
        res1 = Tensor(shape=(3,4,5))
        res1['i,j,k'] = a['i,j,k'] - b['i,j,k']
        res2 = a - b
        assert_array_almost_equal(res1, res2)

    def test_sub_2(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((3,4,5), 12)
        c = range_tensor((3,4,5), 7)
        res1 = Tensor(shape=(3,4,5))
        res1['i,j,k'] = a['i,j,k'] - (b['i,j,k'] + c['i,j,k'])
        res2 = a - (b + c)
        assert_array_almost_equal(res1, res2)

    def test_isub_1(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((3,4,5), 12)
        res = a - b
        a['i,j,k'] -= b['i,j,k']
        assert_array_almost_equal(res, a)

    def test_isub_2(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((3,4,5), 12)
        c = range_tensor((3,5,4), 3)
        res = a - (b + c.transpose((0,2,1)))
        a['i,j,k'] -= b['i,j,k'] + c['i,k,j']
        assert_array_almost_equal(res, a)

    def test_idiv_num(self):
        a = range_tensor((3,4,5), 0)
        res = a / 3.0
        a['i,j,k'] /= 3.0
        assert_array_almost_equal(res, a)

    def test_idiv_tens(self):
        shp = (2,3,4,5)
        a = range_tensor(shp)
        b = range_tensor((3,))
        res = Tensor(shape=shp)
        for i,j,k,l in product(*map(xrange, shp)):
            res[i,j,k,l] = a[i,j,k,l] / b[j]
        a['i,j,k,l'] /= b['j']
        assert_array_almost_equal(a, res)

    def test_idiv_tens_2(self):
        shp = (2,3,4,5)
        a = range_tensor(shp)
        b = range_tensor((3,5))
        res = Tensor(shape=shp)
        for i,j,k,l in product(*map(xrange, shp)):
            res[i,j,k,l] = a[i,j,k,l] / b[j,l]
        a['i,j,k,l'] /= b['j,l']
        assert_array_almost_equal(a, res)

    def test_idiv_tens_3(self):
        shp = (2,3,4,5)
        a = range_tensor(shp)
        b = range_tensor(shp, 27)
        res = Tensor(shape=shp)
        for i,j,k,l in product(*map(xrange, shp)):
            res[i,j,k,l] = a[i,j,k,l] / b[i,j,k,l]
        a['i,j,k,l'] /= b['i,j,k,l']
        assert_array_almost_equal(a, res)

    def test_idiv_tens_4(self):
        shp = (3,3,3,3)
        a = range_tensor(shp)
        b = range_tensor(shp, 27)
        res = Tensor(shape=shp)
        for i,j,k,l in product(*map(xrange, shp)):
            res[i,j,k,l] = a[i,j,k,l] / b[j,l,i,k]
        a['i,j,k,l'] /= b['j,l,i,k']
        assert_array_almost_equal(a, res)

    def test_idiv_contract(self):
        a = range_tensor((4,6))
        b = range_tensor((4,5,3),10)
        c = range_tensor((3,5,6),27)
        res = range_tensor((4,6))
        for i, j in product(xrange(4), xrange(6)):
            res[i,j] /= sum(b[i,k,l] * c[l,k,j] for l,k in product(xrange(3), xrange(5)))
        a['i,j'] /= b['i,k,l'] * c['l,k,j']
        assert_array_almost_equal(a, res)

    def test_idiv_sum(self):
        shp = (2,3,4,5)
        a = range_tensor(shp)
        b,c,d,e = map(lambda x: Tensor(np.arange(1,x+1)), shp)
        res = Tensor(shape=shp)
        for i,j,k,l in product(*map(xrange, shp)):
            res[i,j,k,l] = a[i,j,k,l] / (b[i] + c[j] + 37.123 * d[k] - e[l])
        a['i,j,k,l'] /= b['i'] + c['j'] + 37.123 * d['k'] - e['l']
        assert_array_almost_equal(a, res)

    #--------------------------------------------------------------------------------#
    # Tensor sorting

    def test_sort_to(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((5,4,3), 0)
        res1 = a['i,j,k'].sort_to(b['k,j,i'])
        res2 = a.transpose((2,1,0))
        assert_array_almost_equal(res1, res2)

    #--------------------------------------------------------------------------------#
    # EinsumContraction

    def test_contraction_dot_1(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((5,4,3), 0)
        res1 = a['i,j,k'] * b['k,j,i']
        res2 = 0
        for i,j,k in product(xrange(3), xrange(4), xrange(5)):
            res2 += a[i,j,k] * b[k,j,i]
        self.assertEqual(res1, res2)

    def test_contraction_dot_2(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((5,4,3), 0)
        res1 = 37.3 * a['i,j,k'] * b['k,j,i']
        res2 = 0
        for i,j,k in product(xrange(3), xrange(4), xrange(5)):
            res2 += 37.3 * a[i,j,k] * b[k,j,i]
        self.assertEqual(res1, res2)

    def test_contraction_dot_3(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((5,4,3), 0)
        c = range_tensor((3,3), 0)
        res1 = 37.3 * a['i,j,k'] * b['k,j,l'] * c['i,l']
        res2 = 0
        for i,j,k,l in product(xrange(3), xrange(4), xrange(5), xrange(3)):
            res2 += 37.3 * a[i,j,k] * b[k,j,l] * c[i,l]
        self.assertEqual(res1, res2)

    def test_contraction_external(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((5,4,3), 0)
        self.assertEqual((a['i,n,m'] * b['m,n,k']).external_indices, ('i', 'k'))

    def test_contraction_internal(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((5,4,3), 0)
        self.assertEqual((a['i,n,m'] * b['m,n,k']).internal_indices, ('m', 'n'))

    def test_contraction_factorization_1(self):
        EinsumContraction.factorize_contraction = True
        EinsumContraction.print_factorization = True
        a = range_tensor((3,3,3), 1)
        b = range_tensor((3,3,3), 3)
        c = range_tensor((3,3,3), 5)
        d = range_tensor((3,3,3), 7)
        a.name, b.name, c.name, d.name = "abcd"
        res1 = Tensor(shape=(3,3,3,3), name="res1")
        res1['i,j,p,o'] = a['i,j,k'] * b['k,l,m'] * c['l,m,n'] * d['n,o,p']
        res2 = Tensor(shape=(3,3,3,3))
        for i,j,k,l,m,n,o,p in product(*map(xrange, (3,)*8)):
            res2[i,j,p,o] += a[i,j,k] * b[k,l,m] * c[l,m,n] * d[n,o,p]
        assert_array_almost_equal(res1, res2)

    def test_contraction_factorization_2(self):
        EinsumContraction.factorize_contraction = True
        EinsumContraction.print_factorization = True
        a = range_tensor((3,3,3,2), 1)
        b = range_tensor((5,3,4), 3)
        c = range_tensor((4,4), 5)
        d = range_tensor((2,3,2), 7)
        a.name, b.name, c.name, d.name = "abcd"
        res1 = Tensor(shape=(5,4,2,3), name="res1")
        res1['m,o,p,j'] = a['i,j,k,l'] * b['m,k,n'] * c['n,o'] * d['p,i,l']
        res2 = res1.copy_shape(name='res2')
        for i,j,k,l,m,n,o,p in product(*map(xrange, (3,3,3,2,5,4,4,2))):
            print i,j,k,l,m,n,o,p
            res2[m,o,p,j] += a[i,j,k,l] * b[m,k,n] * c[n,o] * d[p,i,l]
        assert_array_almost_equal(res1, res2)

    def test_contraction_factorization_3(self):
        EinsumContraction.factorize_contraction = True
        EinsumContraction.print_factorization = True
        a = range_tensor((3,3,5,3), 1)
        b = range_tensor((5,5), 5)
        c = range_tensor((3,3), 5)
        d = range_tensor((3,3,5,3), 7)
        a.name, b.name, c.name, d.name = "abcd"
        res1 = Tensor(shape=(3,3,3,3), name="res1")
        res1['i,j,k,l'] = a['k,l,m,n'] * b['m,o'] * c['n,p'] * d['i,j,o,p'] \
            - a['j,i,m,n'] * b['m,o'] * c['n,p'] * d['k,l,o,p']
        res2 = res1.copy_shape(name='res2')
        for i,j,k,l,m,n,o,p in product(*map(xrange, (3,3,3,3,5,3,5,3))):
            res2[i,j,k,l] += a[k,l,m,n] * b[m,o] * c[n,p] * d[i,j,o,p] \
                                - a[j,i,m,n] * b[m,o] * c[n,p] * d[k,l,o,p]
        assert_array_almost_equal(res1, res2)

    def test_contraction_factorization_4(self):
        EinsumContraction.factorize_contraction = True
        EinsumContraction.print_factorization = True
        a = range_tensor((3,3,3,5), 1)
        b = range_tensor((5,5), 5)
        c = range_tensor((3,3), 8)
        d = range_tensor((3,3,3,5), 7)
        a.name, b.name, c.name, d.name = "abcd"
        res1 = Tensor(shape=(3,3,3,3), name="res1")
        res1['i,j,k,l'] = 5.3 * a['k,l,n,m'] * b['m,o'] * c['n,p'] * d['i,j,p,o']
        res2 = res1.copy_shape(name='res2')
        for i,j,k,l,m,n,o,p in product(*map(xrange, (3,3,3,3,5,3,5,3))):
            res2[i,j,k,l] += 5.3 * a[k,l,n,m] * b[m,o] * c[n,p] * d[i,j,p,o]
        assert_array_almost_equal(res1, res2)
    #--------------------------------------------------------------------------------#
    # EinsumContraction arithmetic operations

    def test_contraction_neg(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((4,3,2), 0)
        res1 = Tensor(shape=(5,2))
        res2 = Tensor(shape=(5,2))
        res1['k,l'] = -(a['i,j,k'] * b['j,i,l'])
        res2['k,l'] = a['i,j,k'] * b['j,i,l']
        self.assertEqual(res1, -res2)

    def test_contraction_mul_number(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((4,3,2), 0)
        res1 = Tensor(shape=(5,2))
        res2 = Tensor(shape=(5,2))
        res1['k,l'] = -5.7 * (a['i,j,k'] * b['j,i,l'])
        res2['k,l'] = a['i,j,k'] * b['j,i,l']
        self.assertEqual(res1, -5.7 * res2)

    def test_contraction_add(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((4,3,2), 0)
        res = Tensor(shape=(5,2))
        res2 = Tensor(shape=(5,2))
        res['k,l'] = a['i,j,k'] * b['j,i,l'] + a['i,j,k'] * b['j,i,l']
        res2['k,l'] = 2 * a['i,j,k'] * b['j,i,l']
        self.assertSequenceEqual(res, res2)

    def test_contraction_sub(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((4,3,2), 0)
        res = Tensor(shape=(5,2))
        res2 = Tensor(shape=(5,2))
        res['k,l'] = 2 * (a['i,j,k'] * b['j,i,l']) - 7 * a['i,j,k'] * b['j,i,l']
        res2['k,l'] = a['i,j,k'] * b['j,i,l']
        self.assertSequenceEqual(res, -5 * res2)

    #--------------------------------------------------------------------------------#
    # EinsumSum arithmetic operations

    def test_sum_add_sums(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((3,4,5), 13)
        res = Tensor(shape=(3,4,5))
        res['i,j,k'] = (a['i,j,k'] - b['i,j,k']) + (0.37 * b['i,j,k'] - 3.27 * a['i,j,k'])
        assert_array_almost_equal(res, a - b + 0.37 * b - 3.27 * a)

    def test_sum_sub_tens(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((3,4,5), 13)
        res = Tensor(shape=(3,4,5))
        res['i,j,k'] = 3 * (a['i,j,k'] + 2 * b['i,j,k']) - b['i,j,k']
        assert_array_almost_equal(res, 3 * a + 5 * b)

    def test_sum_sub_sums(self):
        a = range_tensor((3,4,5), 0)
        b = range_tensor((3,4,5), 13)
        res = Tensor(shape=(3,4,5))
        res['i,j,k'] = (a['i,j,k'] - b['i,j,k']) - (0.37 * b['i,j,k'] - 3.27 * a['i,j,k'])
        assert_array_almost_equal(res, a - b - 0.37 * b + 3.27 * a)

    def test_sum_sub_2(self):
        shp = (2,3,4,5)
        a = range_tensor(shp)
        b,c,d,e = map(lambda x: Tensor(np.arange(1,x+1)), shp)
        res = Tensor(shape=shp)
        for i,j,k,l in product(*map(xrange, shp)):
            res[i,j,k,l] = b[i] + c[j] + 37.123 * d[k] - e[l]
        a['i,j,k,l'] = b['i'] + c['j'] + 37.123 * d['k'] - e['l']
        assert_array_almost_equal(a, res)

    def test_sum_sub_3(self):
        shp = (2,3,4,5)
        a = range_tensor(shp)
        b,c,d,e = map(lambda x: Tensor(np.arange(1,x+1)), shp)
        res = Tensor(shape=shp)
        for i,j,k,l in product(*map(xrange, shp)):
            res[i,j,k,l] = b[i] + 13.2 * (c[j] + 37.123 * d[k]) - e[l]
        a['i,j,k,l'] = b['i'] + 13.2 * (c['j'] + 37.123 * d['k']) - e['l']
        assert_array_almost_equal(a, res)

    #--------------------------------------------------------------------------------#
    # Complicated sums
    def test_sum_into_1(self):
        a = range_tensor((3,3,3))
        b = range_tensor((3,3,3),23)
        c = range_tensor((3,3), 7)
        EinsumContraction.factorize_contraction = True
        EinsumContraction.print_factorization = True
        a.name, b.name, c.name = "abc"
        res = Tensor(shape=(3,3,3,3), name="res")
        res['i,j,k,l'] = a['i,j,m'] * b['m,k,l'] \
                         + a['i,m,n'] * b['j,n,m'] \
                         + a['j,m,n'] * b['l,n,m'] \
                         + a['k,m,n'] * b['i,n,m'] \
                         + c['i,l'] + a['i,k,j'] \
                         + c['j,m'] * b['m,n,i'] * c['k,n'] \
                         + a['l,m,n'] * a['n,j,k'] * c['m,i'] \
                         + a['l,m,n'] * a['n,o,p'] * a['j,p,o'] * a['m,r,r']
        res2 = Tensor(shape=(3,3,3,3))
        for i,j,k,l in product(xrange(3), repeat=4):
            res2[i,j,k,l] = c[i,l] + a[i,k,j]
            for m in xrange(3):
                res2[i,j,k,l] += a[i,j,m] * b[m,k,l]
                for n in xrange(3):
                    res2[i,j,k,l] += a[i,m,n] * b[j,n,m] \
                                     + a[j,m,n] * b[l,n,m] \
                                     + a[k,m,n] * b[i,n,m] \
                                     + c[j,m] * b[m,n,i] * c[k,n] \
                                     + a[l,m,n] * a[n,j,k] * c[m,i]
                    for o, p, r in product(xrange(3), repeat=3):
                         res2[i,j,k,l] += a[l,m,n] * a[n,o,p] * a[j,p,o] * a[m,r,r]
        assert_array_almost_equal(res, res2)

    #--------------------------------------------------------------------------------#
    # internal contractions

    def test_internal_contraction_1(self):
        a = range_tensor((4,4,4,4))
        res = Tensor(shape=(4,4))
        res['p,q'] = a['p,q,i,i']
        res2 = Tensor(shape=(4,4))
        for p, q, i in product(xrange(4), repeat=3):
            res2[p,q] += a[p,q,i,i]
        assert_array_almost_equal(res, res2)

    def test_internal_contraction_2(self):
        a = range_tensor((4,4,4,4))
        res = Tensor(shape=(4,4))
        res['p,q'] = 2 * a['p,q,i,j']
        res2 = Tensor(shape=(4,4))
        for p, q, i, j in product(xrange(4), repeat=4):
            res2[p,q] += 2 * a[p,q,i,j]
        assert_array_almost_equal(res, res2)

    def test_internal_contraction_3(self):
        a = range_tensor((4,4,4,4))
        res = Tensor(shape=(4,4))
        res['p,q'] = 2 * a['p,q,i,i'] - a['p,i,i,q']
        res2 = Tensor(shape=(4,4))
        for p, q, i in product(xrange(4), repeat=3):
            res2[p,q] += 2 * a[p,q,i,i] - a[p,i,i,q]
        assert_array_almost_equal(res, res2)

    def test_internal_contraction_4(self):
        a = range_tensor((4,4,4,4))
        res = Tensor(shape=(4,4))
        res['p,q'] += 2 * a['p,q,i,i'] - a['p,i,i,q']
        res2 = Tensor(shape=(4,4))
        for p, q, i in product(xrange(4), repeat=3):
            res2[p,q] += 2 * a[p,q,i,i] - a[p,i,i,q]
        assert_array_almost_equal(res, res2)

    def test_internal_contraction_5(self):
        a = range_tensor((4,4,4,4))
        res = Tensor(shape=(4,4))
        res['p,q'] -= a['p,q,i,i']
        res2 = Tensor(shape=(4,4))
        for p, q, i in product(xrange(4), repeat=3):
            res2[p,q] -= a[p,q,i,i]
        assert_array_almost_equal(res, res2)

    #--------------------------------------------------------------------------------#
    # Subindices

    def test_subindices(self):
        my_set = IndexRangeSet()
        DeclareIndexRange('p', 0, 4, index_range_set=my_set).with_subranges(
            IndexRange('a', 2, Ellipsis, index_range_set=my_set),
            IndexRange('i', 0, 2, index_range_set=my_set)
        )
        a = Matrix([
                [ 4,  1, 3, 4],
                [ 2,  5, 7, 8],
                [ 9, 10, 2, 1],
                [11, 12, 1, 2],
            ],
            indices='p,p',
            index_range_set=my_set
        )
        exp_res = Tensor([
                [ 4,  1, 9,  6],
                [ 2,  5, 9, 12],
                [ 9, 10, 2,  1],
                [11, 12, 1,  2],
            ],
            indices='p,p',
            index_range_set=my_set
        )
        a['i_1,a_3'] = a['i_1,i_2'] * a['a_2,a_3']
        self.assertSequenceEqual(a, exp_res)

    def test_subindices_sum(self):
        my_set = IndexRangeSet()
        DeclareIndexRange('p', 0, 4, index_range_set=my_set).with_subranges(
            IndexRange('a', 2, Ellipsis, index_range_set=my_set),
            IndexRange('i', 0, 2, index_range_set=my_set)
        )
        a = Matrix([
            [ 4,  1, 3, 4],
            [ 2,  5, 7, 8],
            [ 9, 10, 2, 1],
            [11, 12, 1, 2],
            ],
            indices='p,p',
            index_range_set=my_set
        )
        exp_res = Tensor([
            [ 4,  1, 6, 2],
            [ 2,  5, 3, 7],
            [ 9, 10, 2, 1],
            [11, 12, 1, 2],
            ],
            indices='p,p',
            index_range_set=my_set
        )
        a['i_1,a_2'] = a['i_1,i_2'] + a['a_1,a_2']
        self.assertSequenceEqual(a, exp_res)

    #--------------------------------------------------------------------------------#
    # Delegation tests

    def setup_delegation(self):
        my_set = IndexRangeSet()
        DeclareIndexRange('p,q,r,s', 0, 5, in_set=my_set).with_subranges(
            IndexRange('a,b', 2, Ellipsis, in_set=my_set),
            IndexRange('i,j', 0, 2, in_set=my_set)
        )
        self.tensor = index_range_tensor('p,q', in_set=my_set)
        self.t4 = index_range_tensor('p,q,r,s', in_set=my_set)

    def test_delegate_zero_1(self):
        self.setup_delegation()
        orig = copy(self.tensor)
        orig[0:2,0:2] = 0
        self.tensor['i,j'].zero()
        assert_array_almost_equal(self.tensor, orig)

    def test_delegate_zero_2(self):
        self.setup_delegation()
        orig = copy(self.tensor)
        orig[0:2,2:] = 0
        self.tensor['i,a'].zero()
        assert_array_almost_equal(self.tensor, orig)

    def test_delegate_diagonal(self):
        self.setup_delegation()
        assert_array_almost_equal(self.tensor["i,j"].diagonal, Vector([1,7]))

    def test_delegate_shape(self):
        self.setup_delegation()
        self.assertSequenceEqual(self.tensor["a,i"].shape, (3,2))
        self.assertSequenceEqual(self.tensor["p,q"].shape, (5,5))
        self.assertSequenceEqual(self.tensor["a,q"].shape, (3,5))
        self.assertSequenceEqual(self.tensor["a,b"].shape, (3,3))

    def test_eq_1(self):
        self.setup_delegation()
        self.t4['p,q,r,s'] += self.t4['p,q,s,r']
        self.assertTrue((self.t4['p,q,r,s'] == self.t4['p,q,s,r']).all())
        # Check that something actually happened
        assert_array_almost_equal(self.t4, self.t4.transpose((0,1,3,2)))

    def test_eq_2(self):
        self.setup_delegation()
        self.t4['i,j,a,b'] += self.t4['i,j,b,a']
        self.assertTrue((self.t4['i,j,a,b'] == self.t4['i,j,b,a']).all())
        # Check that something actually happened
        assert_array_almost_equal(self.t4['i,j,a,b'].sliced_tensor, self.t4['i,j,a,b'].sliced_tensor.transpose((0,1,3,2)))

