/***************************************************************************
 *   Copyright (C) 2004 by Johan Maes - ON4QZ                              *
 *   on4qz@telenet.be                                                      *
 *   http://users.telenet.be/on4qz                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
/* Based on:
	Program: REALFFT.C
  Author: Philip VanBaren
*/

#include "fftdisplay.h"
#include "math.h"

#define FFTAVERAGING 0.1
#define VALAVG 0.05
#define FFTHIGH 2800
#define FFTLOW  300
#define FFTSPAN (FFTHIGH-FFTLOW)


fftDisplay::fftDisplay(QWidget *p): QFrame(p)
{
  eraseColor=QColor(qRgb(0,0,128)).rgb();
	arMag=NULL;
	fftTempArray=NULL;
	im=NULL;
	setFrameStyle( QFrame::Panel | QFrame::Sunken );
	setBackgroundRole(QPalette::Dark);
	showWaterfall=FALSE;
	enabled=FALSE;
  minMagnitude=0;
  avgMagnitude=0;
	maxMagnitude=1;
  in=out=NULL;
}

fftDisplay::~fftDisplay( )
{
	deleteBuffers();	

}

/** delete allocated buffers */
void fftDisplay::deleteBuffers()
{
    fftw_destroy_plan(plan);
    fftw_free(in);
    fftw_free(out);
    delete im;
    delete FFTArray;
    delete [] fftTempArray;
    delete [] arMag;
}


/** must be called before performing any FFT */
void fftDisplay::init(unsigned int fftLength, int samplingrate)
{
  unsigned int i;
  h=contentsRect().height();
  w=contentsRect().width();
  if(in!=NULL) deleteBuffers();
  im=new QImage(w,h,QImage::Format_RGB32);
  im->fill(eraseColor);
  FFTArray=new QPolygon(w);
  fftTempArray=new float[w];
  points = fftLength;
  out = (double *)fftw_malloc(points * sizeof(double));
  in  = (double *)fftw_malloc(points * sizeof(double));
   // create the fftw plan
  plan = fftw_plan_r2r_1d(points, in, out, FFTW_R2HC, FFTW_ESTIMATE);
  arMag=new double[points];
  for (i = 0; i <points; i++)
		{
			arMag[i]=0;
		}
  for (i = 0; i <w; i++) fftTempArray[i]=0.;

/*
*  FFT size is only half the number of data points
*  The full FFT output can be reconstructed from this FFT's output.
*  (This optimization can be made since the data is real.)
*/
  bucketHigIndex=(points*FFTHIGH)/samplingrate;
  bucketLowIndex=(points*FFTLOW) /samplingrate;
  bucketSpanIndex=(points*FFTSPAN)/samplingrate; // equivalent to (points/2*FFTSPAN)/(samplingrate/2)
  bufIndex=0;
  maxMagnitude=0;
  minMagnitude=1000.;
}



/** do the actual FFT calculation */

void fftDisplay::realFFT(short int *iBuffer)
{
	uint i;
  for (i=0;i<points;i++)
		{
      in[i]=(double) iBuffer[i]*(0.5+cos((DSPFLOAT)i*2*M_PI/(DSPFLOAT)points+M_PI)/2);;
		}
  doFFT();
}


void fftDisplay::realFFT(float *iBuffer)
{
	uint i;
  memmove(in,in+points/2,points/2*sizeof(double));
  for (i=0;i<points/2;i++)
		{
      in[points/2+i]=(double) iBuffer[i];
		}
   doFFT();
}

void fftDisplay::doFFT()
{
	uint i;
  double re,im,tmp;
  fftw_execute(plan);
  double maxMag=0;
  double minMag=200000;

  arMag[0]=0;
  for (i = 1; i <points/2; i++)
  	{
      re=out[i];
      im=out[points-i];
      tmp=sqrt(re*re+im*im);
      arMag[i]=tmp;
      if(maxMag<tmp) maxMag=tmp;
      if(minMag>tmp) minMag=tmp;
      avgMagnitude=avgMagnitude*(1-VALAVG)+VALAVG*tmp;
    }
  maxMagnitude=maxMagnitude*(1-VALAVG)+VALAVG*maxMag;
  minMagnitude=minMagnitude*(1-VALAVG)+VALAVG*minMag;
  addToLog(QString("doFFT: min=%1,max=%2 avg=%3").arg(minMagnitude).arg(maxMagnitude).arg(avgMagnitude),DBFFT);
	enabled=TRUE;
}


void fftDisplay::paintEvent(QPaintEvent *p)
{
  draw(TRUE);
  QFrame::paintEvent(p);
}

/*
 We have n points representing the range of 0 (DC) up to half of the sampling frequency.
 bucketHigIndex=(points*FFTHIGH)/samplingrate;
 bucketLowIndex=(points*FFTLOW) /samplingrate;
 bucketSpanIndex=(points*FFTSPAN)/samplingrate
 */


void fftDisplay::draw(bool update)
{
	
	int i;
  int ind;
  int t;
  addToLog("fftDisplay: draw",DBFFT);
  QColor c;
	if(!enabled) return;
	if(update)
		{
    addToLog("fftDisplay: draw update",DBFFT);
			if(showWaterfall)
				{
					for (i=(h-2);i>=0;i--) memmove(im->scanLine(i+1),im->scanLine(i),w*sizeof(uint));
   			} 
      uint *ptr=(uint *)im->scanLine(0);

  // pseudo-coloring
		if (maxMagnitude==0) maxMagnitude=1;
    addToLog(QString("%1,%2").arg(maxMagnitude).arg(avgMagnitude),DBFFT);
    for (i=0; i<w;i++)
			{
        ind=bucketLowIndex+(i*bucketSpanIndex)/(w-1);

  //      double ttt= arMag[ind]/(maxMagnitude);
        double ttt=(arMag[ind]-4*minMagnitude)/(0.6*maxMagnitude-4*minMagnitude);
       if(ttt>fftTempArray[i]) fftTempArray[i]=ttt;
       else fftTempArray[i]=fftTempArray[i]*(1-FFTAVERAGING)+FFTAVERAGING*(ttt); // maximum of 1;
        if(fftTempArray[i]<0) fftTempArray[i]=0; //ranging  from 0db to -90db
        if(fftTempArray[i]>1) fftTempArray[i]=1;

        int val=(uint)(fftTempArray[i]*255);
				if(showWaterfall)
					{
						c.setHsv(255-val,255,255);
						ptr[i]=c.rgb();
					}
				else
					{
            t=h-(val*h)/255;
						FFTArray->setPoint(i,i,t);
            addToLog(QString("%1,%2").arg(i).arg(t),DBFFT);
					}
			}
		if(!showWaterfall)
			{
				QPen pn(Qt::red);
				pn.setWidth(2);
				im->fill(eraseColor);
				QPainter p(im);
				p.setPen(pn);
        t=((1200-FFTLOW)*w/FFTSPAN);
        p.drawLine(t,0,t,h);
				pn.setColor(Qt::yellow);
				p.setPen(pn);
        t=((1500-FFTLOW)*w/FFTSPAN);
        p.drawLine(t,0,t,h);
        t=((2300-FFTLOW)*w/FFTSPAN);
        p.drawLine(t,0,t,h);
				pn.setColor(Qt::white);
				p.setPen(pn);
				for (i=0;i<7;i++)
					{
            t=(((1600-FFTLOW+i*100))*w/FFTSPAN);
            p.drawLine(t,0,t,h);
					}
				pn.setWidth(2);
        pn.setColor(Qt::yellow);
				p.setPen(pn);
				p.drawPolyline(*FFTArray);
				
			}
		}
  QPainter p(this);
//p.drawImage(QRect(0,0,w,h),*im);
  p.drawImage(0,0,*im);
}

void fftDisplay::mousePressEvent( QMouseEvent *e )
{
  if (e->button() == Qt::LeftButton)
    {
      showWaterfall=!showWaterfall;
      //im->fill(eraseColor);
    }
}
