% File src/library/base/man/tempfile.Rd
% Part of the R package, https://www.R-project.org
% Copyright 1995-2023 R Core Team
% Distributed under GPL 2 or later

\name{tempfile}
\alias{tempfile}
\alias{tempdir}
\title{Create Names for Temporary Files}
\description{
  \code{tempfile} returns a vector of character strings which can be used as
  names for temporary files.
}
\usage{% ../R/temp.R
tempfile(pattern = "file", tmpdir = tempdir(), fileext = "")
tempdir(check = FALSE)
}
\arguments{
  \item{pattern}{a non-empty character vector giving the initial part
    of the name.}
  \item{tmpdir}{a non-empty character vector giving the directory name.}
  \item{fileext}{a non-empty character vector giving the file extension.}
  \item{check}{\code{\link{logical}} indicating if \code{tmpdir()}
    should be checked and recreated if no longer valid.}
}
\value{
  For \code{tempfile} a character vector giving the names of possible
  (temporary) files.  Note that no files are generated by \code{tempfile}.

  For \code{tempdir}, the path of the per-session temporary directory.

  On Windows, both will use a backslash as the path separator.

  On a Unix-alike, the value will be an absolute path (unless
  \code{tmpdir} is set to a relative path), but it need not be canonical
  (see \code{\link{normalizePath}}) and on macOS it often is not.  
}
\details{
  The length of the result is the maximum of the lengths of the three
  arguments; values of shorter arguments are recycled.

  The names are very likely to be unique among calls to \code{tempfile}
  in an \R session and across simultaneous \R sessions (unless
  \code{tmpdir} is specified).  The filenames are guaranteed not to be
  currently in use.

  The file name is made by concatenating the path given by
  \code{tmpdir}, the \code{pattern} string, a random string in hex and
  a suffix of \code{fileext}.

  By default, \code{tmpdir} will be the directory given by
  \code{tempdir()}.  This will be a subdirectory of the per-session
  temporary directory found by the following rule when the \R session is
  started.  The environment variables \env{TMPDIR}, \env{TMP} and
  \env{TEMP} are checked in turn and the first found which points to a
  writable directory is used:
#ifdef unix
  if none succeeds \file{/tmp} is used.  The path must not contain spaces.
#endif
#ifdef windows
  if none succeeds the value of \env{R_USER} (see
  \code{\link{Rconsole}}) is used.  If the path to the directory
  contains a space in any of the components, the path returned will use
  the short names version of the path (if that does not contain a space).
#endif
  Note that setting any of these environment variables in the \R session
  has no effect on \code{tempdir()}: the per-session temporary directory
  is created before the interpreter is started.
}

\section{Note on parallel use}{
  \R processes forked by functions such as \code{\link{mclapply}} and
  \code{\link{makeForkCluster}} in package \pkg{parallel} share a
  per-session temporary directory.  Further, the \sQuote{guaranteed not
    to be currently in use} applies only at the time of asking, and two
  children could ask simultaneously.  This is circumvented by ensuring
  that \code{tempfile} calls in different children try different names.
}

\source{
  The final component of \code{tempdir()} is created by the POSIX system
  call \code{mkdtemp}, or if this is not available (e.g.\sspace{}on
  Windows) a version derived from the source code of GNU \code{glibc}.

  It will be of the form \file{RtmpXXXXXX} where the last 6 characters
  are replaced in a platform-specific way.  POSIX only requires that the
  replacements be ASCII, which allows \code{.} (so the value may appear
  to have a file extension) and \link{regexp} metacharacters such as
  \code{+}.  Most commonly the replacements are from the \link{regexp}
  pattern \code{[A-Za-z0-9]}, but \code{.} \emph{has} been seen.
}

\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth & Brooks/Cole.
}
\seealso{\code{\link{unlink}} for deleting files.}
\examples{\donttest{
tempfile(c("ab", "a b c"))   # give file name with spaces in!

tempfile("plot", fileext = c(".ps", ".pdf"))

tempdir() # works on all platforms with a platform-dependent result
}

## Show how 'check' is working on some platforms:
if(exists("I'm brave") && `I'm brave` &&
   identical(.Platform$OS.type, "unix") && grepl("^/tmp/", tempdir())) {
  cat("Current tempdir(): ", tempdir(), "\n")
  cat("Removing it :", file.remove(tempdir()),
      "; dir.exists(tempdir()):", dir.exists(tempdir()), "\n")
  cat("and now  tempdir(check = TRUE) :", tempdir(check = TRUE),"\n")
}

}
\keyword{file}
