
/**************************************************************************
 *                                                                        *
 *  Regina - A Normal Surface Theory Calculator                           *
 *  Computational Engine                                                  *
 *                                                                        *
 *  Copyright (c) 1999-2014, Ben Burton                                   *
 *  For further details contact Ben Burton (bab@debian.org).              *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or         *
 *  modify it under the terms of the GNU General Public License as        *
 *  published by the Free Software Foundation; either version 2 of the    *
 *  License, or (at your option) any later version.                       *
 *                                                                        *
 *  As an exception, when this program is distributed through (i) the     *
 *  App Store by Apple Inc.; (ii) the Mac App Store by Apple Inc.; or     *
 *  (iii) Google Play by Google Inc., then that store may impose any      *
 *  digital rights management, device limits and/or redistribution        *
 *  restrictions that are required by its terms of service.               *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful, but   *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *  General Public License for more details.                              *
 *                                                                        *
 *  You should have received a copy of the GNU General Public             *
 *  License along with this program; if not, write to the Free            *
 *  Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,       *
 *  MA 02110-1301, USA.                                                   *
 *                                                                        *
 **************************************************************************/

/* end stub */

/*! \file maths/nperm5.h
 *  \brief Deals with permutations of {0,1,2,3,4}.
 */

#ifndef __NPERM5_H
#ifndef __DOXYGEN
#define __NPERM5_H
#endif

#include <string>
#include "regina-core.h"

namespace regina {

/**
 * \weakgroup maths
 * @{
 */

/**
 * Represents a permutation of {0,1,2,3,4}.
 * Amongst other things, such permutations are used in describing
 * simplex gluings in 4-manifold triangulations.  NPerm5 objects are small
 * enough to pass about by value instead of by reference.
 *
 * Each permutation has an internal code, and this code is sufficient to
 * reconstruct the permutation.
 * Thus the internal code may be a useful means for passing
 * permutation objects to and from the engine.
 *
 * The internal code is an unsigned integer.  The lowest three bits represent
 * the image of 0, the next lowest three bits represent the image of 1 and so
 * on.
 */
class REGINA_API NPerm5 {
    public:
        /**
         * Contains all possible permutations of five elements.
         *
         * The permutations with even indices in the array are the even
         * permutations, and those with odd indices in the array are the
         * odd permutations.
         *
         * Note that the permutations are not necessarily in
         * lexicographical order.
         */
        static const NPerm5 S5[120];

        /**
         * A dimension-agnostic alias for NPerm5::S5.  In general, for
         * each \a K the class NPermK will define an alias \a Sn
         * that references the list of all permutations NPermK::SK.
         */
        static const NPerm5* Sn;

        /**
         * Contains all possible permutations of five elements in
         * lexicographical order.
         */
        static const NPerm5 orderedS5[120];

        /**
         * A dimension-agnostic alias for NPerm5::orderedS5.  In general, for
         * each \a K the class NPermK will define an alias \a orderedSn
         * that references the list of all permutations NPermK::orderedSK.
         */
        static const NPerm5* orderedSn;

        /**
         * Contains the inverses of the permutations in the array \a S5.
         *
         * Specifically, the inverse of permutation <tt>S5[i]</tt> is
         * the permutation <tt>S5[ invS5[i] ]</tt>.
         */
        static const unsigned invS5[120];

        /**
         * A dimension-agnostic alias for NPerm5::invS5.  In general, for
         * each \a K the class NPermK will define an alias \a invSn
         * that references the list of all permutations NPermK::invSK.
         */
        static const unsigned* invSn;

        /**
         * Contains all possible permutations of four elements.
         * In each permutation, 4 maps to 4.
         *
         * The permutations with even indices in the array are the even
         * permutations, and those with odd indices in the array are the
         * odd permutations.
         *
         * For all permutation classes (NPerm4, NPerm5 and so on), the
         * S4 array stores the same permutations in the same order (but
         * of course using different data types).
         *
         * Note that the permutations are not necessarily in
         * lexicographical order.  For the corresponding inverse array,
         * see NPerm4::invS4.
         */
        static const NPerm5 S4[24];

        /**
         * A dimension-agnostic alias for NPerm5::S4.  In general, for
         * each \a K the class NPermK will define an alias \a Sn_1
         * that references the list of all permutations NPermK::S(K-1).
         */
        static const NPerm5* Sn_1;

        /**
         * Contains all possible permutations of four elements in
         * lexicographical order.  In each permutation, 4 maps to 4.
         */
        static const NPerm5 orderedS4[24];

        /**
         * Contains all possible permutations of three elements.
         * In each permutation, 3 maps to 3 and 4 maps to 4.
         *
         * The permutations with even indices in the array are the even
         * permutations, and those with odd indices in the array are the
         * odd permutations.
         *
         * For all permutation classes (NPerm4, NPerm5 and so on), the
         * S3 array stores the same permutations in the same order (but
         * of course using different data types).
         *
         * Note that the permutations are not necessarily in
         * lexicographical order.  For the corresponding inverse array,
         * see NPerm3::invS3.
         */
        static const NPerm5 S3[6];

        /**
         * Contains all possible permutations of three elements in
         * lexicographical order.  In each permutation, 3 maps to 3 and
         * 4 maps to 4.
         */
        static const NPerm5 orderedS3[6];

        /**
         * Contains all possible permutations of two elements.
         * In each permutation, 2 maps to 2, 3 maps to 3, and 4 maps to 4.
         *
         * The permutations with even indices in the array are the even
         * permutations, and those with odd indices in the array are the
         * odd permutations.
         *
         * For all permutation classes (NPerm4, NPerm5 and so on), the
         * S2 array stores the same permutations in the same order (but
         * of course using different data types).
         *
         * Note that these permutations are already in lexicographical order.
         */
        static const NPerm5 S2[2];

        enum {
            /**
             * The total number of permutations on five elements.
             * This is the size of the array Sn.
             *
             * \ifacespython Not present.
             */
            nPerms = 120,

            /**
             * The total number of permutations on four elements.
             * This is the size of the array Sn_1.
             *
             * \ifacespython Not present.
             */
            nPerms_1 = 24
        };

    private:
        unsigned code;
            /**< The internal code representing this permutation. */

    public:
        /**
         * Creates the identity permutation.
         */
        NPerm5();

        /**
         * Creates the transposition of \a a and \a b.
         * Note that \a a and \a b need not be distinct.
         *
         * \pre \a a and \a b are in {0,1,2,3,4}.
         *
         * @param a the element to switch with \a b.
         * @param b the element to switch with \a a.
         */
        NPerm5(int a, int b);

        /**
         * Creates a permutation mapping (0,1,2,3,4) to
         * (<i>a</i>,<i>b</i>,<i>c</i>,<i>d</i>,<i>e</i>) respectively.
         *
         * \pre {<i>a</i>,<i>b</i>,<i>c</i>,<i>d</i>,<i>e</i>} = {0,1,2,3,4}.
         *
         * @param a the desired image of 0.
         * @param b the desired image of 1.
         * @param c the desired image of 2.
         * @param d the desired image of 3.
         * @param e the desired image of 4.
         */
        NPerm5(int a, int b, int c, int d, int e);

        /**
         * Creates a permutation mapping \a i to \a image[i] for each
         * \a i = 0,1,2,3,4.
         *
         * \pre The array \a image contains five elements, which are
         * 0, 1, 2, 3 and 4 in some order.
         *
         * \ifacespython Not present.
         *
         * @param image the array of images.
         */
        NPerm5(const int* image);

        /**
         * Creates a permutation mapping
         * (<i>a0</i>,<i>b0</i>,<i>c0</i>,<i>d0</i>,<i>e0</i>) to
         * (<i>a1</i>,<i>b1</i>,<i>c1</i>,<i>d1</i>,<i>e1</i>) respectively.
         *
         * \pre {<i>a0</i>,<i>b0</i>,<i>c0</i>,<i>d0</i>,<i>e0</i>} =
         * {<i>a1</i>,<i>b1</i>,<i>c1</i>,<i>d1</i>,<i>e1</i>} =
         * {0,1,2,3,4}.
         *
         * @param a0 the desired preimage of <i>a1</i>.
         * @param b0 the desired preimage of <i>b1</i>.
         * @param c0 the desired preimage of <i>c1</i>.
         * @param d0 the desired preimage of <i>d1</i>.
         * @param e0 the desired preimage of <i>e1</i>.
         * @param a1 the desired image of <i>a0</i>.
         * @param b1 the desired image of <i>b0</i>.
         * @param c1 the desired image of <i>c0</i>.
         * @param d1 the desired image of <i>d0</i>.
         * @param e1 the desired image of <i>e0</i>.
         */
        NPerm5(int a0, int a1, int b0, int b1, int c0, int c1, int d0, int d1,
            int e0, int e1);

        /**
         * Creates a permutation that is a clone of the given
         * permutation.
         *
         * @param cloneMe the permutation to clone.
         */
        NPerm5(const NPerm5& cloneMe);

        /**
         * Returns the internal code representing this permutation.
         * Note that the internal code is sufficient to reproduce the
         * entire permutation.
         *
         * The code returned will be a valid permutation code as
         * determined by isPermCode().
         *
         * @return the internal code.
         */
        unsigned getPermCode() const;

        /**
         * Sets this permutation to that represented by the given
         * internal code.
         *
         * \pre the given code is a valid permutation code; see
         * isPermCode() for details.
         *
         * @param newCode the internal code that will determine the
         * new value of this permutation.
         */
        void setPermCode(unsigned newCode);

        /**
         * Creates a permutation from the given internal code.
         *
         * \pre the given code is a valid permutation code; see
         * isPermCode() for details.
         *
         * @param newCode the internal code for the new permutation.
         * @return the permutation reprsented by the given internal code.
         */
        static NPerm5 fromPermCode(unsigned newCode);

        /**
         * Determines whether the given integer is a valid internal
         * permutation code.  Valid permutation codes can be passed to
         * setPermCode() or fromPermCode(), and are returned by getPermCode().
         *
         * @return \c true if and only if the given code is a valid
         * internal permutation code.
         */
        static bool isPermCode(unsigned newCode);

        /**
         * Sets this permutation to be equal to the given permutation.
         *
         * @param cloneMe the permutation whose value will be assigned
         * to this permutation.
         * @return a reference to this permutation.
         */
        NPerm5& operator = (const NPerm5& cloneMe);

        /**
         * Returns the composition of this permutation with the given
         * permutation.  If this permutation is <i>p</i>, the
         * resulting permutation will be <i>p o q</i>, satisfying
         * <tt>(p*q)[x] == p[q[x]]</tt>.
         *
         * @param q the permutation with which to compose this.
         * @return the composition of both permutations.
         */
        NPerm5 operator * (const NPerm5& q) const;

        /**
         * Finds the inverse of this permutation.
         *
         * @return the inverse of this permutation.
         */
        NPerm5 inverse() const;

        /**
         * Determines the sign of this permutation.
         *
         * @return 1 if this permutation is even, or -1 if this
         * permutation is odd.
         */
        int sign() const;

        /**
         * Determines the image of the given integer under this
         * permutation.
         *
         * @param source the integer whose image we wish to find.  This
         * should be between 0 and 4 inclusive.
         * @return the image of \a source.
         */
        int operator[](int source) const;

        /**
         * Determines the preimage of the given integer under this
         * permutation.
         *
         * @param image the integer whose preimage we wish to find.  This
         * should be between 0 and 4 inclusive.
         * @return the preimage of \a image.
         */
        int preImageOf(int image) const;

        /**
         * Determines if this is equal to the given permutation.
         * This is true if and only if both permutations have the same
         * images for 0, 1, 2, 3 and 4.
         *
         * @param other the permutation with which to compare this.
         * @return \c true if and only if this and the given permutation
         * are equal.
         */
        bool operator == (const NPerm5& other) const;

        /**
         * Determines if this differs from the given permutation.
         * This is true if and only if the two permutations have
         * different images for at least one of 0, 1, 2, 3 or 4.
         *
         * @param other the permutation with which to compare this.
         * @return \c true if and only if this and the given permutation
         * differ.
         */
        bool operator != (const NPerm5& other) const;

        /**
         * Lexicographically compares the images of (0,1,2,3,4) under this
         * and the given permutation.
         *
         * @param other the permutation with which to compare this.
         * @return -1 if this permutation produces a smaller image, 0 if
         * the permutations are equal and 1 if this permutation produces
         * a greater image.
         */
        int compareWith(const NPerm5& other) const;

        /**
         * Determines if this is the identity permutation.
         * This is true if and only if each of 0, 1, 2, 3 and 4 is
         * mapped to itself.
         *
         * @return \c true if and only if this is the identity
         * permutation.
         */
        bool isIdentity() const;

        /**
         * A deprecated alias for str(), which returns a string representation
         * of this permutation.
         *
         * \deprecated This routine has (at long last) been deprecated;
         * use the simpler-to-type str() instead.
         *
         * @return a string representation of this permutation.
         */
        std::string toString() const;
        /**
         * Returns a string representation of this permutation.
         * The representation will consist of five adjacent digits
         * representing the images of 0, 1, 2, 3 and 4 respectively.
         * An example of a string representation is <tt>30421</tt>.
         *
         * @return a string representation of this permutation.
         */
        std::string str() const;

        /**
         * Returns a string representation of this permutation with only
         * the images of 0 and 1.  The resulting string will therefore
         * have length two.
         *
         * @return a truncated string representation of this permutation.
         */
        std::string trunc2() const;

        /**
         * Returns a string representation of this permutation with only
         * the images of 0, 1 and 2.  The resulting string will therefore
         * have length three.
         *
         * @return a truncated string representation of this permutation.
         */
        std::string trunc3() const;

        /**
         * Returns a string representation of this permutation with only
         * the images of 0, 1, 2 and 3.  The resulting string will therefore
         * have length four.
         *
         * @return a truncated string representation of this permutation.
         */
        std::string trunc4() const;

        /**
         * Returns the index of this permutation in the NPerm5::S5 array.
         *
         * @return the index \a i for which this permutation is equal to
         * NPerm5::S5[i].  This will be between 0 and 119 inclusive.
         *
         * @author Ryan Budney
         */
        int S5Index() const;

        /**
         * Returns the index of this permutation in the NPerm5::S5 array.
         * This is a dimension-agnostic alias for S5Index().
         *
         * @return the index \a i for which this permutation is equal to
         * NPerm5::S5[i].  This will be between 0 and 119 inclusive.
         */
        int SnIndex() const;

        /**
         * Returns the index of this permutation in the NPerm5::orderedS5 array.
         *
         * @return the index \a i for which this permutation is equal to
         * NPerm5::orderedS5[i].  This will be between 0 and 119 inclusive.
         *
         * @author Ryan Budney
         */
        int orderedS5Index() const;

        /**
         * Returns the index of this permutation in the NPerm5::orderedS5 array.
         * This is a dimension-agnostic alias for orderedS5Index().
         *
         * @return the index \a i for which this permutation is equal to
         * NPerm5::orderedS5[i].  This will be between 0 and 119 inclusive.
         */
        int orderedSnIndex() const;

    private:
        /**
         * Creates a permutation from the given internal code.
         *
         * \pre the given code is a valid permutation code; see
         * isPermCode() for details.
         *
         * @param newCode the internal code from which the new
         * permutation will be created.
         */
        NPerm5(unsigned newCode);

        /**
         * Determines the image of the given integer under this
         * permutation.
         *
         * @param source the integer whose image we wish to find.  This
         * should be between 0 and 4 inclusive.
         * @return the image of \a source.
         */
        int imageOf(int source) const;

    friend std::ostream& operator << (std::ostream& out, const NPerm5& p);
};

/**
 * Writes a string representation of the given permutation to the given
 * output stream.  The format will be the same as is used by
 * NPerm5::str().
 *
 * @param out the output stream to which to write.
 * @param p the permutation to write.
 * @return a reference to \a out.
 */
inline REGINA_API std::ostream& operator << (std::ostream& out,
        const NPerm5& p) {
    return (out << p.str());
}

/*@}*/

// Inline functions for NPerm5

inline NPerm5::NPerm5() : code(18056) {
}

inline NPerm5::NPerm5(unsigned newCode) : code(newCode) {
}

inline NPerm5::NPerm5(int a, int b) {
    code = 18056;
    code += ((a << (3*b)) - (b << (3*b)));
    code += ((b << (3*a)) - (a << (3*a)));
}

inline NPerm5::NPerm5(int a, int b, int c, int d, int e) {
    code = (e << 12) | (d << 9) | (c << 6) | (b << 3) | a;
}

inline NPerm5::NPerm5(const int* image) {
    code = (image[4] << 12) | (image[3] << 9) | (image[2] << 6) |
        (image[1] << 3) | image[0];
}

inline NPerm5::NPerm5(int a0, int a1, int b0, int b1,
        int c0, int c1, int d0, int d1, int e0, int e1) {
    code =
        (a1 << (3*a0)) |
        (b1 << (3*b0)) |
        (c1 << (3*c0)) |
        (d1 << (3*d0)) |
        (e1 << (3*e0));
}

inline NPerm5::NPerm5(const NPerm5& cloneMe) : code(cloneMe.code) {
}

inline unsigned NPerm5::getPermCode() const {
    return code;
}

inline void NPerm5::setPermCode(unsigned newCode) {
    code = newCode;
}

inline NPerm5 NPerm5::fromPermCode(unsigned newCode) {
    return NPerm5(newCode);
}

inline NPerm5& NPerm5::operator = (const NPerm5& cloneMe) {
    code = cloneMe.code;
    return *this;
}

inline NPerm5 NPerm5::operator *(const NPerm5& q) const {
    return NPerm5(imageOf(q[0]), imageOf(q[1]), imageOf(q[2]),
        imageOf(q[3]), imageOf(q[4]));
}

inline NPerm5 NPerm5::inverse() const {
    // Specify the inverse by its internal code.
    return NPerm5(static_cast<unsigned>(
        (1 << (3*imageOf(1))) |
        (2 << (3*imageOf(2))) |
        (3 << (3*imageOf(3))) |
        (4 << (3*imageOf(4)))));
}

inline int NPerm5::operator[](int source) const {
    return (code >> (3*source)) & 7;
}

inline int NPerm5::preImageOf(int image) const {
    if (( code       & 7) == static_cast<unsigned>(image)) return 0;
    if (((code >> 3) & 7) == static_cast<unsigned>(image)) return 1;
    if (((code >> 6) & 7) == static_cast<unsigned>(image)) return 2;
    if (((code >> 9) & 7) == static_cast<unsigned>(image)) return 3;
    return 4;
}

inline bool NPerm5::operator == (const NPerm5& other) const {
    return (code == other.code);
}

inline bool NPerm5::operator != (const NPerm5& other) const {
    return (code != other.code);
}

inline std::string NPerm5::toString() const {
    return str();
}

inline bool NPerm5::isIdentity() const {
    return (code == 18056);
}

inline int NPerm5::imageOf(int source) const {
    return (code >> (3*source)) & 7;
}

inline int NPerm5::SnIndex() const {
    return S5Index();
}

inline int NPerm5::orderedSnIndex() const {
    return orderedS5Index();
}

} // namespace regina

#endif

