/*
*  RAL -- Rubrica Addressbook Library
*  file: groups.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <time.h>
#include <glib.h>
#include <glib-object.h>

#include "group.h"
#include "utils.h"


enum { 
  PROP_0,
  GROUP_ID, 
  GROUP_NAME, 
  GROUP_LABEL,
  GROUP_OWNER, 
  GROUP_PIXMAP, 
  GROUP_ENABLED 
};


static gint id = 0;

struct _RGroupPrivate {
  gint   id;

  gchar* name;
  gchar* label;
  gchar* owner;
  gchar* pixmap;
  gboolean enabled;

  gboolean dispose_has_run;
};


#define R_GROUP_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE((obj),   \
                                  R_GROUP_TYPE, RGroupPrivate))


static void r_group_class_init (RGroupClass* klass);
static void r_group_init       (RGroup* obj);

static void r_group_dispose    (RGroup* obj);
static void r_group_finalize   (RGroup* obj);


static void r_group_set_property (GObject* obj, guint property_id, 
				  const GValue* value, GParamSpec* spec);
static void r_group_get_property (GObject* obj, guint property_id,
				  GValue* value, GParamSpec* spec);


GType
r_group_get_type()
{
  static GType group_type = 0;
  
  if (!group_type)
    {
      static const GTypeInfo group_info =
	{
	  sizeof(RGroupClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_group_class_init,
	  NULL,
	  NULL,
	  sizeof(RGroup),
	  0,
	  (GInstanceInitFunc) r_group_init
	};

      group_type = g_type_register_static (G_TYPE_OBJECT, "RGroup",
					   &group_info, 0);
    }
  
  return group_type;
}


static void
r_group_class_init(RGroupClass* klass)
{
  GObjectClass *class; 
  GParamSpec* pspec;

  class           = G_OBJECT_CLASS (klass);
  class->dispose  = (GObjectFinalizeFunc) r_group_dispose;
  class->finalize = (GObjectFinalizeFunc) r_group_finalize;

  class->set_property = r_group_set_property;
  class->get_property = r_group_get_property;

  g_type_class_add_private (klass, sizeof(RGroupPrivate));

  /* class property 
   */
  pspec = g_param_spec_int("id",
			   "group's id",
			   "the id of the groups",
			   G_MINLONG,
			   G_MAXLONG,
			   0,
			   G_PARAM_READABLE);
  g_object_class_install_property(class, GROUP_ID, pspec);

  /**
   * RGroup:group-name
   *
   * the name of the group
   */
  pspec = g_param_spec_string("group-name",  
			      "group name",  
			      "the name of the group",   
			      NULL,     
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);  
  g_object_class_install_property(class, GROUP_NAME, pspec);    
  
  
  /**
   * RGroup:group-label
   *
   * the name of the group
   */
  pspec = g_param_spec_string("group-label",  
			      "group label",  
			      "the label of the group",   
			      NULL,     
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);  
  g_object_class_install_property(class, GROUP_LABEL, pspec);    


  /**
   * RGroup:group-owner
   *
   * the owner (creator) of the group. It can have two values:
   * "r" for the default groups, and "user" for the
   * user's defined groups
   */
  pspec = g_param_spec_string("group-owner",  
			      "group's owner",  
			      "the owner of the group. It can get "
			      "two values: \"rubrica\", if the owner of "
			      "the group is rubrica, \"user\" if the "
			      "group's owner is defined by user", 
			      "rubrica",     /*  default */
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);  
  g_object_class_install_property(class, GROUP_OWNER, pspec);  
  
  /**
   * RGroup:group-pixmap
   *
   * a pixmap for the group
   */
  pspec = g_param_spec_string("group-pixmap",  
			      "group's pixmap",  
			      "the path or uri to the pixmap that will be" 
			      "shown near group's name",   
			      NULL,     
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);  
  g_object_class_install_property(class, GROUP_PIXMAP, pspec);  
  
  /**
   * RGroup:enabled
   *
   * enabled/disabled
   */
  pspec = g_param_spec_boolean("enabled",  
			       "enabled flag",  
			       "check if group is enabled",   
			       0,     
			       G_PARAM_CONSTRUCT | G_PARAM_READWRITE);  
  g_object_class_install_property(class, GROUP_ENABLED, pspec);   
}


static void
r_group_init(RGroup* self)
{
  self->priv = R_GROUP_GET_PRIVATE(self);
  
  self->priv->id      = 0;
  self->priv->name    = NULL;
  self->priv->label   = NULL;
  self->priv->owner   = NULL;
  self->priv->pixmap  = NULL;
  self->priv->enabled = FALSE;

  self->priv->dispose_has_run = FALSE;
}



static void 
r_group_dispose (RGroup* self)
{
  g_return_if_fail(IS_R_GROUP(self));
  
  if (self->priv->dispose_has_run)
    return;
  
  g_free(self->priv->name);
  g_free(self->priv->label);
  g_free(self->priv->owner);
  g_free(self->priv->pixmap);

  self->priv->dispose_has_run = TRUE;
}


static void 
r_group_finalize (RGroup* self)
{
  g_return_if_fail(IS_R_GROUP(self));
}



static void 
r_group_set_property (GObject* obj, guint property_id,
		      const GValue* value, GParamSpec* spec)
{
  RGroup* self = (RGroup*) obj;
  
  switch (property_id) 
    {
    case GROUP_ID:
      break;

    case GROUP_NAME: 
      g_free(self->priv->name);
      self->priv->name = g_value_dup_string(value); 
      break; 
      
    case GROUP_LABEL: 
      g_free(self->priv->label);
      self->priv->label = g_value_dup_string(value); 
      break;       

    case GROUP_OWNER:
      g_free(self->priv->owner); 
      self->priv->owner = g_value_dup_string(value); 
      break;
      
    case GROUP_PIXMAP:
      g_free(self->priv->pixmap); 
      self->priv->pixmap = g_value_dup_string(value); 
      break;
      
    case GROUP_ENABLED:
      self->priv->enabled = g_value_get_boolean(value); 
      break; 
      
    default:
      break;  
    }  
}


static void 
r_group_get_property (GObject* obj, guint property_id,
		      GValue* value, GParamSpec* spec)
{
  RGroup* self = (RGroup*) obj;
  
  switch (property_id) 
    {
    case GROUP_ID:
      g_value_set_int(value, self->priv->id);
      break;

    case GROUP_NAME: 
      g_value_set_string(value, self->priv->name); 
      break; 
      
    case GROUP_LABEL: 
      g_value_set_string(value, self->priv->label); 
      break; 

    case GROUP_OWNER:
      g_value_set_string(value, self->priv->owner); 
      break;
      
    case GROUP_PIXMAP:
      g_value_set_string(value, self->priv->pixmap); 
      break;
      
    case GROUP_ENABLED:
      g_value_set_boolean(value, self->priv->enabled); 
      break; 
      
    default:
      break;  
    }  
}



/*    *************************************  Public
*/

/**
 * r_group_new
 *
 * create a new #RGroup
 *
 * returns: a RGroup*
 */
RGroup*
r_group_new(void)
{
  RGroup* group;

  group = g_object_new(r_group_get_type(), NULL);
  
  group->priv->id = ++id;
  
  return group;
}


/**
 * r_group_free
 * @group: a #RGroup
 *
 * free memory
 */
void
r_group_free(RGroup* group)
{
  g_return_if_fail(IS_R_GROUP(group));

  g_object_unref(group);   
}


/**
 * r_group_copy
 * @group: a #RGroup 
 *
 * copy the given group
 *
 * returns: a new allocated RGroup*
 */
RGroup* 
r_group_copy (RGroup* group)
{
  RGroup* new;
  gchar *name, *label, *owner, *pixmap;
  gboolean enabled;

  g_return_val_if_fail(IS_R_GROUP(group), NULL);

  new = r_group_new();
  
  g_object_get(G_OBJECT(group), "group-name", &name, "group-label", &label,
	       "group-owner", &owner, "group-pixmap", &pixmap, 
	       "enabled", &enabled, NULL);

  g_object_set(G_OBJECT(new), "group-name", name, "group-label", label,
	       "group-owner", owner, "group-pixmap", pixmap, 
	       "enabled", enabled, NULL);


  return new;
}


/**
 * r_group_rename
 * @group: a #RGroup
 * @newname: a gchar
 *
 * rename the given group.
 *
 * returns: a gboolean. %TRUE if group has been renamed, %FALSE otherwise.
 */
gboolean   
r_group_rename (RGroup* group, gchar* newname)
{
  g_return_val_if_fail(IS_R_GROUP(group), FALSE); 
  g_return_val_if_fail(newname != NULL, FALSE); 

  g_object_set(group, "group-name", newname, NULL);
  
  return TRUE;
}


/**
 * r_group_change_pixmap
 * @group: a #RGroup
 * @newpix: a gchar*
 *
 * replace the pixmap's path with the new one
 *
 * returns: a gboolean. %TRUE if pixmap's path has been changed, 
 * %FALSE otherwise.
 */
gboolean   
r_group_change_pixmap (RGroup* group, gchar* newpix)
{
  g_return_val_if_fail(IS_R_GROUP(group), FALSE); 
  g_return_val_if_fail(newpix != NULL, FALSE); 

  g_object_set(group, "group-pixmap", newpix, NULL);
 
  return TRUE;
}


/**
 * r_group_is
 * @group: a #RGroup
 * @name: a gchar*
 *
 * test if the group has the given name
 *
 * returns: a gboolean. %TRUE if group's name and name coincide, 
 * %FALSE otherwise.
 */
gboolean 
r_group_has_name (RGroup* group, gchar* name)
{
  g_return_val_if_fail(IS_R_GROUP(group), FALSE); 
  g_return_val_if_fail(name != NULL, FALSE); 

  return (g_ascii_strcasecmp(group->priv->name, name) == 0); 
}


/**
 * r_group_has_owner
 * @group: a #RGroup
 * @owner: a gchar*
 *
 * check group's owner
 *
 * Returns: %TRUE if "owner" is the group's owner, %FALSE otherwise
 */
gboolean 
r_group_has_owner (RGroup* group, const gchar* owner)
{
  g_return_val_if_fail(IS_R_GROUP(group), FALSE); 
  g_return_val_if_fail(owner != NULL, FALSE); 
  
  return (g_ascii_strcasecmp(group->priv->owner, owner) == 0);   
}
