/*
*  RAL -- Rubrica Addressbook Library
*  file: ref.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gconf/gconf-client.h>
#include <glib/gi18n-lib.h>
#include <gdk/gdkkeysyms.h>
#include <sys/types.h>
#include <unistd.h>

#include "libral.h"

#include "data_view.h"
#include "app.h"
#include "types.h"
#include "themes.h"
#include "preferences.h"

static GdkCursor* hand_cursor;
static GdkCursor* regular_cursor;
static GdkCursor* wait_cursor;      

static gboolean   over_link;


enum {
  PROP_0,
  DATA_FONT,
};

struct _RubricaDataViewPrivate {
  GtkWidget* text1;      // header
  GtkWidget* text2;      // body

  GtkTextBuffer* buf1;   // header's buffer
  GtkTextBuffer* buf2;   // body's buffer
  
  GtkIconTheme*  theme;  
  GtkWidget*     image;

  gchar* font;

  gboolean dispose_has_run;
};


/*    signals enumeration 
 */
enum {
  LAUNCH,
  LAST_SIGNAL   
};

static guint data_view_signals[LAST_SIGNAL] = {0};

#define RUBRICA_DATA_VIEW_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o), \
                                          RUBRICA_DATA_VIEW_TYPE,          \
                                          RubricaDataViewPrivate))


static GObjectClass *parent_class = NULL;


static void rubrica_data_view_class_init   (RubricaDataViewClass* klass);
static void rubrica_data_view_init         (RubricaDataView* obj);

static void rubrica_data_view_finalize     (RubricaDataView* self);
static void rubrica_data_view_dispose      (RubricaDataView* self);


static void rubrica_data_view_set_property (GObject* obj, guint property_id,
					    GValue* value, GParamSpec* spec);
static void rubrica_data_view_get_property (GObject* obj, guint property_id,
					    GValue* value, GParamSpec* spec);


/*  Private
*/  
static GtkTextTagTable* rubrica_data_view_create_tag_table(void);


static void     insert_link             (GtkTextBuffer *buffer, 
					 GtkTextIter *iter, gchar *uri, 
					 RubricaLinkType type);
static void     follow_if_link          (GtkWidget *view, GtkTextIter *iter, 
					 gpointer data);
static void     set_cursor              (GtkWidget* view, gint x, gint y);


static void clean_buffer                (GtkTextBuffer* buffer);
static void data_view_clear_header      (RubricaDataView* view);
static void data_view_clear_body        (RubricaDataView* view);

static void data_view_set_image         (RubricaDataView* view, 
					 GdkPixbuf* pixbuf);

static void data_view_write_card_header (GtkTextBuffer* buffer, gchar* label,
					 gchar* string);
static void data_view_write_pixbuf      (GtkTextBuffer* buffer, 
					 GdkPixbuf* pixbuf, gchar* label);
static void data_view_write_bold        (GtkTextBuffer* buffer, gchar* label, 
					 gchar* string, gboolean bool);
static void data_view_write_line        (GtkTextBuffer* buffer, gchar* label, 
					 gchar* string, gboolean tab);
static void data_view_write_link        (GtkTextBuffer* buffer, gchar* label,
					 gchar* url, RubricaLinkType link);
static void data_view_write             (GtkTextBuffer* buffer, gchar* str);

static void write_company_info  (RubricaDataView* view, RCard* card);
static void write_personal_data (RubricaDataView* view, RPersonalCard* card);
static void write_groups        (RubricaDataView* view, RCard* card);
static void write_addresses     (RubricaDataView* view, RCard* card);
static void write_work          (RubricaDataView* view, RPersonalCard* card);
static void write_net           (RubricaDataView* view, RCard* card);
static void write_telephones    (RubricaDataView* view, RCard* card);
static void write_notes         (RubricaDataView* view, RPersonalCard* card);
static void write_company_notes (RubricaDataView* view, RCompanyCard* card);

static void data_view_show_personal_card (RubricaDataView* view, RCard* card);
static void data_view_show_company_card  (RubricaDataView* view, RCard* card);


/*  Callbacks
*/
static gboolean key_press_event         (GtkWidget *text_view, 
					 GdkEventKey *event, gpointer data);
static gboolean event_after             (GtkWidget *text_view, GdkEvent  *ev,
					 gpointer data);
static gboolean motion_notify_event     (GtkWidget* view, 
					 GdkEventMotion *event, gpointer data);
static gboolean visibility_notify_event (GtkWidget* view, 
					 GdkEventVisibility *event, 
					 gpointer data);


void change_font (GtkTextTag *tag, gpointer data);


void 
change_font (GtkTextTag *tag, gpointer data)
{
  PangoFontDescription* pfd;
  const gchar* font = (gchar*) data;
  gchar* name;

  pfd = pango_font_description_from_string (font); 

  g_object_get(tag, "name", &name, NULL);
  if (g_strrstr(name, "bold"))
    g_object_set(tag, "font-desc", pfd, "weight", PANGO_WEIGHT_BOLD, NULL);
  else
    g_object_set(tag, "font-desc", pfd, NULL);
  
  pango_font_description_free (pfd);  
}

GType
rubrica_data_view_get_type()
{
  static GType data_view_type = 0;
  
  if (!data_view_type)
    {
      static const GTypeInfo data_view_info =
	{
	  sizeof(RubricaDataViewClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_data_view_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaDataView),
	  0,
	  (GInstanceInitFunc) rubrica_data_view_init
	};

      data_view_type = g_type_register_static (GTK_TYPE_VBOX, 
					       "RubricaDataView", 
					       &data_view_info, 0);
    }
  
  return data_view_type;
}


static void 
rubrica_data_view_dispose (RubricaDataView* self)
{
  g_return_if_fail(IS_RUBRICA_DATA_VIEW(self));
  
  if (self->priv->dispose_has_run)
    return;

  self->priv->dispose_has_run = TRUE;  
}


static void 
rubrica_data_view_finalize (RubricaDataView* self)
{
  g_return_if_fail(IS_RUBRICA_DATA_VIEW(self));
}


static void
rubrica_data_view_class_init(RubricaDataViewClass* klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  GParamSpec* pspec;
  
  parent_class = g_type_class_peek_parent (klass);

  object_class->finalize = (GObjectFinalizeFunc) rubrica_data_view_finalize;
  object_class->dispose  = (GObjectFinalizeFunc) rubrica_data_view_dispose;

  object_class->set_property = (gpointer) rubrica_data_view_set_property;
  object_class->get_property = (gpointer) rubrica_data_view_get_property;

  g_type_class_add_private (klass, sizeof(RubricaDataViewPrivate));

  /**
   * RubricaDataView:data-font
   *
   * 
   */
  pspec = g_param_spec_string("data-font", 
			      "data font", 
			      "the font used to display data",  
			      NULL,
			      G_PARAM_READWRITE);
  g_object_class_install_property(object_class, DATA_FONT, pspec);

  data_view_signals[LAUNCH] =
    g_signal_new("launch", 
		 RUBRICA_DATA_VIEW_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RubricaDataViewClass, launch),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__POINTER,
		 G_TYPE_NONE,            /* return type */
		 1,                      /* params      */
		 G_TYPE_POINTER);        /* params type: error code */
}


static void
rubrica_data_view_init(RubricaDataView* self)
{
  GtkWidget* scrollwin;
  GtkTextTagTable* tag_table; 
  GtkWidget* vbox;
  GtkWidget* hbox;

  hand_cursor    = gdk_cursor_new (GDK_HAND2); 
  regular_cursor = gdk_cursor_new (GDK_XTERM);
  wait_cursor    = gdk_cursor_new (GDK_WATCH);
  over_link      = FALSE;

  gtk_box_set_homogeneous(GTK_BOX(self), FALSE);
  gtk_box_set_spacing(GTK_BOX(self), 6);
  gtk_container_set_border_width(GTK_CONTAINER(self), 12);
  
  tag_table = rubrica_data_view_create_tag_table(); 
  
  self->priv = RUBRICA_DATA_VIEW_GET_PRIVATE(self);
  
  self->priv->buf1  = gtk_text_buffer_new(tag_table);
  self->priv->buf2  = gtk_text_buffer_new(tag_table);
  self->priv->text1 = gtk_text_view_new_with_buffer(self->priv->buf1);
  self->priv->text2 = gtk_text_view_new_with_buffer(self->priv->buf2);
  
  self->priv->theme = gtk_icon_theme_get_default(); 
  self->priv->image = gtk_image_new();
  self->priv->font  = NULL;

  vbox = gtk_vbox_new(FALSE, 0);
    
  scrollwin = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollwin), vbox); 
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrollwin),
				      GTK_SHADOW_IN);
  gtk_box_pack_start(GTK_BOX(self), scrollwin, TRUE, TRUE, 0);

  hbox = gtk_hbox_new(FALSE, 0);
  
  gtk_box_pack_start(GTK_BOX(hbox), self->priv->image, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), self->priv->text1, TRUE, TRUE, 0);  
  
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), self->priv->text2, TRUE, TRUE, 0);

  g_object_set(self->priv->text1, 
	       "cursor-visible", FALSE,
	       "editable", FALSE, 
	       "right-margin", 6, 
	       "left-margin", 6,
	       "wrap-mode", GTK_WRAP_WORD, NULL);

  g_object_set(self->priv->text2, 
	       "cursor-visible", FALSE,
	       "editable", FALSE, 
	       "right-margin", 6, 
	       "left-margin", 6,
	       "wrap-mode", GTK_WRAP_WORD, NULL);

  g_signal_connect (G_OBJECT (self->priv->text2), "key-press-event", 
		    G_CALLBACK (key_press_event), self);

  g_signal_connect (G_OBJECT (self->priv->text2), "event-after", 
		    G_CALLBACK (event_after), self);

  g_signal_connect (G_OBJECT (self->priv->text2), "motion-notify-event", 
		    G_CALLBACK (motion_notify_event), self);

  g_signal_connect (G_OBJECT (self->priv->text2), "visibility-notify-event", 
		    G_CALLBACK (visibility_notify_event), self);  

  gtk_widget_show(GTK_WIDGET(self));
  gtk_widget_show(self->priv->image);
  gtk_widget_show(self->priv->text1);
  gtk_widget_show(self->priv->text2);
  
  gtk_widget_show(vbox);
  gtk_widget_show(hbox);
  gtk_widget_show(scrollwin);

  self->priv->dispose_has_run = FALSE;
}


static void 
rubrica_data_view_set_property (GObject* obj, guint property_id,
				GValue* value, GParamSpec* spec)
{
  RubricaDataView* self = (RubricaDataView*) obj;
  RubricaDataViewPrivate* priv = RUBRICA_DATA_VIEW_GET_PRIVATE(self);
  GtkTextTagTable* table;

  switch (property_id) 
    {
    case DATA_FONT:
      if (priv->font)
	g_free(priv->font);

      priv->font = g_value_dup_string(value);
      table = gtk_text_buffer_get_tag_table(priv->buf1);
      gtk_text_tag_table_foreach(table, change_font, priv->font);
      break;

    default: 
      break; 
    } 
} 
 

static void 
rubrica_data_view_get_property (GObject* obj, guint property_id,
				GValue* value, GParamSpec* spec)
{
  RubricaDataView* self = (RubricaDataView*) obj;
  RubricaDataViewPrivate* priv = RUBRICA_DATA_VIEW_GET_PRIVATE(self);

  switch (property_id) 
    {     
    case DATA_FONT:
      g_value_set_string(value, priv->font);
      break;

     default:
      break;  
    }  
}



/*   Private
*/
static GtkTextTagTable* 
rubrica_data_view_create_tag_table(void)
{
  GtkTextTag* tag;
  GtkTextTagTable* tag_table; 
  PangoFontDescription* pfd;
  GConfClient* client;
  const gchar* font = NULL;

  client = gconf_client_get_default();
  font   = gconf_client_get_string(client, RUBRICA_GCONF_FONT, NULL);

  tag_table = gtk_text_tag_table_new();

  tag = gtk_text_tag_new("tabulato");
  g_object_set(tag, "left-margin", 25, "right-margin", 25, NULL);
  gtk_text_tag_table_add(tag_table, tag);

  if (font)
    pfd = pango_font_description_from_string (font); 
  else
    pfd = pango_font_description_from_string ("Serif 10");  
 
  tag = gtk_text_tag_new("header");
  g_object_set(tag, "font-desc", pfd, NULL);
  gtk_text_tag_table_add(tag_table, tag);  

  tag = gtk_text_tag_new("header-bold");
  g_object_set(tag, "font-desc", pfd, "weight", PANGO_WEIGHT_BOLD, NULL);
  gtk_text_tag_table_add(tag_table, tag);  
   
  tag = gtk_text_tag_new("body");
  g_object_set(tag, "font-desc", pfd, NULL);
  gtk_text_tag_table_add(tag_table, tag);

  tag = gtk_text_tag_new("body-bold");
  g_object_set(tag, "font-desc", pfd, "weight", PANGO_WEIGHT_BOLD, NULL);
  gtk_text_tag_table_add(tag_table, tag);
  pango_font_description_free (pfd);  

  g_object_unref(client);    

  return tag_table;
}

/* Looks at all tags covering the position (x, y) in the text view, 
 * and if one of them is a link, change the cursor to the "hands" cursor
 * typically used by web browsers.
 */
static void
set_cursor (GtkWidget* text_view, gint x, gint y)
{
  GtkTextIter iter;
  GSList *tags = NULL, *tagp = NULL;
  gboolean hovering = FALSE;

  gtk_text_view_get_iter_at_location (GTK_TEXT_VIEW(text_view), &iter, x, y);
  
  tags = gtk_text_iter_get_tags (&iter);
  for (tagp = tags;  tagp != NULL;  tagp = tagp->next)
    {
      GtkTextTag *tag = tagp->data;
      RubricaLink* link = NULL;
      	
      link = g_object_get_data (G_OBJECT (tag), "link");
      if (link) 
        {
          hovering = TRUE;
          break;
        }
    }

  if (hovering != over_link)
    {
      GdkWindow* window;

      over_link = hovering;
      
      window = gtk_text_view_get_window(GTK_TEXT_VIEW(text_view), 
					GTK_TEXT_WINDOW_TEXT);

      if (over_link)
        gdk_window_set_cursor(window, hand_cursor);
      else
        gdk_window_set_cursor(window, regular_cursor);
    }

  if (tags) 
    g_slist_free (tags);
}


static void 
insert_link (GtkTextBuffer *buffer, GtkTextIter *iter, gchar *uri, 
	     RubricaLinkType type)
{
  GtkTextTag *tag;
  RubricaLink* link;
  
  tag = gtk_text_buffer_create_tag (buffer, NULL, 
				    "foreground", "blue", 
				    "underline", PANGO_UNDERLINE_SINGLE, 
				    NULL);

  link = g_new0(RubricaLink, 1);
  link->uri  = g_strdup(uri);
  link->type = type;
  g_object_set_data (G_OBJECT (tag), "link", link);

  gtk_text_buffer_insert_with_tags (buffer, iter, uri, -1, tag, NULL);
  gtk_text_buffer_insert_with_tags (buffer, iter, "\n", -1, tag, NULL);
}



/* Looks at all tags covering the position of iter in the text view, 
 * and if one of them is a link, follow it by showing the page identified
 * by the data attached to it.
 */
static void
follow_if_link (GtkWidget *text_view, GtkTextIter *iter, gpointer data)
{
  RubricaDataView *view = (RubricaDataView*) data;
  GSList *tags = NULL;
  GSList *tagp = NULL;

  tags = gtk_text_iter_get_tags (iter);
  for (tagp = tags;  tagp != NULL;  tagp = tagp->next) 
    {
      RubricaLink* link;
      GtkTextTag *tag = tagp->data; 

      link = g_object_get_data (G_OBJECT (tag), "link"); 

      g_signal_emit_by_name(view, "launch", link, G_TYPE_POINTER);
    }

  if (tags)  
    g_slist_free (tags);
}


/* Links can be activated by pressing Enter.
 */
static gboolean 
key_press_event (GtkWidget *text_view, GdkEventKey *event, gpointer data) 
{
  GtkTextIter iter;
  GtkTextBuffer *buffer;

  switch (event->keyval)
    {
      case GDK_Return: 
      case GDK_KP_Enter:
        buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text_view));
        gtk_text_buffer_get_iter_at_mark (buffer, &iter, 
                                          gtk_text_buffer_get_insert (buffer));
        follow_if_link (text_view, &iter, data);
        break;

      default:
        break;
    }

  return FALSE;
}


/* Links can also be activated by clicking.
 */
static gboolean
event_after (GtkWidget *text_view, GdkEvent  *ev, gpointer data)
{
  GtkTextIter start, end, iter;
  GtkTextBuffer *buffer;
  GdkEventButton *event;
  gint x, y;

  if (ev->type != GDK_BUTTON_RELEASE)
    return FALSE;

  event = (GdkEventButton *)ev;

  if (event->button != 1)
    return FALSE;

  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text_view));

  /* we shouldn't follow a link if the user has selected something */
  gtk_text_buffer_get_selection_bounds (buffer, &start, &end);
  if (gtk_text_iter_get_offset (&start) != gtk_text_iter_get_offset (&end))
    return FALSE;

  gtk_text_view_window_to_buffer_coords (GTK_TEXT_VIEW (text_view), 
                                         GTK_TEXT_WINDOW_WIDGET,
                                         event->x, event->y, &x, &y);

  gtk_text_view_get_iter_at_location (GTK_TEXT_VIEW (text_view), &iter, x, y);

  follow_if_link (text_view, &iter, data);

  return FALSE;
}



/* Update the cursor image if the pointer moved. 
 */
static gboolean
motion_notify_event (GtkWidget* text_view, GdkEventMotion *event, 
		     gpointer data)
{
  RubricaDataView* view = (RubricaDataView*) data;
  RubricaDataViewPrivate* priv;
  gint x, y;

  priv = RUBRICA_DATA_VIEW_GET_PRIVATE(view);

  gtk_text_view_window_to_buffer_coords (GTK_TEXT_VIEW (text_view), 
                                         GTK_TEXT_WINDOW_WIDGET,
                                         event->x, event->y, &x, &y);

  set_cursor (text_view, x, y);

  gdk_window_get_pointer (text_view->window, NULL, NULL, NULL);

  return FALSE;
}


/* Also update the cursor image if the window becomes visible
 * (e.g. when a window covering it got iconified).
 */
static gboolean
visibility_notify_event (GtkWidget* text_view, GdkEventVisibility *event, 
			 gpointer data)
{
  RubricaDataViewPrivate* priv = RUBRICA_DATA_VIEW_GET_PRIVATE(data);
  gint wx, wy, bx, by;
   
  gdk_window_get_pointer (priv->text2->window, &wx, &wy, NULL);
  
  gtk_text_view_window_to_buffer_coords (GTK_TEXT_VIEW (priv->text2), 
                                         GTK_TEXT_WINDOW_WIDGET,
                                         wx, wy, &bx, &by);

  set_cursor (text_view, bx, by);

  return FALSE;
}


static void 
clean_buffer(GtkTextBuffer* buffer)
{
  GtkTextIter start, end;
  
  gtk_text_buffer_get_bounds(buffer, &start, &end);
  gtk_text_buffer_delete(buffer, &start, &end);
  gtk_text_buffer_get_iter_at_offset(buffer, &start, 0);  
}


static void 
data_view_clear_header(RubricaDataView* view)
{
  RubricaDataViewPrivate* priv = RUBRICA_DATA_VIEW_GET_PRIVATE(view);
  
  clean_buffer(priv->buf1);
}


static void 
data_view_clear_body(RubricaDataView* view)
{
  RubricaDataViewPrivate* priv = RUBRICA_DATA_VIEW_GET_PRIVATE(view);
  
  clean_buffer(priv->buf2);
}


static void 
data_view_set_image(RubricaDataView* view, GdkPixbuf* pixbuf)
{
  RubricaDataViewPrivate* priv;

  g_return_if_fail(IS_RUBRICA_DATA_VIEW(view));
  g_return_if_fail(pixbuf != NULL);

  priv = RUBRICA_DATA_VIEW_GET_PRIVATE(view);

  gtk_image_set_from_pixbuf(GTK_IMAGE(priv->image), pixbuf);
  gdk_pixbuf_unref(pixbuf);
}


static void 
data_view_write_card_header(GtkTextBuffer* buffer, gchar* label, gchar* string)
{
  GtkTextIter iter;

  g_return_if_fail(label != NULL);
  g_return_if_fail(string != NULL);

  gtk_text_buffer_get_end_iter (buffer, &iter);
  
  gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, label, -1, 
					    "header-bold", NULL);
  gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, string, -1, 
					    "header", NULL);
  gtk_text_buffer_insert(buffer, &iter, "\n", -1);
}


static void 
data_view_write_pixbuf(GtkTextBuffer* buffer, GdkPixbuf* pixbuf, gchar* label)
{
  GtkTextIter iter;
  
  gtk_text_buffer_get_end_iter (buffer, &iter);
  
  gtk_text_buffer_insert_pixbuf(buffer, &iter, pixbuf);      
  gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, label, -1, 
					    "body-bold", NULL);  
  gtk_text_buffer_insert(buffer, &iter, "\n", -1);
}


static void 
data_view_write_bold(GtkTextBuffer* buffer, gchar* label, gchar* string,
		     gboolean bool)
{
  GtkTextIter iter;
  
  gtk_text_buffer_get_end_iter (buffer, &iter);
  
  if (bool)
    gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, label, -1,
					      "body-bold", "tabulato", NULL);  
  else
    gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, label, -1,
					      "body-bold", NULL);    
  
  gtk_text_buffer_insert(buffer, &iter, string, -1);
  gtk_text_buffer_insert(buffer, &iter, "\n", -1);
}


static void 
data_view_write_line(GtkTextBuffer* buffer, gchar* label, 
		     gchar* string, gboolean tab)
{
  if (string && (g_ascii_strcasecmp(string, "") != 0))
    {
      gchar* lbl;

      lbl = g_strdup_printf("%s: ", label);
      
      data_view_write_bold(buffer, lbl, string, tab);  
      
      g_free(lbl);     
      g_free(string);      
    }
}


static void 
data_view_write_link(GtkTextBuffer* buffer, gchar* label, gchar* url, 
		     RubricaLinkType link)
{
  GtkTextIter iter;
  gchar* str;
  
  str = g_strdup_printf("%s: ", label);
  
  gtk_text_buffer_get_end_iter (buffer, &iter);
    
  gtk_text_buffer_insert_with_tags_by_name (buffer, &iter, str, -1, 
					    "body-bold", "tabulato", NULL);
  insert_link(buffer, &iter, url, link);
}


static void 
data_view_write(GtkTextBuffer* buffer, gchar* str)
{
  GtkTextIter iter;
  
  g_return_if_fail(str != NULL);

  gtk_text_buffer_get_end_iter (buffer, &iter);
  gtk_text_buffer_insert_with_tags_by_name(buffer, &iter, str, -1, 
					   "body", "tabulato", NULL);
}


static void 
write_company_info(RubricaDataView* view, RCard* card)
{
  RubricaDataViewPrivate* priv;
  GtkTextBuffer* header;
  GdkPixbuf* pixbuf = NULL;   
  gchar *name, *logo, *vat;
  gchar *label, *string;
    
  priv = RUBRICA_DATA_VIEW_GET_PRIVATE(view);
  header = priv->buf1;

  g_object_get(card, "card-name", &name, "company-logo", &logo, 
	       "company-vat", &vat, NULL);
  
  if (g_file_test(logo, G_FILE_TEST_EXISTS))
    pixbuf = gdk_pixbuf_new_from_file_at_size(logo, 128, 128, NULL);
  else
    pixbuf = gtk_icon_theme_load_icon(priv->theme, "card_photo", 128, 0, NULL);

  data_view_set_image(view, pixbuf);

  label  = g_strdup_printf("%s: ", _("Card"));  
  string = g_strdup_printf("%s\n\n", name);  
  data_view_write_card_header(header, label, string);
  g_free(label);
  g_free(string);  

  if (vat && (g_ascii_strcasecmp(vat, "") != 0))
    {
      string = g_strdup_printf("%s: ", _("Taxpayer number/VAT"));
      data_view_write_bold(header, string, vat, FALSE);  
      
      g_free(string);
    }
}


static void 
write_personal_data(RubricaDataView* view, RPersonalCard* card)
{
  RubricaDataViewPrivate* priv;
  RDate* date; 
  GtkTextBuffer* header;
  GdkPixbuf* pixbuf = NULL;  
  RContact* contact;
  gchar* card_name;
  gchar *first, *last, *middle, *nick, *title, *prefix, *prof, *photo;
  gchar *label, *string;
  
  priv    = RUBRICA_DATA_VIEW_GET_PRIVATE(view);
  header  = priv->buf1;
  contact = r_personal_card_get_contact(R_PERSONAL_CARD(card));

  g_object_get(contact, 
	       "first-name",  &first,  "last-name", &last, 
	       "middle-name", &middle, "nick-name", &nick, 
	       "title",       &title,  "prefix",    &prefix, 
	       "profession",  &prof,   "photo",     &photo, NULL);
  g_object_get(R_CARD(card), "card-name", &card_name, NULL);

  /* display photo */
  if (g_file_test(photo, G_FILE_TEST_EXISTS))
    pixbuf = gdk_pixbuf_new_from_file_at_size(photo, 128, 128, NULL);
  else
    pixbuf = gtk_icon_theme_load_icon(priv->theme, "card_photo", 128, 0, NULL);
  data_view_set_image(view, pixbuf);

  /* display card's name */
  label  = g_strdup_printf("%s: ", _("Card"));  
  string = g_strdup_printf("%s\n", card_name);
  data_view_write_card_header(header, label, string);
  g_free(label);
  g_free(string);
  
  /* display contact's data */  
  if (first || middle || last)
    {
      gchar *str;

      label = g_strdup_printf("%s: ", _("Name"));
      str   = g_strdup_printf("%s %s %s", 
			      last   ? last   : "", 
			      first  ? first  : "", 
			      middle ? middle : "");
      data_view_write_bold(header, label, str, FALSE);
      
      g_free(last);
      g_free(first);
      g_free(middle);
      g_free(label);
      g_free(str);
    }

  data_view_write_line(header, _("Nickname"),   nick,   FALSE);
  data_view_write_line(header, _("Title"),      title,  FALSE);
  data_view_write_line(header, _("Prefix"),     prefix, FALSE );
  data_view_write_line(header, _("Profession"), prof,   FALSE);
  
  date = r_contact_get_birthday(contact);
  if (r_date_is_valid(date))
    {
      label = g_strdup_printf("%s: ", _("Birthday"));
      data_view_write_bold(header, label, r_date_get_human_date(date), FALSE);
      
      g_free(label);
    }
}



static void 
write_groups(RubricaDataView* view, RCard* card)
{
  RubricaDataViewPrivate* priv;
  GtkTextBuffer* body;
  gpointer group = NULL;
    
  priv = RUBRICA_DATA_VIEW_GET_PRIVATE(view);
  body = priv->buf2;

  group = r_card_get_group(R_CARD(card));
  if (group)
    {
      data_view_write(body, "\n");
      data_view_write_bold(body, _("Card's groups"), "", FALSE);

      for (; group; group = r_card_get_next_group(R_CARD(card)))
	{
	  gchar *str, *name = NULL; 
	  
	  g_object_get(R_GROUP(group), "group-name",  &name, NULL);
	 
	  str = g_strdup_printf("%s\n", _(name)); 
	  data_view_write(body, str);

	  g_free(str);
	  g_free(name);
	}
    }
}


static void 
write_addresses(RubricaDataView* view, RCard* card)
{
  RubricaDataViewPrivate* priv;
  GtkTextBuffer* body;
  RAddress* address;
  GdkPixbuf* pixbuf = NULL;    

  priv = RUBRICA_DATA_VIEW_GET_PRIVATE(view);
  body = priv->buf2;

  address = r_card_get_address(R_CARD(card));
  if (address)
    {
      pixbuf = gtk_icon_theme_load_icon (priv->theme, "mailbox", 20, 0, NULL);

      data_view_write(body, "\n");
      data_view_write_pixbuf(body, pixbuf, _("Addresses"));      
      gdk_pixbuf_unref(pixbuf);

      for (; address; address = r_card_get_next_address(R_CARD(card)))
	{
	  RAddressType type;
	  gchar *adtype, *street, *number, *zip;
	  gchar *city, *province, *state, *country;
	  
	  g_object_get(address, "address-type", &type, 
		       "street", &street, "street-number", &number, 
		       "zip", &zip, "city", &city,
		       "province", &province, "state", &state, 
		       "country", &country, NULL);
	  
	  if ((street && (g_ascii_strcasecmp(street, "") != 0))   ||
	      (zip && (g_ascii_strcasecmp(zip, "") != 0))         ||
	      (city && (g_ascii_strcasecmp(city, "") != 0))       ||
	      (state && (g_ascii_strcasecmp(state, "") != 0))     ||
	      (country && (g_ascii_strcasecmp(country, "") != 0)))
	    {
	      GString* string;
	      
	      string = g_string_new("");
	      
	      adtype = r_address_lookup_enum2str(type);
	      data_view_write_bold(body, _(adtype), "", TRUE);
	      
	      if (street && (g_ascii_strcasecmp(street, "") != 0))
		{
		  g_string_printf(string, "%s", street);
		  
		  if (number && (g_ascii_strcasecmp(number, "") != 0))
		    g_string_append_printf(string, ", %s", number);
		}
	      
	      if ((zip && (g_ascii_strcasecmp(zip, "") != 0))   ||
		  (city && (g_ascii_strcasecmp(city, "") != 0)) ||
		  (province && (g_ascii_strcasecmp(province, "") != 0)))
		{
		  g_string_append_printf(string, "\n");
		  
		  if (zip && (g_ascii_strcasecmp(zip, "") != 0)) 
		    g_string_append_printf(string, "%s ", zip);
		  
		  if (city && (g_ascii_strcasecmp(city, "") != 0))
		    g_string_append_printf(string, "%s", city);
		  
		  if (province && (g_ascii_strcasecmp(province, "") != 0))
		    g_string_append_printf(string, ", (%s) ", province);
		}

	      if ((state && (g_ascii_strcasecmp(state, "") != 0))   ||
		  (country && (g_ascii_strcasecmp(country, "") != 0)))
		{
		  g_string_append_printf(string, "\n");
		  
		  if (state && (g_ascii_strcasecmp(state, "") != 0))
		    g_string_append_printf(string, "%s ", state);
		  
		  if (country && (g_ascii_strcasecmp(country, "") != 0))
		    g_string_append_printf(string, "%s",country);
		}

	      data_view_write(body, string->str);
	      data_view_write(body, "\n");
	      
	      g_string_free(string, TRUE);     
	    }
	}
    }
}



static void 
write_work(RubricaDataView* view, RPersonalCard* card)
{
  RubricaDataViewPrivate* priv;
  GtkTextBuffer* body;
  RWork* work = NULL;
  GdkPixbuf* pixbuf = NULL;  
  gchar *assignment = NULL, *organization = NULL;
  gchar *department = NULL, *subdep = NULL, *manager = NULL;
  gchar *collaborator = NULL, *mphone = NULL, *cphone = NULL;
  
  priv = RUBRICA_DATA_VIEW_GET_PRIVATE(view);
  body = priv->buf2;
  
  work = r_personal_card_get_work(R_PERSONAL_CARD(card));
  if (work && r_work_have_data(work))
    {
      g_object_get(work, 
		   "assignment",         &assignment, 
		   "organization",       &organization, 
		   "department",         &department,
		   "sub-department",     &subdep, 
		   "manager-name",       &manager,
		   "collaborator",       &collaborator,
		   "manager-phone",      &mphone,
		   "collaborator-phone", &cphone, NULL);
      
      pixbuf = gtk_icon_theme_load_icon (priv->theme, "role", 20, 0, NULL);

      data_view_write(body, "\n");
      data_view_write_pixbuf(body, pixbuf, _("Work"));
      gdk_pixbuf_unref(pixbuf);

      data_view_write_line(body, _("Assignment"), assignment, TRUE);
      data_view_write_line(body, _("Organization"), organization, TRUE);
      data_view_write_line(body, _("Department"), department, TRUE);
      data_view_write_line(body, _("Subdepartment"), subdep, TRUE);
      data_view_write_line(body, _("Manager"), manager, TRUE);
      data_view_write_line(body, _("Manager's telephone"), collaborator, TRUE);
      data_view_write_line(body, _("Collaborator"), mphone, TRUE);
      data_view_write_line(body, _("Collaborator's telephone"), cphone, TRUE);
    }
}



static void 
write_net (RubricaDataView* view, RCard* card)
{
  RubricaDataViewPrivate* priv;
  GtkTextBuffer* body;
  GdkPixbuf* pixbuf = NULL;  
  RNetAddress* net = NULL;
  RNetAddressType type;
  RubricaLinkType linktype = UNKNOWN_LINK;
  gchar* url;
  
  priv = RUBRICA_DATA_VIEW_GET_PRIVATE(view);
  body = priv->buf2;

  net = r_card_get_net_address(R_CARD(card));
  if (net)
    {
      pixbuf = gtk_icon_theme_load_icon (priv->theme, "net", 20, 0, NULL);
      
      data_view_write(body, "\n");
      data_view_write_pixbuf(body, pixbuf, _("Net"));
      gdk_pixbuf_unref(pixbuf);
      
      for (; net; net = r_card_get_next_net_address(R_CARD(card)))
	{	  
	  gchar* ttype;
	  g_object_get(R_NET_ADDRESS(net), 
		       "url", &url, "url-type", &type, NULL);
	  	  
	  switch(type)
	    {
	    case R_NET_ADDRESS_WEB:
	      linktype = WEB_LINK;
	      break;
	      
	    case R_NET_ADDRESS_EMAIL:  
	      linktype = EMAIL_LINK;
	      break;
	      
	    case R_NET_ADDRESS_EKIGA:  
	      linktype = TELEPHONE_LINK;
	      break;
	      
	    case R_NET_ADDRESS_IRC_AIM:  
	      linktype = IRC_LINK;
	      break;
	      
	    case R_NET_ADDRESS_IRC_ICQ:  
	      linktype = IRC_LINK;
	      break;
	      
	    case R_NET_ADDRESS_IRC_JABBER:  
	      linktype = IRC_LINK;
	      break;
	      
	    case R_NET_ADDRESS_IRC_YAHOO:  
	      linktype = IRC_LINK;
	      break;
	      
	    case R_NET_ADDRESS_IRC_MSN:  
	      linktype = IRC_LINK;
	      break;
	      
	      
	    default:
	      break;
	    }

	  ttype = r_net_address_decode_type(type);
	  data_view_write_link(body, _(ttype), url, linktype);
	}
    }
}


static void 
write_telephones(RubricaDataView* view, RCard* card)
{
  RubricaDataViewPrivate* priv;
  GtkTextBuffer* body;
  RTelephone* tel = NULL;
  GdkPixbuf* pixbuf = NULL;  

  priv = RUBRICA_DATA_VIEW_GET_PRIVATE(view);
  body = priv->buf2;

  tel = r_card_get_telephone(R_CARD(card));
  if (tel)
    {
      pixbuf = gtk_icon_theme_load_icon (priv->theme, "phone", 20, 0, NULL);

      data_view_write(body, "\n");
      data_view_write_pixbuf(body, pixbuf, _("Telephone"));
      gdk_pixbuf_unref(pixbuf);

      for (; tel; tel = r_card_get_next_telephone(R_CARD(card)))
	{
	  RTelephoneType ttype;
	  gchar* num;
	  
	  g_object_get(R_TELEPHONE(tel), 
		       "telephone-number", &num,
		       "telephone-type",   &ttype, NULL);

	  data_view_write_link(body, _(r_telephone_lookup_enum2str(ttype)),
			       num, TELEPHONE_LINK);
	  
	  g_free(num);    	
	}
    }
}



static void
write_notes(RubricaDataView* view, RPersonalCard* card)
{
  RubricaDataViewPrivate* priv;
  GtkTextBuffer* body;
  RNotes* notes;
  GdkPixbuf* pixbuf = NULL;  
  gchar *partner, *other_notes, *pubkey;
  gboolean has_partner;

  priv  = RUBRICA_DATA_VIEW_GET_PRIVATE(view);
  body  = priv->buf2;
  notes = r_personal_card_get_notes(R_PERSONAL_CARD(card));

  if (notes && r_notes_have_data(notes))
    {       
      g_object_get(notes, 
		   "has-partner",  &has_partner, 
		   "partner-name", &partner,
		   "other-notes",  &other_notes, 
		   "pubkey",       &pubkey, NULL);

      pixbuf = gtk_icon_theme_load_icon (priv->theme, "notes", 20, 0, NULL);
      
      data_view_write(body, "\n");
      data_view_write_pixbuf(body, pixbuf, _("Notes"));
      gdk_pixbuf_unref(pixbuf);

      if (has_partner)
	{   
	  data_view_write(body, _("Contact has a partner"));
	  data_view_write(body, "\n");
	  
	  data_view_write_line(body, _("partner's name"), partner, TRUE);
	  
	  if (r_notes_know_birthday(notes))
	    {	  
	      data_view_write_line(body, _("partner's birthday"), 
				   r_notes_get_birth_date(notes), TRUE);
	    }
	  
	  if (r_notes_know_anniversary(notes))
	    {
	      data_view_write_line(body, _("anniversary"), 
				   r_notes_get_anniversary_date(notes), TRUE);
	    }
	}
      
      data_view_write_line(body, _("Public key"), pubkey, TRUE);
      data_view_write_line(body, _("Other notes"), other_notes, TRUE);  
    }
}


static void
write_company_notes(RubricaDataView* view, RCompanyCard* card)
{  
  RubricaDataViewPrivate* priv;
  GtkTextBuffer* body;
  gchar* text = NULL;
 
  priv = RUBRICA_DATA_VIEW_GET_PRIVATE(view);
  body = priv->buf2;

  g_object_get(card, "company-notes", &text, NULL);
  if (text)
    {
      GdkPixbuf* pixbuf = NULL;  
   
      pixbuf = gtk_icon_theme_load_icon (priv->theme, "notes", 20, 0, NULL);
      data_view_write(body, "\n");
      data_view_write_pixbuf(body, pixbuf, _("Notes"));
      data_view_write(body, "\n");
      gdk_pixbuf_unref(pixbuf);

      data_view_write(body, text);   
    }
}



static void
data_view_show_personal_card(RubricaDataView* view, RCard* card)
{
  RubricaDataViewPrivate* priv;

  priv = RUBRICA_DATA_VIEW_GET_PRIVATE(view);

  /* Clean header and body buffers */
  data_view_clear_header(view); 
  data_view_clear_body(view); 

  write_personal_data (view, R_PERSONAL_CARD(card));
  write_groups        (view, card);
  write_addresses     (view, card);
  write_work          (view, R_PERSONAL_CARD(card));
  write_net           (view, card);
  write_telephones    (view, card);
  write_notes         (view, R_PERSONAL_CARD(card));
}



static void
data_view_show_company_card(RubricaDataView* view, RCard* card)
{
  RubricaDataViewPrivate* priv;

  priv = RUBRICA_DATA_VIEW_GET_PRIVATE(view);

  data_view_clear_header(view); 
  data_view_clear_body(view); 

  write_company_info  (view, card);
  write_groups        (view, card);
  write_addresses     (view, card);
  write_net           (view, card);
  write_telephones    (view, card);
  write_company_notes (view, R_COMPANY_CARD(card));  
}






/*   Public
*/
/**
 * rubrica_data_view_new
 *
 * create a #RubricaDataView
 *
 * returns: a new #RubricaDataView*
 */
GtkWidget*
rubrica_data_view_new()
{
  GtkWidget* view;

  view = GTK_WIDGET(g_object_new(RUBRICA_DATA_VIEW_TYPE, NULL));
  
  return view;
}


void 
rubrica_data_view_show_card (RubricaDataView* view, RCard* card)
{
  g_return_if_fail(IS_RUBRICA_DATA_VIEW(view));
  g_return_if_fail(IS_R_CARD(card));
  
  if (r_card_is_personal(card))
    data_view_show_personal_card(view, card);
  else
    data_view_show_company_card(view, card);    
}
