/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "PeriodicActionThread.h"
#include <stdlib.h>
#include "Lock.h"

#ifdef _LINUX
#define __max(a,b)	(((a)>(b))?(a):(b))
#endif

//How long we wait for the thread to terminate when stop() is called.
const unsigned long PeriodicActionThread::WAIT_STOP_TIMEOUT = 100 * SECOND;
const unsigned long PeriodicActionThread::UNDEFINED_INTERVAL = 0;

//ctor. nResumeDelay is the delay after calling resume() and before doing the next action.
PeriodicActionThread::PeriodicActionThread(const char *pName,
					   unsigned long nActionInterval, /*= UNDEFINED_INTERVAL*/
					   unsigned long nResumeDelay /*= 0*/)
{
  m_pName = (pName != NULL) ? pName : "Action";
  m_nActionInterval = nActionInterval;
  m_nResumeDelay = nResumeDelay;
  m_bRunning = false;
  m_CurrentState = STATE_RUNNING;
}

//dtor
PeriodicActionThread::~PeriodicActionThread()
{
  m_bRunning = false; //Cause thread to stop
}

//Stop the periodic action thread
bool PeriodicActionThread::stop()
{
  if (GetState() == STATE_STOPPING)
    return false; //Already stopping
  ChangeState(STATE_STOPPING);
  return wait(WAIT_STOP_TIMEOUT);
}

//Suspend the thread and wait until resume() is called.
void PeriodicActionThread::suspend()
{
  ChangeState(STATE_SUSPENDED);
}

//Resume the suspended thread after suspend() was called.
void PeriodicActionThread::resume()
{
  ChangeState(STATE_RESUMING);
}

//Stop waiting for the next periodic action, and do the action now.
void PeriodicActionThread::actNow()
{
  /*All we need to do is signal the state machine the state has changed (even though
    it didn't really change) and it will stop waiting and do the action. After this,
    it will go back to doing the action periodically as always.*/

  ChangeState(STATE_RUNNING);
}

//The actual function that runs the thread.
void PeriodicActionThread::run()
{
  State state = GetState();
  unsigned long nDelay = 0;

  m_bRunning = true;
  do
    {
      switch (state)
	{
	case STATE_RUNNING:
	  {
	    long nTimeBeforeAction = elapsedTime();
	    action();
	    long nActionElapsedTime = elapsedTime() - nTimeBeforeAction;

	    //nActionElapsedTime milliseconds have passed already. We deduct
	    //this time from our delay. If too much time passed, we don't
	    //wait at all.
	    nDelay = __max((signed long)(m_nActionInterval - nActionElapsedTime), 0);
	    break;
	  }
	case STATE_SUSPENDED:
	  nDelay = WAIT_INFINITE;
	  break;
	case STATE_RESUMING:
	  nDelay = m_nResumeDelay;
	  //Next time we wait, the timeout will bring us into the running
	  //state - we return to normally performing the periodic action.
	  state = STATE_RUNNING;
	  break;
	case STATE_STOPPING:
	  cleanup();
	  m_bRunning = false;
	  break;
	}

      if (m_bRunning && m_ChangeStateEvent.wait(nDelay))
	{
	  //Event was set - this means the state has changed.
	  state = GetState();
	}
    } while (m_bRunning);

}

//Stop the periodic action thread from within itself. Can be used by derived classes.
void PeriodicActionThread::internalStop()
{
  m_bRunning = false;
}

//Change the internal state of the periodic action thread's state machine
void PeriodicActionThread::ChangeState(State newState)
{
  Lock lock(m_StateSemaphore);
  m_CurrentState = newState;
  m_ChangeStateEvent.set();
}

//Get the internal state of the periodic action thread's state machine
PeriodicActionThread::State PeriodicActionThread::GetState()
{
  Lock lock(m_StateSemaphore);
  return m_CurrentState;
}
