/************************************************************************/
/*                                                                      */
/*    vspline - a set of generic tools for creation and evaluation      */
/*              of uniform b-splines                                    */
/*                                                                      */
/*            Copyright 2015 - 2017 by Kay F. Jahnke                    */
/*                                                                      */
/*    The git repository for this software is at                        */
/*                                                                      */
/*    https://bitbucket.org/kfj/vspline                                 */
/*                                                                      */
/*    Please direct questions, bug reports, and contributions to        */
/*                                                                      */
/*    kfjahnke+vspline@gmail.com                                        */
/*                                                                      */
/*    Permission is hereby granted, free of charge, to any person       */
/*    obtaining a copy of this software and associated documentation    */
/*    files (the "Software"), to deal in the Software without           */
/*    restriction, including without limitation the rights to use,      */
/*    copy, modify, merge, publish, distribute, sublicense, and/or      */
/*    sell copies of the Software, and to permit persons to whom the    */
/*    Software is furnished to do so, subject to the following          */
/*    conditions:                                                       */
/*                                                                      */
/*    The above copyright notice and this permission notice shall be    */
/*    included in all copies or substantial portions of the             */
/*    Software.                                                         */
/*                                                                      */
/*    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND    */
/*    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES   */
/*    OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND          */
/*    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT       */
/*    HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,      */
/*    WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING      */
/*    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR     */
/*    OTHER DEALINGS IN THE SOFTWARE.                                   */
/*                                                                      */
/************************************************************************/

/// \file thread_pool.h
///
/// \brief provides a thread pool for vspline's multithread() routine
///
/// class thread_pool aims to provide a simple and straightforward implementation
/// of a thread pool for multithread() in multithread.h, but the class might find
/// use elsewhere. The operation is simple, I think of it as 'piranha mode' ;)
///
/// a set of worker threads is launched which wait for 'tasks', which come in the shape
/// of std::function<void()>, from a queue. When woken, a worker thread tries to obtain
/// a task. If it succeeds, the task is executed, and the worker thread tries to get
/// another task. If none is to be had, it goes to sleep, waiting to be woken once
/// there are new tasks.

#include <thread>
#include <mutex>
#include <queue>
#include <condition_variable>
#include <iostream>

namespace vspline
{
  
class thread_pool
{
  // used to switch off the worker threads at program termination.
  // access under task_mutex.

  bool stay_alive = true ;

  // the thread pool itself is held in this variable. The pool
  // does not change after construction

  std::vector < std::thread * > pool ;
  
public:

  // mutex and condition variable for interaction with the task queue
  // and stay_alive

  std::mutex task_mutex ;
  std::condition_variable task_cv ;
  
  // queue to hold tasks. access under task_mutex

  std::queue < std::function < void() > > task_queue ;

private:
  
  /// code to run a worker thread
  /// We use a thread pool of worker threads. These threads have a very 
  /// simple cycle: They try and obtain a task (std::function<void()>). 
  /// If there is one to be had, it is invoked, otherwise they wait on
  /// task_cv. When woken up, the flag stay_alive is checked, and if it
  /// is found to be false, the worker thread ends.
  
  void worker_thread()
  {
    while ( true )
    {
      // under task_mutex, check stay_alive and try to obtain a task
      std::unique_lock<std::mutex> task_lock ( task_mutex ) ;

      if ( ! stay_alive )
      {
        task_lock.unlock() ;
        break ; // die
      }

      if ( task_queue.size() )
      {
        // there are tasks in the queue, take one
        auto task = task_queue.front() ;
        task_queue.pop() ;
        task_lock.unlock() ;
        // got a task, perform it, then try for another one
        task() ;
      }
      else
      {
        // no luck. wait.
        task_cv.wait ( task_lock ) ; // simply wait, spurious alert is okay
      }
      // start next cycle, either after having completed a job
      // or after having been woken by an alert
    }
  }

public:
  
  thread_pool ( int nthreads = 4 * std::thread::hardware_concurrency() )
  {
    // to launch a thread with a method, we need to bind it to the object:
    std::function < void() > wf = std::bind ( &thread_pool::worker_thread , this ) ;
    
    // now we can fill the pool with worker threads
    for ( int t = 0 ; t < nthreads ; t++ )
      pool.push_back ( new std::thread ( wf ) ) ;
  }

  int get_nthreads() const
  {
    return pool.size() ;
  }

  ~thread_pool()
  {
    {
      // under task_mutex, set stay_alive to false
      
      std::lock_guard<std::mutex> task_lock ( task_mutex ) ;
      stay_alive = false ;      
    }

    // wake all inactive worker threads,
    // join all worker threads once they are finished

    task_cv.notify_all() ;
    
    for ( auto threadp : pool )
    {
      threadp->join() ;
    }
    
    // once all are joined, delete their std::thread object

    for ( auto threadp : pool )
    {
      delete threadp ;
    }
  }
} ;

} ; // end of namespace vspline

